/*-
 * customsInt.h --
 *	Definitions internal to the customs daemon.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 *
 *	"$Id: customsInt.h,v 1.35 2001/02/17 01:09:58 stolcke Exp $ ICSI (Berkeley)"
 */
#ifndef _CUSTOMSINT_H_
#define _CUSTOMSINT_H_

#include    <sys/time.h>

#include    "sprite.h"
#include    "customs.h"
#include    "log.h"
#include    "xlog.h"

/*
 * Extended Avail structure including a timestamp for clock synchrony
 * check.  This helps detect machines with clocks that a badly off and
 * should therefore be avoided if DOUBLECHECK_TIMEOUT is in effect.
 * XXX: The first items in this structure must agree with Avail to
 * maintain backwards.
 * Also added: job count, to allow MCA to handle EXPORT_EXCLUSIVE.
 */
typedef struct {
    struct in_addr	addr;  	    /* The address of the host */
    Time_Interval	interval;   /* If the master doesn't get another
				     * avail packet after this interval,
				     * the host will be considered down */
    Rpc_Long 	  	avail;	    /* 0 if available. One of the AVAIL_*
				     * constants if not */
    Rpc_Long    	rating;	    /* Availability index (high => better) */
    Rpc_Long    	clock;	    /* Client time stamp */
    Rpc_Long		nJobs;	    /* Number of jobs running */
} Avail;

typedef struct {
    Rpc_Long 	  	avail;	    /* 0 if available. One of the AVAIL_*
				     * constants if not */
    Rpc_Long    	rating;	    /* Availability index (high => better) */
} AllocReply;

/*
 * Data sent with CUSTOMS_ALLOC
 */
typedef struct {
    ExportPermit	permit;	    /* export permit to be issued to client */
    Rpc_ULong		flags;	    /* EXPORT_* flags from client */
} AllocData;


#define MAX_REG_SIZE	    1024

#define CUSTOMSINT_RETRY    2
#define CUSTOMSINT_URETRY   500000
#define CUSTOMSINT_NRETRY   3

/*
 * Access check
 */
#ifndef USE_RESERVED_PORTS
#define CustomsReserved(proc, addr, msg)	/* no checking */
#else
#define CustomsReserved(proc, addr, msg) \
	if (ntohs((addr)->sin_port) >= IPPORT_RESERVED) {		\
	    if (verbose) {						\
		xlog (XLOG_WARNING, "Access to %s from %d@%s denied",	\
			proc, ntohs((addr)->sin_port),			\
			InetNtoA((addr)->sin_addr));			\
	    }								\
	    Log_Send(LOG_ACCESS, 1, xdr_sockaddr_in, addr);		\
	    Rpc_Error(msg, RPC_ACCESS);					\
	    return;							\
	}
#endif /* USE_RESERVED_PORTS */

#define CustomsCheckToken(msg, len, data) \
    if ((len) != 0 && (len) != sizeof(elect_Token)) {			\
	Rpc_Error(msg, RPC_BADARGS);					\
	return;								\
    } else if ((len) != 0 && *(Rpc_Long *)(data) != elect_Token) {	\
	/*								\
	 * Not our type -- ignore					\
	 */								\
	return;								\
    }

/*
 * EXTERNAL DECLARATIONS
 */

#define Local(sinPtr) Rpc_IsLocal((sinPtr))

/*
 * customs.c:
 */
extern char 	  	    localhost[];    /* Name of this machine */
extern struct sockaddr_in   localAddr;	    /* Real internet address of
					     * udp socket (i.e. not 127.1) */
extern struct timeval	    retryTimeOut;   /* Timeout for each try */
extern Boolean	  	    amMaster;	    /* TRUE if acting as the MCA */
extern Boolean	  	    canBeMaster;    /* TRUE if we are allowed to be
					     * the MCA */
extern struct sockaddr_in   serverAddr;     /* Address of server host */
extern Boolean	  	    verbose;	    /* TRUE if should print lots of
					     * messages */
extern int  	  	    udpSocket;	    /* Socket we use for udp rpc
					     * calls and replies */
extern short	    	    udpPort;	    /* Local customs UDP service port*/
extern int  	  	    tcpSocket;	    /* Service socket for handing tcp
					     * rpc calls. */
extern short	    	    tcpPort;	    /* Local TCP service port */
extern char 	  	    *regPacket;	    /* Our registration packet */
extern int  	  	    regPacketLen;   /* The length of it */
extern Lst 	  	    clients;	    /* Names of clients we support */
extern Lst 	  	    attributes;	    /* List of attribute strings */
extern Rpc_ULong    	    arch;   	    /* Architecture code */

void			    Customs_SetProcTitle();
					    /* Set process title for ps(1) */

/*
 * avail.c:
 */
extern int		    maxImports;	    /* Maximum number of imports */
extern int		    cpuLimit;	    /* CPU time limit for imports */
extern int		    memLimit;	    /* Memory use limit for imports */
extern int		    niceLevel;	    /* Minimum nice level for imports*/
extern int		    npriLevel;	    /* Non-deg. priority for imports */
extern int		    checkUser;	    /* User access checking level */
extern int		    localJobs;	    /* Local jobs are allowed */
extern int		    maxExclusives;  /* Maximum number of 'exclusives' */

extern double		    avail_LoadBias; /* Bias for load correction */
extern Boolean		    avail_Exclusive;/* No of 'exclusive' jobs running */

void	    	  	    Avail_Init();   /* Initialize availability module*/
Boolean	    	  	    Avail_Send();   /* Send an availability packet to
					     * the master */
int	    	  	    Avail_Local();  /* Check local availability */
extern int  	    	    avail_Bias;	    /* Bias for availability "rating"*/

/*
 * import.c:
 */
extern Job_Stats	    stats;	    /* Accumulated job statistics */
void	    	  	    Import_Init();  /* Initialize importation */
int	    	  	    Import_Kill();  /* Signal imported jobs */
int	    	  	    Import_NJobs(); /* Return the number of active */
					    /* imported jobs */

/*
 * mca.c:
 */
void	    	  	    MCA_Init();	    /* Set up to act as master */
void	    	  	    MCA_Cancel();   /* Cancel mastery */
void	    	  	    MCA_HostInt();  /* Process an internal Host req */

/*
 * election.c
 */
void	    	  	    Elect_Init();   	/* Initialization */
void	    	  	    Elect_GetMaster();	/* Find MCA */
Boolean	    	  	    Elect_InProgress();	/* See if an election is going
						 * on. */
extern struct sockaddr_in   masterAddr;     /* Address of master's socket */
extern Rpc_Long		    elect_Token;    /* Token to pass during
					     * elections */

/*
 * log.c
 */
void	    	  	    Log_Init();
void	    	  	    Log_Send _ANSI_ARGS_((Rpc_Proc, int, ...));

/*
 * swap.c
 */
extern void 	  	    Swap_Timeval();  	/* struct tv */
extern void		    Swap_Avail();    	/* Avail_Data */
extern void		    Swap_AllocReply(); 	/* Alloc_Reply */
extern void		    Swap_AvailInt(); 	/* AvailClock */
extern void		    Swap_Host();       	/* Host_Data */
extern void		    Swap_ExportPermit();/* ExportPermit */
extern void		    Swap_AllocData();   /* AllocData */
extern void		    Swap_WayBill();  	/* WayBill */
extern void		    Swap_Kill();	/* Kill_Data */
extern void		    Swap_RegPacket();	/* reg packet (free-form) */
extern void		    Swap_Info();    	/* Info packet (free-form) */
extern void		    Swap_Version();    	/* Version packet */
extern void		    Swap_Jobs();    	/* Job list */
extern void		    Swap_Stats();    	/* Job stats */
extern void		    Swap_SockAddr();    /* Internet socket address */

/*
 * os-*.c
 */
extern int		    OS_Init();		/* Initialize OS module */
extern void		    OS_Exit();		/* Deinitialize */
extern int		    OS_Idle();		/* Idle time */
extern unsigned int	    OS_Load();		/* Load value */
extern int		    OS_Swap();		/* Free swap space */
extern int		    OS_Proc();		/* Free process slots */

#endif /* _CUSTOMSINT_H_ */
