//
// "$Id: cupsps.cpp,v 1.2 2005/06/17 20:54:21 mike Exp $"
//
//   PostScript class for the CUPS printer driver for Windows.
//
//   Copyright 2001-2005 by Easy Software Products.
//
//   These coded instructions, statements, and computer programs are the
//   property of Easy Software Products and are protected by Federal
//   copyright law.  Distribution and use rights are outlined in the file
//   "LICENSE.txt" which should have been included with this file.  If this
//   file is missing or damaged please contact Easy Software Products
//   at:
//
//       Attn: CUPS Licensing Information
//       Easy Software Products
//       44141 Airport View Drive, Suite 204
//       Hollywood, Maryland 20636 USA
//
//       Voice: (301) 373-9600
//       EMail: cups-info@cups.org
//         WWW: http://www.cups.org/
//
// Contents:
//
//   cupsPS::cupsPS()         - Create a new instance of the PostScript class.
//   cupsPS::~cupsPS()        - Destroy an instance of the PostScript class.
//   cupsPS::AddRef()         - Increment the reference count of the class.
//   cupsPS::Command()        - Insert PostScript commands into the output.
//   cupsPS::DevMode()        - Return device information.
//   cupsPS::DisableDriver()  - Disable the driver.
//   cupsPS::DisablePDEV()    - Disable the printer.
//   cupsPS::EnableDriver()   - Enable the driver.
//   cupsPS::EnablePDEV()     - Enable the printer.
//   cupsPS::GetInfo()        - Get driver information.
//   cupsPS::PublishDriverInterface()
//                            - Get the primary driver interface.
//   cupsPS::QueryInterface() - Return a pointer to the interface class.
//   cupsPS::Release()        - Decrement the reference count and possibly delete the class.
//   cupsPS::ResetPDEV()      - Reset the printer driver.
//   DllInitialize()          - Initialize the DLL (not used).
//   cups_utf16to8()          - Convert a UTF-16 string to UTF-8.
//

//
// Include necessary headers...
//

#define CUPS_CLASS_PREFIX L"[cupsPS]"
#define CUPS_CLASS_NAME	cupsPS
#define CUPS_CLASS_ID	CLSID_OEMRENDER

#include "../common/driver.h"


class cupsPS : public IPrintOemPS	//// CUPS PostScript module class
{
  protected:

  long			ref;		// Reference count
  IPrintOemDriverPS	*help;		// Pointer to primary driver

  public:

  STDMETHOD(QueryInterface)(THIS_ REFIID riid, LPVOID FAR *obj);
  STDMETHOD_(ULONG,AddRef)(THIS);
  STDMETHOD_(ULONG,Release)(THIS);

  STDMETHOD(PublishDriverInterface)(THIS_ IUnknown *pIUnknown);
  STDMETHOD(EnableDriver)(THIS_ DWORD driver_version, DWORD size,
                          PDRVENABLEDATA pded);
  STDMETHOD(DisableDriver)(THIS);
  STDMETHOD(EnablePDEV)(THIS_ PDEVOBJ obj, PWSTR printer_name,
                        ULONG num_patterns, HSURF *patterns,
                        ULONG num_gdi_info, GDIINFO *gdi_info,
                        ULONG num_dev_info, DEVINFO *dev_info,
                        DRVENABLEDATA *pded, OUT PDEVOEM *dm);
  STDMETHOD(DisablePDEV)(THIS_ PDEVOBJ dm);
  STDMETHOD(ResetPDEV)(THIS_ PDEVOBJ olddm, PDEVOBJ newdm);
  STDMETHOD(GetInfo)(THIS_ DWORD mode, PVOID buffer, DWORD size,
                     PDWORD needed);
  STDMETHOD(DevMode)(THIS_ DWORD mode, POEMDMPARAM param);
  STDMETHOD(Command)(THIS_ PDEVOBJ dm, DWORD index, PVOID data,
                     DWORD size, OUT DWORD *result);

  cupsPS();
  ~cupsPS();
};


#include "../common/driver.cpp"


//
// Local functions...
//

static char *cups_utf16to8(const WCHAR *utf16, char *utf8, int utf8size);


//
// 'cupsPS::cupsPS()' - Create a new instance of the PostScript class.
//

cupsPS::cupsPS()
{
  cupsDebug(0, L"cupsPS::cupsPS()\n");

  // Bump component count...
  InterlockedIncrement(&num_components);

  // Initialize class data...
  ref  = 1;
  help = NULL; 
}


//
// 'cupsPS::~cupsPS()' - Destroy an instance of the PostScript class.
//

cupsPS::~cupsPS()
{
  cupsDebug(0, L"cupsPS::~cupsPS()\n");

  // Release helper interface...
  if (help)
  {
    help->Release();
    help = NULL;
  }

  // Bump component count...
  InterlockedDecrement(&num_components);
}


//
// 'cupsPS::AddRef()' - Increment the reference count of the class.
//

ULONG __stdcall				// O - New reference count
cupsPS::AddRef()
{
  cupsDebug(0, L"cupsPS::AddRef()\n");

  return (InterlockedIncrement(&ref));
}


//
// 'cupsPS::Command()' - Insert PostScript commands into the output.
//

HRESULT __stdcall			// O - S_OK on success, E_FAIL on failure
cupsPS::Command(PDEVOBJ   pdevobj,	// I - Printer device
		DWORD     dwIndex,	// I - Section index
		PVOID     pData,	// I - Not used
		DWORD     cbSize,	// I - Not used
		OUT DWORD *pdwResult)	// O - Result of command
{
  char		buffer[1024],		// Buffer for job ticket data
		utf8[512];		// Quoted UTF-8 value
  DWORD		bytes;			// Bytes written
  cups_dev_t	*dm;			// DEVMODE data


  cupsDebug(4, L"cupsPS::Command(pdevobj=%p, dwIndex=%d, pData=%p, cbSize=%d, pdwResult=%p)\n",
            pdevobj, dwIndex, pData, cbSize, pdwResult);

  // See where we are...
  if (dwIndex == PSINJECT_PSADOBE)
  {
    cupsDebug(0, L"PSINJECT_PSADOBE");

    // Add the additional job-billing and page-label options here...
    dm = (cups_dev_t *)(pdevobj->pOEMDM);

    if (dm->page_label[0])
    {
      // Output the page-label job ticket...
      sprintf(buffer, "%%cupsJobTicket: page-label=\"%s\"\n",
              cups_utf16to8(dm->page_label, utf8, sizeof(utf8)));
      help->DrvWriteSpoolBuf(pdevobj, buffer, strlen(buffer), &bytes);

      cupsDebug(0, L"Wrote: %%cupsJobTicket: page-label=\"...\"\n");
    }

    if (dm->job_billing[0])
    {
      // Output the job-billing job ticket...
      sprintf(buffer, "%%cupsJobTicket: job-billing=\"%s\"\n",
              cups_utf16to8(dm->job_billing, utf8, sizeof(utf8)));
      help->DrvWriteSpoolBuf(pdevobj, buffer, strlen(buffer), &bytes);

      cupsDebug(0, L"Wrote: %%cupsJobTicket: job-billing=\"\"\n");
    }

    // Return with no errors...
    *pdwResult = ERROR_SUCCESS;
  }
  else
  {
    // Indicate that we don't implement this particular part...
    *pdwResult = ERROR_NOT_SUPPORTED;
  }

  cupsDebug(-4, L"returning S_OK...\n");

  return (S_OK);
}


//
// 'cupsPS::DevMode()' - Return device information.
//

HRESULT __stdcall			// O - S_OK on success, E_FAIL on error
cupsPS::DevMode(DWORD       dwMode,	// I - Information requested
                POEMDMPARAM pOemDMParam)// O - Data
{
  cupsDebug(0, L"cupsPS::DevMode(dwMode=%d, pOemDMParam=%p)\n", dwMode,
            pOemDMParam);

  return (cupsDevMode(dwMode, pOemDMParam));
}


//
// 'cupsPS::DisableDriver()' - Disable the driver.
//

HRESULT __stdcall			// O - E_NOTIMPL
cupsPS::DisableDriver(THIS)
{
  cupsDebug(0, L"cupsPS::DisableDriver()\n");

  return (E_NOTIMPL);
}


//
// 'cupsPS::DisablePDEV()' - Disable the printer.
//

HRESULT __stdcall			// O - E_NOTIMPL
cupsPS::DisablePDEV(PDEVOBJ pdevobj)	// I - Printer device
{
  cupsDebug(0, L"cupsPS::DisablePDEV()\n");

  return (E_NOTIMPL);
}


//
// 'cupsPS::EnableDriver()' - Enable the driver.
//

HRESULT __stdcall			// O - E_NOTIMPL
cupsPS::EnableDriver(
    DWORD          dwDriverVersion,	// I - Driver version
    DWORD          cbSize,		// I - Size of buffer
    PDRVENABLEDATA pded)		// I - Driver enable data
{
  cupsDebug(0, L"cupsPS::EnableDriver()\n");

  return (E_NOTIMPL);
}


//
// 'cupsPS::EnablePDEV()' - Enable the printer.
//

HRESULT __stdcall			// O - E_NOTIMPL
cupsPS::EnablePDEV(
    PDEVOBJ       pdevobj,		// I - Printer device
    PWSTR         pPrinterName,		// I - Printer name
    ULONG         cPatterns,		// I - Number of patterns
    HSURF         *phsurfPatterns,	// I - Patterns
    ULONG         cjGdiInfo,		// I - Number of GDI contexts
    GDIINFO       *pGdiInfo,		// I - GDI contexts
    ULONG         cjDevInfo,		// I - Number of device infos
    DEVINFO       *pDevInfo,		// I - Device infos
    DRVENABLEDATA *pded,		// I - Driver enable data
    OUT PDEVOEM   *pDevOem)		// O - Printer driver data
{
  cupsDebug(0, L"cupsPS::EnablePDEV()\n");

  return (E_NOTIMPL);
}


//
// 'cupsPS::GetInfo()' - Get driver information.
//

HRESULT __stdcall			// O - S_OK on success, E_FAIL on error
cupsPS::GetInfo(DWORD  dwMode,		// I - Information requested
		PVOID  pBuffer,		// I - Output buffer
		DWORD  cbSize,		// I - Size of output buffer
		PDWORD pcbNeeded)	// O - Actual size required
{
  PUBLISHERINFO *pi;			// Publisher information


  cupsDebug(0, L"cupsPS::GetInfo(dwMode=%d, pBuffer=%p, cbSize=%d, pcbNeeded=%p)\n",
            dwMode, pBuffer, cbSize, pcbNeeded);

  // Check arguments...
  if (!pcbNeeded || 
      (dwMode != OEMGI_GETSIGNATURE && dwMode != OEMGI_GETVERSION &&
       dwMode != OEMGI_GETPUBLISHERINFO))
  {
    SetLastError(ERROR_INVALID_PARAMETER);
    return (E_FAIL);
  }

  // Set actual size required...
  if (dwMode == OEMGI_GETPUBLISHERINFO)
    *pcbNeeded = sizeof(PUBLISHERINFO);
  else
    *pcbNeeded = sizeof(DWORD);

  // See if we have room...
  if (cbSize < *pcbNeeded || !pBuffer)
  {
    // Not enough space, or we have no buffer to work with...
    SetLastError(ERROR_INSUFFICIENT_BUFFER);
    return (E_FAIL);
  }

  // Copy the requested information into the buffer...
  switch (dwMode)
  {
    case OEMGI_GETSIGNATURE : // OEM DLL Signature
        *(PDWORD)pBuffer = CUPS_SIGNATURE;
        break;

    case OEMGI_GETVERSION : // OEM DLL version
        *(PDWORD)pBuffer = CUPS_WINVERSION;
        break;

    case OEMGI_GETPUBLISHERINFO : // Publisher information
        pi                  = (PUBLISHERINFO *)pBuffer;
	pi->dwMode          = OEM_MODE_PUBLISHER;
	pi->wMinoutlinePPEM = 10;
	pi->wMaxbitmapPPEM  = 20;
        break;
  }

  // Return with no errors...
  return (S_OK);
}


//
// 'cupsPS::PublishDriverInterface()' - Get the primary driver interface.
//

HRESULT __stdcall			// O - S_OK on success, E_FAIL on error
cupsPS::PublishDriverInterface(
    IUnknown *pIUnknown)		// I - Driver information
{
  cupsDebug(0, L"cupsPS::PublishDriverInterface(pIUnknown=%p)\n", pIUnknown);

  // Initialize the primary driver interface pointer as needed...
  if (!help)
  {
    // Get help functions...
    if (pIUnknown->QueryInterface(IID_IPrintOemDriverPS,
                                  (void **)&help) != S_OK)
    {
      // No helper functions, return an error...
      help = NULL;

      return (E_FAIL);
    }
  }

  // Everything worked...
  return (S_OK);
}


//
// 'cupsPS::QueryInterface()' - Return a pointer to the interface class.
//

HRESULT __stdcall			// O - S_OK on success, E_NOINTERFACE on error
cupsPS::QueryInterface(const IID &iid,	// I - Interface ID
                       void      **data)// O - Pointer to interface class
{    
  cupsDebug(4, L"cupsPS::QueryInterface(iid=%p, data=%p)\n", iid, data);

  if (iid == IID_IUnknown)
  {
    AddRef();

    *data = static_cast<IUnknown*>(this); 

    cupsDebug(-4, L"returning IUnknown + S_OK...\n");

    return (S_OK);
  }
  else if (iid == IID_IPrintOemPS)
  {
    AddRef();

    *data = static_cast<IPrintOemPS*>(this); 

    cupsDebug(-4, L"returning IPrintOemPS + S_OK...\n");

    return (S_OK);
  }
  else
  {
    *data = NULL;

    cupsDebug(-4, L"returning E_NOINTERFACE...\n");

    return (E_NOINTERFACE);
  }
}


//
// 'cupsPS::Release()' - Decrement the reference count and possibly delete the class.
//

ULONG __stdcall				// O - New reference count
cupsPS::Release() 
{
  cupsDebug(0, L"cupsPS::Release()\n");

  if (!InterlockedDecrement(&ref))
  {
    delete this;
    return (0);
  }
  else
    return (ref);
}


//
// 'cupsPS::ResetPDEV()' - Reset the printer driver.
//

HRESULT __stdcall			// O - E_NOTIMPL
cupsPS::ResetPDEV(PDEVOBJ pdevobjOld,	// I - Old device
                  PDEVOBJ pdevobjNew)	// I - New device
{
  cupsDebug(0, L"cupsPS::ResetPDEV()\n");

  return (E_NOTIMPL);
}


//
// 'DllInitialize()' - Initialize the DLL (not used).
//

BOOL WINAPI				// O - TRUE on success
DllInitialize(ULONG reason)		// I - Reason for call (not used)
{
  cupsDebug(0, L"DllInitialize(reason=%lu)\n", reason);

  return (TRUE);
}


//
// 'cups_utf16to8()' - Convert a UTF-16 string to UTF-8.
//

static char *				// O - UTF-8 string
cups_utf16to8(const WCHAR *utf16,	// I - UTF-16 string
              char        *utf8,	// I - UTF-8 string buffer
	      int         utf8size)	// I - Size of UTF-8 string buffer
{
  char		*utf8ptr;		// Pointer into UTF-8 value
  unsigned	ch;			// Current character


  cupsDebug(0, L"cups_utf16to8(utf16=\"%s\", utf8=%p, utf8size=%d)\n",
            utf16, utf8, utf8size);

  for (utf8ptr = utf8, utf8size --; utf8size > 0 && *utf16;)
  {
    ch = *utf16++;

    if (ch >= 0xd800 && ch <= 0xdbff && *utf16)
    {
      // Multi-word UTF-16 char...
      ch = (((ch & 0x3ff) << 10) | (*utf16++ & 0x3ff)) + 0x10000;
    }

    if (ch == '\'' || ch == '\"' || ch == '\\')
    {
      // Quoted special character...
      if (utf8size < 2)
        break;

      utf8size -= 2;

      *utf8ptr++ = '\\';
      *utf8ptr++ = ch;
    }
    else if (ch < 0x80)
    {
      // Normal ASCII character...
      utf8size --;

      *utf8ptr++ = ch;
    }
    else if (ch < 0x800)
    {
      // Two-byte UTF-8 character...
      if (utf8size < 2)
        break;

      utf8size -= 2;

      *utf8ptr++ = 0xc0 | (ch >> 6);
      *utf8ptr++ = 0x80 | (ch & 0x3f);
    }
    else if (ch < 0x10000)
    {
      // Three-byte UTF-8 character...
      if (utf8size < 3)
        break;

      utf8size -= 3;

      *utf8ptr++ = 0xe0 | (ch >> 12);
      *utf8ptr++ = 0x80 | ((ch >> 6) & 0x3f);
      *utf8ptr++ = 0x80 | (ch & 0x3f);
    }
    else
    {
      // Four-byte UTF-8 character...
      if (utf8size < 4)
        break;

      utf8size -= 4;

      *utf8ptr++ = 0xf0 | (ch >> 18);
      *utf8ptr++ = 0x80 | ((ch >> 12) & 0x3f);
      *utf8ptr++ = 0x80 | ((ch >> 6) & 0x3f);
      *utf8ptr++ = 0x80 | (ch & 0x3f);
    }
  }

  *utf8ptr = '\0';

  return (utf8);
}


//
// End of "$Id: cupsps.cpp,v 1.2 2005/06/17 20:54:21 mike Exp $".
//
