/*
    centralwidget.cpp  -  Central widget for the KBB main window

    copyright   : (c) 2001 by Martijn Klingens
    email       : mklingens@yahoo.com

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#include <qsplitter.h>
#include <qpushbutton.h>
#include <qwidgetstack.h>
#include <qlayout.h>

#include <kdialog.h>
#include <kdebug.h>
#include <kcombobox.h>
#include <klistview.h>
#include <klineeditdlg.h>

#include "kbbprefs.h"
#include "bugsystem.h"
#include "packagelvi.h"
#include "buglvi.h"
#include "msginputdialog.h"
#include "packageselectdialog.h"
#include "cwbugdetails.h"
#include "bugcommand.h"
#include "severityselectdialog.h"
#include "cwsearchwidget.h"
#include "cwbuglistcontainer.h"
#include "cwbugdetailscontainer.h"

#include "centralwidget.h"

using namespace KBugBusterMainWindow;

CentralWidget::CentralWidget( const QCString& initialPackage, const QCString& initialBug, QWidget *parent , const char * name )
: QWidget( parent, name )
{
    // Master layout
    ( new QVBoxLayout( this, 0,
                       KDialog::spacingHint() ) )->setAutoAdd( true );

    // Create QSplitter children
    m_vertSplitter = new QSplitter( QSplitter::Vertical, this );
    m_listPane     = new CWBugListContainer( m_vertSplitter );
    m_horSplitter  = new QSplitter( QSplitter::Horizontal,m_vertSplitter );
    m_searchPane   = new CWSearchWidget( m_horSplitter );
    m_bugPane      = new CWBugDetailsContainer( m_horSplitter );

    m_searchPane->hide();
//    m_listPane->hide();

    m_searchPane->setSizePolicy( QSizePolicy( QSizePolicy::Minimum,
                                                    QSizePolicy::Minimum ) );
    m_horSplitter->setResizeMode( m_searchPane, QSplitter::FollowSizeHint );

    connect( m_listPane, SIGNAL( resetProgressBar() ),
             SIGNAL( resetProgressBar() ) );
    connect( m_bugPane, SIGNAL( resetProgressBar() ),
             SIGNAL( resetProgressBar() ) );

    // Start the proper jobs for loading the package lists
    connect( BugSystem::self(),
             SIGNAL( packageListAvailable( const Package::List & ) ),
             SLOT( updatePackageList( const Package::List & ) ) );
    connect( BugSystem::self(),
             SIGNAL( bugListAvailable( const Package &, const Bug::List & ) ),
             SLOT( updateBugList( const Package &, const Bug::List & ) ) );
    connect( BugSystem::self(),
             SIGNAL( bugDetailsAvailable( const Bug &, const BugDetails & ) ),
             SLOT( updateBugDetails( const Bug &, const BugDetails & ) ) );

    connect( m_bugPane, SIGNAL( signalCloseBug() ), SLOT( closeBug() ) );
    connect( m_bugPane, SIGNAL( signalCloseBugSilently() ), SLOT( closeBugSilently() ) );
    connect( m_bugPane, SIGNAL( signalReopenBug() ), SLOT( reopenBug() ) );
    connect( m_bugPane, SIGNAL( signalReassignBug() ), SLOT( reassignBug() ) );
    connect( m_bugPane, SIGNAL( signalTitleBug() ), SLOT( titleBug() ) );
    connect( m_bugPane, SIGNAL( signalSeverityBug() ), SLOT( severityBug() ) );
    connect( m_bugPane, SIGNAL( signalReplyBug() ), SLOT( replyBug() ) );
    connect( m_bugPane, SIGNAL( signalReplyPrivateBug() ), SLOT( replyPrivateBug() ) );

    connect( m_bugPane, SIGNAL( signalClearCommand() ), SLOT( clearCommand() ) );

    // Add the selection slots for the listviews
    connect( m_searchPane->m_searchPackages,
             SIGNAL( activated( const QString & ) ),
             SLOT( slotRetrieveBugList( const QString & ) ) );

    connect( m_listPane, SIGNAL( executed( const Bug & ) ),
             SLOT( slotRetrieveBugDetails( const Bug & ) ) );
    connect( m_listPane, SIGNAL( currentChanged( const Bug & ) ),
             SLOT( slotSetActiveBug( const Bug & ) ) );

    connect( m_listPane, SIGNAL( searchPackage() ), SIGNAL( searchPackage() ) );
    connect( m_bugPane, SIGNAL( searchBugNumber() ), SIGNAL( searchBugNumber() ) );

    BugSystem::self()->retrievePackageList();
    QString currentPackageName = initialPackage.isEmpty() ? KBBPrefs::instance()->mCurrentPackage : initialPackage;
    if ( !currentPackageName.isEmpty() )
    {
        m_currentPackage = BugSystem::self()->package( currentPackageName );
        BugSystem::self()->retrieveBugList( m_currentPackage );

        if ( initialBug.isEmpty() )
        {
            QString currentBugNumber = KBBPrefs::instance()->mCurrentBug;
            if ( !currentBugNumber.isEmpty() ) {
                m_currentBug = BugSystem::self()->bug( m_currentPackage, currentBugNumber );
                BugSystem::self()->retrieveBugDetails( m_currentBug );
            }
        }
    }
    if ( !initialBug.isEmpty() )
    {
        // ### bad way to instanciating a bug object! doesn't restore details!
        m_currentBug = Bug::fromNumber(initialBug); // bug number specified on cmdline. Is it a problem that we don't have details ?
        BugSystem::self()->retrieveBugDetails( m_currentBug );
    }
}

CentralWidget::~CentralWidget()
{
//    kdDebug() << "CentralWidget::~CentralWidget()" << endl;
}

void CentralWidget::readConfig()
{
    m_horSplitter->setSizes( KBBPrefs::instance()->mSplitter2 );
    m_vertSplitter->setSizes( KBBPrefs::instance()->mSplitter1 );
}

void CentralWidget::writeConfig()
{
#if 0
    kdDebug() << "m_vertSplitter" << endl;
    QValueList<int> sizes = m_vertSplitter->sizes();
    QValueList<int>::ConstIterator it;
    for( it = sizes.begin(); it != sizes.end(); ++it ) {
        kdDebug() << "  " << (*it) << endl;
    }
#endif

    KBBPrefs::instance()->mSplitter1 = m_vertSplitter->sizes();
    KBBPrefs::instance()->mSplitter2 = m_horSplitter->sizes();

    KBBPrefs::instance()->mCurrentPackage = m_currentPackage.name();
    KBBPrefs::instance()->mCurrentBug = m_currentBug.number();
}

void CentralWidget::slotRetrieveBugList( const QString &package )
{
    if( package.isNull() || m_packageList[ package ].isNull() ||
        m_packageList[ package ] == m_currentPackage )
    {
        return; // Nothing to do
    }

//    m_bugPane->enableButtons( Bug() );
//    m_listPane->m_listMergeBtn->setEnabled( false );
//    m_listPane->m_listUnmergeBtn->setEnabled( false );

    m_currentPackage = m_packageList[ package ];

    BugSystem::self()->retrieveBugList( m_currentPackage );
}

void CentralWidget::slotRetrieveBugDetails( const Bug &bug )
{
    if( m_currentBug == bug )
        return; // Nothing to do

    m_currentBug = bug;
    BugSystem::self()->retrieveBugDetails( m_currentBug );
}

void CentralWidget::slotSetActiveBug( const Bug &bug )
{
    if( bug.isNull() )
    {
        return;
    }

    if( m_activeBug == bug )
        return; // Nothing to do

    m_activeBug = bug;
}

void CentralWidget::updatePackageList( const Package::List &pkgs )
{
    // ### needs proper implementation ;-)

    m_searchPane->m_searchPackages->clear();
    m_searchPane->m_searchPackages->completionObject()->clear();
//    m_bugPane->m_bugDetails->m_bugPackage->clear();
    emit resetProgressBar();

    Package::List::ConstIterator it = pkgs.begin();
    for ( ; it != pkgs.end(); ++it )
    {
        m_packageList[ ( *it ).name() ] = *it;
        m_searchPane->m_searchPackages->insertItem( ( *it ).name() );
        m_searchPane->m_searchPackages->
                  completionObject()->addItem( ( *it ).name() );
//        m_bugPane->m_bugDetails->m_bugPackage->insertItem( ( *it ).name() );
    }

/*
    if( m_bugPane->m_bugStack->id(
                m_bugPane->m_bugStack->visibleWidget() ) != 0 )
    {
        m_bugPane->m_bugDetails->m_bugPackage->setCurrentItem( -1 );
    }
*/
}

void CentralWidget::updateBugList( const Package &pkg, const Bug::List &bugs )
{
    m_listPane->setBugList( pkg, bugs );
}

void CentralWidget::updateBugDetails( const Bug &bug, const BugDetails &bd )
{
    m_bugPane->setBug( bug, bd );
}

void CentralWidget::slotReloadPackageList()
{
    BugSystem::self()->cache()->invalidatePackageList();
    BugSystem::self()->retrievePackageList();
}

void CentralWidget::slotReloadPackage()
{
    if (!m_currentPackage.isNull()) {
        BugSystem::self()->cache()->invalidateBugList( m_currentPackage );
        BugSystem::self()->retrieveBugList( m_currentPackage );
    }
}

void CentralWidget::slotReloadBug()
{
    if (!m_currentBug.isNull()) {
        BugSystem::self()->cache()->invalidateBugDetails( m_currentBug );
        BugSystem::self()->retrieveBugDetails( m_currentBug );
    }
}

void CentralWidget::mergeBugs()
{
    QStringList bugNumbers = m_listPane->selectedBugs();
    if ( bugNumbers.count() >= 2 ) {
      BugSystem::self()->queueCommand(
          new BugCommandMerge( bugNumbers, m_currentPackage ) );
    }
}

void CentralWidget::unmergeBugs()
{
    BugSystem::self()->queueCommand(
        new BugCommandUnmerge( m_currentBug, m_currentPackage ) );
}

void CentralWidget::closeBug()
{
    MsgInputDialog *dlg = new MsgInputDialog( MsgInputDialog::Close,
                                              m_currentBug, m_currentPackage,
					      this );
    dlg->show();
}

void CentralWidget::closeBugSilently()
{
    BugSystem::self()->queueCommand(
            new BugCommandCloseSilently( m_currentBug, m_currentPackage ) );
}

void CentralWidget::reopenBug()
{
    BugSystem::self()->queueCommand(
        new BugCommandReopen( m_currentBug, m_currentPackage ) );
}

void CentralWidget::reassignBug()
{
    Package package = PackageSelectDialog::getPackage( this );

    if ( package.isNull() ) {
        return;
    }

    BugSystem::self()->queueCommand(
        new BugCommandReassign( m_currentBug, package.name(), m_currentPackage ) );
}

void CentralWidget::titleBug()
{
    bool ok = false;
    QString title = KLineEditDlg::getText( i18n("Change Bug Title"),
                                           i18n( "Please enter new title" ),
                                           m_currentBug.title(), &ok, this );
    if ( ok && !title.isEmpty() ) {
        BugSystem::self()->queueCommand(
            new BugCommandRetitle( m_currentBug, title, m_currentPackage ) );
    }
}

void CentralWidget::severityBug()
{
    SeveritySelectDialog *dlg = new SeveritySelectDialog( this );
    dlg->setSeverity( m_currentBug.severity() );
    int result = dlg->exec();
    if ( result == QDialog::Accepted ) {
        BugSystem::self()->queueCommand(
            new BugCommandSeverity( m_currentBug,
                                    dlg->selectedSeverityAsString(), m_currentPackage ) );
    }
}

void CentralWidget::replyBug()
{
    MsgInputDialog *dlg = new MsgInputDialog( MsgInputDialog::Reply,
                                              m_currentBug, m_currentPackage,
					      this );
    dlg->show();
}

void CentralWidget::replyPrivateBug()
{
    MsgInputDialog *dlg = new MsgInputDialog( MsgInputDialog::ReplyPrivate,
                                              m_currentBug, m_currentPackage,
					      this );
    dlg->show();
}

void CentralWidget::clearCommand()
{
  BugSystem::self()->clearCommands( m_currentBug.number() );
}

void CentralWidget::searchBugByTitle( int options, const QString& pattern )
{
  m_listPane->searchBugByTitle( options, pattern );
}

#include "centralwidget.moc"

/* vim: set et ts=4 sw=4 softtabstop=4: */
