#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <scsi/scsi.h>
#include <scsi/sg.h>
#include <glib.h>
#include <signal.h>
#include <gtk/gtk.h>

#include "nautilus-cd-burner.h"

static char *last_error = NULL;

struct cdrecord_output {
	GMainLoop *loop;
	int result;
	int pid;
	int stdin;
	GString *line;
	GString *stderr;
	gboolean changed_text;
	gboolean send_return;
	gboolean expect_cdrecord_to_die;
};

static void
cancel_cdrecord (gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	kill (cdrecord_output->pid, SIGINT);

	cdrecord_output->result = RESULT_CANCEL;
	g_main_loop_quit (cdrecord_output->loop);
}

static void
reload_dialog_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	if (cdrecord_output->send_return) {
		write (cdrecord_output->stdin, "\n", 1);
	} else {
		kill (cdrecord_output->pid, SIGUSR1);
	}
}

static void
insert_cd_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	cdrecord_output->result = RESULT_RETRY;
	g_main_loop_quit (cdrecord_output->loop);
}


static gboolean  
cdrecord_stdout_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	struct cdrecord_output *cdrecord_output = data;
	char *line;
	char buf[1];
	unsigned int track, mb_written, mb_total;
	GIOStatus status;
	GtkWidget *reload_dialog;
	
	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	
	if (status == G_IO_STATUS_NORMAL) {
		if (cdrecord_output->line) {
			g_string_append (cdrecord_output->line, line);
			g_free (line);
			line = g_string_free (cdrecord_output->line, FALSE);
			cdrecord_output->line = NULL;
		}
		
		if (sscanf (line, "Track %2u: %d of %d MB written",
			    &track, &mb_written, &mb_total) == 3) {
			if (!cdrecord_output->changed_text) {
				cd_progress_set_text (_("Writing CD"));
			}

			cd_progress_set_fraction ((mb_total > 0) ? ((double)mb_written/mb_total) * 0.98 : 0.0);
		} else if (g_str_has_prefix (line, "Re-load disk and hit <CR>") ||
			   g_str_has_prefix (line, "send SIGUSR1 to continue")) {
			reload_dialog = gtk_message_dialog_new (cd_progress_get_window (),
								GTK_DIALOG_DESTROY_WITH_PARENT,
								GTK_MESSAGE_INFO,
								GTK_BUTTONS_OK,
								_("Please reload the CD in the CD writer."));
			gtk_window_set_title (GTK_WINDOW (reload_dialog),
					      _("Reload CD"));
			cdrecord_output->send_return = (*line == 'R');
			g_signal_connect (reload_dialog, "response", (GCallback)reload_dialog_response, cdrecord_output);
			gtk_widget_show (reload_dialog);
		} else if (g_str_has_prefix (line, "Fixating...")) {
			cd_progress_set_text (_("Fixating CD"));
		} else if (g_str_has_prefix (line, "Fixating time:")) {
			cd_progress_set_fraction (1.0);
			cdrecord_output->result = RESULT_FINISHED;
		} else if (g_str_has_prefix (line, "Last chance to quit, ")) {
			cd_progress_set_cancel_func (cancel_cdrecord, TRUE, &cdrecord_output);
		}

		g_free (line);
		
	} else if (status == G_IO_STATUS_AGAIN) {
		/* A non-terminated line was read, read the data into the buffer. */
		status = g_io_channel_read_chars (source, buf, 1, NULL, NULL);
		if (status == G_IO_STATUS_NORMAL) {
			if (cdrecord_output->line == NULL) {
				cdrecord_output->line = g_string_new (NULL);
			}
			g_string_append_c (cdrecord_output->line, buf[0]);
		}
	} else if (status == G_IO_STATUS_EOF) {
		return FALSE;
	}

	return TRUE;
}

static gboolean  
cdrecord_stderr_read (GIOChannel   *source,
		     GIOCondition  condition,
		     gpointer      data)
{
	struct cdrecord_output *cdrecord_output = data;
	char *line;
	GIOStatus status;
	GtkWidget *insert_cd_dialog;
	
	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);

	/* TODO: Handle errors */
	if (status == G_IO_STATUS_NORMAL) {
		g_string_prepend (cdrecord_output->stderr, line);
		if (strstr (line, "No disk / Wrong disk!") != NULL) {
			insert_cd_dialog = gtk_message_dialog_new (cd_progress_get_window (),
								   GTK_DIALOG_DESTROY_WITH_PARENT,
								   GTK_MESSAGE_INFO,
								   GTK_BUTTONS_OK,
								   _("Please insert a blank CD in the CD writer."));
			gtk_window_set_title (GTK_WINDOW (insert_cd_dialog),
					      _("Insert blank CD"));
			g_signal_connect (insert_cd_dialog, "response", (GCallback)insert_cd_response, cdrecord_output);
			gtk_widget_show (insert_cd_dialog);
			cdrecord_output->expect_cdrecord_to_die = TRUE;
		} else if (strstr (line, "Data may not fit on current disk") != NULL) {
			last_error = g_strdup (_("The files selected did not fit on the CD"));
		}
		g_free (line);
	} else if (status == G_IO_STATUS_EOF) {
		if (!cdrecord_output->expect_cdrecord_to_die) {
			g_main_loop_quit (cdrecord_output->loop);
		}
		return FALSE;
	} else {
		g_print ("cdrecord stderr read failed, status: %d\n", status);
	}

	return TRUE;
}

static void
details_clicked (GtkButton *button, gpointer data)
{
	struct cdrecord_output *cdrecord_output = data;
	GtkWidget *dialog, *label, *text, *scrolled_window;
	GtkTextBuffer *buffer;

	
	dialog = gtk_dialog_new_with_buttons (_("CD writing error details"),
					      cd_progress_get_window (),
					      GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
					      GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
					      NULL);

	label = gtk_label_new (_("Detailed error output from cdrecord:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    label,
			    FALSE, FALSE, 0);

	buffer = gtk_text_buffer_new (NULL);
	gtk_text_buffer_set_text (buffer, cdrecord_output->stderr->str, -1);
	
	text = gtk_text_view_new_with_buffer (buffer);
	g_object_unref (buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text), FALSE);
	gtk_widget_show (text);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scrolled_window), text);
	gtk_widget_show (scrolled_window);
	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    scrolled_window,
			    TRUE, TRUE, 0);


	g_signal_connect (dialog, "response", (GCallback)gtk_widget_destroy, NULL);

	gtk_widget_show (dialog);
}

int
write_iso (CDDrive *recorder,
	   const char *filename,
	   int speed,
	   gboolean eject,
	   gboolean dummy_write)
{
	struct cdrecord_output cdrecord_output;
	const char *argv[20]; /* Shouldn't need more than 20 arguments */
	char *speed_str, *dev_str;
	int stdout_pipe, stderr_pipe;
	guint stdout_tag, stderr_tag;
	GIOChannel *channel;
	GtkWidget *button, *dialog;
	GError *error;
	int i;

	i = 0;
	argv[i++] = "cdrecord";
	speed_str = g_strdup_printf ("speed=%d", speed);
	if (speed != 0) {
		argv[i++] = speed_str;
	}
	dev_str = g_strdup_printf ("dev=%s", recorder->cdrecord_id);
	argv[i++] = dev_str;
	if (dummy_write) {
		argv[i++] = "-dummy";
	}
	if (eject) {
		argv[i++] = "-eject";
	}
	argv[i++] = "-v";
	argv[i++] = "-data";
	argv[i++] = filename;
	argv[i++] = NULL;

	cdrecord_output.stderr = NULL;
 retry:
	cdrecord_output.result = RESULT_ERROR;
	cdrecord_output.expect_cdrecord_to_die = FALSE;
	cdrecord_output.line = NULL;
	if (cdrecord_output.stderr != NULL) {
		g_string_truncate (cdrecord_output.stderr, 0);
	} else {
		cdrecord_output.stderr = g_string_new (NULL);
	}
	
	cd_progress_set_text (_("Preparing to write CD"));
	cdrecord_output.changed_text = FALSE;
	cd_progress_set_fraction (0.0);
	cd_progress_set_image_spinning (TRUE);
	cd_progress_set_cancel_func (cancel_cdrecord, FALSE, &cdrecord_output);

	error = NULL;
	if (!g_spawn_async_with_pipes  (NULL,
					(char **)argv,
					NULL,
					G_SPAWN_SEARCH_PATH,
					NULL, NULL,
					&cdrecord_output.pid,
					&cdrecord_output.stdin,
					&stdout_pipe,
					&stderr_pipe,
					&error)) {
		g_warning ("cdrecord command failed: %s\n", error->message);
		g_error_free (error);
		/* TODO: Better error handling */
	} else {
		/* Make sure we don't block on a read. */
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);

		cdrecord_output.loop = g_main_loop_new (NULL, FALSE);
	
		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stdout_read,
					     &cdrecord_output);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stderr_read,
					     &cdrecord_output);
		g_io_channel_unref (channel);
		
		cd_progress_set_cancel_func (cancel_cdrecord, FALSE, &cdrecord_output);
		
		g_main_loop_run (cdrecord_output.loop);
		g_main_loop_unref (cdrecord_output.loop);
		
		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);

		if (cdrecord_output.result == RESULT_RETRY) {
			goto retry;
		}
	}

	g_free (speed_str);
	g_free (dev_str);
	
	
	if (cdrecord_output.result == RESULT_ERROR) {
		dialog = gtk_message_dialog_new (cd_progress_get_window (),
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 last_error ? _("There was an error writing to the CD:\n%s") : _("There was an error writing to the CD"),
						 last_error);
		gtk_window_set_title (GTK_WINDOW (dialog),
				      _("Error writing to CD"));
		button = gtk_button_new_with_mnemonic (_("_Details"));
		gtk_widget_show (button);
		gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area),
				  button,
				  FALSE, TRUE, 0);
		g_signal_connect (button, "clicked", (GCallback)details_clicked, &cdrecord_output);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
	}

	g_string_free (cdrecord_output.stderr, TRUE);
	cd_progress_set_image_spinning (FALSE);

	return cdrecord_output.result;
}
