#if !defined (__EXCEPTION_HPP)
#define __EXCEPTION_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if !defined IN_COMMON_HPP
   #error except.hpp is included by common.hpp only.
#endif

#include <string>
//
// Base exception class
//


namespace corelinux
{


   DECLARE_CLASS( Exception );

   // Define a reference to line number

   DECLARE_TYPE( Dword, LineNum );


   /**
   Exception is the base exception class used in the CoreLinux++ libraries.
   It is provided to support a rich base from which domain Exceptions may
   derive.
   */

   class Exception 
   {
   
   public:
   
   
      /// Exception Severity States

      enum Severity
      {
         CONTINUABLE = 1,     /// System can continue processing
         THREADFATAL,         /// Exception may prove to be thread fatal
         PROCESSFATAL,        /// Exception may prove to be process fatal
         THREADTERMINATE,     /// System should kill thread
         PROCESSTERMINATE     /// System should exit
      };
   
                     
                        /**
                        Default Constructor
                        @param why describes why the exception was thrown
                        @param file The source module throwing the exception
                        @param line The line of source throwing the exception
                        @param severity The Exception::Severity of the Exception
                        @param outOfMemory An out of memory indicator
                        */

                        Exception
                        (
                           CharCptr why,
                           CharCptr file,
                           LineNum  line,
                           Severity severity = Exception::CONTINUABLE,
                           bool     outOfMemory = false 
                        );
   
                        /**
                        Copy constructor
                        @param Exception const reference
                        */
                     
                        Exception( ExceptionCref crOther );
   
                        /// Virtual Destructor
   
      virtual           ~Exception(void);
   
         //
         // Operator overloads
         //
   
               /**
               Assignment operator overload
               @param  Exception const reference
               @return Exception reference to self
               */
            
               ExceptionRef   operator = ( ExceptionCref otherRef );
            
               /**
               Comparisson operator overload
               @param Exception const reference
               @return true if equal, false otherwise
               */

               bool           operator==( ExceptionCref otherRef );
            
         //
         // Accessor methods
         //
   
               /**
               Accessor
               @return Const reference to module name where 
               Exception was thrown
               */
            
               const std::string &    getFile( void ) const;
            
               /**
               Accessor
               @return Const reference to line number in module where 
               Exception was thrown
               */
            
               LineNumCref       getLine( void ) const;
            
               /**
               Accessor
               @return Const reference to Exception explanation.
               */
            
               const std::string &    getWhy( void ) const;
            
               /**
               Accessor
               @return Const reference to Severity of Exception.
               */
            
               const Severity &  getSeverity( void ) const;
            
               /**
               Accessor
               @return Const reference to the unwind stack description.
               */
            
               const std::string &    getUnwind( void ) const;   
            
               /**
               Accessor
               @return true if out of memory exception, false otherwise
               */
            
               bool           isOutOfMemory( void ) const 
                  { return theOutOfMemoryFlag;}
            
         //
         // Mutator methods
         //
   
               /** 
                  Append unwind information to the Exception.
                  Clients should use this service to identify
                  themselves and specify any changes to severity.
               */   
            
               void           addUnwindInfo( CharCptr unwindInfo );
            
               ///   Change the severity to Severity::THREADFATAL
            
               void           setThreadFatalSeverity( void );
            
               ///   Change the severity to Severity::PROCESSFATAL
            
               void           setProcessFatalSeverity( void );
            
               /**
               Change the severity to Severity::THREADTERMINATE. This is
               useful to the catcher that the thread should be
               cleaned up.
               */
            
               void           setThreadTerminateSeverity( void );
            
               /**
               Change the severity to Severity::PROCESSTERMINATE. This is
               useful to the catcher that the process should exit
               */
            
               void           setProcessTerminateSeverity( void );
   
   protected:
   
                        /**
                           Exceptions must have a reason.
                           Default constructor is not allowed.
                        */   
                  
                        Exception( void );
   
                        /**
                        Exception constructor for use by derivations
                        */
                     
                        Exception
                        (
                           CharCptr file,
                           LineNum  line,
                           Severity severity = Exception::CONTINUABLE,
                           bool     outOfMemory = false 
                        );
   
               /// Changes the exception reason
            
               void     setWhy( const std::string & );

               /// Changes the exception reason

               void     setWhy( CharCptr );
   
   private:
   
   
   private:
   
               // Reason why the exception is being thrown.
      
               std::string      theReason;
   
               // File that threw the exception.
            
               std::string      theFile;
            
               // Severity of the exception.
            
               Severity    theSeverity;
            
               // Unwind information added as exception is unwound from 
               // stack.
            
               std::string      theUnwindInfo;
            
               // Line number in the file throwing the exception.
            
               LineNum     theLine;
            
               // Flag that indicates if there is a low memory situation.
            
               bool        theOutOfMemoryFlag;
   };

}


#endif  // !defined __EXCEPTION_HPP

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.2 $
   $Date: 2000/07/28 01:37:09 $
   $Locker:  $
*/


