/*
 *  @(#)DefaultAttributeSet.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.defimpl;

import net.sourceforge.groboutils.pmti.v1.IAttributeSet;
import net.sourceforge.groboutils.pmti.v1.IAttribute;

import java.util.Hashtable;
import java.util.Enumeration;


/**
 * Contains a queryable set of attribute name-value pairs associated with
 * issue types.  Implementations may have direct accessors for these
 * attributes.  Common attributes include: who reported the issue, a list
 * of comments written about this issue, and a history of the changes the
 * issue has gone through.  All <tt>IAttributeSet</tt> implementations must be
 * immutable, unless they also implement <tt>IEditableAttributeSet</tt>.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:51:58 $
 * @since      July 12, 2002
 */
public class DefaultAttributeSet implements IAttributeSet
{
    private Hashtable attribs;
    
    
    public DefaultAttributeSet( IAttribute[] a )
    {
        if (a == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        
        this.attribs = new Hashtable();
        for (int i = 0; i < a.length; ++i)
        {
            if (a[i] != null)
            {
                this.attribs.put( a[i].getInfo().getName(), a[i] );
            }
        }
    }
    
    
    
    /**
     * Returns a list of all attributes.
     */
    public IAttribute[] getAttributes()
    {
        IAttribute a[] = createAttributeArray( this.attribs.size() );
        Enumeration enum = this.attribs.elements();
        for (int i = 0; enum.hasMoreElements(); ++i)
        {
            a[i] = (IAttribute)enum.nextElement();
        }
        return a;
    }
    
    
    /**
     * 
     */
    public String[] getAttributeNames()
    {
        String s[] = new String[ this.attribs.size() ];
        Enumeration enum = this.attribs.keys();
        for (int i = 0; enum.hasMoreElements(); ++i)
        {
            s[i] = (String)enum.nextElement();
        }
        return s;
    }
    
    
    /**
     * Finds the attribute for the given name.
     *
     * @return the attribute for the name, or <tt>null</tt> if no such
     *      attribute exists.
     */
    public IAttribute getAttribute( String name )
    {
        if (name == null)
        {
            return null;
        }
        return (IAttribute)this.attribs.get( name );
    }
    
    
    protected IAttribute[] createAttributeArray( int size )
    {
        return new IAttribute[ size ];
    }
}

