package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.Visitor;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.transaction.GlobalTransaction;

import java.util.HashMap;
import java.util.Map;

/**
 * Implements functionality defined by {@link org.jboss.cache.CacheSPI#removeNode(org.jboss.cache.Fqn)}
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class RemoveNodeCommand extends AbstractVersionedDataCommand
{
   public static final int METHOD_ID = 5;
   public static final int VERSIONED_METHOD_ID = 40;
   private static final Log log = LogFactory.getLog(RemoveNodeCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /*parameters*/
   private boolean skipSendingNodeEvents = false;
   protected Fqn parentFqn;
   protected NodeSPI targetNode;
   protected Map originalData;

   public RemoveNodeCommand(GlobalTransaction globalTransaction, Fqn fqn)
   {
      this.globalTransaction = globalTransaction;
      this.fqn = fqn;
   }

   public RemoveNodeCommand()
   {
   }

   /**
    * Removes the node referenced by the specified Fqn.
    */
   public Object perform(InvocationContext ctx)
   {
      if (trace) log.trace("perform(" + globalTransaction + ", \"" + fqn + ")");
      // Find the node
      targetNode = dataContainer.peekVersioned(fqn, dataVersion, true);
      if (targetNode == null)
      {
         if (trace) log.trace("node " + fqn + " not found");
         return false;
      }
      notifyBeforeRemove(targetNode, ctx);

      NodeSPI parentNode = targetNode.getParent();
      boolean found = targetNode.isValid() && !targetNode.isDeleted();
      targetNode.markAsDeleted(true, true);

      if (globalTransaction != null && found)
      {
         prepareForRollback(parentNode);
      }
      notifyAfterRemove(ctx);
      return found;
   }

   private void prepareForRollback(NodeSPI parentNode)
   {
      parentFqn = parentNode.getFqn();
      Map targetData = targetNode.getDataDirect();
      if (!targetData.isEmpty())
      {
         originalData = new HashMap(targetNode.getDataDirect());
      }
   }

   private void notifyBeforeRemove(NodeSPI n, InvocationContext ctx)
   {
      if (!skipSendingNodeEvents)
      {
         notifier.notifyNodeRemoved(fqn, true, n.getDataDirect(), ctx);
      }
   }

   private void notifyAfterRemove(InvocationContext ctx)
   {
      if (!skipSendingNodeEvents)
      {
         notifier.notifyNodeRemoved(fqn, false, null, ctx);
      }
   }

   public void rollback()
   {
      if (targetNode != null)
      {
         Object childName = targetNode.getFqn().getLastElement();
         if (trace)
         {
            log.trace("rollback(parent: " + parentFqn + ", child: " + childName + ", node=" + targetNode + ")");
         }

         if (parentFqn == null || childName == null)
         {
            log.error("parent fqn or childName or childNode was null");
            return;
         }
         NodeSPI parentNode = dataContainer.peek(parentFqn);
         if (parentNode == null)
         {
            log.warn("node " + parentFqn + " not found");
            return;
         }
         parentNode.addChild(childName, targetNode);
         targetNode.markAsDeleted(false, true);
         targetNode.clearDataDirect();
         if (originalData != null) targetNode.putAllDirect(originalData);
         targetNode.setValid(true, true);
      }
   }

   public Object acceptVisitor(InvocationContext ctx, Visitor visitor) throws Throwable
   {
      return visitor.visitRemoveNodeCommand(ctx, this);
   }

   public boolean isSkipSendingNodeEvents()
   {
      return skipSendingNodeEvents;
   }

   public int getCommandId()
   {
      return isVersioned() ? VERSIONED_METHOD_ID : METHOD_ID;
   }

   @Override
   public Object[] getParameters()
   {
      if (isVersioned())
         return new Object[]{globalTransaction, fqn, true, skipSendingNodeEvents, dataVersion};
      else
         return new Object[]{globalTransaction, fqn, true, skipSendingNodeEvents};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      globalTransaction = (GlobalTransaction) args[0];
      fqn = (Fqn) args[1];
      skipSendingNodeEvents = (Boolean) args[3];
      if (isVersionedId(commandId)) dataVersion = (DataVersion) args[4];
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;

      RemoveNodeCommand that = (RemoveNodeCommand) o;

      if (skipSendingNodeEvents != that.skipSendingNodeEvents) return false;
      if (globalTransaction != null ? !globalTransaction.equals(that.globalTransaction) : that.globalTransaction != null)
         return false;

      return true;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + (globalTransaction != null ? globalTransaction.hashCode() : 0);
      result = 31 * result + (skipSendingNodeEvents ? 1 : 0);
      return result;
   }

   @Override
   protected boolean isVersionedId(int id)
   {
      return id == VERSIONED_METHOD_ID;
   }

   public void setSkipSendingNodeEvents(boolean skipSendingNodeEvents)
   {
      this.skipSendingNodeEvents = skipSendingNodeEvents;
   }

   @Override
   public String toString()
   {
      return "RemoveNodeCommand{" +
            "fqn=" + fqn +
            ", dataVersion=" + dataVersion +
            ", globalTransaction=" + globalTransaction +
            ", skipSendingNodeEvents=" + skipSendingNodeEvents +
            ", parentFqn=" + parentFqn +
            ", targetNode=" + targetNode +
            '}';
   }
}
