/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.options;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.Option;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNull;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.transaction.TransactionManager;

/**
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
@Test(groups = {"functional"})
public class CacheModeLocalSimpleTest
{
   private CacheSPI<Object, Object> cache1, cache2;
   private Option cacheModeLocal;

   @BeforeMethod(alwaysRun = true)
   public void setUp() throws Exception
   {
      cache1 = (CacheSPI<Object, Object>) new DefaultCacheFactory().createCache(false);
      Configuration c = cache1.getConfiguration();
      c.setCacheMode("REPL_SYNC");
      c.setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");

      cache2 = (CacheSPI<Object, Object>) new DefaultCacheFactory().createCache(false);
      c = cache2.getConfiguration();
      c.setCacheMode("REPL_SYNC");
      c.setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");

      cacheModeLocal = new Option();
      cacheModeLocal.setCacheModeLocal(true);
   }

   @AfterMethod(alwaysRun = true)
   public void tearDown()
   {
      TestingUtil.killCaches(cache1, cache2);
   }

   public void testCacheModeLocalWithTx() throws Exception
   {
      doTest(false);
   }

   public void testCacheModeLocalOptimisticWithTx() throws Exception
   {
      doTest(true);
   }

   private void doTest(boolean optimistic) throws Exception
   {
      if (optimistic)
      {
         cache1.getConfiguration().setNodeLockingScheme(Configuration.NodeLockingScheme.OPTIMISTIC);
         cache2.getConfiguration().setNodeLockingScheme(Configuration.NodeLockingScheme.OPTIMISTIC);
         cache1.getConfiguration().setSyncCommitPhase(true);
         cache1.getConfiguration().setSyncRollbackPhase(true);
         cache2.getConfiguration().setSyncCommitPhase(true);
         cache2.getConfiguration().setSyncRollbackPhase(true);
      }

      cache1.start();
      cache2.start();

      TestingUtil.blockUntilViewsReceived(10000, cache1, cache2);

      TransactionManager mgr = cache1.getTransactionManager();
      mgr.begin();

      cache1.put(Fqn.fromString("/replicate"), "k", "v");
      cache1.getInvocationContext().getOptionOverrides().setCacheModeLocal(true);
      cache1.put(Fqn.fromString("/not-replicate"), "k", "v");

      mgr.commit();
      Thread.sleep(3000);

      assertEquals("v", cache1.get("/replicate", "k"));
      assertEquals("v", cache1.get("/not-replicate", "k"));

      assertEquals("v", cache2.get("/replicate", "k"));
      assertNull(cache2.get("/not-replicate", "k"));
   }
}
