/* This is -*- C -*- */
/* $Id: guppi-boxplot-state.c,v 1.11 2001/05/06 07:56:28 trow Exp $ */

/*
 * guppi-boxplot-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-rgb.h>
#include <guppi-memory.h>
#include "guppi-boxplot-state.h"
#include "guppi-boxplot-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_HORIZONTAL,
  ARG_VERTICAL,
  ARG_IQR_FACTOR,
  ARG_FRAME_COLOR,
  ARG_LINE_THICKNESS,
  ARG_FILL_BOX,
  ARG_BOX_COLOR,
  ARG_BOX_SIZE,
  ARG_TICK_SIZE
};

static void
guppi_boxplot_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiBoxplotState *state = GUPPI_BOXPLOT_STATE (obj);

  switch (arg_id) {

  case ARG_HORIZONTAL:
    GTK_VALUE_BOOL (*arg) = guppi_boxplot_state_horizontal (state);
    break;

  case ARG_VERTICAL:
    GTK_VALUE_BOOL (*arg) = guppi_boxplot_state_vertical (state);
    break;

  case ARG_IQR_FACTOR:
    GTK_VALUE_DOUBLE (*arg) = guppi_boxplot_state_iqr_factor (state);
    break;

  case ARG_FRAME_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_boxplot_state_frame_color (state);
    break;

  case ARG_LINE_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_boxplot_state_line_thickness (state);
    break;

  case ARG_FILL_BOX:
    GTK_VALUE_BOOL (*arg) = guppi_boxplot_state_fill_box (state);
    break;

  case ARG_BOX_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_boxplot_state_box_color (state);
    break;

  case ARG_BOX_SIZE:
    GTK_VALUE_DOUBLE (*arg) = guppi_boxplot_state_box_size (state);
    break;

  case ARG_TICK_SIZE:
    GTK_VALUE_DOUBLE (*arg) = guppi_boxplot_state_tick_size (state);
    break;

  default:
    break;
  };
}

static void
guppi_boxplot_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiBoxplotState *state = GUPPI_BOXPLOT_STATE (obj);

  switch (arg_id) {

  case ARG_HORIZONTAL:
    guppi_boxplot_state_set_horizontal (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_VERTICAL:
    guppi_boxplot_state_set_vertical (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_IQR_FACTOR:
    guppi_boxplot_state_set_iqr_factor (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_FRAME_COLOR:
    guppi_boxplot_state_set_frame_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_LINE_THICKNESS:
    guppi_boxplot_state_set_line_thickness (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_FILL_BOX:
    guppi_boxplot_state_set_fill_box (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_BOX_COLOR:
    guppi_boxplot_state_set_box_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_BOX_SIZE:
    guppi_boxplot_state_set_box_size (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_TICK_SIZE:
    guppi_boxplot_state_set_tick_size (state, GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_boxplot_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_boxplot_state_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

static GuppiElementView *
make_view (GuppiElementState * state)
{
  return GUPPI_ELEMENT_VIEW (guppi_type_new (GUPPI_TYPE_BOXPLOT_VIEW));
}

/***************************************************************************/

static void
guppi_boxplot_state_class_init (GuppiBoxplotStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_boxplot_state_get_arg;
  object_class->set_arg = guppi_boxplot_state_set_arg;
  object_class->destroy = guppi_boxplot_state_destroy;
  object_class->finalize = guppi_boxplot_state_finalize;

  state_class->make_view = make_view;

  gtk_object_add_arg_type ("GuppiBoxplotState::horizontal",
			   GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_HORIZONTAL);
  gtk_object_add_arg_type ("GuppiBoxplotState::vertical",
			   GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_VERTICAL);
  gtk_object_add_arg_type ("GuppiBoxplotState::iqr_factor",
			   GTK_TYPE_DOUBLE, GTK_ARG_READWRITE,
			   ARG_IQR_FACTOR);
  gtk_object_add_arg_type ("GuppiBoxplotState::frame_color", GTK_TYPE_UINT,
			   GTK_ARG_READWRITE, ARG_FRAME_COLOR);
  gtk_object_add_arg_type ("GuppiBoxplotState::line_thickness",
			   GTK_TYPE_DOUBLE, GTK_ARG_READWRITE,
			   ARG_LINE_THICKNESS);
  gtk_object_add_arg_type ("GuppiBoxplotState::fill_box", GTK_TYPE_BOOL,
			   GTK_ARG_READWRITE, ARG_FILL_BOX);
  gtk_object_add_arg_type ("GuppiBoxplotState::box_color", GTK_TYPE_UINT,
			   GTK_ARG_READWRITE, ARG_BOX_COLOR);
  gtk_object_add_arg_type ("GuppiBoxplotState::box_size", GTK_TYPE_DOUBLE,
			   GTK_ARG_READWRITE, ARG_BOX_SIZE);
  gtk_object_add_arg_type ("GuppiBoxplotState::tick_size", GTK_TYPE_DOUBLE,
			   GTK_ARG_READWRITE, ARG_TICK_SIZE);
}

static void
data_change_cb (GuppiShared * sh, GuppiBoxplotState * state)
{
  GuppiSeqScalar *sd = GUPPI_SEQ_SCALAR0 (guppi_shared_get (sh));

  if (sd == NULL || guppi_seq_size (GUPPI_SEQ (sd)) == 0) {
    state->ready = FALSE;
  } else {
    state->md = guppi_seq_scalar_median (sd);
    state->q1 = guppi_seq_scalar_q1 (sd);
    state->q3 = guppi_seq_scalar_q3 (sd);
    state->whisker_lower = guppi_seq_scalar_percentile (sd, 0.05);
    state->whisker_upper = guppi_seq_scalar_percentile (sd, 0.95);
    state->ready = TRUE;
  }

  guppi_element_state_changed_size (GUPPI_ELEMENT_STATE (state));
  guppi_element_state_changed (GUPPI_ELEMENT_STATE (state));
}

static void
guppi_boxplot_state_init (GuppiBoxplotState * obj)
{
  GuppiShared *sh = guppi_shared_data ();

  gtk_signal_connect (GTK_OBJECT (sh),
		      "changed", GTK_SIGNAL_FUNC (data_change_cb), obj);

  guppi_element_state_add_shared (GUPPI_ELEMENT_STATE (obj), SHARED_DATA, sh);


  /* Some helpful (?) defaults */
  obj->horizontal = TRUE;
  obj->iqr_factor = 0.2;
  obj->frame_color = RGBA_BLACK;
  obj->line_thickness = 72 / 64.0;
  obj->box_color = RGBA_YELLOW;
  obj->fill_box = TRUE;
  obj->box_size = 72 / 8.0;
  obj->tick_size = 72 / 12.0;
}

GtkType guppi_boxplot_state_get_type (void)
{
  static GtkType guppi_boxplot_state_type = 0;
  if (!guppi_boxplot_state_type) {
    static const GtkTypeInfo guppi_boxplot_state_info = {
      "GuppiBoxplotState",
      sizeof (GuppiBoxplotState),
      sizeof (GuppiBoxplotStateClass),
      (GtkClassInitFunc) guppi_boxplot_state_class_init,
      (GtkObjectInitFunc) guppi_boxplot_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_boxplot_state_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &guppi_boxplot_state_info);
  }
  return guppi_boxplot_state_type;
}

GuppiElementState *
guppi_boxplot_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_boxplot_state_get_type ()));
}

GuppiSeqScalar *
guppi_boxplot_state_data (GuppiBoxplotState * state)
{
  GtkObject *obj;
  g_return_val_if_fail (state != NULL
			&& GUPPI_IS_BOXPLOT_STATE (state), NULL);
  obj =
    guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state), SHARED_DATA);
  return GUPPI_SEQ_SCALAR0 (obj);
}

void
guppi_boxplot_state_set_data (GuppiBoxplotState * state,
			      GuppiSeqScalar * data)
{
  g_return_if_fail (state != NULL && GUPPI_IS_BOXPLOT_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_SEQ_SCALAR (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_DATA, data);
}

void
guppi_boxplot_state_set_horizontal (GuppiBoxplotState * state, gboolean x)
{
  g_return_if_fail (state != NULL);

  if (state->horizontal != x) {
    state->horizontal = x;
    guppi_element_state_changed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_vertical (GuppiBoxplotState * state, gboolean x)
{
  guppi_boxplot_state_set_horizontal (state, !x);
}

void
guppi_boxplot_state_set_iqr_factor (GuppiBoxplotState * state, double f)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (f >= 0);

  if (state->iqr_factor != f) {
    state->iqr_factor = f;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_frame_color (GuppiBoxplotState * state, guint32 c)
{
  g_return_if_fail (state != NULL);

  if (state->frame_color != c) {
    state->frame_color = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_line_thickness (GuppiBoxplotState * state, double t)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (t > 0);

  if (state->line_thickness != t) {
    state->line_thickness = t;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_fill_box (GuppiBoxplotState * state, gboolean x)
{
  g_return_if_fail (state != NULL);

  if (state->fill_box != x) {
    state->fill_box = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_box_color (GuppiBoxplotState * state, guint32 x)
{
  g_return_if_fail (state != NULL);

  if (state->box_color != x) {
    state->box_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_box_size (GuppiBoxplotState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (x > 0);

  if (state->box_size != x) {
    state->box_size = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_boxplot_state_set_tick_size (GuppiBoxplotState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (x > 0);

  if (state->tick_size != x) {
    state->tick_size = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

/* $Id: guppi-boxplot-state.c,v 1.11 2001/05/06 07:56:28 trow Exp $ */
