/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-pricebars-item.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <guppi-convenient.h>
#include <guppi-rgb.h>
#include "guppi-pricebars-state.h"
#include "guppi-pricebars-view.h"
#include "guppi-pricebars-item.h"
#include "guppi-pricebars-tools.h"

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_pricebars_item_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_pricebars_item_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_pricebars_item_destroy (GtkObject *obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_pricebars_item_finalize (GtkObject *obj)
{
  GuppiPricebarsItem *item = GUPPI_PRICEBARS_ITEM (obj);


  guppi_free (item->t_op_buf);
  guppi_free (item->t_hi_buf);
  guppi_free (item->t_lo_buf);
  guppi_free (item->t_cl_buf);

  guppi_free (item->op_buf);
  guppi_free (item->hi_buf);
  guppi_free (item->lo_buf);
  guppi_free (item->cl_buf);
  
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static void
state_changed (GuppiCanvasItem *gci)
{
  guppi_canvas_item_request_update (gci);
}

static void
view_changed (GuppiCanvasItem *gci)
{
  guppi_canvas_item_request_update (gci);
}

static void
update (GuppiCanvasItem *gci, double aff[6], ArtSVP *clip_path, gint flags)
{
  GuppiPricebarsItem *item = GUPPI_PRICEBARS_ITEM (gci);
  double cx0, cx1;
  
  guppi_canvas_item_vp2c_d (gci, 0, 0, &cx0, NULL);
  guppi_canvas_item_vp2c_d (gci, 1, 0, &cx1, NULL);
  
  item->pixels_per_day = cx1 - cx0;
}

static void
render (GuppiCanvasItem *gci, GnomeCanvasBuf *buf)
{
  GuppiPricebarsItem* pb_item;
  GuppiPricebarsState *state;
  GuppiElementView *view;
  GuppiPriceSeries *ser;
  double sc;
  gint r, g, b, a;
  GDate dt, dt2;
  GDate start_date, end_date;
  guint32 color;
  gint horiz_tick_size;

  pb_item = GUPPI_PRICEBARS_ITEM (gci);
  state = GUPPI_PRICEBARS_STATE (guppi_canvas_item_state (gci));
  view = guppi_canvas_item_view (gci);
  ser = guppi_pricebars_state_data (state);
  sc = guppi_canvas_item_scale (gci);

  if (ser == NULL || guppi_date_indexed_empty (GUPPI_DATE_INDEXED (ser)))
    return;

  UINT_TO_RGBA (guppi_pricebars_state_color (state), &r, &g, &b, &a);

  /* Find the date range to be rendered */

  guppi_pricebars_view_date_range (GUPPI_PRICEBARS_VIEW (view),
				   &start_date, &end_date);

  color = guppi_pricebars_state_color (state);

  horiz_tick_size = MIN ((gint)floor (pb_item->pixels_per_day/2), 5);

  dt = start_date;
  while (g_date_lteq (&dt, &end_date)) {
    gint i, No=0, Nh, Nl, Nc=0;
    dt2 = dt;
    g_date_add_days (&dt2, pb_item->buf_size-1);

    /* We make some dangerous assumptions here. */

    if (horiz_tick_size > 0)
      No = guppi_price_series_get_range_timecoded (ser, PRICE_OPEN,
						   &dt, &dt2,
						   pb_item->t_op_buf,
						   pb_item->op_buf,
						   pb_item->buf_size);

    Nh = guppi_price_series_get_range_timecoded (ser, PRICE_HIGH,
						 &dt, &dt2,
						 pb_item->t_hi_buf,
						 pb_item->hi_buf,
						 pb_item->buf_size);


    Nl = guppi_price_series_get_range_timecoded (ser, PRICE_LOW,
						 &dt, &dt2,
						 pb_item->t_lo_buf,
						 pb_item->lo_buf,
						 pb_item->buf_size);


    if (horiz_tick_size > 0)
      Nc = guppi_price_series_get_range_timecoded (ser, PRICE_CLOSE,
						   &dt, &dt2,
						   pb_item->t_cl_buf,
						   pb_item->cl_buf,
						   pb_item->buf_size);

    if (horiz_tick_size > 0) {
      guppi_canvas_item_x_vp2c_d_bulk (gci, pb_item->t_op_buf, pb_item->t_op_buf, No);
      guppi_canvas_item_x_vp2c_d_bulk (gci, pb_item->t_cl_buf, pb_item->t_cl_buf, Nc);
    }

    guppi_canvas_item_x_vp2c_d_bulk (gci, pb_item->t_hi_buf, pb_item->t_hi_buf, Nh);
    guppi_canvas_item_x_vp2c_d_bulk (gci, pb_item->t_lo_buf, pb_item->t_lo_buf, Nl);


    if (horiz_tick_size > 0) {
      guppi_canvas_item_y_vp2c_d_bulk (gci, pb_item->op_buf, pb_item->op_buf, No);
      guppi_canvas_item_y_vp2c_d_bulk (gci, pb_item->cl_buf, pb_item->cl_buf, Nc);
    }

    guppi_canvas_item_y_vp2c_d_bulk (gci, pb_item->hi_buf, pb_item->hi_buf, Nh);
    guppi_canvas_item_y_vp2c_d_bulk (gci, pb_item->lo_buf, pb_item->lo_buf, Nl);



    if (horiz_tick_size > 0) {
      for (i=0; i<No; ++i) {
	gint jul = (gint)pb_item->t_op_buf[i];
	guppi_paint_horiz (buf,
			   jul-horiz_tick_size, jul,
			   (gint)pb_item->op_buf[i], color);
      }
      
      for (i=0; i<Nc; ++i) {
	gint jul = (gint)pb_item->t_cl_buf[i];
	guppi_paint_horiz (buf,
			   jul+1, jul+horiz_tick_size+1,
			   (gint)pb_item->cl_buf[i], color);
      }
    }

    for (i = 0; i < MIN(Nh,Nl); ++i) {

      if (pb_item->t_hi_buf[i] == pb_item->t_lo_buf[i]) {
	gint jul = (gint)pb_item->t_hi_buf[i];
	guppi_paint_vert (buf, jul,
			  (gint)pb_item->lo_buf[i]-1,
			  (gint)pb_item->hi_buf[i]+1,
			  color);
      }
    }

    dt = dt2;
    g_date_add_days (&dt, 1);
  }

}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static void
guppi_pricebars_item_class_init (GuppiPricebarsItemClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *)klass;
  GuppiCanvasItemClass *gci_class = GUPPI_CANVAS_ITEM_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_CANVAS_ITEM);

  object_class->get_arg = guppi_pricebars_item_get_arg;
  object_class->set_arg = guppi_pricebars_item_set_arg;
  object_class->destroy = guppi_pricebars_item_destroy;
  object_class->finalize = guppi_pricebars_item_finalize;

  gci_class->state_changed = state_changed;
  gci_class->view_changed = view_changed;
  gci_class->guppi_update = update;
  gci_class->guppi_render = render;
  gci_class->uses_vp_coordinates = TRUE;

  guppi_canvas_item_class_set_item_class_toolkit (gci_class,
						  guppi_pricebars_toolkit_default ());

}

static void
guppi_pricebars_item_init (GuppiPricebarsItem *obj)
{
  obj->buf_size = 1024;


  obj->t_op_buf = guppi_new (double, obj->buf_size);
  obj->t_hi_buf = guppi_new (double, obj->buf_size);
  obj->t_lo_buf = guppi_new (double, obj->buf_size);
  obj->t_cl_buf = guppi_new (double, obj->buf_size);

  obj->op_buf = guppi_new (double, obj->buf_size);
  obj->hi_buf = guppi_new (double, obj->buf_size);
  obj->lo_buf = guppi_new (double, obj->buf_size);
  obj->cl_buf = guppi_new (double, obj->buf_size);
}

GtkType
guppi_pricebars_item_get_type (void)
{
  static GtkType guppi_pricebars_item_type = 0;
  if (!guppi_pricebars_item_type) {
    static const GtkTypeInfo guppi_pricebars_item_info = {
      "GuppiPricebarsItem",
      sizeof (GuppiPricebarsItem),
      sizeof (GuppiPricebarsItemClass),
      (GtkClassInitFunc)guppi_pricebars_item_class_init,
      (GtkObjectInitFunc)guppi_pricebars_item_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_pricebars_item_type = gtk_type_unique (GUPPI_TYPE_CANVAS_ITEM, &guppi_pricebars_item_info);
  }
  return guppi_pricebars_item_type;
}

