/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: guppi-polynomial.c,v 1.2 2001/05/04 06:11:43 trow Exp $ */

/*
 * guppi-polynomial.c
 *
 * Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#include <config.h>
#include <stdlib.h>
#include <math.h>
#include "guppi-debug.h"
#include "guppi-memory.h"
#include "guppi-polynomial.h"

#define POLYEPSILON 1e-12

static void
guppi_polynomial_sanitize (GuppiPolynomial *p)
{
  if (p == NULL)
    return;

  while (p->d > 0 && fabs (p->c[p->d]) < POLYEPSILON)
    --p->d;
}

GuppiPolynomial *
guppi_polynomial_new (gint degree, ...)
{
  GuppiPolynomial *p;
  va_list args;
  gint i;

  g_return_val_if_fail (degree >= 0, NULL);

  p = guppi_new0 (GuppiPolynomial, 1);
  p->d = degree;
  p->N = degree;
  p->c = guppi_new0 (double, p->N+1);

  va_start (args, degree);
  for (i=0; i<degree+1; ++i) {
    p->c[i] = va_arg (args, double);
  }
  va_end (args);

  return p;
}

GuppiPolynomial *
guppi_polynomial_newv (gint degree, const double *vec)
{
  GuppiPolynomial *p;
  gint i;

  g_return_val_if_fail (degree >= 0, NULL);
  g_return_val_if_fail (vec, NULL);

  p = guppi_new0 (GuppiPolynomial, 1);
  p->d = degree;
  p->N = degree;
  p->c = guppi_new0 (double, p->N+1);
  for (i=0; i<degree+1; ++i) {
    p->c[i] = vec[i];
  }

  return p;
}

GuppiPolynomial *
guppi_polynomial_new_linear (double c0, double c1)
{
  return guppi_polynomial_new (1, c0, c1);
}

GuppiPolynomial *
guppi_polynomial_new_quadratic (double c0, double c1, double c2)
{
  return guppi_polynomial_new (2, c0, c1, c2);
}

GuppiPolynomial *
guppi_polynomial_copy (GuppiPolynomial *p)
{
  GuppiPolynomial *cpy;

  if (p == NULL)
    return NULL;

  cpy = guppi_new0 (GuppiPolynomial, 1);
  cpy->d = p->d;
  cpy->N = p->d;
  cpy->c = guppi_new0 (double, cpy->N+1);

  memcpy (cpy->c, p->c, sizeof (double) * (cpy->N+1));

  return cpy;
}

void
guppi_polynomial_free (GuppiPolynomial *p)
{
  if (p) {
    guppi_free (p->c);
    guppi_free (p);
  }
}

double
guppi_polynomial_coefficient (GuppiPolynomial *p, gint i)
{
  g_return_val_if_fail (p != NULL, 0);

  return 0 <= i && i <= p->d ? p->c[i] : 0;
}

double
guppi_polynomial_eval (GuppiPolynomial *p, double x)
{
  double run;
  gint i;

  g_return_val_if_fail (p != NULL, 0);

  i = p->d;
  run = p->c[i];
  --i;
  while (i >= 0) {
    run = run * x + p->c[i];
    --i;
  }
  return run;
}

void
guppi_polynomial_eval_many (GuppiPolynomial *p, gint N, double *src, double *dest)
{
  double run, x;
  gint i, j;

  g_return_if_fail (p != NULL);
  if (N == 0)
    return;
  g_return_if_fail (N > 0);
  g_return_if_fail (src != NULL);
  g_return_if_fail (dest != NULL);

  for (j = 0; j < N; ++j) {

    x = src[j];

    i = p->d;
    run = p->c[i];
    --i;

    while (i >= 0) {
      run = run * x + p->c[i];
      --i;
    }

    dest[j] = x;

  }
}

void
guppi_polynomial_D (GuppiPolynomial *p)
{
  gint i;
  g_return_if_fail (p != NULL);

  if (p->d == 0) {
    p->c[0] = 0;
    return;
  }

  for (i = 1; i <= p->d; ++i) {
    p->c[i-1] = i * p->c[i];
  }
   
  --p->d;
}

void
guppi_polynomial_normalize (GuppiPolynomial *p)
{
  gint i;

  g_return_if_fail (p != NULL);

  for (i = 0; i < p->d; ++i) {
    p->c[i] /= p->c[p->d];
  }
  p->c[p->d] = 1.0;
}

void
guppi_polynomial_modulo (GuppiPolynomial *p, GuppiPolynomial *mod)
{
  double factor;
  g_return_if_fail (p != NULL);
  g_return_if_fail (mod != NULL);

  if (p->d < mod->d)
    return;

  factor = p->c[p->d] / mod->c[mod->d];

  guppi_FIXME ();

  g_assert_not_reached ();
}

void
guppi_polynomial_deflate (GuppiPolynomial *p, double x0)
{
  gint i;
  double old, t;

  g_return_if_fail (p != NULL);
  g_return_if_fail (p->d != 0);

  old = p->c[p->d-1];
  p->c[p->d-1] = p->c[p->d];
  for (i = p->d-2; i >= 0; --i) {
    t = p->c[i];
    p->c[i] = p->c[i+1] * x0 + old;
    old = t;
  }
  --p->d;

  guppi_polynomial_sanitize (p);
}

void
guppi_polynomial_deflate_complex (GuppiPolynomial *p, double re, double im)
{
  gint i;
  double a, b, older, old, t;

  g_return_if_fail (p != NULL);
  g_return_if_fail (p->d >= 2);

  a = 2*re;
  b = -(re*re + im*im);

  older = p->c[p->d-2];
  old   = p->c[p->d-3];

  p->c[p->d-2] = p->c[p->d];
  p->c[p->d-3] = p->c[p->d-1] + a * p->c[p->d-2];

  for (i = p->d-4; i >= 0; --i) {
    t = p->c[i];

    p->c[i] = older + a * p->c[i+1] + b * p->c[i+2];
    
    older = old;
    old = t;
  }

  p->d -= 2;

  guppi_polynomial_sanitize (p);
}

double
guppi_polynomial_newton_polish (GuppiPolynomial *p, double x, gint iter, double epsilon)
{
  double best_x, best_abs_y;
  double fx, Dfx, abs_fx;
  gint i;

  g_return_val_if_fail (p != NULL, x);

  best_x = x;
  best_abs_y = 1e+8;

  while (iter > 0 && best_abs_y > epsilon) {
    
    /* We simultaneously evaluate the polynomial
       and its first derivative. */
    i = p->d;
    fx = p->c[i];
    Dfx = i * p->c[i];
    --i;
    while (i >= 1) {
      fx = fx * x + p->c[i];
      Dfx = Dfx * x + i * p->c[i];
      --i;
    }
    fx = fx * x + p->c[0];

    abs_fx = fabs (fx);

    if (abs_fx < best_abs_y) {
      best_abs_y = abs_fx;
      best_x = x;
    }
    g_print ("%d: %g %g %g\n", iter, x, fx, Dfx);
    
    if (fabs (Dfx) < epsilon)
      break;

    x -= fx / Dfx;

    --iter;
  }

  return best_x;
}

double
guppi_polynomial_gershgorin_radius (GuppiPolynomial *p)
{
  double max_c, top_c, q;
  gint i;

  g_return_val_if_fail (p != NULL, 0);
  
  if (p->d == 0)
    return 0;

  max_c = 0;
  top_c = p->c[p->d];
  for (i = 0; i < p->d; ++i) {
    q = fabs (p->c[i] / top_c);
    if (q > max_c)
      max_c = q;
  }
  
  return max_c + 1;
}


void
guppi_polynomial_spew (GuppiPolynomial *p)
{
  gint i;

  g_return_if_fail (p != NULL);

  for (i = p->d; i >= 0; --i) {

    if (fabs (p->c[i] - 1) > 1e-16 || i == 0)
      g_print ("%g ", p->c[i]);

    if (i) {
      if (i == 1)
	g_print ("x + ");
      else
	g_print ("x^%d + ", i);
    }
  }

  g_print ("\n");
}

xmlNodePtr
guppi_polynomial_export_xml (GuppiPolynomial *p, GuppiXMLDocument *doc)
{
  xmlNodePtr poly_node;
  xmlNodePtr term_node;
  gint i;

  g_return_val_if_fail (p != NULL, NULL);
  g_return_val_if_fail (doc != NULL, NULL);

  poly_node = guppi_xml_new_node (doc, "Polynomial");
  guppi_xml_set_propertyf (poly_node, "degree", "%d", guppi_polynomial_degree (p));

  for (i = 0; i <= guppi_polynomial_degree (p); ++i)
    if (fabs (p->c[i]) >= POLYEPSILON) {
      term_node = guppi_xml_new_text_nodef (doc, "term", "%g", p->c[i]);
      guppi_xml_set_propertyf (term_node, "degree", "%d", i);
      xmlAddChild (poly_node, term_node);
    }

  return poly_node;
}

GuppiPolynomial *
guppi_polynomial_import_xml (GuppiXMLDocument *doc, xmlNodePtr node)
{
  GuppiPolynomial *p;
  gint i, N;
  gchar *buf;
  
  g_return_val_if_fail (doc != NULL, NULL);
  g_return_val_if_fail (node != NULL, NULL);

  if (strcmp (node->name, "Polynomial"))
    return NULL;

  buf = xmlGetProp (node, "degree");
  N = buf ? atoi (buf) : 0;
  free (buf);

  p = guppi_new0 (GuppiPolynomial, 1);
  p->d = N;
  p->N = MIN (N, 3);
  p->c = guppi_new0 (double, p->N+1);

  node = node->xmlChildrenNode;

  while (node) {

    if (!strcmp (node->name, "term")) {

      buf = xmlGetProp (node, "degree");
      i = buf ? atoi (buf) : 0;
      free (buf);

      buf = xmlNodeListGetString (doc->doc, node->xmlChildrenNode, 1);
      if (0 <= i && i <= N)
	p->c[i] = atof (buf);
      free (buf);
    }
    
    node = node->next;
  }

  return p;
}

