/*
 * accessible-tree-model.c: A tree model for Accessibles
 *
 * Authors:
 *    Michael Meeks
 *
 * Copyright 2002 Sun Microsystems, Inc.
 */
#include <config.h>
#include <stdio.h>
#include <string.h>
#include <bonobo/bonobo-exception.h>

#include "graphics.h"
#include "accessible-tree-model.h"

#undef DEBUG_TREE

/* FIXME: leaks references like a sieve ! */

static GObjectClass *parent_class;

/* Store the accessible on 'iter->user_data */

#define ITER_ACCESSIBLE(i) ((i)->user_data)

static gboolean
make_iter_invalid (GtkTreeIter *iter)
{
	iter->stamp = 0;
	iter->user_data = NULL;
	iter->user_data2 = NULL;
	iter->user_data3 = NULL;

	return FALSE;
}

static gboolean
make_iter (GtkTreeModel *model, Accessible *node, GtkTreeIter *iter)
{
	if (!node)
		return make_iter_invalid (iter);

	iter->stamp = ACCESSIBLE_TREE_MODEL (model)->stamp;
	ITER_ACCESSIBLE (iter) = node;
	iter->user_data2 = NULL;
	iter->user_data3 = NULL;

	return TRUE;
}

static void
make_root_path_iter (AccessibleTreeModel *model,
		     GtkTreePath        **path,
		     GtkTreeIter         *iter)
{
	make_iter ((GtkTreeModel *)model, model->root_node, iter);

	*path = gtk_tree_path_new ();
	gtk_tree_path_append_index (*path, 0);
}

static int
accessible_tree_model_get_n_columns (GtkTreeModel *model)
{
	return ACCESSIBLE_TREE_MODEL_NUM_COLS;
}

static GType
accessible_tree_model_get_column_type (GtkTreeModel *model, int index)
{
	switch (index) {
	case ACCESSIBLE_TREE_MODEL_NAME_COL:
		return G_TYPE_STRING;
	case ACCESSIBLE_TREE_MODEL_ROLE_COL:
		return G_TYPE_STRING;
	case ACCESSIBLE_TREE_MODEL_DESCR_COL:
		return G_TYPE_STRING;
	case ACCESSIBLE_TREE_MODEL_ICON_COL:
		return GDK_TYPE_PIXBUF;
	default:
		g_assert_not_reached ();
	}
	
	return G_TYPE_INVALID;
}

static gboolean
accessible_tree_model_get_iter (GtkTreeModel *model,
				GtkTreeIter  *iter,
				GtkTreePath  *path)
{
	int depth, i;
	int *indices;
	GtkTreeIter parent;

	depth = gtk_tree_path_get_depth (path);
	indices = gtk_tree_path_get_indices (path);

	if (!gtk_tree_model_iter_nth_child (model, iter, NULL, indices [0])) {
		g_warning ("Foo !");
		return FALSE;
	}

	for (i = 1; i < depth; i++) {
		parent = *iter;

		if (!gtk_tree_model_iter_nth_child (model, iter,
						    &parent, indices [i]))
			return FALSE;
	}

	return TRUE;
}

static GtkTreePath *
accessible_tree_model_get_path (GtkTreeModel *model, GtkTreeIter *iter)
{
	Accessible *node, *parent;
	GtkTreePath *path;
	AccessibleTreeModel *tree_model;

	node = ITER_ACCESSIBLE (iter);
	tree_model = ACCESSIBLE_TREE_MODEL (model);

	Accessible_ref (node);

	path = gtk_tree_path_new ();

#ifdef DEBUG_TREE	
	fprintf (stderr, "root node  %p (%s) (%s)\n",
		 tree_model->root_node,
		 Accessible_getRoleName (tree_model->root_node),
		 Accessible_getName (tree_model->root_node));
#endif

	while (node != tree_model->root_node && node != NULL) {
		gtk_tree_path_prepend_index (
			path, Accessible_getIndexInParent (node));
		parent = Accessible_getParent (node);
#ifdef DEBUG_TREE	
		fprintf (stderr, "%p parent of %p [%s] (%s) (%s)\n",
			 parent,
			 node,
			 accessible_is_base_accessible (node) ? "ok" : "broken",
			 Accessible_getRoleName (node),
			 Accessible_getName (node));
#endif
		Accessible_unref (node);
		node = parent;
	}

#warning FIXME: at-spis hierarchy is dead broken ...
#if 0
	if (node == NULL) { /* exception - whole tree broken */
		gtk_tree_path_free (path);
		path = NULL;
	} else
#endif
		gtk_tree_path_prepend_index (path, 0);

	Accessible_unref (node);

	return path;
}

static void
accessible_tree_model_get_value (GtkTreeModel *model,
				 GtkTreeIter  *iter,
				 int           column,
				 GValue       *value)
{
	Accessible *node;

	node = ITER_ACCESSIBLE (iter);

	switch (column) {
	case ACCESSIBLE_TREE_MODEL_NAME_COL:
		g_value_init (value, G_TYPE_STRING);
		if (!node)
			g_value_set_static_string (value, "<No accessible>");
		else {
			char *txt;

			txt = Accessible_getName (node);
			if (!txt || txt [0] == '\0')
				g_value_set_string (value, "<no name>");
			else
				g_value_set_string (value, txt);
			SPI_freeString (txt);
		}
		break;

	case ACCESSIBLE_TREE_MODEL_ROLE_COL:
		g_value_init (value, G_TYPE_STRING);
		if (!node)
			g_value_set_static_string (value, "<No accessible>");
		else {
			char *txt;

			txt = Accessible_getRoleName (node);
			if (!txt || txt [0] == '\0')
				g_value_set_string (value, "<no role>");
			else
				g_value_set_string (value, txt);
			SPI_freeString (txt);
		}
		break;

	case ACCESSIBLE_TREE_MODEL_DESCR_COL:
		g_value_init (value, G_TYPE_STRING);
		if (!node)
			g_value_set_static_string (value, "<No accessible>");
		else {
			char *txt;

			txt = Accessible_getDescription (node);
			if (!txt || txt [0] == '\0')
				g_value_set_string (value, "<no description>");
			else
				g_value_set_string (value, txt);
			SPI_freeString (txt);
		}
		break;

	case ACCESSIBLE_TREE_MODEL_ICON_COL: {
		GdkPixbuf *pixbuf;
		AccessibleRole role;

		if (!node)
			role = SPI_ROLE_INVALID;
		else
			role = Accessible_getRole (node);

		g_value_init (value, GDK_TYPE_PIXBUF);

		pixbuf = get_pixbuf_for_role (role);
		g_value_set_object (value, pixbuf);

		break;
	}
	default:
		g_assert_not_reached ();
	}
}

static gboolean
accessible_tree_model_iter_next (GtkTreeModel *model,
				 GtkTreeIter  *iter)
{
	int idx;
	Accessible *next;

	/* FIXME: Dog slow roundtrip wise */
	idx = Accessible_getIndexInParent (ITER_ACCESSIBLE (iter));

	next = Accessible_getChildAtIndex (
		Accessible_getParent (ITER_ACCESSIBLE (iter)),
		idx + 1);

	if (next != ITER_ACCESSIBLE (iter))
		return make_iter (model, next, iter);
	else
		return make_iter_invalid (iter);
}

static gboolean
accessible_tree_model_iter_children (GtkTreeModel *model,
				     GtkTreeIter  *iter,
				     GtkTreeIter  *parent_iter)
{
	return make_iter (
		model,
		Accessible_getChildAtIndex (
			ITER_ACCESSIBLE (parent_iter),
			0),
		iter);
}

static gboolean
accessible_tree_model_iter_parent (GtkTreeModel *model,
				   GtkTreeIter  *iter,
				   GtkTreeIter  *child_iter)
{
	return make_iter (
		model,
		Accessible_getParent (
			ITER_ACCESSIBLE (child_iter)),
		iter);
}

static gboolean
accessible_tree_model_iter_has_child (GtkTreeModel *model,
				      GtkTreeIter  *iter)
{
	return Accessible_getChildCount (ITER_ACCESSIBLE (iter));
}

static int
accessible_tree_model_iter_n_children (GtkTreeModel *model,
				       GtkTreeIter  *iter)
{
	return Accessible_getChildCount (ITER_ACCESSIBLE (iter));
}

static gboolean
accessible_tree_model_iter_nth_child (GtkTreeModel *model,
				      GtkTreeIter  *iter,
				      GtkTreeIter  *parent_iter,
				      int           n)
{
	if (!parent_iter) {
		AccessibleTreeModel *tree_model;

		tree_model = ACCESSIBLE_TREE_MODEL (model);

		ITER_ACCESSIBLE (iter) = tree_model->root_node;

		return (tree_model->root_node != NULL);
	}

	return make_iter (
		model,
		Accessible_getChildAtIndex (
			ITER_ACCESSIBLE (parent_iter), n),
		iter);
}

static void
accessible_tree_model_ref_node (GtkTreeModel *model,
				GtkTreeIter  *iter)
{
	Accessible_ref (ITER_ACCESSIBLE (iter));
}

static void
accessible_tree_model_unref_node (GtkTreeModel *model, GtkTreeIter *iter)
{
	Accessible_unref (ITER_ACCESSIBLE (iter));
}

gboolean
accessible_is_base_accessible (Accessible *accessible)
{
	gboolean ret;
	CORBA_Environment ev;
	CORBA_Object *a = (CORBA_Object *) accessible;

	if (!a)
		return TRUE;

	CORBA_exception_init (&ev);
	ret = CORBA_Object_is_a (*a, "IDL:Accessibility/Accessible:1.0", &ev);
	if (BONOBO_EX (&ev))
		ret = FALSE;
	CORBA_exception_free (&ev);

	return ret;
}

GtkTreeModel *
accessible_tree_model_new (Accessible *root)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	GtkTreeModel *model;

	g_return_val_if_fail (root != NULL, NULL);
	
	model = g_object_new (ACCESSIBLE_TYPE_TREE_MODEL, NULL);

	g_assert (accessible_is_base_accessible (root));

	ACCESSIBLE_TREE_MODEL (model)->root_node = root;

	make_root_path_iter (
		ACCESSIBLE_TREE_MODEL (model), &path, &iter);

	gtk_tree_model_row_inserted (model, path, &iter);

	gtk_tree_path_free (path);

	return model;
}

static void
report_global_event (const AccessibleEvent *event, void *user_data)
{
	GtkTreeIter   iter;
	GtkTreePath  *path;
	GtkTreeModel *model = user_data;

#ifdef DEBUG_TREE
	{
		char *txt;
		
		g_warning ("Event '%s' '%p' ('%s') %ld %ld",
			   event->type, event->source,
			   (txt = accessible_get_path (event->source)),
			   event->detail1, event->detail2);
		g_free (txt);
	}
#endif

	make_iter (model, event->source, &iter);

	path = gtk_tree_model_get_path (model, &iter);

	if (!path) {
		g_warning ("Whole tree is dead or node not in this tree ! (sigh)");
	} else
		gtk_tree_model_row_changed (model, path, &iter);

/*	gtk_tree_model_row_deleted (
		GTK_TREE_MODEL (model), path);
	gtk_tree_model_row_inserted (
	GTK_TREE_MODEL (model), path, &iter); */
}

static void
accessible_tree_model_init (AccessibleTreeModel *model)
{
	model->stamp = 1;

	model->event_listener = SPI_createAccessibleEventListener (
		report_global_event, model);
	SPI_registerGlobalEventListener (
		model->event_listener, "object:children-changed");
}

static void
accessible_tree_model_dispose (GObject *object)
{
	AccessibleTreeModel *model;

	model = ACCESSIBLE_TREE_MODEL (object);

	if (model->root_node != NULL) {
		Accessible_unref (model->root_node);
		model->root_node = NULL;
	}

	if (model->event_listener != NULL) {
		SPI_deregisterGlobalEventListenerAll (
			model->event_listener);
		AccessibleEventListener_unref (
			model->event_listener);
		model->event_listener = NULL;
	}


	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
accessible_tree_model_class_init (GObjectClass *gobject_class)
{
	parent_class = g_type_class_peek_parent (gobject_class);

	gobject_class->dispose = accessible_tree_model_dispose;
}

static void
accessible_tree_model_tree_model_init (GtkTreeModelIface *iface)
{
	iface->get_n_columns = accessible_tree_model_get_n_columns;
	iface->get_column_type = accessible_tree_model_get_column_type;
	iface->get_iter = accessible_tree_model_get_iter;
	iface->get_path = accessible_tree_model_get_path;
	iface->get_value = accessible_tree_model_get_value;
	iface->iter_next = accessible_tree_model_iter_next;
	iface->iter_children = accessible_tree_model_iter_children;
	iface->iter_has_child = accessible_tree_model_iter_has_child;
	iface->iter_n_children = accessible_tree_model_iter_n_children;
	iface->iter_nth_child = accessible_tree_model_iter_nth_child;
	iface->iter_parent = accessible_tree_model_iter_parent;
	iface->ref_node = accessible_tree_model_ref_node;
	iface->unref_node = accessible_tree_model_unref_node;
}

GType
accessible_tree_model_get_type (void)
{
	static GType object_type = 0;

	if (object_type == 0) {
		static const GTypeInfo object_info = {
			sizeof (AccessibleTreeModelClass),
			NULL,
			NULL,
			(GClassInitFunc) accessible_tree_model_class_init,
			NULL,
			NULL,
			sizeof (AccessibleTreeModel),
			0,
			(GInstanceInitFunc) accessible_tree_model_init,
		};

		static const GInterfaceInfo tree_model_info = {
			(GInterfaceInitFunc) accessible_tree_model_tree_model_init,
			NULL,
			NULL
		};

		object_type = g_type_register_static (
			G_TYPE_OBJECT, "AccessibleTreeModel", &object_info, 0);
		g_type_add_interface_static (object_type,
					     GTK_TYPE_TREE_MODEL,
					     &tree_model_info);
	}

	return object_type;
}

Accessible *
accessible_tree_get_from_iter (GtkTreeIter *iter)
{
	g_return_val_if_fail (iter != NULL, NULL);

	return ITER_ACCESSIBLE (iter);
}


/**
 * accessible_to_path:
 * @accessible: the accessible
 * 
 *   Convert the @accessible's location into a
 * path of sorts, try to be more robust than
 * pure integer offets by using names etc.
 * 
 * Return value: the g_allocated path, or NULL on
 * fatal exception
 **/
char *
accessible_get_path (Accessible *accessible)
{
	GString *path;
	gboolean error = FALSE;
	GSList  *l, *parents = NULL;

	while (accessible) {
		parents = g_slist_prepend (parents, accessible);
		accessible = Accessible_getParent (accessible);
	}

	path = g_string_new ("");

	for (l = parents; l; l = l->next) {
		long  idx;
		char *name, *role;
		Accessible *a = l->data;
		AccessibleRole r;

		g_string_append_c (path, '/');

		idx = Accessible_getIndexInParent (a);
		g_string_printf (path, "%ld", idx);

		name = Accessible_getName (a);
		if (name)
			g_string_append (path, name);
		else {
			error = TRUE;
			goto out;
		}
		SPI_freeString (name);
		g_string_append_c (path, ',');

		role = Accessible_getRoleName (a);
		if (role)
			g_string_append (path, role);
		else {
			error = TRUE;
			goto out;
		}
		SPI_freeString (role);
		g_string_append_c (path, ',');

		r = Accessible_getRole (a);
		g_string_append_printf (path, "%d", r);
	}

 out:
	for (l = parents; l; l = l->next)
		Accessible_unref (l->data);
	g_slist_free (parents);

	return g_string_free (path, error);
}

/**
 * accessible_from_path:
 * @path: the path
 * 
 * retreive the accessible with the given path
 * 
 * Return value: an accessible with that path.
 **/
Accessible *
accessible_from_path (const char *path)
{
	g_warning (G_STRLOC "implement me");
	return NULL;
}
