/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  Copyright (C) 2000, 2001, 2002, 2003 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: ephy-window.c,v 1.144 2003/12/08 15:21:31 chpe Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ephy-window.h"
#include "ephy-command-manager.h"
#include "ephy-bookmarks-menu.h"
#include "ephy-favorites-menu.h"
#include "ephy-state.h"
#include "ppview-toolbar.h"
#include "window-commands.h"
#include "find-dialog.h"
#include "print-dialog.h"
#include "ephy-shell.h"
#include "eel-gconf-extensions.h"
#include "ephy-prefs.h"
#include "ephy-zoom.h"
#include "ephy-debug.h"
#include "ephy-file-helpers.h"
#include "statusbar.h"
#include "toolbar.h"
#include "popup-commands.h"
#include "ephy-encoding-menu.h"
#include "ephy-tabs-menu.h"
#include "ephy-stock-icons.h"
#include "ephy-extension.h"
#include "ephy-favicon-cache.h"

#include <string.h>
#include <glib/gi18n.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkactiongroup.h>
#include <gtk/gtktoggleaction.h>
#include <gtk/gtkuimanager.h>
#include <gtk/gtktoggleaction.h>

static void ephy_window_class_init		(EphyWindowClass *klass);
static void ephy_window_init			(EphyWindow *gs);
static void ephy_window_finalize		(GObject *object);
static void ephy_window_show			(GtkWidget *widget);
static void ephy_window_notebook_switch_page_cb	(GtkNotebook *notebook,
						 GtkNotebookPage *page,
						 guint page_num,
						 EphyWindow *window);

static GtkActionEntry ephy_menu_entries [] = {

	/* Toplevel */
	{ "File", NULL, N_("_File") },
	{ "Edit", NULL, N_("_Edit") },
	{ "View", NULL, N_("_View") },
	{ "Bookmarks", NULL, N_("_Bookmarks") },
	{ "Go", NULL, N_("_Go") },
	{ "Tabs", NULL, N_("_Tabs") },
	{ "Help", NULL, N_("_Help") },
	{ "PopupAction", NULL, "" },

	/* File menu */
	{ "FileNewWindow", GTK_STOCK_NEW, N_("_New Window"), "<control>N",
	  N_("Open a new window"),
	  G_CALLBACK (window_cmd_file_new_window) },
	{ "FileNewTab", STOCK_NEW_TAB, N_("New _Tab"), "<control>T",
	  N_("Open a new tab"),
	  G_CALLBACK (window_cmd_file_new_tab) },
	{ "FileOpen", GTK_STOCK_OPEN, N_("_Open..."), "<control>O",
	  N_("Open a file"),
	  G_CALLBACK (window_cmd_file_open) },
	{ "FileSaveAs", GTK_STOCK_SAVE_AS, N_("Save _As..."), "<shift><control>S",
	  N_("Save the current page"),
	  G_CALLBACK (window_cmd_file_save_as) },
	{ "FilePrintSetup", NULL, N_("Print Set_up..."), NULL,
	  N_("Setup the page settings for printing"),
	  G_CALLBACK (window_cmd_file_print_setup) },
	{ "FilePrintPreview", GTK_STOCK_PRINT_PREVIEW, N_("Print Pre_view"),"<control><shift>P",
	  N_("Print preview"),
	  G_CALLBACK (window_cmd_file_print_preview) },
	{ "FilePrint", GTK_STOCK_PRINT, N_("_Print..."), "<control>P",
	  N_("Print the current page"),
	  G_CALLBACK (window_cmd_file_print) },
	{ "FileSendTo", STOCK_SEND_MAIL, N_("S_end To..."), NULL,
	  N_("Send a link of the current page"),
	  G_CALLBACK (window_cmd_file_send_to) },
	{ "FileCloseWindow", GTK_STOCK_CLOSE, N_("_Close"), "<control>W",
	  N_("Close this window"),
	  G_CALLBACK (window_cmd_file_close_window) },

	/* Edit menu */
	{ "EditUndo", GTK_STOCK_UNDO, N_("_Undo"), "<control>Z",
	  N_("Undo the last action"),
	  G_CALLBACK (window_cmd_edit_undo) },
	{ "EditRedo", GTK_STOCK_REDO, N_("Re_do"), "<shift><control>Z",
	  N_("Redo the last undone action"),
	  G_CALLBACK (window_cmd_edit_redo) },
	{ "EditCut", GTK_STOCK_CUT, N_("Cu_t"), "<control>X",
	  N_("Cut the selection"),
	  G_CALLBACK (window_cmd_edit_cut) },
	{ "EditCopy", GTK_STOCK_COPY, N_("_Copy"), "<control>C",
	  N_("Copy the selection"),
	  G_CALLBACK (window_cmd_edit_copy) },
	{ "EditPaste", GTK_STOCK_PASTE, N_("_Paste"), "<control>V",
	  N_("Paste clipboard"),
	  G_CALLBACK (window_cmd_edit_paste) },
	{ "EditSelectAll", NULL, N_("Select _All"), "<control>A",
	  N_("Select the entire page"),
	  G_CALLBACK (window_cmd_edit_select_all) },
	{ "EditFind", GTK_STOCK_FIND, N_("_Find..."), "<control>F",
	  N_("Find a word or phrase in the page"),
	  G_CALLBACK (window_cmd_edit_find) },
	{ "EditFindNext", NULL, N_("Find Ne_xt"), "<control>G",
	  N_("Find next occurrence of the word or phrase"),
	  G_CALLBACK (window_cmd_edit_find_next) },
	{ "EditFindPrev", NULL, N_("Find Pre_vious"), "<shift><control>G",
	  N_("Find previous occurrence of the word or phrase"),
	  G_CALLBACK (window_cmd_edit_find_prev) },
	{ "EditPersonalData", NULL, N_("P_ersonal Data"), NULL,
	  N_("View and remove cookies and passwords"),
	  G_CALLBACK (window_cmd_edit_personal_data) },
	{ "EditToolbar", NULL, N_("T_oolbars"), NULL,
	  N_("Customize toolbars"),
	  G_CALLBACK (window_cmd_edit_toolbar) },
	{ "EditPrefs", GTK_STOCK_PREFERENCES, N_("P_references"), NULL,
	  N_("Configure the web browser"),
	  G_CALLBACK (window_cmd_edit_prefs) },

	/* View menu */
	{ "ViewStop", GTK_STOCK_STOP, N_("_Stop"), "Escape",
	  N_("Stop current data transfer"),
	  G_CALLBACK (window_cmd_view_stop) },
	{ "ViewReload", GTK_STOCK_REFRESH, N_("_Reload"), "<control>R",
	  N_("Display the latest content of the current page"),
	  G_CALLBACK (window_cmd_view_reload) },
	{ "ViewZoomIn", GTK_STOCK_ZOOM_IN, N_("Zoom _In"), "<control>plus",
	  N_("Increase the text size"),
	  G_CALLBACK (window_cmd_view_zoom_in) },
	{ "ViewZoomOut", GTK_STOCK_ZOOM_OUT, N_("Zoom _Out"), "<control>minus",
	  N_("Decrease the text size"),
	  G_CALLBACK (window_cmd_view_zoom_out) },
	{ "ViewZoomNormal", GTK_STOCK_ZOOM_100, N_("_Normal Size"), NULL,
	  N_("Use the normal text size"),
	  G_CALLBACK (window_cmd_view_zoom_normal) },
	{ "ViewEncoding", NULL, N_("Text _Encoding"), NULL,
	  N_("Change the text encoding"),
	  NULL },
	{ "ViewPageSource", STOCK_VIEW_SOURCE, N_("_Page Source"), "<control>U",
	  N_("View the source code of the page"),
	  G_CALLBACK (window_cmd_view_page_source) },

	/* Bookmarks menu */
	{ "FileBookmarkPage", STOCK_ADD_BOOKMARK, N_("_Add Bookmark..."), "<control>D",
	  N_("Add a bookmark for the current page"),
	  G_CALLBACK (window_cmd_file_bookmark_page) },
	{ "GoBookmarks", EPHY_STOCK_BOOKMARKS, N_("_Edit Bookmarks"), "<control>B",
	  N_("Open the bookmarks window"),
	  G_CALLBACK (window_cmd_go_bookmarks) },

	/* Go menu */
	{ "GoBack", GTK_STOCK_GO_BACK, N_("_Back"), "<alt>Left",
	  N_("Go to the previous visited page"),
	  G_CALLBACK (window_cmd_go_back) },
	{ "GoForward", GTK_STOCK_GO_FORWARD, N_("_Forward"), "<alt>Right",
	  N_("Go to the next visited page"),
	  G_CALLBACK (window_cmd_go_forward) },
	{ "GoUp", GTK_STOCK_GO_UP, N_("_Up"), "<alt>Up",
	  N_("Go up one level"),
	  G_CALLBACK (window_cmd_go_up) },
	{ "GoHome", GTK_STOCK_HOME, N_("_Home"), "<alt>Home",
	  N_("Go to the home page"),
	  G_CALLBACK (window_cmd_go_home) },
	{ "GoLocation", NULL, N_("_Location..."), "<control>L",
	  N_("Go to a specified location"),
	  G_CALLBACK (window_cmd_go_location) },
	{ "GoHistory", EPHY_STOCK_HISTORY, N_("H_istory"), "<control>H",
	  N_("Open the history window"),
	  G_CALLBACK (window_cmd_go_history) },

	/* Tabs menu */
	{ "TabsPrevious", NULL, N_("_Previous Tab"), "<control>Page_Up",
	  N_("Activate previous tab"),
	  G_CALLBACK (window_cmd_tabs_previous) },
	{ "TabsNext", NULL, N_("_Next Tab"), "<control>Page_Down",
	  N_("Activate next tab"),
	  G_CALLBACK (window_cmd_tabs_next) },
	{ "TabsMoveLeft", NULL, N_("Move Tab _Left"), "<shift><control>Page_Up",
	  N_("Move current tab to left"),
	  G_CALLBACK (window_cmd_tabs_move_left) },
	{ "TabsMoveRight", NULL, N_("Move Tab _Right"), "<shift><control>Page_Down",
	  N_("Move current tab to right"),
	  G_CALLBACK (window_cmd_tabs_move_right) },
	{ "TabsDetach", NULL, N_("_Detach Tab"), "<shift><control>M",
	  N_("Detach current tab"),
	  G_CALLBACK (window_cmd_tabs_detach) },

	/* Help menu */
	{"HelpContents", GTK_STOCK_HELP, N_("_Contents"), "F1",
	 N_("Display web browser help"),
	 G_CALLBACK (window_cmd_help_contents) },
	{ "HelpAbout", GNOME_STOCK_ABOUT, N_("_About"), NULL,
	  N_("Display credits for the web browser creators"),
	  G_CALLBACK (window_cmd_help_about) },
};
static guint ephy_menu_n_entries = G_N_ELEMENTS (ephy_menu_entries);

static GtkToggleActionEntry ephy_menu_toggle_entries [] =
{
	/* View Menu */
	{ "ViewToolbar", NULL, N_("_Toolbar"), "<shift><control>T",
	  N_("Show or hide toolbar"),
	  G_CALLBACK (window_cmd_view_toolbar), TRUE },
	{ "ViewBookmarksBar", NULL, N_("_Bookmarks Bar"), NULL,
	  N_("Show or hide bookmarks bar"),
	  G_CALLBACK (window_cmd_view_bookmarks_bar), TRUE },
	{ "ViewStatusbar", NULL, N_("St_atusbar"), NULL,
	  N_("Show or hide statusbar"),
	  G_CALLBACK (window_cmd_view_statusbar), TRUE },
	{ "ViewFullscreen", STOCK_FULLSCREEN, N_("_Fullscreen"), "F11",
	  N_("Browse at full screen"),
	  G_CALLBACK (window_cmd_view_fullscreen), FALSE}
};
static guint ephy_menu_n_toggle_entries = G_N_ELEMENTS (ephy_menu_toggle_entries);

static GtkActionEntry ephy_popups_entries [] = {
	/* Document */
	{ "SaveBackgroundAs", NULL, N_("_Save Background As..."), NULL,
	  NULL, G_CALLBACK (popup_cmd_save_background_as) },
	{ "ContextBookmarkPage", STOCK_ADD_BOOKMARK, N_("Add Boo_kmark..."), "<control>D",
	  N_("Add a bookmark for the current page"),
	  G_CALLBACK (window_cmd_file_bookmark_page) },

	/* Framed document */
	{ "OpenFrame", NULL, N_("_Open Frame"), NULL,
	  NULL, G_CALLBACK (popup_cmd_open_frame) },
	{ "OpenFrameInNewWindow", NULL, N_("Open Frame in _New Window"), NULL,
	  NULL, G_CALLBACK (popup_cmd_frame_in_new_window) },
	{ "OpenFrameInNewTab", NULL, N_("Open Frame in New _Tab"), NULL,
	  NULL, G_CALLBACK (popup_cmd_frame_in_new_tab) },

	/* Links */
	{ "OpenLink", GTK_STOCK_OPEN, N_("_Open Link"),
	  NULL, NULL, G_CALLBACK (popup_cmd_open_link) },
	{ "OpenLinkInNewWindow", NULL, N_("Open Link in _New Window"), NULL,
	  NULL, G_CALLBACK (popup_cmd_link_in_new_window) },
	{ "OpenLinkInNewTab", NULL, N_("Open Link in New _Tab"),
	  NULL, NULL, G_CALLBACK (popup_cmd_link_in_new_tab) },
	{ "DownloadLink", GTK_STOCK_SAVE, N_("_Download Link"), NULL,
	  NULL, G_CALLBACK (popup_cmd_download_link) },
	{ "BookmarkLink", STOCK_ADD_BOOKMARK, N_("_Bookmark Link..."),
	  NULL, NULL, G_CALLBACK (popup_cmd_bookmark_link) },
	{ "CopyLinkAddress", NULL, N_("_Copy Link Address"), NULL,
	  NULL, G_CALLBACK (popup_cmd_copy_link_address) },

	/* Images */
	{ "OpenImage", GTK_STOCK_OPEN, N_("Open _Image"), GTK_STOCK_OPEN,
	  NULL, G_CALLBACK (popup_cmd_open_image) },
	{ "OpenImageInNewWindow", NULL, N_("Open Image in New _Window"), NULL,
	  NULL, G_CALLBACK (popup_cmd_image_in_new_window) },
	{ "OpenImageInNewTab", NULL, N_("Open Image in New T_ab"), NULL,
	  NULL, G_CALLBACK (popup_cmd_image_in_new_tab) },
	{ "SaveImageAs", GTK_STOCK_SAVE_AS, N_("_Save Image As..."), NULL,
	  NULL, G_CALLBACK (popup_cmd_save_image_as) },
	{ "SetImageAsBackground", NULL, N_("_Use Image As Background"), NULL,
	  NULL, G_CALLBACK (popup_cmd_set_image_as_background) },
	{ "CopyImageLocation", NULL, N_("Copy I_mage Address"), NULL,
	  NULL, G_CALLBACK (popup_cmd_copy_image_location) },
};
static guint ephy_popups_n_entries = G_N_ELEMENTS (ephy_popups_entries);

#define CONF_LOCKDOWN_DISABLE_JAVASCRIPT_CHROME  "/apps/epiphany/lockdown/disable_javascript_chrome"
#define CONF_LOCKDOWN_HIDE_MENUBAR "/apps/epiphany/lockdown/hide_menubar"

#define EPHY_WINDOW_GET_PRIVATE(object)(G_TYPE_INSTANCE_GET_PRIVATE ((object), EPHY_TYPE_WINDOW, EphyWindowPrivate))

struct EphyWindowPrivate
{
	GtkWidget *main_vbox;
	GtkWidget *menu_dock;
	GtkWidget *exit_fullscreen_popup;
	Toolbar *toolbar;
	GtkWidget *statusbar;
	GtkActionGroup *action_group;
	GtkActionGroup *popups_action_group;
	EphyFavoritesMenu *fav_menu;
	EphyEncodingMenu *enc_menu;
	EphyTabsMenu *tabs_menu;
	EphyBookmarksMenu *bmk_menu;
	PPViewToolbar *ppview_toolbar;
	GtkNotebook *notebook;
	EphyTab *active_tab;
	EphyDialog *find_dialog;
	EphyDialog *print_dialog;
	EmbedChromeMask requested_chrome_mask;
	EmbedChromeMask actual_chrome_mask;
	gboolean closing;
	gboolean is_fullscreen;
	gboolean is_ppview;
	gboolean has_size;
	guint num_tabs;

	guint disable_js_chrome_notifier_id;
	guint show_toolbars_notifier_id;
	guint show_bookmarks_bar_notifier_id;
	guint show_statusbar_notifier_id;
	guint hide_menubar_notifier_id;
};

enum
{
	PROP_0,
	PROP_ACTIVE_TAB
};

static GObjectClass *parent_class = NULL;

GType
ephy_window_get_type (void)
{
        static GType ephy_window_type = 0;

        if (ephy_window_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (EphyWindowClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) ephy_window_class_init,
                        NULL,
                        NULL, /* class_data */
                        sizeof (EphyWindow),
                        0, /* n_preallocs */
                        (GInstanceInitFunc) ephy_window_init
                };

                ephy_window_type = g_type_register_static (GTK_TYPE_WINDOW,
							   "EphyWindow",
							   &our_info, 0);
        }
        return ephy_window_type;
}

static void
ephy_window_destroy (GtkObject *gtkobject)
{
	EphyWindow *window = EPHY_WINDOW (gtkobject);

	LOG ("EphyWindow destroy %p", window)

	if (window->priv->closing == FALSE)
	{
		EphyExtension *manager;

		window->priv->closing = TRUE;

		/* Let the extensions detach themselves from the window */
		manager = EPHY_EXTENSION (ephy_shell_get_extensions_manager (ephy_shell));
		ephy_extension_detach_window (manager, window);
	}

	if (window->priv->exit_fullscreen_popup)
	{
		gtk_widget_destroy (window->priv->exit_fullscreen_popup);
		window->priv->exit_fullscreen_popup = NULL;
	}

        GTK_OBJECT_CLASS (parent_class)->destroy (gtkobject);
}

static void
ephy_window_selection_received_cb (GtkWidget *widget,
				   GtkSelectionData *selection_data,
				   guint time, EphyWindow *window)
{
	EphyTab *tab;

	if (selection_data->length <= 0 || selection_data->data == NULL)
		return;

	tab = ephy_window_get_active_tab (window);

	ephy_embed_load_url (ephy_tab_get_embed (tab), selection_data->data);
}

static void
add_widget (GtkUIManager *merge, GtkWidget *widget, EphyWindow *window)
{
	gtk_box_pack_start (GTK_BOX (window->priv->menu_dock),
			    widget, FALSE, FALSE, 0);
}

static void
update_exit_fullscreen_popup_position (EphyWindow *window)
{
	GdkRectangle screen_rect;
	int popup_height;

	gtk_window_get_size (GTK_WINDOW (window->priv->exit_fullscreen_popup),
                             NULL, &popup_height);

	gdk_screen_get_monitor_geometry (gdk_screen_get_default (),
                        gdk_screen_get_monitor_at_window
                        (gdk_screen_get_default (),
                         GTK_WIDGET (window)->window),
                         &screen_rect);

	gtk_window_move (GTK_WINDOW (window->priv->exit_fullscreen_popup),
                        screen_rect.x, screen_rect.height - popup_height);
}

static void
size_changed_cb (GdkScreen *screen, EphyWindow *window)
{
	update_exit_fullscreen_popup_position (window);
}

static void
exit_fullscreen_button_clicked_cb (GtkWidget *button, EphyWindow *window)
{
	gtk_window_unfullscreen (GTK_WINDOW (window));
}

static void
update_chromes_visibility (EphyWindow *window)
{
	GtkWidget *menubar;
	gboolean fullscreen;
	EmbedChromeMask flags = window->priv->actual_chrome_mask;

	fullscreen = window->priv->is_fullscreen;

	menubar = gtk_ui_manager_get_widget
		(GTK_UI_MANAGER (window->ui_merge), "/menubar");
	g_assert (menubar != NULL);

	if (!fullscreen && flags & EMBED_CHROME_MENUBARON)
	{
		gtk_widget_show (menubar);
	}
	else
	{
		gtk_widget_hide (menubar);
	}

	toolbar_set_visibility (window->priv->toolbar,
				flags & EMBED_CHROME_TOOLBARON,
				flags & EMBED_CHROME_BOOKMARKSBARON);


	if (!fullscreen && flags & EMBED_CHROME_STATUSBARON)
	{
		gtk_widget_show (window->priv->statusbar);
	}
	else
	{
		gtk_widget_hide (window->priv->statusbar);
	}

	if ((flags & EMBED_CHROME_PPVIEWTOOLBARON) != FALSE)
	{
		if (!window->priv->ppview_toolbar)
		{
			window->priv->ppview_toolbar = ppview_toolbar_new (window);
		}
	}
	else
	{
		if (window->priv->ppview_toolbar)
		{
			g_object_unref (window->priv->ppview_toolbar);
			window->priv->ppview_toolbar = NULL;
		}
	}
}

static void
update_chrome(EphyWindow *window)
{
	EmbedChromeMask chrome_mask;

	if (window->priv->is_ppview)
	{
		window->priv->actual_chrome_mask = EMBED_CHROME_PPVIEWTOOLBARON;
	}
	else
	{
		if (eel_gconf_get_boolean (CONF_LOCKDOWN_DISABLE_JAVASCRIPT_CHROME))
		{
			chrome_mask = EMBED_CHROME_DEFAULT;
		}
		else
		{
			chrome_mask = window->priv->requested_chrome_mask;
		}

		if ((chrome_mask & EMBED_CHROME_BOOKMARKSBAR_DEFAULT) ||
		    (chrome_mask & EMBED_CHROME_DEFAULT))
		{
			/* keep only not layout flags */
			chrome_mask &= (EMBED_CHROME_WINDOWRAISED |
					EMBED_CHROME_WINDOWLOWERED |
					EMBED_CHROME_CENTERSCREEN |
					EMBED_CHROME_OPENASDIALOG |
					EMBED_CHROME_OPENASPOPUP);

			/* Load defaults */
			if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_TOOLBARS))
			{
				chrome_mask |= EMBED_CHROME_TOOLBARON;
			}
			if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_BOOKMARKS_BAR))
			{
				chrome_mask |= EMBED_CHROME_BOOKMARKSBARON;
			}
			if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_STATUSBAR))
			{
				chrome_mask |= EMBED_CHROME_STATUSBARON;
			}
			if (!eel_gconf_get_boolean (CONF_LOCKDOWN_HIDE_MENUBAR))
			{
				chrome_mask |= EMBED_CHROME_MENUBARON;
			}
		}
		window->priv->actual_chrome_mask = chrome_mask;
	}

	update_chromes_visibility (window);
}

static void
ephy_window_fullscreen (EphyWindow *window)
{
	GtkWidget *popup, *button, *icon, *label, *hbox;

	window->priv->is_fullscreen = TRUE;

	popup = gtk_window_new (GTK_WINDOW_POPUP);
	window->priv->exit_fullscreen_popup = popup;

	button = gtk_button_new ();
	g_signal_connect (button, "clicked",
			  G_CALLBACK (exit_fullscreen_button_clicked_cb),
			  window);
	gtk_widget_show (button);
	gtk_container_add (GTK_CONTAINER (popup), button);

	hbox = gtk_hbox_new (FALSE, 2);
	gtk_widget_show (hbox);
	gtk_container_add (GTK_CONTAINER (button), hbox);

	icon = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_BUTTON);
	gtk_widget_show (icon);
	gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);

	label = gtk_label_new (_("Exit Fullscreen"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	update_exit_fullscreen_popup_position (window);

	gtk_widget_show (popup);

	g_signal_connect (G_OBJECT (gdk_screen_get_default ()),
                          "size-changed", G_CALLBACK (size_changed_cb),
			  popup);

	update_chromes_visibility (window);
}

static void
ephy_window_unfullscreen (EphyWindow *window)
{
	window->priv->is_fullscreen = FALSE;

	g_signal_handlers_disconnect_by_func (G_OBJECT (gdk_screen_get_default ()),
                                              G_CALLBACK (size_changed_cb),
					      window);

	gtk_widget_destroy (window->priv->exit_fullscreen_popup);
	window->priv->exit_fullscreen_popup = NULL;

	update_chromes_visibility (window);
}

static gboolean
ephy_window_state_event_cb (GtkWidget *widget, GdkEventWindowState *event, EphyWindow *window)
{
	if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN)
	{
		GtkActionGroup *action_group;
		GtkAction *action;
		gboolean fullscreen;

		fullscreen = event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN;

		if (fullscreen)
		{
			ephy_window_fullscreen (window);
		}
		else
		{
			ephy_window_unfullscreen (window);
		}

		action_group = window->priv->action_group;

		action = gtk_action_group_get_action (action_group, "ViewFullscreen");
		gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), fullscreen);

		action = gtk_action_group_get_action (action_group, "EditToolbar");
		g_object_set (action, "sensitive", !fullscreen, NULL);
	}

	return FALSE;
}


static void
update_edit_actions_sensitivity (EphyWindow *window)
{
	GtkWidget *widget = gtk_window_get_focus (GTK_WINDOW (window));
	GtkActionGroup *action_group;
	GtkAction *action;
	gboolean can_copy, can_cut, can_undo, can_redo, can_paste;

	if (GTK_IS_EDITABLE (widget))
	{
		gboolean has_selection;

		has_selection = gtk_editable_get_selection_bounds
			(GTK_EDITABLE (widget), NULL, NULL);

		can_copy = has_selection;
		can_cut = has_selection;
		can_paste = TRUE;
		can_undo = FALSE;
		can_redo = FALSE;
	}
	else
	{
		EphyEmbed *embed;

		embed = ephy_window_get_active_embed (window);
		g_return_if_fail (embed != NULL);

		can_copy = ephy_command_manager_can_do_command
				(EPHY_COMMAND_MANAGER (embed), "cmd_copy");
		can_cut = ephy_command_manager_can_do_command
				(EPHY_COMMAND_MANAGER (embed), "cmd_cut");
		can_paste = ephy_command_manager_can_do_command
				(EPHY_COMMAND_MANAGER (embed), "cmd_paste");
		can_undo = ephy_command_manager_can_do_command
				(EPHY_COMMAND_MANAGER (embed), "cmd_undo");
		can_redo = ephy_command_manager_can_do_command
				(EPHY_COMMAND_MANAGER (embed), "cmd_redo");
	}

	action_group = window->priv->action_group;

	action = gtk_action_group_get_action (action_group, "EditCopy");
	g_object_set (action, "sensitive", can_copy, NULL);
	action = gtk_action_group_get_action (action_group, "EditCut");
	g_object_set (action, "sensitive", can_cut, NULL);
	action = gtk_action_group_get_action (action_group, "EditPaste");
	g_object_set (action, "sensitive", can_paste, NULL);
	action = gtk_action_group_get_action (action_group, "EditUndo");
	g_object_set (action, "sensitive", can_undo, NULL);
	action = gtk_action_group_get_action (action_group, "EditRedo");
	g_object_set (action, "sensitive", can_redo, NULL);
}

static void
edit_menu_show_cb (GtkWidget *menu,
		   EphyWindow *window)
{
	update_edit_actions_sensitivity (window);
}

static void
edit_menu_hide_cb (GtkWidget *menu,
		   EphyWindow *window)
{
	GtkActionGroup *action_group;
	GtkAction *action;

	action_group = window->priv->action_group;

	action = gtk_action_group_get_action (action_group, "EditCopy");
	g_object_set (action, "sensitive", TRUE, NULL);
	action = gtk_action_group_get_action (action_group, "EditCut");
	g_object_set (action, "sensitive", TRUE, NULL);
	action = gtk_action_group_get_action (action_group, "EditUndo");
	g_object_set (action, "sensitive", TRUE, NULL);
	action = gtk_action_group_get_action (action_group, "EditRedo");
	g_object_set (action, "sensitive", TRUE, NULL);
}

static void
init_menu_updaters (EphyWindow *window)
{
	GtkWidget *edit_menu_item, *edit_menu;

	edit_menu_item = gtk_ui_manager_get_widget
		(GTK_UI_MANAGER (window->ui_merge), "/menubar/EditMenu");
	edit_menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (edit_menu_item));

	g_signal_connect (edit_menu, "show",
			  G_CALLBACK (edit_menu_show_cb), window);
	g_signal_connect (edit_menu, "hide",
			  G_CALLBACK (edit_menu_hide_cb), window);
}

static void
setup_window (EphyWindow *window)
{
	GtkActionGroup *action_group;
	GtkAction *action;
	GtkUIManager *merge;
	GError *err = NULL;

	window->priv->main_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (window->priv->main_vbox);
	gtk_container_add (GTK_CONTAINER (window),
			   window->priv->main_vbox);

	window->priv->menu_dock = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (window->priv->menu_dock);
	gtk_box_pack_start (GTK_BOX (window->priv->main_vbox),
			    GTK_WIDGET (window->priv->menu_dock),
			    FALSE, TRUE, 0);

	merge = gtk_ui_manager_new ();

	action_group = gtk_action_group_new ("WindowActions");
	gtk_action_group_set_translation_domain (action_group, NULL);
	gtk_action_group_add_actions (action_group, ephy_menu_entries,
				      ephy_menu_n_entries, window);
	gtk_action_group_add_toggle_actions (action_group,
					     ephy_menu_toggle_entries,
					     ephy_menu_n_toggle_entries,
					     window);
	gtk_ui_manager_insert_action_group (merge, action_group, 0);
	window->priv->action_group = action_group;
	action = gtk_action_group_get_action (action_group, "FileOpen");
	g_object_set (action, "short_label", _("Open"), NULL);
	action = gtk_action_group_get_action (action_group, "FileSaveAs");
	g_object_set (action, "short_label", _("Save As"), NULL);
	action = gtk_action_group_get_action (action_group, "FilePrint");
	g_object_set (action, "short_label", _("Print"), NULL);
	action = gtk_action_group_get_action (action_group, "FileBookmarkPage");
	g_object_set (action, "short_label", _("Bookmark"), NULL);
	action = gtk_action_group_get_action (action_group, "EditFind");
	g_object_set (action, "short_label", _("Find"), NULL);
	action = gtk_action_group_get_action (action_group, "GoBookmarks");
	g_object_set (action, "short_label", _("Bookmarks"), NULL);

	action = gtk_action_group_get_action (action_group, "EditFind");
	g_object_set (action, "is_important", TRUE, NULL);
	action = gtk_action_group_get_action (action_group, "GoHome");
	g_object_set (action, "is_important", TRUE, NULL);
	action = gtk_action_group_get_action (action_group, "GoBookmarks");
	g_object_set (action, "is_important", TRUE, NULL);

	action = gtk_action_group_get_action (action_group, "ViewEncoding");
	g_object_set (action, "hide_if_empty", FALSE, NULL);

	action_group = gtk_action_group_new ("PopupsActions");
	gtk_action_group_set_translation_domain (action_group, NULL);
	gtk_action_group_add_actions (action_group, ephy_popups_entries,
				      ephy_popups_n_entries, window);
	gtk_ui_manager_insert_action_group (merge, action_group, 0);
	window->priv->popups_action_group = action_group;

	window->ui_merge = G_OBJECT (merge);
	g_signal_connect (merge, "add_widget", G_CALLBACK (add_widget), window);
	gtk_window_add_accel_group (GTK_WINDOW (window),
				    gtk_ui_manager_get_accel_group (merge));

	gtk_ui_manager_add_ui_from_file
		(merge, ephy_file ("epiphany-ui.xml"), &err);
	if (err != NULL)
	{
		g_warning ("Could not merge epiphany-ui.xml: %s", err->message);
		g_clear_error (&err);
        }

	gtk_ui_manager_ensure_update (GTK_UI_MANAGER (window->ui_merge));

	init_menu_updaters (window);

	window->priv->toolbar = toolbar_new (window);
	gtk_widget_show (GTK_WIDGET (window->priv->toolbar));
	gtk_box_pack_end (GTK_BOX (window->priv->menu_dock),
			  GTK_WIDGET (window->priv->toolbar),
			  FALSE, FALSE, 0);
	g_signal_connect (window,
			  "selection-received",
			  G_CALLBACK (ephy_window_selection_received_cb),
			  window);
	g_signal_connect (window, "window-state-event",
			  G_CALLBACK (ephy_window_state_event_cb),
			  window);
}

static void
sync_tab_address (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	const char *address;

	if (window->priv->closing) return;

	address = ephy_tab_get_location (tab);

	if (address == NULL)
	{
		address = "";
	}

	toolbar_set_location (window->priv->toolbar, address);
}

static void
sync_tab_icon (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	const char *address;
	EphyFaviconCache *cache;
	GdkPixbuf *pixbuf = NULL;

	if (window->priv->closing) return;

	cache = EPHY_FAVICON_CACHE
		(ephy_embed_shell_get_favicon_cache
			(EPHY_EMBED_SHELL (ephy_shell)));

	address = ephy_tab_get_icon_address (tab);

	if (address)
	{
		pixbuf = ephy_favicon_cache_get (cache, address);
	}

	gtk_window_set_icon (GTK_WINDOW (window), pixbuf);

	toolbar_update_favicon (window->priv->toolbar);

	if (pixbuf)
	{
		g_object_unref (pixbuf);
	}
}

static void
sync_tab_load_progress (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	if (window->priv->closing) return;

	statusbar_set_progress (EPHY_STATUSBAR (window->priv->statusbar),
				ephy_tab_get_load_percent (tab));
}

static void
sync_tab_message (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	if (window->priv->closing) return;

	statusbar_set_message (EPHY_STATUSBAR (window->priv->statusbar),
			       ephy_tab_get_status_message (tab));
}

static void
sync_tab_navigation (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	TabNavigationFlags flags;
	GtkActionGroup *action_group;
	GtkAction *action;
	gboolean up = FALSE, back = FALSE, forward = FALSE;

	if (window->priv->closing) return;

	flags = ephy_tab_get_navigation_flags (tab);

	if (flags & TAB_NAV_UP)
	{
		up = TRUE;
	}
	if (flags & TAB_NAV_BACK)
	{
		back = TRUE;
	}
	if (flags & TAB_NAV_FORWARD)
	{
		forward = TRUE;
	}

	action_group = window->priv->action_group;
	action = gtk_action_group_get_action (action_group, "GoUp");
	g_object_set (action, "sensitive", up, NULL);
	action = gtk_action_group_get_action (action_group, "GoBack");
	g_object_set (action, "sensitive", back, NULL);
	action = gtk_action_group_get_action (action_group, "GoForward");
	g_object_set (action, "sensitive", forward, NULL);

	toolbar_update_navigation_actions (window->priv->toolbar,
					   back, forward, up);
}

static void
sync_tab_security (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	EphyEmbed *embed;
	EmbedSecurityLevel level;
	char *description = NULL;
	char *state = NULL;
	gboolean secure;
	char *tooltip;

	if (window->priv->closing) return;

	embed = ephy_tab_get_embed (tab);

	ephy_embed_get_security_level (embed, &level, &description);

	if (level != ephy_tab_get_security_level (tab))
	{
		/* something is VERY wrong here! */
		level = STATE_IS_UNKNOWN;
		description = NULL;
	}

	secure = FALSE;
	switch (level)
	{
		case STATE_IS_UNKNOWN:
			state = _("Unknown");
			break;
		case STATE_IS_INSECURE:
			state = _("Insecure");
			break;
		case STATE_IS_BROKEN:
			state = _("Broken");
			break;
		case STATE_IS_SECURE_MED:
			state = _("Medium");
			secure = TRUE;
			break;
		case STATE_IS_SECURE_LOW:
			state = _("Low");
			secure = TRUE;
			break;
		case STATE_IS_SECURE_HIGH:
			state = _("High");
			secure = TRUE;
			break;
		default:
			g_assert_not_reached ();
			break;
	}

	if (description != NULL)
	{
		tooltip = g_strdup_printf (_("Security level: %s\n%s"),
					   state, description);
		g_free (description);
	}
	else
	{
		tooltip = g_strdup_printf (_("Security level: %s"), state);

	}

	statusbar_set_security_state (EPHY_STATUSBAR (window->priv->statusbar),
			              secure, tooltip);
	g_free (tooltip);

}

static void
sync_tab_load_status (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	GtkAction *action;
	gboolean status;

	if (window->priv->closing) return;

	action = gtk_action_group_get_action (window->priv->action_group, "ViewStop");

	status = ephy_tab_get_load_status (tab);
	g_object_set (action, "sensitive", status, NULL);

	if (status)
	{
		toolbar_spinner_start (window->priv->toolbar);
	}
	else
	{
		toolbar_spinner_stop (window->priv->toolbar);
	}
}

static void
sync_tab_title (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	const char *title;

	if (window->priv->closing) return;

	title = ephy_tab_get_title (tab);

	if (title)
	{
		gtk_window_set_title (GTK_WINDOW(window),
				      title);
	}
}

static void
sync_tab_visibility (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	GList *l, *tabs;
	gboolean visible = FALSE;

	if (window->priv->closing) return;

	tabs = ephy_window_get_tabs (window);
	for (l = tabs; l != NULL; l = l->next)
	{
		EphyTab *tab = EPHY_TAB(l->data);
		g_return_if_fail (EPHY_IS_TAB(tab));

		if (ephy_tab_get_visibility (tab))
		{
			visible = TRUE;
			break;
		}
	}
	g_list_free (tabs);

	if (visible)
	{
		gtk_widget_show (GTK_WIDGET(window));
	}
	else
	{
		gtk_widget_hide (GTK_WIDGET (window));
	}
}

static void
sync_tab_zoom (EphyTab *tab, GParamSpec *pspec, EphyWindow *window)
{
	GtkActionGroup *action_group;
	GtkAction *action;
	gboolean can_zoom_in = TRUE, can_zoom_out = TRUE, can_zoom_normal = FALSE;
	float zoom;

	if (window->priv->closing) return;

	zoom = ephy_tab_get_zoom (tab);

	if (zoom >= ZOOM_MAXIMAL)
	{
		can_zoom_in = FALSE;
	}
	if (zoom <= ZOOM_MINIMAL)
	{
		can_zoom_out = FALSE;
	}
	if (zoom != 1.0)
	{
		can_zoom_normal = TRUE;
	}

	toolbar_update_zoom (window->priv->toolbar, zoom);

	action_group = window->priv->action_group;
	action = gtk_action_group_get_action (action_group, "ViewZoomIn");
	g_object_set (action, "sensitive", can_zoom_in, NULL);
	action = gtk_action_group_get_action (action_group, "ViewZoomOut");
	g_object_set (action, "sensitive", can_zoom_out, NULL);
	action = gtk_action_group_get_action (action_group, "ViewZoomNormal");
	g_object_set (action, "sensitive", can_zoom_normal, NULL);
}

static void
popup_menu_at_coords (GtkMenu *menu, gint *x, gint *y, gboolean *push_in,
		      gpointer user_data)
{
	EphyEmbedEvent *event = user_data;

	*x = event->x;
	*y = event->y;
	*push_in = TRUE;
}

static void
show_embed_popup (EphyWindow *window, EphyTab *tab, EphyEmbedEvent *event)
{
	GtkActionGroup *action_group;
	GtkAction *action;
	EmbedEventContext context;
	const char *popup;
	const GValue *value;
	gboolean framed, has_background;
	GtkWidget *widget;
	EphyEmbedEventType type;

	/* Do not show the menu in print preview mode */
	if (window->priv->is_ppview)
	{
		return;
	}

	ephy_embed_event_get_property (event, "framed_page", &value);
	framed = g_value_get_int (value);

	has_background = ephy_embed_event_has_property (event, "background_image");

	context = ephy_embed_event_get_context (event);

	if ((context & EMBED_CONTEXT_LINK) &&
	    (context & EMBED_CONTEXT_IMAGE))
	{
		popup = "/EphyImageLinkPopup";
	}
	else if (context & EMBED_CONTEXT_LINK)
	{
		popup = "/EphyLinkPopup";
	}
	else if (context & EMBED_CONTEXT_IMAGE)
	{
		popup = "/EphyImagePopup";
	}
#if MOZILLA_SNAPSHOT > 12
	else if (context & EMBED_CONTEXT_INPUT)
	{
		update_edit_actions_sensitivity (window);
		popup = "/EphyInputPopup";
	}
#endif
	else
	{
		popup = framed ? "/EphyFramedDocumentPopup" :
				 "/EphyDocumentPopup";
	}

	action_group = window->priv->popups_action_group;
	action = gtk_action_group_get_action (action_group, "SaveBackgroundAs");
	g_object_set (action, "sensitive", has_background,
			      "visible", has_background, NULL);

	widget = gtk_ui_manager_get_widget (GTK_UI_MANAGER (window->ui_merge),
				            popup);
	g_return_if_fail (widget != NULL);

	g_object_set_data_full (G_OBJECT (window), "context_event",
				g_object_ref (event),
				(GDestroyNotify)g_object_unref);

	type = ephy_embed_event_get_event_type (event);
	if (type == EPHY_EMBED_EVENT_KEY)
	{
		gtk_menu_popup (GTK_MENU (widget), NULL, NULL,
				popup_menu_at_coords, event, 2,
				gtk_get_current_event_time ());
	}
	else
	{
		gtk_menu_popup (GTK_MENU (widget), NULL, NULL,
				NULL, NULL, 2,
				gtk_get_current_event_time ());
	}
}

static gboolean
tab_context_menu_cb (EphyEmbed *embed,
		     EphyEmbedEvent *event,
		     EphyWindow *window)
{
	EphyTab *tab;

	g_return_val_if_fail (EPHY_IS_WINDOW (window), FALSE);
	g_return_val_if_fail (EPHY_IS_EMBED (embed), FALSE);
	g_return_val_if_fail (EPHY_IS_EMBED_EVENT(event), FALSE);

	tab = EPHY_TAB (g_object_get_data (G_OBJECT (embed), "EphyTab"));
	g_return_val_if_fail (EPHY_IS_TAB (tab), FALSE);
	g_return_val_if_fail (window->priv->active_tab == tab, FALSE);

	window = ephy_tab_get_window (tab);
	g_return_val_if_fail (window != NULL, FALSE);

	show_embed_popup (window, tab, event);

	return FALSE;
}

static void
ephy_window_set_active_tab (EphyWindow *window, EphyTab *new_tab)
{
	EphyTab *old_tab;
	EphyEmbed *embed;
	GtkToggleAction *action;

	g_return_if_fail (EPHY_IS_WINDOW (window));
	if (ephy_tab_get_window (new_tab) != window) return;

	old_tab = window->priv->active_tab;

	if (old_tab == new_tab) return;

	if (old_tab)
	{
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_address),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_icon),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_load_progress),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_load_status),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_message),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_navigation),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_security),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_title),
						      window);
		g_signal_handlers_disconnect_by_func (G_OBJECT (old_tab),
						      G_CALLBACK (sync_tab_zoom),
						      window);

		embed = ephy_tab_get_embed (old_tab);
		g_signal_handlers_disconnect_by_func (G_OBJECT (embed),
						      G_CALLBACK (tab_context_menu_cb),
						      window);

		action = GTK_TOGGLE_ACTION (ephy_tab_get_action (old_tab));
		gtk_toggle_action_set_active (action, FALSE);
	}

	window->priv->active_tab = new_tab;

	if (new_tab)
	{
		sync_tab_address	(new_tab, NULL, window);
		sync_tab_icon		(new_tab, NULL, window);
		sync_tab_load_progress	(new_tab, NULL, window);
		sync_tab_load_status	(new_tab, NULL, window);
		sync_tab_message	(new_tab, NULL, window);
		sync_tab_navigation	(new_tab, NULL, window);
		sync_tab_security	(new_tab, NULL, window);
		sync_tab_title		(new_tab, NULL, window);
		sync_tab_zoom		(new_tab, NULL, window);

		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::address",
					 G_CALLBACK (sync_tab_address),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::icon",
					 G_CALLBACK (sync_tab_icon),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::load-progress",
					 G_CALLBACK (sync_tab_load_progress),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::load-status",
					 G_CALLBACK (sync_tab_load_status),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::message",
					 G_CALLBACK (sync_tab_message),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::navigation",
					 G_CALLBACK (sync_tab_navigation),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::security-level",
					 G_CALLBACK (sync_tab_security),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::title",
					 G_CALLBACK (sync_tab_title),
					 window, 0);
		g_signal_connect_object (G_OBJECT (new_tab),
					 "notify::zoom",
					 G_CALLBACK (sync_tab_zoom),
					 window, 0);

		embed = ephy_tab_get_embed (new_tab);
		g_signal_connect_object (embed, "ge_context_menu",
					 G_CALLBACK (tab_context_menu_cb),
					 window, 0);

		action = GTK_TOGGLE_ACTION (ephy_tab_get_action (new_tab));
		gtk_toggle_action_set_active (action, TRUE);

		g_object_notify (G_OBJECT (window), "active-tab");
	}
}

static void
update_tabs_menu_sensitivity (EphyWindow *window)
{
	gboolean prev_tab, next_tab, move_left, move_right, detach;
	GtkActionGroup *action_group;
	GtkAction *action;
	int current;
	int last;

	current = gtk_notebook_get_current_page
		(GTK_NOTEBOOK (window->priv->notebook));
	last = gtk_notebook_get_n_pages
		(GTK_NOTEBOOK (window->priv->notebook)) - 1;
	prev_tab = move_left = (current > 0);
	next_tab = move_right = (current < last);
	detach = gtk_notebook_get_n_pages
		(GTK_NOTEBOOK (window->priv->notebook)) > 1;

	action_group = window->priv->action_group;
	action = gtk_action_group_get_action (action_group, "TabsPrevious");
	g_object_set (action, "sensitive", prev_tab, NULL);
	action = gtk_action_group_get_action (action_group, "TabsNext");
	g_object_set (action, "sensitive", next_tab, NULL);
	action = gtk_action_group_get_action (action_group, "TabsMoveLeft");
	g_object_set (action, "sensitive", move_left, NULL);
	action = gtk_action_group_get_action (action_group, "TabsMoveRight");
	g_object_set (action, "sensitive", move_right, NULL);
	action = gtk_action_group_get_action (action_group, "TabsDetach");
	g_object_set (action, "sensitive", detach, NULL);
}

static void
tab_added_cb (EphyNotebook *notebook, GtkWidget *child, EphyWindow *window)
{
	EphyTab *tab;

	g_return_if_fail (EPHY_IS_EMBED (child));
	tab = EPHY_TAB (g_object_get_data (G_OBJECT (child), "EphyTab"));

	window->priv->num_tabs++;

	update_tabs_menu_sensitivity (window);

	g_signal_connect_object (G_OBJECT (tab), "notify::visible",
				 G_CALLBACK (sync_tab_visibility), window, 0);
}

static void
tab_removed_cb (EphyNotebook *notebook, GtkWidget *child, EphyWindow *window)
{
	EphyTab *tab;

	g_return_if_fail (EPHY_IS_EMBED (child));
	tab = EPHY_TAB (g_object_get_data (G_OBJECT (child), "EphyTab"));

	g_signal_handlers_disconnect_by_func (G_OBJECT (tab),
					      G_CALLBACK (sync_tab_visibility),
					      window);	

	window->priv->num_tabs--;

	if (window->priv->num_tabs == 0)
	{
		/* removed the last tab, close the window */
		gtk_widget_destroy (GTK_WIDGET (window));
	}
	else
	{
		update_tabs_menu_sensitivity (window);
	}
}

static void
tab_detached_cb (EphyNotebook *notebook, GtkWidget *child,
		 gpointer data)
{
	EphyWindow *window;

	g_return_if_fail (EPHY_IS_NOTEBOOK (notebook));
	g_return_if_fail (EPHY_IS_EMBED (child));

	window = ephy_window_new ();
	ephy_notebook_move_page (notebook,
				 EPHY_NOTEBOOK (ephy_window_get_notebook (window)),
				 child, 0);
	gtk_widget_show (GTK_WIDGET (window));
}

static void
tabs_reordered_cb (EphyNotebook *notebook, EphyWindow *window)
{
	update_tabs_menu_sensitivity (window);
	ephy_tabs_menu_update (window->priv->tabs_menu);
}

static GtkNotebook *
setup_notebook (EphyWindow *window)
{
	GtkNotebook *notebook;

	notebook = GTK_NOTEBOOK (ephy_notebook_new ());
	gtk_notebook_set_scrollable (notebook, TRUE);
	gtk_notebook_set_show_border (notebook, FALSE);
	gtk_notebook_set_show_tabs (notebook, FALSE);

	g_signal_connect_after (G_OBJECT (notebook), "switch_page",
				G_CALLBACK (
				ephy_window_notebook_switch_page_cb),
				window);

	g_signal_connect (G_OBJECT (notebook), "tab_added",
			  G_CALLBACK (tab_added_cb), window);
	g_signal_connect (G_OBJECT (notebook), "tab_removed",
			  G_CALLBACK (tab_removed_cb), window);
	g_signal_connect (G_OBJECT (notebook), "tab_detached",
			  G_CALLBACK (tab_detached_cb), NULL);
	g_signal_connect (G_OBJECT (notebook), "tabs_reordered",
			  G_CALLBACK (tabs_reordered_cb), window);

	gtk_widget_show (GTK_WIDGET (notebook));

	return notebook;
}

static void
ephy_window_set_property (GObject *object,
			  guint prop_id,
			  const GValue *value,
			  GParamSpec *pspec)
{
	EphyWindow *window = EPHY_WINDOW (object);

	switch (prop_id)
	{
		case PROP_ACTIVE_TAB:
			ephy_window_set_active_tab (window, g_value_get_object (value));
			break;
	}
}

static void
ephy_window_get_property (GObject *object,
			  guint prop_id,
			  GValue *value,
			  GParamSpec *pspec)
{
	EphyWindow *window = EPHY_WINDOW (object);

	switch (prop_id)
	{
		case PROP_ACTIVE_TAB:
			g_value_set_object (value, window->priv->active_tab);
			break;
	}
}

static void
ephy_window_class_init (EphyWindowClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtkobject_class = GTK_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

        parent_class = g_type_class_peek_parent (klass);

        object_class->finalize = ephy_window_finalize;
	object_class->get_property = ephy_window_get_property;
	object_class->set_property = ephy_window_set_property;

	gtkobject_class->destroy = ephy_window_destroy;

	widget_class->show = ephy_window_show;

	g_object_class_install_property (object_class,
					 PROP_ACTIVE_TAB,
					 g_param_spec_object ("active-tab",
							      "active-tab",
							      "Active tab",
							      EPHY_TYPE_TAB,
							      G_PARAM_READWRITE));

	g_type_class_add_private (object_class, sizeof(EphyWindowPrivate));
}

static void
ensure_default_icon (void)
{
	static gboolean ephy_has_default_icon = FALSE;
	GtkIconTheme *icon_theme;
	GtkIconInfo *icon_info;
	const char *icon_file;

	if (ephy_has_default_icon) return;

	/* FIXME listen on icon changes */
	/* FIXME MultiHead: icon theme is per-display, not global */
	icon_theme = gtk_icon_theme_get_default ();
	icon_info = gtk_icon_theme_lookup_icon (icon_theme, "web-browser", -1, 0);

	if (icon_info)
	{

		icon_file = gtk_icon_info_get_filename (icon_info);
		if (icon_file)
		{
			gtk_window_set_default_icon_from_file (icon_file, NULL);
		}

		gtk_icon_info_free (icon_info);
	}
	else
	{
		g_warning ("Web browser gnome icon not found");
	}

	ephy_has_default_icon = TRUE;
}

static void
update_layout_toggles (EphyWindow *window)
{
	GtkActionGroup *action_group = GTK_ACTION_GROUP (window->priv->action_group);
	GtkAction *action;

	action = gtk_action_group_get_action (action_group, "ViewToolbar");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action),
				      eel_gconf_get_boolean (CONF_WINDOWS_SHOW_TOOLBARS));

	action = gtk_action_group_get_action (action_group, "ViewBookmarksBar");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action),
				      eel_gconf_get_boolean (CONF_WINDOWS_SHOW_BOOKMARKS_BAR));

	action = gtk_action_group_get_action (action_group, "ViewStatusbar");
	gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action),
				      eel_gconf_get_boolean (CONF_WINDOWS_SHOW_STATUSBAR));
}

static void
chrome_notifier (GConfClient *client,
		 guint cnxn_id,
		 GConfEntry *entry,
		 EphyWindow *window)
{
	update_chrome (window);
	update_layout_toggles(window);
}

static void
ephy_window_init (EphyWindow *window)
{
	EphyExtension *manager;

	LOG ("EphyWindow initialising %p", window)

	window->priv = EPHY_WINDOW_GET_PRIVATE (window);

	window->priv->active_tab = NULL;
	window->priv->requested_chrome_mask = EMBED_CHROME_DEFAULT;
	window->priv->actual_chrome_mask = 0;
	window->priv->closing = FALSE;
	window->priv->ppview_toolbar = NULL;
	window->priv->exit_fullscreen_popup = NULL;
	window->priv->num_tabs = 0;
	window->priv->is_fullscreen = FALSE;
	window->priv->is_ppview = FALSE;
	window->priv->has_size = FALSE;

	ensure_default_icon ();

	/* Setup the window and connect verbs */
	setup_window (window);

	/* Setup window contents */
	window->priv->notebook = setup_notebook (window);
	gtk_box_pack_start (GTK_BOX (window->priv->main_vbox),
			    GTK_WIDGET (window->priv->notebook),
			    TRUE, TRUE, 0);

	window->priv->statusbar = statusbar_new ();
	gtk_widget_show (window->priv->statusbar);
	gtk_box_pack_start (GTK_BOX (window->priv->main_vbox),
			    GTK_WIDGET (window->priv->statusbar),
			    FALSE, TRUE, 0);

	g_object_ref (ephy_shell);

	/* Initialize the menus */
	window->priv->tabs_menu = ephy_tabs_menu_new (window);
	window->priv->fav_menu = ephy_favorites_menu_new (window);
	window->priv->enc_menu = ephy_encoding_menu_new (window);
	window->priv->bmk_menu = ephy_bookmarks_menu_new (window);

	window->priv->disable_js_chrome_notifier_id = eel_gconf_notification_add
		(CONF_LOCKDOWN_DISABLE_JAVASCRIPT_CHROME,
		 (GConfClientNotifyFunc)chrome_notifier, window);

	window->priv->show_toolbars_notifier_id = eel_gconf_notification_add
		(CONF_WINDOWS_SHOW_TOOLBARS,
		 (GConfClientNotifyFunc)chrome_notifier, window);

	window->priv->show_bookmarks_bar_notifier_id = eel_gconf_notification_add
		(CONF_WINDOWS_SHOW_BOOKMARKS_BAR,
		 (GConfClientNotifyFunc)chrome_notifier, window);

	window->priv->show_statusbar_notifier_id = eel_gconf_notification_add
		(CONF_WINDOWS_SHOW_STATUSBAR,
		 (GConfClientNotifyFunc)chrome_notifier, window);

	window->priv->hide_menubar_notifier_id = eel_gconf_notification_add
		(CONF_LOCKDOWN_HIDE_MENUBAR,
		 (GConfClientNotifyFunc)chrome_notifier, window);

	/* Once the window is fully created let the extensions attach to it */
	manager = EPHY_EXTENSION (ephy_shell_get_extensions_manager (ephy_shell));
	ephy_extension_attach_window (manager, window);
}

static void
ephy_window_finalize (GObject *object)
{
        EphyWindow *window = EPHY_WINDOW (object);

	eel_gconf_notification_remove (window->priv->disable_js_chrome_notifier_id);
	eel_gconf_notification_remove (window->priv->show_toolbars_notifier_id);
	eel_gconf_notification_remove (window->priv->show_bookmarks_bar_notifier_id);
	eel_gconf_notification_remove (window->priv->show_statusbar_notifier_id);
	eel_gconf_notification_remove (window->priv->hide_menubar_notifier_id);

	if (window->priv->find_dialog)
	{
		g_object_unref (G_OBJECT (window->priv->find_dialog));
	}

	if (window->priv->print_dialog)
	{
		g_object_unref (G_OBJECT (window->priv->print_dialog));
	}

	g_object_unref (window->priv->fav_menu);
	g_object_unref (window->priv->enc_menu);
	g_object_unref (window->priv->tabs_menu);
	g_object_unref (window->priv->bmk_menu);

	if (window->priv->ppview_toolbar)
	{
		g_object_unref (window->priv->ppview_toolbar);
	}

	g_object_unref (window->priv->action_group);
	g_object_unref (window->ui_merge);

        G_OBJECT_CLASS (parent_class)->finalize (object);

	LOG ("Ephy Window finalized %p", window)

	g_object_unref (ephy_shell);
}

EphyWindow *
ephy_window_new (void)
{
	return EPHY_WINDOW (g_object_new (EPHY_TYPE_WINDOW, NULL));
}

void
ephy_window_request_chrome (EphyWindow *window,
			    EmbedChromeMask flags)
{
	window->priv->requested_chrome_mask = flags;
	update_chrome (window);
}

void
ephy_window_set_print_preview (EphyWindow *window, gboolean enabled)
{
	window->priv->is_ppview = enabled;
	update_chrome (window);
	ephy_notebook_set_show_tabs (EPHY_NOTEBOOK (window->priv->notebook), !enabled);
}

GtkWidget *
ephy_window_get_notebook (EphyWindow *window)
{
	g_return_val_if_fail (EPHY_IS_WINDOW (window), NULL);

	return GTK_WIDGET (window->priv->notebook);
}

void
ephy_window_add_tab (EphyWindow *window,
		     EphyTab *tab,
		     gint position,
		     gboolean jump_to)
{
	GtkWidget *widget;

	g_return_if_fail (EPHY_IS_WINDOW (window));
	g_return_if_fail (EPHY_IS_TAB (tab));

	widget = GTK_WIDGET(ephy_tab_get_embed (tab));

	ephy_notebook_insert_page (EPHY_NOTEBOOK (window->priv->notebook),
				   widget, position, jump_to);
}

void
ephy_window_jump_to_tab (EphyWindow *window,
			   EphyTab *tab)
{
	GtkWidget *widget;
	int page;

	widget = GTK_WIDGET(ephy_tab_get_embed (tab));

	page = gtk_notebook_page_num
		(window->priv->notebook, widget);
	gtk_notebook_set_current_page
		(window->priv->notebook, page);
}

static EphyTab *
real_get_active_tab (EphyWindow *window, int page_num)
{
	EphyTab *tab;
	GtkWidget *embed_widget;

	if (page_num == -1)
	{
		page_num = gtk_notebook_get_current_page (window->priv->notebook);
	}
	embed_widget = gtk_notebook_get_nth_page (window->priv->notebook,
						  page_num);

	g_return_val_if_fail (GTK_IS_WIDGET (embed_widget), NULL);
	tab = g_object_get_data (G_OBJECT (embed_widget), "EphyTab");
	g_return_val_if_fail (EPHY_IS_TAB (tab), NULL);

	return tab;
}

void
ephy_window_remove_tab (EphyWindow *window,
		        EphyTab *tab)
{
	GtkWidget *embed;

	g_return_if_fail (EPHY_IS_WINDOW (window));
	g_return_if_fail (EPHY_IS_TAB (tab));

	embed = GTK_WIDGET (ephy_tab_get_embed (tab));

	ephy_notebook_remove_page (EPHY_NOTEBOOK (window->priv->notebook),
				  embed);
}

/**
 * ephy_window_load_url:
 * @window: a #EphyWindow
 * @url: the url to load
 *
 * Load a new url in the active tab of the window.
 * Unlike ephy_embed_load_url this function activate
 * the embed.
 *
 **/
void
ephy_window_load_url (EphyWindow *window,
			const char *url)
{
	EphyEmbed *embed;

        g_return_if_fail (EPHY_IS_WINDOW(window));
	embed = ephy_window_get_active_embed (window);
        g_return_if_fail (embed != NULL);
        g_return_if_fail (url != NULL);

        ephy_embed_load_url (embed, url);
	ephy_embed_activate (embed);
}

void ephy_window_activate_location (EphyWindow *window)
{
	toolbar_activate_location (window->priv->toolbar);
}

void
ephy_window_show (GtkWidget *widget)
{
	EphyWindow *window = EPHY_WINDOW(widget);

	update_chrome (window);
	update_layout_toggles(window);

	if (!window->priv->has_size)
	{
		gboolean keep_state = TRUE;

		/* Do not keep state of sized popups */
		if (window->priv->actual_chrome_mask & EMBED_CHROME_OPENASPOPUP)
		{
			EphyTab *tab;
			int width, height;

			tab = ephy_window_get_active_tab (EPHY_WINDOW (window));
			g_return_if_fail (tab != NULL);

			ephy_tab_get_size (tab, &width, &height);
			if (width != -1 || height != -1)
			{
				keep_state = FALSE;
			}
		}

		if (keep_state)
		{
			ephy_state_add_window (widget, "main_window", 600, 500,
					       EPHY_STATE_WINDOW_SAVE_SIZE);
		}

		window->priv->has_size = TRUE;
	}

	GTK_WIDGET_CLASS (parent_class)->show (widget);
}

EphyTab *
ephy_window_get_active_tab (EphyWindow *window)
{
	g_return_val_if_fail (EPHY_IS_WINDOW (window), NULL);

	return window->priv->active_tab;
}

EphyEmbed *
ephy_window_get_active_embed (EphyWindow *window)
{
	EphyTab *tab;

	g_return_val_if_fail (EPHY_IS_WINDOW (window), NULL);

	tab = ephy_window_get_active_tab (window);

	if (tab)
	{
		return ephy_tab_get_embed (tab);
	}

	return NULL;
}

GList *
ephy_window_get_tabs (EphyWindow *window)
{
	GList *tabs = NULL;
	GtkWidget *w;
	int i = 0;

	while ((w = gtk_notebook_get_nth_page (window->priv->notebook, i)) != NULL)
	{
		EphyTab *tab;

		tab = g_object_get_data (G_OBJECT (w), "EphyTab");
		g_return_val_if_fail (EPHY_IS_TAB (G_OBJECT (tab)), NULL);

		tabs = g_list_prepend (tabs, tab);
		i++;
	}

	return g_list_reverse (tabs);
}

static void
update_embed_dialogs (EphyWindow *window,
		      EphyTab *tab)
{
	EphyEmbed *embed;
	EphyDialog *find_dialog = window->priv->find_dialog;
	EphyDialog *print_dialog = window->priv->print_dialog;

	embed = ephy_tab_get_embed (tab);

	if (find_dialog)
	{
		ephy_embed_dialog_set_embed
			(EPHY_EMBED_DIALOG(find_dialog),
			 embed);
	}

	if (print_dialog)
	{
		ephy_embed_dialog_set_embed
			(EPHY_EMBED_DIALOG(print_dialog),
			 embed);
	}
}

static void
ephy_window_notebook_switch_page_cb (GtkNotebook *notebook,
				     GtkNotebookPage *page,
				     guint page_num,
				     EphyWindow *window)
{
	EphyTab *tab;

	g_return_if_fail (EPHY_IS_WINDOW (window));
	if (window->priv->closing) return;

	/* get the new tab */
	tab = real_get_active_tab (window, page_num);

	/* update new tab */
	ephy_window_set_active_tab (window, tab);

	update_embed_dialogs (window, tab);

	/* update window controls */
	update_tabs_menu_sensitivity (window);
}

void
ephy_window_find (EphyWindow *window)
{
	EphyDialog *dialog;
	EphyEmbed *embed;

	if (window->priv->find_dialog == NULL)
	{
		embed = ephy_window_get_active_embed (window);
		g_return_if_fail (GTK_IS_WINDOW(window));

		dialog = find_dialog_new_with_parent (GTK_WIDGET(window),
						      embed);
		window->priv->find_dialog = dialog;

		g_object_add_weak_pointer(G_OBJECT (dialog),
					  (gpointer *) &window->priv->find_dialog);
	}

	ephy_dialog_show (window->priv->find_dialog);
}

void
ephy_window_print (EphyWindow *window)
{
	if (window->priv->print_dialog == NULL)
	{
		EphyDialog *dialog;
		EphyEmbed *embed;

		embed = ephy_window_get_active_embed (window);
		dialog = ephy_print_dialog_new  (GTK_WIDGET (window), embed, FALSE);

		window->priv->print_dialog = dialog;
		g_object_add_weak_pointer(G_OBJECT (dialog),
					  (gpointer *) &window->priv->print_dialog);
	}

	ephy_dialog_show (window->priv->print_dialog);
}

void
ephy_window_set_zoom (EphyWindow *window,
		      float zoom)
{
	EphyEmbed *embed;
	float current_zoom = 1.0;

        g_return_if_fail (EPHY_IS_WINDOW (window));

	embed = ephy_window_get_active_embed (window);
        g_return_if_fail (embed != NULL);

	current_zoom = ephy_embed_zoom_get (embed);

	if (zoom == ZOOM_IN)
	{
		zoom = ephy_zoom_get_changed_zoom_level (current_zoom, 1);
	}
	else if (zoom == ZOOM_OUT)
	{
		zoom = ephy_zoom_get_changed_zoom_level (current_zoom, -1);
	}

	if (zoom != current_zoom)
	{
		ephy_embed_zoom_set (embed, zoom, TRUE);
	}
}

Toolbar *
ephy_window_get_toolbar (EphyWindow *window)
{
	return window->priv->toolbar;
}
