/*
 *  Copyright (C) 2001 Philip Langdale
 *  Copyright (C) 2003 Marco Pesenti Gritti
 *  Copyright (C) 2003 Xan Lopez
 *  Copyright (C) 2004 Christian Persch
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: ContentHandler.cpp,v 1.68 2004/11/18 13:50:46 chpe Exp $
 */

#include "mozilla-config.h"

#include "config.h"

#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkstock.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <glib/gi18n.h>

#include <nsMemory.h>
#include <nsIURL.h>
#include <nsILocalFile.h>
#include <nsIMIMEInfo.h>
#include <nsIInterfaceRequestorUtils.h>
#include <nsCExternalHandlerService.h>

#ifdef ALLOW_PRIVATE_API
#include <nsIServiceManager.h>
#endif

#include "ephy-prefs.h"
#include "ephy-embed-single.h"
#include "ephy-embed-shell.h"
#include "ephy-file-chooser.h"
#include "ephy-stock-icons.h"
#include "ephy-gui.h"
#include "ephy-debug.h"
#include "eel-gconf-extensions.h"

#include "ContentHandler.h"
#include "MozDownload.h"
#include "EphyUtils.h"

#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
GContentHandler::GContentHandler()
{
	LOG ("GContentHandler ctor (%p)", this)
}
#else
GContentHandler::GContentHandler() : mMimeType(nsnull)
{
	LOG ("GContentHandler ctor (%p)", this)
}
#endif

GContentHandler::~GContentHandler()
{
	LOG ("GContentHandler dtor (%p)", this)

#ifndef MOZ_NSIMIMEINFO_NSACSTRING_
	if (mMimeType)
	{
		nsMemory::Free (mMimeType);
	}
#endif
}

NS_IMPL_ISUPPORTS1(GContentHandler, nsIHelperAppLauncherDialog)

/* void show (in nsIHelperAppLauncher aLauncher, in nsISupports aContext); */
NS_IMETHODIMP GContentHandler::Show(nsIHelperAppLauncher *aLauncher,
				    nsISupports *aContext,
				    PRBool aForced)
{
	nsresult rv;
	EphyEmbedSingle *single;
	gboolean handled = FALSE;

	mContext = aContext;
	mLauncher = aLauncher;
	rv = Init ();
	NS_ENSURE_SUCCESS (rv, rv);

	single = EPHY_EMBED_SINGLE (ephy_embed_shell_get_embed_single (embed_shell));
#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
	g_signal_emit_by_name (single, "handle_content", mMimeType.get(),
			       mUrl.get(), &handled);
#else
	g_signal_emit_by_name (single, "handle_content", mMimeType,
			       mUrl.get(), &handled);
#endif

	if (!handled)
	{
		MIMEInitiateAction ();
	}
	else
	{
		mLauncher->Cancel ();
	}

	return NS_OK;
}

/* nsILocalFile promptForSaveToFile (in nsISupports aWindowContext, in wstring aDefaultFile, in wstring aSuggestedFileExtension); */
NS_IMETHODIMP GContentHandler::PromptForSaveToFile(
				    nsIHelperAppLauncher *aLauncher,			    
				    nsISupports *aWindowContext,
				    const PRUnichar *aDefaultFile,
				    const PRUnichar *aSuggestedFileExtension,
				    nsILocalFile **_retval)
{
	EphyFileChooser *dialog;
	gint response;
	char *filename = NULL;
	nsEmbedCString defaultFile;

	NS_UTF16ToCString (nsEmbedString (aDefaultFile),
			   NS_CSTRING_ENCODING_UTF8, defaultFile);

	if (mAction != CONTENT_ACTION_SAVEAS)
	{
		return BuildDownloadPath (defaultFile.get(), _retval);
	}

	nsCOMPtr<nsIDOMWindow> parentDOMWindow = do_GetInterface (aWindowContext);
	GtkWidget *parentWindow = GTK_WIDGET (EphyUtils::FindGtkParent (parentDOMWindow));

	dialog = ephy_file_chooser_new (_("Save"), parentWindow,
					GTK_FILE_CHOOSER_ACTION_SAVE,
					CONF_STATE_SAVE_DIR,
					EPHY_FILE_FILTER_ALL);
	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), defaultFile.get());

	/* FIXME: modal -- mozilla sucks! */
	do
	{
		g_free (filename);
		response = gtk_dialog_run (GTK_DIALOG (dialog));
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	} while (response == GTK_RESPONSE_ACCEPT
		 && !ephy_gui_confirm_overwrite_file (GTK_WIDGET (dialog), filename));

	if (response == GTK_RESPONSE_ACCEPT)
	{
		nsCOMPtr <nsILocalFile> destFile (do_CreateInstance(NS_LOCAL_FILE_CONTRACTID));
		NS_ENSURE_TRUE (destFile, NS_ERROR_FAILURE);

		destFile->InitWithNativePath (nsEmbedCString (filename));
		g_free (filename);

		NS_IF_ADDREF (*_retval = destFile);

		gtk_widget_destroy (GTK_WIDGET (dialog));

		return NS_OK;
	}
	else
	{
		gtk_widget_destroy (GTK_WIDGET (dialog));
		g_free (filename);

		return NS_ERROR_FAILURE;
	}
}

NS_METHOD GContentHandler::Init ()
{
	nsresult rv;

	NS_ENSURE_TRUE (mLauncher, NS_ERROR_FAILURE);

	nsCOMPtr<nsIMIMEInfo> MIMEInfo;
	mLauncher->GetMIMEInfo (getter_AddRefs(MIMEInfo));
	NS_ENSURE_TRUE (MIMEInfo, NS_ERROR_FAILURE);

#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
	rv = MIMEInfo->GetMIMEType (mMimeType);
#else
	rv = MIMEInfo->GetMIMEType (&mMimeType);
#endif

	nsCOMPtr<nsIURI> uri;
	mLauncher->GetSource (getter_AddRefs(uri));
	NS_ENSURE_TRUE (uri, NS_ERROR_FAILURE);
	
	uri->GetSpec (mUrl);

	return NS_OK;
}

static void
response_cb (GtkWidget *dialog,
	     int response,
	     GContentHandler *self)
{
	gtk_widget_destroy (dialog);

	if (response > 0)
	{
		self->mAction = (ContentAction) response;
	}
	else
	{
		self->mAction = CONTENT_ACTION_NONE;
	}

	self->MIMEDoAction ();
}

static void
release_cb (GContentHandler *data)
{
	NS_RELEASE (data);
}

NS_METHOD GContentHandler::MIMEConfirmAction ()
{
	GtkWidget *dialog;
	const char *action_label, *primary, *secondary;
	//int response;

	nsCOMPtr<nsIDOMWindow> parentDOMWindow = do_GetInterface (mContext);
	GtkWindow *parentWindow = GTK_WINDOW (EphyUtils::FindGtkParent(parentDOMWindow));

	action_label =  (mAction == CONTENT_ACTION_OPEN) ||
			(mAction == CONTENT_ACTION_OPEN_TMP) ?
			GTK_STOCK_OPEN : EPHY_STOCK_DOWNLOAD;

	if (mPermission == EPHY_MIME_PERMISSION_UNSAFE && mHelperApp)
	{
		primary = _("Download the unsafe file?");
		secondary = _("This type of file could potentially damage "
			      "your documents or invade your privacy. "
			      "It's not safe to open it directly. "
			      "You can save it instead.");
	}
	else if (mAction == CONTENT_ACTION_OPEN_TMP)
	{
		primary = _("Open the file in another application?");
		secondary = _("It's not possible to view this file type "
			      "directly in the browser. You can open it with "
			      "another application or save it.");
	}
	else
	{
		primary = _("Download the file?");
		secondary = _("It's not possible to view this file because "
			      "there is no application installed that can open"
			      " it. You can save it instead.");
	}

	dialog = gtk_message_dialog_new
		(parentWindow,
		 GTK_DIALOG_DESTROY_WITH_PARENT,
		 GTK_MESSAGE_WARNING,
		 GTK_BUTTONS_NONE,	
		 primary);

	gtk_message_dialog_format_secondary_text
		(GTK_MESSAGE_DIALOG (dialog), secondary);

	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       _("_Save As..."), CONTENT_ACTION_SAVEAS);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       GTK_STOCK_CANCEL, CONTENT_ACTION_NONE);
	gtk_dialog_add_button (GTK_DIALOG (dialog),
			       action_label, mAction);

	gtk_window_set_icon_name (GTK_WINDOW (dialog), "web-browser");

	gtk_dialog_set_default_response (GTK_DIALOG (dialog), (guint) mAction);

	NS_ADDREF (this);
	g_signal_connect_data (dialog, "response",
			       G_CALLBACK (response_cb), this,
			       (GClosureNotify) release_cb, (GConnectFlags) 0);
	gtk_window_present (GTK_WINDOW (dialog));

	return NS_OK;
}

NS_METHOD GContentHandler::MIMEInitiateAction (void)
{
	gboolean auto_downloads;

	if (eel_gconf_get_boolean (CONF_LOCKDOWN_DISABLE_SAVE_TO_DISK)) return NS_OK;

	auto_downloads = eel_gconf_get_boolean (CONF_AUTO_DOWNLOADS);

#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
	mHelperApp = gnome_vfs_mime_get_default_application (mMimeType.get());
	mPermission = ephy_embed_shell_check_mime (embed_shell, mMimeType.get());
#else
	mHelperApp = gnome_vfs_mime_get_default_application (mMimeType);
	mPermission = ephy_embed_shell_check_mime (embed_shell, mMimeType);
#endif

	if (auto_downloads)
	{
		mAction = CONTENT_ACTION_OPEN;
	}
	else
	{
		mAction = CONTENT_ACTION_OPEN_TMP;
	}

	if (!mHelperApp || mPermission != EPHY_MIME_PERMISSION_SAFE)
	{
		mAction = CONTENT_ACTION_DOWNLOAD;
	}

	if (!auto_downloads || mAction == CONTENT_ACTION_DOWNLOAD)
	{
		MIMEConfirmAction ();
	}
	else
	{
		MIMEDoAction ();
	}

	return NS_OK;
}

NS_METHOD GContentHandler::MIMEDoAction (void)
{
	nsCOMPtr<nsIMIMEInfo> mimeInfo;
	mLauncher->GetMIMEInfo(getter_AddRefs(mimeInfo));
	NS_ENSURE_TRUE (mimeInfo, NS_ERROR_FAILURE);

	if (mAction == CONTENT_ACTION_OPEN ||
	    mAction == CONTENT_ACTION_OPEN_TMP	)
	{
		nsEmbedString desc;

		NS_CStringToUTF16 (nsEmbedCString ("gnome-default"),
			           NS_CSTRING_ENCODING_UTF8, desc);

		/* HACK we use the application description to ask
		   MozDownload to open the file when download
		   is finished */
#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
		mimeInfo->SetApplicationDescription (desc);
#else
		mimeInfo->SetApplicationDescription (desc.get());
#endif
	}
	else
	{
#ifdef MOZ_NSIMIMEINFO_NSACSTRING_
		mimeInfo->SetApplicationDescription (nsEmbedString ());
#else
		mimeInfo->SetApplicationDescription (nsnull);
#endif
	}

	if (mAction == CONTENT_ACTION_OPEN)
	{
		mLauncher->SaveToDisk (nsnull, PR_FALSE);
	}
	else if (mAction == CONTENT_ACTION_OPEN_TMP)
	{
		mLauncher->LaunchWithApplication (nsnull, PR_FALSE);
	}
	else if (mAction == CONTENT_ACTION_NONE)
	{
		mLauncher->Cancel ();
	}
	else
	{
		mLauncher->SaveToDisk (nsnull, PR_FALSE);
	}

	return NS_OK;
}
