/*  -*- Mode: C -*- */
/*  ======================================================================
  File: icalrestriction.c
    
 (C) COPYRIGHT 1999 Eric Busboom
 http://www.softwarestudio.org

 The contents of this file are subject to the Mozilla Public License
 Version 1.0 (the "License"); you may not use this file except in
 compliance with the License. You may obtain a copy of the License at
 http://www.mozilla.org/MPL/
 
 Software distributed under the License is distributed on an "AS IS"
 basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 the License for the specific language governing rights and
 limitations under the License.
 

 ======================================================================*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "icalenums.h"
#include "icalrestriction.h"

#define TMP_BUF_SIZE 1024

/* Define the structs for the restrictions. these data are filled out
in machine generated code below */

typedef int (*restriction_func)(icalcomponent* comp);

typedef struct icalrestriction_property_record {
	icalproperty_method method;
	icalcomponent_kind component;
	icalproperty_kind property;
	icalrestriction_kind restriction;
	restriction_func function;
} icalrestriction_property_record;


typedef struct icalrestriction_component_record {
	icalproperty_method method;
	icalcomponent_kind component;
	icalcomponent_kind subcomponent;
	icalrestriction_kind restriction;
	restriction_func function;
} icalrestriction_component_record;

icalrestriction_component_record icalrestriction_component_records[];
icalrestriction_property_record icalrestriction_property_records[]; 

/* The each row gives the result of comparing a restriction against a
   count. The columns in each row represent 0,1,2+. '-1' indicates
   'invalid, 'don't care' or 'needs more analysis' So, for
   ICAL_RESTRICTION_ONE, if there is 1 of a property with that
   restriction, it passes, but if there are 0 or 2+, it fails. */

char compare_map[ICAL_RESTRICTION_UNKNOWN+1][3] = {
    { 1, 1, 1},/*ICAL_RESTRICTION_NONE*/
    { 1, 0, 0},/*ICAL_RESTRICTION_ZERO*/
    { 0, 1, 0},/*ICAL_RESTRICTION_ONE*/
    { 1, 1, 1},/*ICAL_RESTRICTION_ZEROPLUS*/
    { 0, 1, 1},/*ICAL_RESTRICTION_ONEPLUS*/
    { 1, 1, 0},/*ICAL_RESTRICTION_ZEROORONE*/
    { 1, 1, 0},/*ICAL_RESTRICTION_ONEEXCLUSIVE*/
    { 1, 1, 0},/*ICAL_RESTRICTION_ONEMUTUAL*/
    { 1, 1, 1} /*ICAL_RESTRICTION_UNKNOWN*/
};

char restr_string_map[ICAL_RESTRICTION_UNKNOWN+1][60] = {
    "unknown number",/*ICAL_RESTRICTION_NONE*/
    "0",/*ICAL_RESTRICTION_ZERO*/
    "1",/*ICAL_RESTRICTION_ONE*/
    "zero or more",/*ICAL_RESTRICTION_ZEROPLUS*/
    "one or more" ,/*ICAL_RESTRICTION_ONEPLUS*/
    "zero or one",/*ICAL_RESTRICTION_ZEROORONE*/
    "zero or one, exclusive with another property",/*ICAL_RESTRICTION_ONEEXCLUSIVE*/
    "zero or one, mutual with another property",/*ICAL_RESTRICTION_ONEMUTUAL*/
    "unknown number" /*ICAL_RESTRICTION_UNKNOWN*/
};


int
icalrestriction_compare(icalrestriction_kind restr, int count){

    if ( restr < ICAL_RESTRICTION_NONE || restr > ICAL_RESTRICTION_UNKNOWN
	 || count < 0){
	return -1;
    }

    if (count > 2) {
	count = 2;
    }

    return compare_map[restr][count];

}

int icalrestriction_check_mutual(icalcomponent *comp){
    /* HACK. This function does nothing */
    return 1;
}

int icalrestriction_check_exclusive(icalcomponent *comp){
    /* HACK. This function does nothing */
    return 1;
}

int icalrestriction_check_component(icalproperty_method method,
				    icalcomponent* comp)
{
    icalproperty_kind kind;
    icalcomponent_kind comp_kind;
    icalrestriction_kind restr;
    int count;
    int compare;
    int valid = 1;

    comp_kind = icalcomponent_isa(comp);

    /* Check all of the properties in this component */

    for(kind = ICAL_ANY_PROPERTY+1; kind != ICAL_NO_PROPERTY; kind++){
	count = icalcomponent_count_properties(comp, kind);
	
	restr = icalrestriction_get_property_restriction(method, 
							 comp_kind,
							 kind);


	if(restr == ICAL_RESTRICTION_ONEEXCLUSIVE ) {
	    /* HACK. Treat this as a 0/1 restriction */
	    restr = ICAL_RESTRICTION_ZEROORONE;
	    compare = icalrestriction_compare(restr,count);
	} else if (restr == ICAL_RESTRICTION_ONEMUTUAL ) {
          	    /* HACK. Treat this as a 0/1 restriction */
	    restr = ICAL_RESTRICTION_ZEROORONE;
	    compare = icalrestriction_compare(restr,count);
	} else {
	    compare = icalrestriction_compare(restr,count);
	}

	assert(compare != -1);


	if (compare == 0){
	    char temp[TMP_BUF_SIZE];
	    
	    snprintf(temp, TMP_BUF_SIZE,"Failed iTIP restrictions for property %s. Expected %s instances of the property and got %d",
		    icalenum_property_kind_to_string(kind),
		    restr_string_map[restr], count);
	    
	    icalcomponent_add_property
		(comp,
		 icalproperty_vanew_xlicerror(
		     temp,
		     icalparameter_new_xlicerrortype(ICAL_XLICERRORTYPE_INVALIDITIP),
		     0));   
	} 


	valid = valid && compare;
    }



    return valid;


}

int icalrestriction_check(icalcomponent* outer_comp)
{
    icalcomponent_kind comp_kind;
    icalproperty_method method;
    icalcomponent* inner_comp;
    icalproperty *method_prop;
    int valid;
    
    icalerror_check_arg_rz( (outer_comp!=0), "outer comp");


    /* Get the Method value from the outer component */

    comp_kind = icalcomponent_isa(outer_comp);

    if (comp_kind != ICAL_VCALENDAR_COMPONENT){
	icalerror_set_errno(ICAL_BADARG_ERROR);
	return 0;
    }

    method_prop = icalcomponent_get_first_property(outer_comp,
						   ICAL_METHOD_PROPERTY);

    if (method_prop == 0){
	method = ICAL_METHOD_NONE;
    } else {
	method = icalproperty_get_method(method_prop);
    }
   

    /* Check the VCALENDAR wrapper */
    valid = icalrestriction_check_component(ICAL_METHOD_NONE,outer_comp);


    /* Now check the inner components */

    for(inner_comp= icalcomponent_get_first_component(outer_comp,
						      ICAL_ANY_COMPONENT);
	inner_comp != 0;
	inner_comp= icalcomponent_get_next_component(outer_comp,
						     ICAL_ANY_COMPONENT)){

	valid = valid && icalrestriction_check_component(method,inner_comp);

    }


    return valid;

}

icalrestriction_kind
icalrestriction_get_property_restriction(icalproperty_method method,
					 icalcomponent_kind component,
					 icalproperty_kind property)
{
    int i;

    for(i = 0;
	icalrestriction_property_records[i].restriction != ICAL_RESTRICTION_NONE;
	i++){

	if (method == icalrestriction_property_records[i].method &&
	    component == icalrestriction_property_records[i].component &&
	    property ==  icalrestriction_property_records[i].property ){
	    return  icalrestriction_property_records[i].restriction;
	}
    }
	
    return ICAL_RESTRICTION_UNKNOWN;
}

icalrestriction_kind
icalrestriction_get_component_restriction(icalproperty_method method,
					  icalcomponent_kind component,
					  icalcomponent_kind subcomponent)
{

    int i;

    for(i = 0;
	icalrestriction_component_records[i].restriction != ICAL_RESTRICTION_NONE;
	i++){

	if (method == icalrestriction_component_records[i].method &&
	    component == icalrestriction_component_records[i].component &&
	    subcomponent ==  icalrestriction_component_records[i].subcomponent ){
	    return  icalrestriction_component_records[i].restriction;
	}
    }
	
    return ICAL_RESTRICTION_UNKNOWN;
}

/* Everything below this line is machine generated. Do not edit. */
icalrestriction_property_record icalrestriction_property_records[] = {
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,icalrestriction_check_exclusive},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,icalrestriction_check_exclusive},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,icalrestriction_check_exclusive},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,icalrestriction_check_exclusive},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS ,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_VCALENDAR_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE, icalrestriction_check_exclusive},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VEVENT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTODO_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEEXCLUSIVE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VJOURNAL_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VFREEBUSY_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_VTIMEZONE_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XSTANDARD_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDAYLIGHT_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XAUDIOALARM_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XDISPLAYALARM_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ONEPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XEMAILALARM_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_ACTION_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_ATTACH_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_ATTENDEE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_CALSCALE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_CATEGORIES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_CLASS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_COMMENT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_COMPLETED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_CONTACT_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_CREATED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DESCRIPTION_PROPERTY,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DTEND_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DTSTAMP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DTSTART_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DUE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_DURATION_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_REPEAT_PROPERTY,ICAL_RESTRICTION_ONEMUTUAL,icalrestriction_check_mutual},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_EXDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_EXRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_FREEBUSY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_GEO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_LASTMODIFIED_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_LOCATION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_METHOD_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_ORGANIZER_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_PERCENTCOMPLETE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_PRIORITY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_PRODID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_RDATE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_RECURRENCEID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_RELATEDTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_REQUESTSTATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_RESOURCES_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_RRULE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_SEQUENCE_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_STATUS_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_SUMMARY_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TRANSP_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TRIGGER_PROPERTY,ICAL_RESTRICTION_ONE,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TZID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TZNAME_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TZOFFSETFROM_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TZOFFSETTO_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_TZURL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_UID_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_URL_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_VERSION_PROPERTY,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_XPROCEDUREALARM_COMPONENT,ICAL_X_PROPERTY,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_NONE,ICAL_NO_COMPONENT,ICAL_NO_PROPERTY,ICAL_RESTRICTION_NONE}
};
icalrestriction_component_record icalrestriction_component_records[] = {
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VEVENT_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VFREEBUSY_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VFREEBUSY_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VFREEBUSY_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VTODO_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REQUEST,ICAL_VTODO_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REPLY,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_VJOURNAL_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_REFRESH,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_COUNTER,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_DECLINECOUNTER,ICAL_VTODO_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_PUBLISH,ICAL_VJOURNAL_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROORONE,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_ADD,ICAL_VJOURNAL_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_VTIMEZONE_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_X_COMPONENT,ICAL_RESTRICTION_ZEROPLUS,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_VALARM_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_VEVENT_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_VFREEBUSY_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_CANCEL,ICAL_VJOURNAL_COMPONENT,ICAL_VTODO_COMPONENT,ICAL_RESTRICTION_ZERO,0},
    {ICAL_METHOD_NONE,ICAL_NO_COMPONENT,ICAL_NO_COMPONENT,ICAL_RESTRICTION_NONE}
};
