/* sqlqueryform.c
 *
 * Copyright (C) 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlqueryform.h"
#include "choicecombo.h"

static void sql_query_form_class_init (SqlQueryFormClass * class);
static void sql_query_form_init (SqlQueryForm * qfm);
static void sql_query_form_destroy (GtkObject * object);
static void sql_query_form_post_init (SqlQueryForm * qfm);
static void sql_query_form_update (SqlQueryForm * qfm);
static void update_buttons_sensitiveness (SqlQueryForm * qfm);

/* structures for each node in the form */
typedef enum
{
	NODE_FIELD,
	NODE_LINK
}
QueryNodeType;

typedef struct _ANode ANode;
struct _ANode
{
	GtkWidget *widget;	/* if NODE_LINK, only the nodes with possible othernodes
				   will have widget!=NULL, to be able to have ONE ONLY
				   ANode with a widget for a SqlMemFlinkink */
	QueryNodeType type;
	gint pos;		/* num of column in the SqlQueryRes tabular */
	gboolean modified;
	QueryElement *elt;	/* CAUTION: if NODE_LINK, elt will point to 
				   the QueryElement which is represented, not 
				   what is displayed in the ChoiceCombo widget */
	SqlMemFlink *link;
	GSList *choicelist;	/* only for NODE_LINK */
	GSList *othernodes;	/* list of the other nodes sharing the same table and
				   which are NODE_LINK */
};

/* structure to hold the information for each possible entry in a 
   ChoiceCombo */
typedef struct _ChoiceEntry ChoiceEntry;
struct _ChoiceEntry
{
	gchar *linkval;
	gchar *printval;
};

/*
 *
 * Main functions for the object
 *
 */

/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

enum
{
	ACTION_REFRESH,
	ACTION_INSERT,
	ACTION_VIEWALL,
	ACTION_CHOOSE,
	LAST_SIGNAL
};

static gint sql_query_form_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

guint
sql_query_form_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query_Form",
			sizeof (SqlQueryForm),
			sizeof (SqlQueryFormClass),
			(GtkClassInitFunc) sql_query_form_class_init,
			(GtkObjectInitFunc) sql_query_form_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_vbox_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_form_class_init (SqlQueryFormClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_query_form_signals[ACTION_REFRESH] =
		gtk_signal_new ("action_refresh",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryFormClass,
						   action_refresh),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_form_signals[ACTION_INSERT] =
		gtk_signal_new ("action_insert",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryFormClass,
						   action_insert),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_form_signals[ACTION_VIEWALL] =
		gtk_signal_new ("action_viewall",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryFormClass,
						   action_viewall),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_form_signals[ACTION_CHOOSE] =
		gtk_signal_new ("action_choose",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryFormClass,
						   action_choose),
				gtk_marshal_NONE__UINT, GTK_TYPE_NONE, 1,
				GTK_TYPE_ULONG);


	gtk_object_class_add_signals (object_class, sql_query_form_signals,
				      LAST_SIGNAL);
	class->action_refresh = NULL;
	class->action_insert = NULL;
	class->action_viewall = NULL;
	class->action_choose = NULL;

	object_class->destroy = sql_query_form_destroy;
	parent_class = gtk_type_class (gtk_object_get_type ());
}

static void
sql_query_form_init (SqlQueryForm * qfm)
{
	qfm->qx = NULL;
	qfm->table = NULL;
	qfm->nodes = NULL;
	qfm->inform = NULL;
	qfm->numrow = 0;
	qfm->modified = FALSE;
	qfm->outdated = FALSE;
}

static void qfm_do_action_commit (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_first (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_prev (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_next (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_last (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_insert (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_refresh (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_delete (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_viewall (GtkWidget * button, SqlQueryForm * qfm);
static void qfm_do_action_choose (GtkWidget * button, SqlQueryForm * qfm);
GtkWidget *
sql_query_form_new (SqlQueryExec * exec, gulong numrow)
{
	GtkObject *obj;
	SqlQueryForm *qfm;
	GtkWidget *wid;
	SqlQueryActions action;

	g_return_val_if_fail (exec != NULL, NULL);
	g_return_val_if_fail (IS_SQL_QUERY_EXEC (exec), NULL);

	if (!exec->env->modif_table) {
		gnome_app_warning (GNOME_APP (exec->q->conf->app),
				   _
				   ("You must first choose a table to modify."));
		return NULL;
	}

	obj = gtk_type_new (sql_query_form_get_type ());
	qfm = SQL_QUERY_FORM (obj);
	qfm->qx = exec;
	qfm->numrow = numrow;
	qfm->new_entry = numrow ? FALSE : TRUE;

	if (!exec->qres && !qfm->new_entry)
		g_error ("SqlQueryRes is NULL! %s line %d", __FILE__,
			 __LINE__);

	sql_query_form_post_init (qfm);

	/* now the buttons as configured by the SqlQueryEnv */
	wid = sql_query_exec_get_action_buttons (exec);
	gtk_box_pack_start (GTK_BOX (qfm), wid, FALSE, FALSE, 0);
	gtk_object_set_data (GTK_OBJECT (qfm), "buttons", wid);
	for (action = QUERY_ACTION_FIRST; action <= QUERY_ACTION_LAST_ENUM;
	     action *= 2) {
		GtkWidget *button;
		gchar *str;
		str = g_strdup_printf ("%ld", action);
		button = gtk_object_get_data (GTK_OBJECT (wid), str);
		if (button) {
			switch (action) {
			case QUERY_ACTION_DELETE:
				if (!qfm->new_entry) {
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_delete),
							    qfm);
				}
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_INSERT:
				/* gtk_signal_connect(GTK_OBJECT(button), "clicked", */
				/*                         GTK_SIGNAL_FUNC(qfm_do_action_insert), qfm); */
			case QUERY_ACTION_EDIT:
				gtk_widget_destroy (button);
				gtk_object_set_data (GTK_OBJECT (wid), str,
						     NULL);
				break;
			case QUERY_ACTION_COMMIT:
				gtk_signal_connect (GTK_OBJECT (button),
						    "clicked",
						    GTK_SIGNAL_FUNC
						    (qfm_do_action_commit),
						    qfm);
				gtk_widget_set_sensitive (button, FALSE);
				break;
			case QUERY_ACTION_FIRST:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_first),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_PREV:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_prev),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_NEXT:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_next),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_LAST:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_last),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_REFRESH:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_refresh),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_CHOOSE:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_choose),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			case QUERY_ACTION_VIEWALL:
				if (!qfm->new_entry)
					gtk_signal_connect (GTK_OBJECT
							    (button),
							    "clicked",
							    GTK_SIGNAL_FUNC
							    (qfm_do_action_viewall),
							    qfm);
				else {
					gtk_widget_destroy (button);
					gtk_object_set_data (GTK_OBJECT (wid),
							     str, NULL);
				}
				break;
			}
		}
		g_free (str);
	}

	update_buttons_sensitiveness (qfm);

	/* FIXME: connect to the "not_valid" signal of SqlQueryExec object to freeze the widget */
	return GTK_WIDGET (obj);
}

static void
sql_query_form_destroy (GtkObject * object)
{
	SqlQueryForm *qfm;
	GSList *list, *hold;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY_FORM (object));

	qfm = SQL_QUERY_FORM (object);

	g_print ("sql_query_form_destroy\n");
	if (qfm->inform)
		g_slist_free (qfm->inform);
	while (qfm->nodes) {
		/* destroying the choicelist */
		list = ((ANode *) (qfm->nodes->data))->choicelist;
		while (list) {
			g_free (((ChoiceEntry *) (list->data))->linkval);
			g_free (((ChoiceEntry *) (list->data))->printval);
			g_free ((ChoiceEntry *) (list->data));
			hold = list;
			list = g_slist_remove_link (list, list);
			g_slist_free_1 (hold);
		}
		g_free (qfm->nodes->data);
		hold = qfm->nodes;
		qfm->nodes = g_slist_remove_link (qfm->nodes, qfm->nodes);
		g_slist_free_1 (hold);
	}

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
	g_print ("sql_query_form_destroy END\n");
}

void
sql_query_form_refresh (SqlQueryForm * qfm)
{
	sql_query_form_update (qfm);
	update_buttons_sensitiveness (qfm);
}

static gint
m_GCompareFunc (ANode * a, ANode * b)
{
	if (a->pos < b->pos)
		return -1;
	if (a->pos > b->pos)
		return 1;
	return 0;
}

/* this function is called when an update is needed: when the form is first created, 
   after a button move, or on a refresh */
static void
sql_query_form_update (SqlQueryForm * qfm)
{
	GSList *list;
	gchar *str;
	GtkWidget *wid;
	ANode *node;

	if (!qfm->qx->env->modif_table || qfm->new_entry)
		return;

	list = qfm->inform;
	while (list) {
		node = (ANode *) (list->data);
		wid = NULL;
		str = g_strdup_printf ("n%d",
				       g_slist_position (qfm->inform, list));
		wid = gtk_object_get_data (GTK_OBJECT (qfm->table), str);
		g_free (str);

		if (wid) {
			SqlDataDisplayFns *fns;
			GdaField *gfield = NULL;

			if (!qfm->new_entry)
				gfield = sql_query_res_get_gdafield (qfm->qx->
								     qres,
								     qfm->
								     numrow,
								     node->
								     pos);
			fns = gtk_object_get_data (GTK_OBJECT (wid), "fns");
			if (fns)
				(fns->
				 gdafield_to_widget_update) (DATA_DISPLAY
							     (wid), gfield);
			else {	/* choosing the right item to display in the ChoiceCombo */
				GString *string;
				GSList *list2;

				fns = sql_access_get_object_display_fns (qfm->
									 qx->
									 q->
									 conf->
									 srv,
									 GTK_OBJECT
									 (node->
									  elt->
									  main));
				str = (fns->gdafield_to_str) (gfield);
				if (str) {
					string = g_string_new (str);
					g_free (str);
				}
				else
					string = g_string_new ("");
				list2 = node->othernodes;
				while (list2) {
					fns = sql_access_get_object_display_fns (qfm->qx->q->conf->srv, GTK_OBJECT (((ANode *) (list2->data))->elt->main));
					if (!qfm->new_entry)
						gfield = sql_query_res_get_gdafield (qfm->qx->qres, qfm->numrow, ((ANode *) (list2->data))->pos);
					else
						gfield = NULL;
					str = (fns->gdafield_to_str) (gfield);
					if (str) {
						g_string_sprintfa (string,
								   " / %s",
								   str);
						g_free (str);
					}
					else
						g_string_append (string,
								 " / -");
					list2 = g_slist_next (list2);
				}
				choice_combo_set_selection_txt (CHOICE_COMBO
								(wid),
								string->str);
				g_string_free (string, FALSE);
			}
		}
		list = g_slist_next (list);
	}
}

static void sql_query_form_modif_cb (GtkObject * obj, SqlQueryForm * qfm);
static void sql_query_form_modif_2_cb (GtkObject * obj, gpointer ptr,
				       SqlQueryForm * qfm);
static void
sql_query_form_post_init (SqlQueryForm * qfm)
{
	GtkWidget *table, *wid;
	GSList *list, *inform, *choicelist, *list2;
	gint i, nbrows;
	gboolean takeit;
	ANode *node;
	QueryElement *elt;
	QueryLink *qlink;
	ChoiceEntry *ce;
	gchar *str;
	SqlMemTable *t;
	SqlQueryRes *res;
	SqlDataDisplayFns *fns;
	GdaField *gfield;

	/* determining which element will be in the form */
	inform = NULL;
	i = 0;
	list = qfm->qx->q->objects;
	/* choose among the SqlQuery the objects (in q->objects) which
	   will appear on the form for modification */
	while (list) {
		elt = (QueryElement *) (list->data);
		takeit = FALSE;

		if (elt->print_name) {
			if ((elt->type == QUERY_ELT_FIELD)) {
				SqlMemTable *t;
				t = sql_db_find_table_from_field (qfm->qx->q->
								  conf->db,
								  SQL_MEM_FIELD
								  (elt->
								   main));
				if (qfm->qx->env->modif_table
				    && (t == qfm->qx->env->modif_table))
					takeit = TRUE;
			}

			if (takeit) {
				node = (ANode *) g_malloc (sizeof (ANode));
				qfm->nodes = g_slist_append (qfm->nodes, node);	/* 1st list */
				node->pos = i;
				node->type = NODE_FIELD;
				node->elt = elt;
				node->link = NULL;
				node->modified = FALSE;
				node->othernodes = NULL;
				node->choicelist = NULL;
				node->widget = NULL;
				inform = g_slist_append (inform, node);	/* 2nd list */
			}
			i++;
		}
		/*i++; */
		list = g_slist_next (list);
	}
	/* select some other fields of the table to be modified, but this
	   time through a link (using the ChoiceCombo widget). These are fields
	   belonging to the table to modify, for which some links from other tables
	   come TO that field, and those links are declared as being used. */
	list = qfm->qx->q->depend_list;
	while (list) {
		qlink = (QueryLink *) (list->data);
		if (qlink->useit) {
			if (g_slist_find
			    (qfm->qx->env->modif_table->fields,
			     qlink->link->to)) {
				SqlMemTable *t;
				GSList *list2;
				/* does any field of the table containing the FROM field 
				   appear in the list of printed elements? */
				t = sql_db_find_table_from_field (qfm->qx->q->
								  conf->db,
								  qlink->
								  link->from);
				if (t) {
					list2 = t->fields;
					while (list2) {
						if ((i =
						     sql_query_is_field_printed
						     (qfm->qx->q,
						      SQL_MEM_FIELD (list2->
								     data))) >
						    -1) {
							node = (ANode *)
								g_malloc
								(sizeof
								 (ANode));
							qfm->nodes =
								g_slist_append
								(qfm->nodes,
								 node);
							node->pos = i;
							node->type =
								NODE_LINK;
							node->elt =
								sql_query_get_query_element
								(qfm->qx->q,
								 i);
							node->link =
								qlink->link;
							node->modified =
								FALSE;
							node->othernodes =
								NULL;
							node->choicelist =
								NULL;
							node->widget = NULL;
							inform = g_slist_append (inform, node);
						}
						list2 = g_slist_next (list2);
					}
				}
			}
		}
		list = g_slist_next (list);
	}

	/* reordering all the items to make them appear in the same order as they
	   were specified in the SELECT query */
	inform = g_slist_sort (inform, (GCompareFunc) m_GCompareFunc);

	/* grouping all the NODE_LINK "ANodes" which actually refer to the same 
	   table to avoid stupid display */
	nbrows = g_slist_length (inform);
	list = inform;
	while (list) {
		SqlMemTable *t1, *t2;

		node = (ANode *) (list->data);
		if ((node->type == NODE_LINK) && (!node->modified)) {
			t1 = sql_db_find_table_from_field (qfm->qx->q->conf->
							   db,
							   node->link->from);
			list2 = g_slist_next (list);
			while (list2) {
				ANode *node2;
				node2 = (ANode *) (list2->data);
				if ((node2->type == NODE_LINK)
				    && (!node2->modified)) {
					t2 = sql_db_find_table_from_field
						(qfm->qx->q->conf->db,
						 node2->link->from);
					if (t1 == t2) {	/* lists need to be grouped */
						node->othernodes =
							g_slist_append (node->
									othernodes,
									node2);
						node2->modified = TRUE;
					}
				}
				list2 = g_slist_next (list2);
			}
		}
		list = g_slist_next (list);
	}
	/* removing any NODE_LINK "ANode" for which linklist is NULL */
	list = inform;
	while (list) {
		node = (ANode *) (list->data);
		if ((node->type == NODE_LINK) && (node->modified)) {
			inform = g_slist_remove_link (inform, list);
			g_slist_free_1 (list);
			node->modified = FALSE;
			list = inform;
		}
		list = g_slist_next (list);
	}
	qfm->inform = inform;

	/* Creating the table and the widgets in that table */
	nbrows = g_slist_length (inform);
	if (nbrows == 0) {
		wid = gtk_label_new (_("There is nothing to be modified\n"
				       "in this query"));
		gtk_box_pack_start (GTK_BOX (qfm), wid, FALSE, TRUE,
				    GNOME_PAD);
		gtk_widget_show (wid);
	}
	else {
		table = gtk_table_new (nbrows, 2, FALSE);
		qfm->table = table;
		gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD);
		gtk_box_pack_start (GTK_BOX (qfm), table, TRUE, TRUE,
				    GNOME_PAD);
		gtk_widget_show (table);

		/* filling the table */
		list = inform;
		i = 0;
		while (list) {
			GString *string, *title, *order;

			node = (ANode *) (list->data);
			switch (node->type) {
			case NODE_FIELD:
		       /*** FIELD ***/
				/* if the form is an insert form, and if there is a sequence which links
				   to the field, then we don't display the field for input */
				if (qfm->new_entry) {
					if (!
					    (node->elt->type ==
					     QUERY_ELT_FIELD))
						g_error ("ELT is not a QUERY_ELT_FIELD in %s line %d", __FILE__, __LINE__);
					if (sql_db_find_sequence_to_field
					    (qfm->qx->q->conf->db,
					     SQL_MEM_FIELD (node->elt->
							    main))) {
						node->widget = NULL;
						break;
					}
				}
				str = g_strdup_printf ("%s: ",
						       node->elt->name);
				wid = gtk_label_new (str);
				g_free (str);
				gtk_table_attach (GTK_TABLE (table), wid,
						  0, 1, i, i + 1,
						  GTK_FILL | GTK_SHRINK, 0, 0,
						  0);
				gtk_widget_show (wid);
				fns = sql_access_get_object_display_fns (qfm->
									 qx->
									 q->
									 conf->
									 srv,
									 GTK_OBJECT
									 (node->
									  elt->
									  main));
				if (!qfm->new_entry) {
					gfield = sql_query_res_get_gdafield
						(qfm->qx->qres, qfm->numrow,
						 node->pos);
					wid = GTK_WIDGET ((fns->
							   gdafield_to_widget)
							  (gfield));
				}
				else {
					/* FIXME: maybe it would be worthwile to set a default value for a field
					   if there is a WHERE condition like field=value */
					if (!
					    (node->elt->type ==
					     QUERY_ELT_FIELD))
						g_error ("ELT is not a QUERY_ELT_FIELD in %s line %d", __FILE__, __LINE__);
					if (SQL_MEM_FIELD (node->elt->main)->
					    default_val)
						wid = GTK_WIDGET ((fns->
								   sql_to_widget)
								  (SQL_MEM_FIELD (node->elt->main)->default_val));
					else
						wid = GTK_WIDGET ((fns->
								   gdafield_to_widget)
								  (NULL));
				}
				gtk_object_set_data (GTK_OBJECT (wid), "fns",
						     fns);

				str = g_strdup_printf ("n%d",
						       g_slist_position
						       (inform, list));
				gtk_object_set_data (GTK_OBJECT (table), str,
						     wid);
				g_free (str);

				if ((fns->
				     use_free_space) (DATA_DISPLAY (wid)))
					gtk_table_attach (GTK_TABLE (table),
							  wid, 1, 2, i, i + 1,
							  GTK_FILL |
							  GTK_EXPAND |
							  GTK_SHRINK,
							  GTK_FILL |
							  GTK_EXPAND |
							  GTK_SHRINK, 0, 0);
				else
					gtk_table_attach (GTK_TABLE (table),
							  wid, 1, 2, i, i + 1,
							  GTK_FILL |
							  GTK_EXPAND |
							  GTK_SHRINK, 0, 0,
							  0);

				if (IS_DATA_DISPLAY (wid))
					gtk_signal_connect (GTK_OBJECT (wid),
							    "contents_modified",
							    GTK_SIGNAL_FUNC
							    (sql_query_form_modif_cb),
							    qfm);
				else
					g_warning
						("the returned widget is not of type DataDisplay");
				node->widget = wid;
				break;
			case NODE_LINK:
		      /*** LINK ***/
				title = g_string_new (node->elt->name);

				/* build the list for that combo */
				/* table t common for the entire linklist */
				t = sql_db_find_table_from_field (qfm->qx->q->
								  conf->db,
								  node->link->
								  from);
				order = g_string_new (SQL_MEM_FIELD
						      (node->elt->main)->
						      name);
				string = g_string_new ("SELECT DISTINCT ");
				g_string_sprintfa (string, "%s, %s",
						   node->link->from->name,
						   SQL_MEM_FIELD (node->elt->
								  main)->
						   name);
				list2 = node->othernodes;
				while (list2) {
					g_string_sprintfa (title, "/\n%s",
							   ((ANode *) (list2->
								       data))->
							   elt->name);
					g_string_sprintfa (string, ", %s",
							   SQL_MEM_FIELD (((ANode *) (list2->data))->elt->main)->name);
					g_string_sprintfa (order, ", %s",
							   SQL_MEM_FIELD (((ANode *) (list2->data))->elt->main)->name);
					list2 = g_slist_next (list2);
				}
				g_string_sprintfa (string,
						   " FROM %s ORDER BY %s",
						   t->name, order->str);
				g_string_free (order, TRUE);
				res = sql_access_do_query (qfm->qx->q->conf->
							   srv, string->str);
				g_string_free (string, TRUE);
				choicelist = NULL;
				if (res) {
					gint i, nrows;
					gint ncols = 0, j;

					ncols = g_slist_length (node->
								othernodes);
					nrows = sql_query_res_get_nbtuples
						(res);
					for (i = 1; i <= nrows; i++) {
						ce = g_new0 (ChoiceEntry, 1);
						fns = sql_access_get_object_display_fns (qfm->qx->q->conf->srv, GTK_OBJECT (node->elt->main));
						gfield = sql_query_res_get_gdafield (res, i, 0);
						ce->linkval = (fns->gdafield_to_sql) (gfield);	/* to_str */
						if (!ce->linkval) {
							ce->linkval =
								g_strdup ("");
							g_warning
								("(fns->gdafield_to_sql)(gfield) returned NULL in %s line %d",
								 __FILE__,
								 __LINE__);
						}
						gfield = sql_query_res_get_gdafield (res, i, 1);
						str = (fns->
						       gdafield_to_str)
						   (gfield);
						if (str) {
							string = g_string_new
								(str);
							g_free (str);
						}
						else
							string = g_string_new
								("");
						list2 = node->othernodes;
						for (j = 1; j <= ncols; j++) {
							fns = sql_access_get_object_display_fns (qfm->qx->q->conf->srv, GTK_OBJECT (((ANode *) (list2->data))->elt->main));
							gfield = sql_query_res_get_gdafield (res, i, 1 + j);
							str = (fns->
							       gdafield_to_str)
							   (gfield);
							if (str) {
								g_string_sprintfa
									(string,
									 " / %s",
									 str);
								g_free (str);
							}
							else
								g_string_append
									(string,
									 " / -");
							list2 = g_slist_next
								(list2);
						}
						ce->printval = string->str;
						g_string_free (string, FALSE);
						choicelist =
							g_slist_append
							(choicelist, ce);
					}
					sql_query_res_free (res);
				}

				g_string_sprintfa (title, ": ");
				wid = gtk_label_new (title->str);
				g_string_free (title, TRUE);
				gtk_table_attach (GTK_TABLE (table), wid,
						  0, 1, i, i + 1,
						  GTK_FILL | GTK_SHRINK, 0, 0,
						  0);
				gtk_widget_show (wid);
				wid = GTK_WIDGET (choice_combo_new ());

				/* setting the ChoiceCombo contents */
				choice_combo_set_content (CHOICE_COMBO (wid),
							  choicelist,
							  GTK_STRUCT_OFFSET
							  (ChoiceEntry,
							   printval));
				node->choicelist = choicelist;
				/* choosing the right item to display in the ChoiceCombo is
				   done when we call the sql_query_form_update() function */

				str = g_strdup_printf ("n%d",
						       g_slist_position
						       (inform, list));
				gtk_object_set_data (GTK_OBJECT (table), str,
						     wid);
				g_free (str);

				gtk_table_attach (GTK_TABLE (table), wid,
						  1, 2, i, i + 1,
						  GTK_FILL | GTK_EXPAND |
						  GTK_SHRINK, 0, 0, 0);
				gtk_signal_connect (GTK_OBJECT (wid),
						    "selection_changed",
						    GTK_SIGNAL_FUNC
						    (sql_query_form_modif_2_cb),
						    qfm);
				node->widget = wid;
				break;
			default:
				wid = NULL;	/* should never happen! */
				break;
			}

			if (wid) {
				gtk_object_set_data (GTK_OBJECT (wid), "node",
						     node);
				gtk_widget_show (wid);
				i++;
			}
			list = g_slist_next (list);
		}
	}

	sql_query_form_update (qfm);
}

/* called whenever an entry in the QueryForm is modified */
static void
sql_query_form_modif_cb (GtkObject * obj, SqlQueryForm * qfm)
{
	ANode *node;
	GSList *list;
	GtkWidget *button;

	node = gtk_object_get_data (GTK_OBJECT (obj), "node");
	node->modified = TRUE;
	list = node->othernodes;
	while (list) {
		((ANode *) (list->data))->modified = TRUE;
		list = g_slist_next (list);
	}

	button = gtk_object_get_data (GTK_OBJECT (qfm), "buttons");
	if (button) {
		gchar *str;
		str = g_strdup_printf ("%d", QUERY_ACTION_COMMIT);
		button = gtk_object_get_data (GTK_OBJECT (button), str);
		g_free (str);
		if (button)
			gtk_widget_set_sensitive (button, TRUE);
	}
}

static void
sql_query_form_modif_2_cb (GtkObject * obj, gpointer ptr, SqlQueryForm * qfm)
{
	sql_query_form_modif_cb (obj, qfm);
}




static void qfm_do_action_insert_into (SqlQueryForm * qfm);
static void
dummy_cb (gint reply, gpointer data)
{
}
static void
qfm_do_action_commit (GtkWidget * button, SqlQueryForm * qfm)
{
	GSList *list;
	ANode *node;
	gchar *str;
	GString *query, *where, *tmpstr;
	gboolean firstup, firstwh;
	SqlQueryRes *res;
	SqlDataDisplayFns *fns;
	GdaField *gfield;

	if (qfm->new_entry) {	/* insertion is handled by another function */
		qfm_do_action_insert_into (qfm);
		return;
	}

	query = g_string_new ("UPDATE ");
	where = g_string_new ("WHERE ");
	g_string_sprintfa (query, "%s \nset ",
			   qfm->qx->env->modif_table->name);
	firstup = TRUE;
	firstwh = TRUE;
	list = qfm->nodes;
	while (list) {
		node = (ANode *) (list->data);

		switch (node->type) {
		case NODE_FIELD:	/* node node->elt->name */
			if (node->modified) {
				/* UPDATE */
				/* set to str */
				fns = sql_access_get_object_display_fns (qfm->
									 qx->
									 q->
									 conf->
									 srv,
									 GTK_OBJECT
									 (node->
									  elt->
									  main));
				str = (fns->
				       widget_to_sql) (DATA_DISPLAY (node->
								     widget));
				if (str) {
					if (firstup) {
						firstup = FALSE;
						g_string_sprintfa (query,
								   "%s = %s",
								   SQL_MEM_FIELD
								   (node->
								    elt->
								    main)->
								   name, str);
					}
					else
						g_string_sprintfa (query,
								   ", %s = %s",
								   SQL_MEM_FIELD
								   (node->
								    elt->
								    main)->
								   name, str);
					g_free (str);
				}
			}
			/* WHERE */
			fns = sql_access_get_object_display_fns (qfm->qx->q->
								 conf->srv,
								 GTK_OBJECT
								 (node->elt->
								  main));
			if (!qfm->new_entry)
				gfield = sql_query_res_get_gdafield (qfm->qx->
								     qres,
								     qfm->
								     numrow,
								     node->
								     pos);
			else
				gfield = NULL;
			str = (fns->gdafield_to_sql) (gfield);
			if (!str || !strcmp (str, "''")) {	/* str == '' */
				if (firstwh)
					firstwh = FALSE;
				else
					g_string_sprintfa (where, "AND ");

				if (str)
					g_string_sprintfa (where,
							   "(%s = %s OR %s is null) ",
							   SQL_MEM_FIELD
							   (node->elt->main)->
							   name, str,
							   SQL_MEM_FIELD
							   (node->elt->main)->
							   name);
				else
					g_string_sprintfa (where,
							   "%s is null ",
							   SQL_MEM_FIELD
							   (node->elt->main)->
							   name);
			}
			else {	/* str != '' */
				if (firstwh) {
					firstwh = FALSE;
					g_string_sprintfa (where, "%s = %s ",
							   SQL_MEM_FIELD
							   (node->elt->main)->
							   name, str);
				}
				else
					g_string_sprintfa (where,
							   "AND %s = %s ",
							   SQL_MEM_FIELD
							   (node->elt->main)->
							   name, str);
			}
			if (str)
				g_free (str);
			break;
		case NODE_LINK:
			if (node->widget) {
				/* node node->link->to->name */
				if (node->modified) {
					ChoiceEntry *ce;
					ce = choice_combo_get_selection
						(CHOICE_COMBO (node->widget));
					/* set to ce->linkval */
					if (firstup) {
						firstup = FALSE;
						g_string_sprintfa (query,
								   "%s = %s",
								   node->
								   link->to->
								   name,
								   ce->
								   linkval);
					}
					else
						g_string_sprintfa (query,
								   ", %s = %s",
								   node->
								   link->to->
								   name,
								   ce->
								   linkval);
				}
			}
			break;
		}

		list = g_slist_next (list);
	}
	g_string_sprintfa (query, " %s", where->str);

	tmpstr = g_string_new ("SELECT count(*) FROM ");
	g_string_sprintfa (tmpstr, "%s %s", qfm->qx->env->modif_table->name,
			   where->str);
	g_string_free (where, TRUE);

	res = sql_access_do_query (qfm->qx->q->conf->srv, tmpstr->str);
	g_string_free (tmpstr, TRUE);
	if (res) {
		GtkWidget *dlg;
		gint i;

		i = atoi (sql_query_res_get_item (res, 1, 0));
		if (i == 0) {
			gnome_app_message (GNOME_APP (qfm->qx->q->conf->app),
					   _
					   ("There is no data to be modified, the "
					    "modifications have been cancelled"));
			sql_query_form_update (qfm);
		}
		else {
			if (i > 1)
				str = g_strdup_printf (_
						       ("%d tuples will be modified.\n"
							"Update query is:\n%s\n"
							"Do You want to continue modification?"),
						       i, query->str);
			else
				str = g_strdup_printf (_
						       ("One tuple will be modified.\n"
							"Update query is:\n%s\n"
							"Do You want to continue modification?"),
						       query->str);
			dlg = gnome_question_dialog (str,
						     (GnomeReplyCallback)
						     dummy_cb, NULL);
			g_free (str);
			gtk_object_destroy (GTK_OBJECT (res));
			i = gnome_dialog_run_and_close (GNOME_DIALOG (dlg));
			if (!i) {	/* saving */
				/*g_print("Action Query is #%s#\n", query->str); */
				sql_access_do_query (qfm->qx->q->conf->srv,
						     query->str);
				qfm_do_action_refresh (NULL, qfm);
			}
		}
	}
	else {
		gnome_app_error (GNOME_APP (qfm->qx->q->conf->app),
				 _("An error has occured while trying to "
				   "update the data, "
				   "modifications have been cancelled."));
		sql_query_form_update (qfm);
	}
	g_string_free (query, TRUE);
}

static void
qfm_do_action_insert_into (SqlQueryForm * qfm)
{
	GSList *list;
	ANode *node;
	gchar *str;
	GString *query, *values;
	gboolean firstin, firstval;
	SqlDataDisplayFns *fns;
	GdaField *gfield;
	GtkWidget *dlg;
	gint i;

	query = g_string_new ("INSERT INTO ");
	values = g_string_new ("VALUES (");
	g_string_sprintfa (query, "%s \n(", qfm->qx->env->modif_table->name);
	firstin = TRUE;
	firstval = TRUE;
	list = qfm->nodes;
	while (list) {
		node = (ANode *) (list->data);

		switch (node->type) {
		case NODE_FIELD:	/* node node->elt->name */
			if (node->widget) {
				fns = sql_access_get_object_display_fns (qfm->
									 qx->
									 q->
									 conf->
									 srv,
									 GTK_OBJECT
									 (node->
									  elt->
									  main));
				str = (fns->
				       widget_to_sql) (DATA_DISPLAY (node->
								     widget));
			}
			else {	/* here the value comes from a sequence */
				SqlMemSeq *seq;
				seq = sql_db_find_sequence_to_field (qfm->qx->
								     q->conf->
								     db,
								     SQL_MEM_FIELD
								     (node->
								      elt->
								      main));
				if (SQL_MEM_FIELD (node->elt->main)->
				    default_val)
					str = g_strdup (SQL_MEM_FIELD
							(node->elt->main)->
							default_val);
				else
					str = g_strdup_printf ("nextval('%s')", seq->name);	/* FIXME: Postgres only! */
			}
			if (str) {
				if (firstin)
					firstin = FALSE;
				else
					g_string_append (query, " ,");
				g_string_append (query,
						 SQL_MEM_FIELD (node->elt->
								main)->name);
				if (firstval)
					firstval = FALSE;
				else
					g_string_append (values, " ,");
				g_string_sprintfa (values, "%s", str);
				g_free (str);
			}
			break;
		case NODE_LINK:
			if (node->widget) {
				/* node node->link->to->name */
				/* set to ce->linkval */
				ChoiceEntry *ce;

				ce = choice_combo_get_selection (CHOICE_COMBO
								 (node->
								  widget));

				if (firstin)
					firstin = FALSE;
				else
					g_string_append (query, " ,");
				g_string_append (query, node->link->to->name);
				if (firstval)
					firstval = FALSE;
				else
					g_string_append (values, " ,");
				g_string_sprintfa (values, "%s", ce->linkval);
			}
			break;
		}

		list = g_slist_next (list);
	}
	g_string_append (values, ")");
	g_string_sprintfa (query, ") %s", values->str);
	g_string_free (values, TRUE);

	str = g_strdup_printf (_("Insertion query is:\n%s\n"
				 "Do You want to proceed?"), query->str);
	dlg = gnome_question_dialog (str, (GnomeReplyCallback) dummy_cb,
				     NULL);
	g_free (str);
	i = gnome_dialog_run_and_close (GNOME_DIALOG (dlg));
	if (!i) {		/* saving */
		/*g_print("Action Query is #%s#\n", query->str); */
		sql_access_do_query (qfm->qx->q->conf->srv, query->str);
		qfm_do_action_refresh (NULL, qfm);
		gtk_object_destroy (GTK_OBJECT (qfm));
	}
	g_string_free (query, TRUE);
}

static void
qfm_do_action_first (GtkWidget * button, SqlQueryForm * qfm)
{
	qfm->numrow = 1;
	sql_query_form_update (qfm);
	qfm->modified = FALSE;
	update_buttons_sensitiveness (qfm);
}

static void
qfm_do_action_prev (GtkWidget * button, SqlQueryForm * qfm)
{
	qfm->numrow--;
	sql_query_form_update (qfm);
	qfm->modified = FALSE;
	update_buttons_sensitiveness (qfm);
}

static void
qfm_do_action_next (GtkWidget * button, SqlQueryForm * qfm)
{
	qfm->numrow++;
	sql_query_form_update (qfm);
	qfm->modified = FALSE;
	update_buttons_sensitiveness (qfm);
}

static void
qfm_do_action_last (GtkWidget * button, SqlQueryForm * qfm)
{
	qfm->numrow = qfm->qx->qres->nbtuples;
	sql_query_form_update (qfm);
	qfm->modified = FALSE;
	update_buttons_sensitiveness (qfm);
}

static void
qfm_do_action_insert (GtkWidget * button, SqlQueryForm * qfm)
{
	gtk_signal_emit (GTK_OBJECT (qfm),
			 sql_query_form_signals[ACTION_INSERT]);
}

static void
qfm_do_action_refresh (GtkWidget * button, SqlQueryForm * qfm)
{
	gtk_signal_emit (GTK_OBJECT (qfm),
			 sql_query_form_signals[ACTION_REFRESH]);
}

static void
qfm_do_action_delete (GtkWidget * button, SqlQueryForm * qfm)
{
	gchar *str;
	str = g_strdup_printf (_("To be implemented in %s, line %d"),
			       __FILE__, __LINE__);
	gnome_app_message (GNOME_APP (qfm->qx->q->conf->app), str);
	g_free (str);
}

static void
qfm_do_action_viewall (GtkWidget * button, SqlQueryForm * qfm)
{
	gtk_signal_emit (GTK_OBJECT (qfm),
			 sql_query_form_signals[ACTION_VIEWALL]);
}

static void
qfm_do_action_choose (GtkWidget * button, SqlQueryForm * qfm)
{
	gtk_signal_emit (GTK_OBJECT (qfm),
			 sql_query_form_signals[ACTION_CHOOSE], qfm->numrow);
}

static void
update_buttons_sensitiveness (SqlQueryForm * qfm)
{
	gchar *str;
	GtkWidget *buttons, *button;
	gboolean at_begin = FALSE;
	gboolean at_end = FALSE;

	if (qfm->numrow <= 1)
		at_begin = TRUE;

	if (qfm->numrow >= qfm->qx->qres->nbtuples)
		at_end = TRUE;

	buttons = gtk_object_get_data (GTK_OBJECT (qfm), "buttons");

	str = g_strdup_printf ("%ld", QUERY_ACTION_FIRST);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, at_begin ? FALSE : TRUE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_PREV);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, at_begin ? FALSE : TRUE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_NEXT);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, at_end ? FALSE : TRUE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_LAST);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, at_end ? FALSE : TRUE);

	str = g_strdup_printf ("%ld", QUERY_ACTION_COMMIT);
	button = gtk_object_get_data (GTK_OBJECT (buttons), str);
	g_free (str);
	if (button)
		gtk_widget_set_sensitive (button, FALSE);
}
