/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-source-viewer-manager.h"

/* FIXME: Breakpoints need to be restored in a source viewer that is destroyed
 * and recreated */

struct _GdfSourceViewerManagerPriv {
    GtkWidget *notebook;
    GtkWidget *hbox;
    GtkWidget *sources;

	GdfSourceViewer *prev_sv;
	gboolean has_empty_page : 1;
    gint num_pages;

	GHashTable *breakpoint_locations;

	/* signal connections */
    gint sources_changed_sig;

	gint program_loaded_sig;
	gint program_unloaded_sig;
	gint breakpoint_set_sig;
	gint breakpoint_enabled_sig;
	gint breakpoint_disabled_sig;
	gint breakpoint_deleted_sig;
	gint execution_killed_sig;
	gint execution_exited_sig;
	gint execution_source_line_sig;
    gint frame_changed_sig;
	
    GdfDebuggerClient *dbg;
};

enum {
    ARG_0,
    ARG_DEBUGGER
};

static void source_viewer_manager_init (GdfSourceViewerManager *svm);
static void source_viewer_manager_class_init (GdfSourceViewerManagerClass *klass);
static void source_viewer_manager_destroy (GtkObject *object);
static void get_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void set_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void fill_sources_combo (GdfSourceViewerManager *svm);
static void unload_sources_combo (GdfSourceViewerManager *svm);
static void sources_entry_changed_cb (GtkWidget *entry, 
                                      GdfSourceViewerManager *svm);
static GtkWidget *view_source (GdfSourceViewerManager *svm, 
							   const gchar *source_path);
static GtkWidget *create_source_page (GdfSourceViewerManager *svm,
									  const gchar *source_path);
static void create_empty_page (GdfSourceViewerManager *svm);
static void remove_empty_page (GdfSourceViewerManager *svm);
static gint find_opened_page (GdfSourceViewerManager *svm, 
							  const gchar *source_path);
static void remove_page (GdfSourceViewerManager *svm, gint page_num);
static void clear_manager (GdfSourceViewerManager *svm);
static void add_breakpoint (GdfSourceViewerManager *svm, GDF_Breakpoint *bp);
static void enable_breakpoint (GdfSourceViewerManager *svm,
							   gint bp_num, gboolean enabled);
static void remove_breakpoint (GdfSourceViewerManager *svm, gint bp_num);
static void connect_debugger_signals (GdfSourceViewerManager *svm);
static void disconnect_debugger_signals (GdfSourceViewerManager *svm);
static void program_loaded_cb (GdfDebuggerClient *dbg,
                               GdfSourceViewerManager *svm);
static void program_unloaded_cb (GdfDebuggerClient *dbg,
								 GdfSourceViewerManager *svm);
static void breakpoint_set_cb (GdfDebuggerClient *dbg,
							   gint bp_num,
							   GdfSourceViewerManager *svm);
static void breakpoint_enabled_cb (GdfDebuggerClient *dbg,
								   gint bp_num,
								   GdfSourceViewerManager *svm);
static void breakpoint_disabled_cb (GdfDebuggerClient *dbg,
									gint bp_num,
									GdfSourceViewerManager *svm);
static void breakpoint_deleted_cb (GdfDebuggerClient *dbg,
								   gint bp_num,
								   GdfSourceViewerManager *svm);
static void execution_source_line_cb (GdfDebuggerClient *dbg, 
									  gchar *file, gint line, 
									  GdfSourceViewerManager *svm);
static void execution_exited_cb (GdfDebuggerClient *dbg,
								 gint exit_code, GdfSourceViewerManager *svm);
static void frame_changed_cb (GdfDebuggerClient *dbg, gint new_frame,
                              GdfSourceViewerManager *svm);
static guint bp_hash (gconstpointer v);
static gint bp_equal (gconstpointer v, gconstpointer v2);
static gboolean breakpoint_hash_remove_func (gpointer key,
											 gpointer value,
											 gpointer user_data);

static GtkVBoxClass *parent_class;

/*
 * Public Interface
 */

GtkType
gdf_source_viewer_manager_get_type (void)
{
	static GtkType type = 0;
	
	if (!type) {
		static const GtkTypeInfo info = {
			"GdfSourceViewerManager",
			sizeof (GdfSourceViewerManager),
			sizeof (GdfSourceViewerManagerClass),
			(GtkClassInitFunc) source_viewer_manager_class_init,
			(GtkObjectInitFunc) source_viewer_manager_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL
		};
	
		type = gtk_type_unique (gtk_vbox_get_type (), &info);
	}

	return type;
}

GdfSourceViewerManager *
gdf_source_viewer_manager_new (void)
{
	GdfSourceViewerManager *svm;
	
	svm = gtk_type_new (gdf_source_viewer_manager_get_type ());

	return svm;
}

GtkWidget *
gdf_source_viewer_manager_view_source (GdfSourceViewerManager *svm, 
                                       const gchar *filename)
{
    return view_source (svm, filename);
}

/**
 * gdf_source_viewer_manager_set_debugger:
 * @bpm: The source viewer manager.
 * @dbg: The debugger to associate.
 * 
 * Sets the source viewer manager's associated debugger.  The new debugger 
 * will be ref'ed by the breakpoint manager.  Any previous debugger will be 
 * unref'ed.
 **/
void
gdf_source_viewer_manager_set_debugger (GdfSourceViewerManager *svm,
										 GdfDebuggerClient *dbg)
{
	g_return_if_fail (svm != NULL);
	g_return_if_fail (GDF_IS_SOURCE_VIEWER_MANAGER (svm));
	
	if (svm->priv->dbg) {
		disconnect_debugger_signals (svm);
		
		gtk_object_unref (GTK_OBJECT (svm->priv->dbg));
	}

    if (svm->priv->breakpoint_locations) {
        g_hash_table_foreach_remove (svm->priv->breakpoint_locations,
                                     breakpoint_hash_remove_func, NULL);
        g_hash_table_destroy (svm->priv->breakpoint_locations);
    }
	
	svm->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
        if (GTK_OBJECT_FLOATING (GTK_OBJECT (dbg))) 
        gtk_object_sink (GTK_OBJECT (dbg));
        
        connect_debugger_signals (svm);
        
        if (svm->priv->dbg->program_loaded) {
            fill_sources_combo (svm);
            svm->priv->breakpoint_locations =
                g_hash_table_new (bp_hash, bp_equal);
        }
    }
}

/*
 * Class/Object functions
 */

void
source_viewer_manager_class_init (GdfSourceViewerManagerClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;

    gtk_object_add_arg_type ("GdfSourceViewerManager::debugger",
                             GTK_TYPE_OBJECT,
                             GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
                             ARG_DEBUGGER);

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	
	object_class->destroy = source_viewer_manager_destroy;
    object_class->get_arg = get_arg;
    object_class->set_arg = set_arg;
}

void
source_viewer_manager_init (GdfSourceViewerManager *svm)
{
    GtkWidget *label;
	svm->priv = g_new0 (GdfSourceViewerManagerPriv, 1);

	svm->priv->notebook = gtk_notebook_new ();
    gtk_notebook_set_scrollable (GTK_NOTEBOOK (svm->priv->notebook), TRUE);
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (svm->priv->notebook), FALSE);
    svm->priv->hbox = gtk_hbox_new (FALSE, 5);
    svm->priv->sources = gtk_combo_new ();
    gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (svm->priv->sources)->entry),
                            FALSE);

    label = gtk_label_new (N_("Source: "));
    
    gtk_box_pack_start (GTK_BOX (svm->priv->hbox), label, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (svm->priv->hbox), svm->priv->sources, 
                        FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (svm), svm->priv->notebook, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (svm), svm->priv->hbox, FALSE, FALSE, 2);
   
    gtk_widget_show (svm->priv->notebook);
    gtk_widget_show (label);
    gtk_widget_show (svm->priv->sources);
    gtk_widget_show (svm->priv->hbox);

	svm->priv->has_empty_page = FALSE;
    svm->priv->num_pages = 0;
	svm->priv->prev_sv = NULL;
	svm->priv->dbg = NULL;
    svm->priv->sources_changed_sig = -1;
	svm->priv->breakpoint_locations = NULL;
    
	create_empty_page (svm);
}

void
source_viewer_manager_destroy (GtkObject *obj)
{
	GdfSourceViewerManager *svm = GDF_SOURCE_VIEWER_MANAGER (obj);
	
	if (svm->priv->dbg) {
		disconnect_debugger_signals (svm);
		gtk_object_unref (GTK_OBJECT (svm->priv->dbg));
	}
	
    if (svm->priv->breakpoint_locations) {
        g_hash_table_foreach_remove (svm->priv->breakpoint_locations,
                                     breakpoint_hash_remove_func, NULL);
        g_hash_table_destroy (svm->priv->breakpoint_locations);
    }

	g_free (svm->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}


void 
get_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfSourceViewerManager *svm = GDF_SOURCE_VIEWER_MANAGER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (svm->priv->dbg);
        break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfSourceViewerManager *svm = GDF_SOURCE_VIEWER_MANAGER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GDF_TRACE ();
        gdf_source_viewer_manager_set_debugger (svm,
                                                 GTK_VALUE_OBJECT (*arg));
        break;
    default :
        break;
    }   
}

/*
 * Helper functions
 */

void
fill_sources_combo (GdfSourceViewerManager *svm)
{
    GDF_source_file_list* sources = 
        gdf_debugger_client_get_sources (svm->priv->dbg);
    GList *source_names = NULL;
    int i;

    /* Move the sources into the list */
    for (i = 0; i < sources->_length; i++) {
        /* FIXME: Allow the user to decide whether they want headers or not */
        if (sources->_buffer[i][strlen (sources->_buffer[i]) - 1] != 'h')
            source_names = g_list_prepend (source_names, sources->_buffer[i]);
    }

    if (svm->priv->sources_changed_sig != -1) {
        gtk_signal_disconnect (GTK_OBJECT (GTK_COMBO (svm->priv->sources)->entry),
                               svm->priv->sources_changed_sig);
    }
    gtk_combo_set_popdown_strings (GTK_COMBO (svm->priv->sources),
                                   source_names);
    svm->priv->sources_changed_sig = 
        gtk_signal_connect (GTK_OBJECT (GTK_COMBO (svm->priv->sources)->entry), 
                            "changed", 
                            GTK_SIGNAL_FUNC (sources_entry_changed_cb), svm);
   
    CORBA_free (sources);
    g_list_free (source_names);
}

void 
unload_sources_combo (GdfSourceViewerManager *svm)
{
    gtk_signal_disconnect (GTK_OBJECT (GTK_COMBO (svm->priv->sources)->entry),
                           svm->priv->sources_changed_sig);
    svm->priv->sources_changed_sig = -1;
    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (svm->priv->sources)->entry), "");
    gtk_list_clear_items (GTK_LIST (GTK_COMBO (svm->priv->sources)->list),
                          0, -1);
}

/* makes sure the sources entry contains the correct source file */
static void 
check_sources_entry (GdfSourceViewerManager *svm, gchar *filename) 
{
    gchar *text;
    text = strrchr (filename, G_DIR_SEPARATOR) + 1;
    if (!(text - 1)) 
        text = filename;
    
    if (strcmp (gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (svm->priv->sources)->entry)),
                text) != 0) {
        
        gtk_signal_disconnect (GTK_OBJECT (GTK_COMBO (svm->priv->sources)->entry),
                               svm->priv->sources_changed_sig);
        
        gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (svm->priv->sources)->entry),
                            text);
        
        svm->priv->sources_changed_sig = 
            gtk_signal_connect (GTK_OBJECT (GTK_COMBO (svm->priv->sources)->entry), 
                                "changed", 
                                GTK_SIGNAL_FUNC (sources_entry_changed_cb), 
                                svm);
    }
}

void
sources_entry_changed_cb (GtkWidget *entry, GdfSourceViewerManager *svm)
{
    gchar *text, *filename;
    text = gtk_entry_get_text (GTK_ENTRY (entry));
    filename = gdf_debugger_client_get_absolute_source_path (svm->priv->dbg,
                                                              text);
    view_source (svm, filename);
    g_free (filename);
}

GtkWidget *
view_source (GdfSourceViewerManager *svm, const gchar *source_path)
{
	gint page_num;
	GtkWidget *ret;
	
	remove_empty_page (svm);

	if ((page_num = find_opened_page (svm, source_path)) > -1) {
		gtk_notebook_set_page (GTK_NOTEBOOK (svm->priv->notebook), page_num);
		ret = gtk_notebook_get_nth_page (GTK_NOTEBOOK (svm->priv->notebook), 
										 page_num);
	} else {
		ret = create_source_page (svm, source_path);
        if (ret == NULL) {
            GtkWidget *msg = gnome_message_box_new (_("File does not exist."), 
                                                    GNOME_MESSAGE_BOX_ERROR, 
                                                    GNOME_STOCK_BUTTON_OK, 
                                                    NULL);
            gtk_widget_show (msg);

            if (svm->priv->num_pages == 0) 
                create_empty_page (svm);

        } else {
            gtk_notebook_set_page (GTK_NOTEBOOK (svm->priv->notebook), 
                                   svm->priv->num_pages - 1);
        }
	}

    check_sources_entry (svm, (char*)source_path);

	return ret;	
}	

GtkWidget *
create_source_page (GdfSourceViewerManager *svm, const gchar *source_path)
{
	gchar *tab_text;
	GtkWidget *tab_label;
	GtkWidget *source_viewer;

	tab_text = strrchr (source_path, G_DIR_SEPARATOR) + 1;
    
	if (tab_text == 1)
        tab_text = (char*)source_path;
	
	tab_label = gtk_label_new (tab_text);
	gtk_object_set_data_full (GTK_OBJECT (tab_label),
                              "full_filename", 
							  g_strdup (source_path),
							  (GtkDestroyNotify) g_free);

	source_viewer = gdf_source_viewer_new (source_path);

    if (source_viewer == NULL)
        return NULL;

	gdf_source_viewer_set_debugger (GDF_SOURCE_VIEWER (source_viewer), 
									 svm->priv->dbg);
	gtk_notebook_append_page (GTK_NOTEBOOK (svm->priv->notebook), 
                              source_viewer, tab_label);
	svm->priv->num_pages++;

	gtk_widget_show (source_viewer);
	gtk_widget_show (tab_label);

	return source_viewer;
}

void
create_empty_page (GdfSourceViewerManager *svm)
{
	GtkWidget *tab_label;
	GtkWidget *child_label;

    if (!svm->priv->has_empty_page) {
        tab_label = gtk_label_new (_("No file"));
        child_label = gtk_label_new (_("No source file loaded."));
        
        gtk_notebook_append_page (GTK_NOTEBOOK (svm->priv->notebook), 
                                  child_label, tab_label);
        
        gtk_widget_show (tab_label);
        gtk_widget_show (child_label);
        svm->priv->has_empty_page = TRUE;
    }
}

void
remove_empty_page (GdfSourceViewerManager *svm)
{
	if (svm->priv->has_empty_page) {
		gtk_notebook_remove_page (GTK_NOTEBOOK (svm->priv->notebook), 0);
        svm->priv->has_empty_page = FALSE;
    }
}


gint 
find_opened_page (GdfSourceViewerManager *svm, 
				  const gchar *source_path)
{
	GtkWidget *page;
	GtkWidget *label;
	gchar *full_filename;
	gint cur = 0;

    if (svm->priv->num_pages == 0)
        return -1;
	
	page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (svm->priv->notebook),
									  cur);
	
	while (page != NULL) {
		label = 
			gtk_notebook_get_tab_label (GTK_NOTEBOOK (svm->priv->notebook), 
                                        page);
		full_filename =
			gtk_object_get_data (GTK_OBJECT (label), "full_filename");
		if (strcmp (full_filename, source_path) == 0) {
			return cur;
		}
		
		page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (svm->priv->notebook), 
                                          ++cur);
	}
	return -1;
}

void
remove_page (GdfSourceViewerManager *svm, gint page_num)
{
	gtk_notebook_remove_page (GTK_NOTEBOOK (svm->priv->notebook), page_num);
	svm->priv->num_pages--;
    if (svm->priv->num_pages == 0 && !svm->priv->has_empty_page)
        create_empty_page (svm);
}

void
clear_manager (GdfSourceViewerManager *svm)
{
	while (svm->priv->num_pages) {
		remove_page (svm, 0);
	}
}

void
add_breakpoint (GdfSourceViewerManager *svm,
				GDF_Breakpoint *bp)
{
	GtkWidget *sv;
    char *filename;
    
    filename = gdf_debugger_client_get_absolute_source_path (svm->priv->dbg, 
                                                              bp->file_name);
    
	sv = view_source (svm, filename);

    if (!sv) 
        return;
    
	gdf_source_viewer_add_breakpoint (GDF_SOURCE_VIEWER (sv), bp->num, 
									  bp->line_num, bp->enabled);

	/* Store a breakpoint->source path mapping */
	g_hash_table_insert (svm->priv->breakpoint_locations,
						 GINT_TO_POINTER (bp->num),
						 (gpointer)g_strdup (filename));
    g_free (filename);
}

void
enable_breakpoint (GdfSourceViewerManager *svm, gint bp_num, gboolean enabled)
{
	GtkWidget *sv;
	gchar *source_path;
	
	source_path = g_hash_table_lookup (svm->priv->breakpoint_locations, 
									   GINT_TO_POINTER (bp_num));

	sv = view_source (svm, source_path);
    
    if (!sv) 
        return;

	/* If the source viewer has been destroyed and re-created, it doesn't
	 * have any information about this breakpoint.  In this case we will 
	 * need to re-add the breakpoint. */
	if (gdf_source_viewer_has_breakpoint (GDF_SOURCE_VIEWER (sv), bp_num)) {
		gdf_source_viewer_enable_breakpoint (GDF_SOURCE_VIEWER (sv), bp_num,
											 enabled);
	} else {
		GDF_Breakpoint *bp;
		bp = gdf_debugger_client_get_breakpoint_info (svm->priv->dbg, 
													   bp_num);

		g_assert (bp->enabled == TRUE);
		
		CORBA_free (bp);
	}
}

void
remove_breakpoint (GdfSourceViewerManager *svm, gint bp_num)
{
	gchar *source_path;
	gint page;
	
	source_path = g_hash_table_lookup (svm->priv->breakpoint_locations,
									   GINT_TO_POINTER (bp_num));

	/* Only send this event to opened sources - don't bother reopening a
     * breakpoint just to delete it */
	if ((page = find_opened_page (svm, source_path)) > -1) {
		GtkWidget *sv;
		sv = gtk_notebook_get_nth_page (GTK_NOTEBOOK (svm->priv->notebook), 
                                        page);
		gdf_source_viewer_remove_breakpoint (GDF_SOURCE_VIEWER (sv),
											 bp_num);
	}

	g_hash_table_remove (svm->priv->breakpoint_locations, 
						 GINT_TO_POINTER (bp_num));
}

void
connect_debugger_signals (GdfSourceViewerManager *svm)
{
	svm->priv->program_loaded_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"program_loaded",
							GTK_SIGNAL_FUNC (program_loaded_cb),
							(gpointer)svm);
	svm->priv->program_unloaded_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"program_unloaded",
							GTK_SIGNAL_FUNC (program_unloaded_cb),
							(gpointer)svm);
	svm->priv->breakpoint_set_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"breakpoint_set",
							GTK_SIGNAL_FUNC (breakpoint_set_cb),
							(gpointer)svm);
	svm->priv->breakpoint_enabled_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"breakpoint_enabled",
							GTK_SIGNAL_FUNC (breakpoint_enabled_cb),
							(gpointer)svm);
	svm->priv->breakpoint_disabled_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"breakpoint_disabled",
							GTK_SIGNAL_FUNC (breakpoint_disabled_cb),
							(gpointer)svm);
	svm->priv->breakpoint_deleted_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"breakpoint_deleted",
							GTK_SIGNAL_FUNC (breakpoint_deleted_cb),
							(gpointer)svm);
	svm->priv->execution_source_line_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"execution_source_line",
							GTK_SIGNAL_FUNC (execution_source_line_cb),
							(gpointer)svm);
	svm->priv->execution_exited_sig = 
		gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
							"execution_exited",
							GTK_SIGNAL_FUNC (execution_exited_cb),
							(gpointer)svm);
    svm->priv->frame_changed_sig = 
        gtk_signal_connect (GTK_OBJECT (svm->priv->dbg),
                            "stack_frame_changed",
                            GTK_SIGNAL_FUNC (frame_changed_cb),
                            (gpointer)svm);
}

void
disconnect_debugger_signals (GdfSourceViewerManager *svm)
{
    gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->program_loaded_sig);
    gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->program_unloaded_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->breakpoint_set_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->breakpoint_enabled_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->breakpoint_disabled_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->breakpoint_deleted_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg), 
						   svm->priv->execution_source_line_sig);
	gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
						   svm->priv->execution_exited_sig);
    gtk_signal_disconnect (GTK_OBJECT (svm->priv->dbg),
                           svm->priv->frame_changed_sig);
}

/* 
 * Debugger Callbacks 
 */
void
program_loaded_cb (GdfDebuggerClient *dbg, GdfSourceViewerManager *svm)
{
    svm->priv->num_pages = 0;
	svm->priv->prev_sv = NULL;
	svm->priv->breakpoint_locations = g_hash_table_new (bp_hash, bp_equal);

    fill_sources_combo (svm);
}

void
program_unloaded_cb (GdfDebuggerClient *dbg,
					 GdfSourceViewerManager *svm)
{
	g_hash_table_foreach_remove (svm->priv->breakpoint_locations,
								 breakpoint_hash_remove_func, NULL);
	g_hash_table_destroy (svm->priv->breakpoint_locations);
    svm->priv->breakpoint_locations = NULL;

    unload_sources_combo (svm);
	clear_manager (svm);
}

void 
breakpoint_set_cb (GdfDebuggerClient *dbg,
				   gint bp_num,
				   GdfSourceViewerManager *svm)
{
	GDF_Breakpoint *bp;
	
	bp = gdf_debugger_client_get_breakpoint_info (dbg, bp_num);

	add_breakpoint (svm, bp);

	CORBA_free (bp);
}

void 
breakpoint_enabled_cb (GdfDebuggerClient *dbg,
					   gint bp_num,
					   GdfSourceViewerManager *svm)
{
	enable_breakpoint (svm, bp_num, TRUE);
}

void 
breakpoint_disabled_cb (GdfDebuggerClient *dbg,
						gint bp_num,
						GdfSourceViewerManager *svm)
{
	enable_breakpoint (svm, bp_num, FALSE);
}

void 
breakpoint_deleted_cb (GdfDebuggerClient *dbg,
					   gint bp_num,
					   GdfSourceViewerManager *svm)
{
	remove_breakpoint (svm, bp_num);
}

void 
execution_source_line_cb (GdfDebuggerClient *dbg, 
						  gchar *file, gint line, 
						  GdfSourceViewerManager *svm)
{
	GtkWidget *sv;
	
	if (svm->priv->prev_sv)
		gdf_source_viewer_clear_current_line (svm->priv->prev_sv);
	
	sv = view_source (svm, file);

    if (!sv)
        return;

	gdf_source_viewer_set_current_line (GDF_SOURCE_VIEWER (sv), line);
	
	svm->priv->prev_sv = GDF_SOURCE_VIEWER (sv);
}

void
execution_exited_cb (GdfDebuggerClient *dbg, gint exit_code,
					 GdfSourceViewerManager *svm)
{
	if (svm->priv->prev_sv)
		gdf_source_viewer_clear_current_line (svm->priv->prev_sv);
	
	svm->priv->prev_sv = NULL;
}

void
frame_changed_cb (GdfDebuggerClient *dbg, gint new_frame,
                  GdfSourceViewerManager *svm)
{
    GDF_StackFrame *frame;
    GtkWidget *sv;
    char *full_path;

    frame = gdf_debugger_client_get_frame (dbg, new_frame);
    full_path = 
        gdf_debugger_client_get_absolute_source_path (dbg, 
                                                       frame->location.file);

    sv = view_source (svm, full_path);
    gdf_source_viewer_view_line (GDF_SOURCE_VIEWER (sv), 
                                 frame->location.line);

    CORBA_free (frame);
    g_free (full_path);
}

/* Breakpoint hash functions */

guint 
bp_hash (gconstpointer v)
{
	return (guint)(GPOINTER_TO_INT (v));
}
	
gint 
bp_equal (gconstpointer v, gconstpointer v2)
{
	return GPOINTER_TO_INT (v) == GPOINTER_TO_INT (v2);
}

gboolean 
breakpoint_hash_remove_func (gpointer key,
							 gpointer value,
							 gpointer user_data)
{
	g_free (value);

	return TRUE;
}


