/*  -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Symbol viewing for libdryad.
 * 
 * Copyright (C) 1999 Dave Camp <campd@oit.edu>, 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include "libdryad.h"
#include <string.h>
#include "gdf-libdryad-debugger.h"

static void read_value (GdbInstance *inst, GDF_Symbol *sym);
static void check_type (GDF_Symbol *sym);
static void expand_pointer (GdbInstance *inst, GdfLibdryadSymbolSet *ss,
			    GDF_Symbol *sym);
static void expand_structure (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
			      GDF_Symbol *sym);

static GDF_SymbolSequence *init_seq (void);
static void grow_seq (GDF_SymbolSequence *seq, int amount);

void
gdb_get_locals (GdbInstance *inst, GdfLibdryadSymbolSet *ss)
{
    char *line;
    GList *names = NULL;
    GList *i;
    char *work;

    if (ss->symbols) {
	CORBA_free (ss->symbols);
    }
    ss->set_size = 0;
    
    gdb_execute_command (inst, "info locals");

    ss->symbols = gdb_allocate_symbols ();

    /* Read the names of all the locals */
    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if ((work = strstr (line, " = "))) {
	    *work = '\0';
	    work += 3;
	    
	    names = g_list_prepend (names, g_strdup (line));
	    
	    if (!strncmp (work, "{", 1)) {
		int levels = 1;
		while (levels > 0) {
		    line = gdb_read_line (inst);
		    if (!strncmp (line, "{", 1))
			levels++;
		    if (!strncmp (line, "}", 1))
			levels--;
		}
	    }
	}
	line = gdb_read_line (inst);
    }


    /* Add all the locals to the symbol set */
    names = g_list_reverse (names);

    for (i = names; i != NULL; i = i->next) {
	gdb_add_expression (inst, ss, i->data);
	g_free (i->data);
    }
    g_list_free (names);

    /* Notify the symbol set that it has changed */
    gdf_libdryad_symbol_set_changed (ss);
}

void
gdb_get_expression (GdbInstance *inst, GDF_Symbol *sym, const char *expr)
{
    char *cmd_str;
    
    cmd_str = g_strdup_printf ("output %s", expr);
    gdb_execute_command (inst, cmd_str);
    g_free (cmd_str);
    
    sym->name = CORBA_string_dup (expr);
    sym->expression = CORBA_string_dup (expr);
    read_value (inst, sym);
    check_type (sym);
}

void
gdb_expand_symbol (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		   GDF_Symbol *sym)
{
    g_return_if_fail (sym->expandable);

    if (!sym->expanded) {
	switch (sym->type) {
	case GDF_TYPE_PTR :
	    expand_pointer (inst, ss, sym);
	    break;
	case GDF_TYPE_STRUCT :
	    expand_structure (inst, ss, sym);
	    break;
	default:
	    return;
	}
    }

    sym->expanded = TRUE;
}

static void
update_seq (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
	    GDF_SymbolSequence *seq) 
{
    char *cmd_str;
    char *old_value;
    GDF_Symbol *sym;
    CORBA_boolean was_expandable;

    int i;

    for (i = 0; i < seq->_length; i++) {
	sym = &seq->_buffer[i];
	
	cmd_str = g_strdup_printf ("output %s", sym->expression);
	gdb_execute_command (inst, cmd_str);
	g_free (cmd_str);
	
	old_value = sym->value;
	
	was_expandable = sym->expandable;

	read_value (inst, sym);
	check_type (sym);
	if (strcmp (old_value, sym->value) 
	    || was_expandable != sym->expandable) {
	    CORBA_any *event_any;

	    event_any = gdf_marshal_event_long ("symbol_changed", 
						sym->handle);
	    
	    gdf_event_channel_client_push (ss->event_channel,
					   event_any);
	    CORBA_free (event_any);
	}
	
	CORBA_free (old_value);
	
	update_seq (inst, ss, 
		    (GDF_SymbolSequence*)sym->children._value);
    }
}

void
gdb_update_symbols (GdbInstance *inst, GdfLibdryadSymbolSet *ss)
{
    update_seq (inst, ss, ss->symbols);
}

GDF_SymbolSequence *
gdb_allocate_symbols (void)
{
    return init_seq ();
}

static void
add_expression_to_seq  (GdbInstance *inst,
			GDF_SymbolSequence *seq, 
			GdfLibdryadSymbolSet *ss, 
			const char *expr)
{
    GDF_Symbol *sym;
    
    grow_seq (seq, 1);
    sym = &seq->_buffer[seq->_length - 1];
    gdb_get_expression (inst, sym, expr);
    
    sym->handle = ss->set_size++;
}

void
gdb_add_expression (GdbInstance *inst,
		    GdfLibdryadSymbolSet *ss, 
		    const char *expr)
{
    add_expression_to_seq (inst, ss->symbols, ss, expr);
}

void
read_value (GdbInstance *inst, GDF_Symbol *sym) 
{
    char *line;

    sym->value = NULL;
    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if (!strncmp(line, "\32\32value-begin", strlen("\32\32value-begin"))) {
	    line = gdb_read_line (inst);
	    
	    while (line[0] == '\0' && strcmp (line, "\32\32\error-begin")) {
		line = gdb_read_line (inst);
	    }
	    
	    if (strcmp (line, "\32\32error-begin")) {
		sym->value = CORBA_string_dup (line);
	    } else {
		while (strcmp (line, "\32\32error")) {
		    char *new;
		    line = gdb_read_line (inst);
		    new = g_strdup_printf ("%s%s",
					   sym->value ? sym->value : "", 
					   line);
		    if (sym->value) {
			CORBA_free (sym->value);
		    }
		    sym->value = CORBA_string_dup (new);
		    g_free (new);
		}
		break;
	    }

	    if (!strncmp (line, "{", 1)) {
		int levels = 1;
		while (levels > 0) {
		    line = gdb_read_line (inst);
		    if (!strncmp (line, "{", 1))
			levels++;
		    if (!strncmp (line, "}", 1))
			levels--;
		}
	    }
	    break;
	}
	if (!strcmp (line, "\32\32error-begin")) {
	    while (strcmp (line, "\32\32error")) {
		char *new;
		line = gdb_read_line (inst);
		new = g_strdup_printf ("%s%s",
				       sym->value ? sym->value : "", 
					   line);
		if (sym->value) {
		    CORBA_free (sym->value);
		}
		sym->value = CORBA_string_dup (new);
		g_free (new);
	    }
	}
	line = gdb_read_line (inst);
    }
    g_assert (sym->value != NULL);
}

void
check_type (GDF_Symbol *sym)
{
    /* Check for pointer types */
    char *val = sym->value;
    
    sym->expandable = FALSE;
    sym->type = GDF_TYPE_NORMAL;

    if (val[0] == '(') {
	while (*val != ')') {
            if (*val++ == '*') {  /* FIXME: This is C/C++ specific */
		sym->expandable = TRUE;
		sym->type = GDF_TYPE_PTR;
		return;
	    }
	}
    }

    val = sym->value;
    if (val[0] == '{') {
	sym->expandable = TRUE;
	sym->type = GDF_TYPE_STRUCT;
	return;
    }
    /* FIXME: Check for array types */

    sym->type = GDF_TYPE_NORMAL;
}

static void
add_child_expression (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		      GDF_Symbol *sym, const char *name, const char *expr)
{
    GDF_Symbol *new_sym;
    GDF_SymbolSequence *children;

    children = sym->children._value;
    add_expression_to_seq (inst, children, ss, expr);
    new_sym = &(children->_buffer[children->_length - 1]);
    CORBA_free (new_sym->name);
    new_sym->name = CORBA_string_dup (name);
}

void
expand_pointer (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		GDF_Symbol *sym)
{
    char *new_name;
    char *new_expr;
    new_name = g_strdup_printf ("*%s", sym->name);
    new_expr = g_strdup_printf ("*(%s)", sym->expression);
    add_child_expression (inst, ss, sym, new_name, new_expr);
    g_free (new_expr);
    g_free (new_name);
}

void
expand_structure (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		  GDF_Symbol *sym)
{
    GDF_SymbolSequence *children;
    char *expr;
    char *cmd_str;
    char *line;
    GList *names = NULL;
    GList *i;

    cmd_str = g_strdup_printf ("output %s", sym->expression);
    gdb_execute_command (inst, cmd_str);
    g_free (cmd_str);
    
    children = sym->children._value;
    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if (!strncmp (line, "\32\32field-begin", 
		      strlen ("\32\32field-begin"))) {
	    line = gdb_read_line (inst);
	    names = g_list_prepend (names, g_strdup (line));

	    if (!strncmp (line, "{", 1)) {
		int levels = 1;
		while (levels > 0) {
		    line = gdb_read_line (inst);
		    if (!strncmp (line, "{", 1))
			levels++;
		    if (!strncmp (line, "}", 1))
			levels--;
		}
	    }
	}
	line = gdb_read_line (inst);
    }

    names = g_list_reverse (names);
    for (i = names; i != NULL; i = i->next) {
	g_assert (i != NULL);
	expr = g_strdup_printf ("(%s).%s", sym->expression, (char*)i->data);
	g_assert (i != NULL);
	add_child_expression (inst, ss, sym, i->data, expr);
	g_free (expr);
	g_free (i->data);
    }
    g_list_free (names); 
}

static GDF_SymbolSequence *
init_seq (void)
{
    GDF_SymbolSequence *seq = GDF_SymbolSequence__alloc ();
    seq->_length = 0;
    seq->_maximum = 0;
    seq->_buffer = NULL;
    /* We tell it not to release now, but will set release to TRUE when the
     * buffer is grown for the first time */
    CORBA_sequence_set_release (seq, CORBA_FALSE);
    return seq;
}

void
grow_seq (GDF_SymbolSequence *seq, int amount)
{
    int i;

    int old_max = seq->_maximum;
    GDF_Symbol *old_buf = seq->_buffer;

    seq->_maximum += amount;
    seq->_length += amount;
    seq->_buffer = CORBA_sequence_GDF_Symbol_allocbuf (seq->_maximum);

    if (old_buf) {
	ORBit_mem_info *block;

	for (i = 0; i < old_max; i++) {
	    GDF_Symbol *sym = &(seq->_buffer[i]);
	    GDF_Symbol *old_sym = &old_buf[i];
	    sym->symbol_set = old_sym->symbol_set;
	    sym->handle = old_sym->handle;
	    sym->name = old_sym->name;
	    sym->expression = old_sym->expression;
	    sym->type = old_sym->type;
	    sym->value = old_sym->value;
	    sym->expandable = old_sym->expandable;
	    sym->expanded = old_sym->expanded;
	    sym->children = old_sym->children;
	}

	/* This is somewhat not nice.  Basically, we want to free the previous
	 * buffer, but we don't want the deep free that is done by 
	 * CORBA_free.  To get around this we cheat and free the buffer
	 * ourselves */

	block = PTR_TO_MEMINFO (old_buf);
	g_free (block);
    } else {
	/* _release was previously set to FALSE (because there was no buffer).
	 * set it to true now that a buffer has been created. */
	CORBA_sequence_set_release (seq, CORBA_TRUE);
    }

    for (i = seq->_length - amount; i < seq->_length; i++) {
	GDF_Symbol *sym = &(seq->_buffer[i]);
	sym->symbol_set = -1;
	sym->handle = -1;
	sym->type = GDF_TYPE_NORMAL;
	sym->expandable = CORBA_FALSE;
	sym->expanded = CORBA_FALSE;
	sym->children._type = TC_CORBA_sequence_GDF_Symbol;
	sym->children._value = init_seq ();
	CORBA_any_set_release (&sym->children, CORBA_TRUE);
    }
}












