/*  -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Implements a GDF Debugger based on Libdryad.
 * 
 * Copyright (C) 1999 Dave Camp <campd@oit.edu>, 
 *                    Martin Baulig <martin@home-of-linux.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

/** 
 * IMPORTANT NOTE: This debugger is ugly, and doesn't really work well.  It
 * is not intended to be used for long; a more permanent solution will be 
 * coming at some point.
 **/

#include <config.h>
#include <gnome.h>
#include <bonobo.h>
#include <string.h>

#include "gdf-libdryad-debugger.h"
#include "libdryad.h"
#include "../../lib/gnome-debug.h"


static CORBA_Object create_libdryad_debugger (BonoboObject *object);
static void gdf_libdryad_debugger_destroy (GtkObject *object);
static void gdf_libdryad_debugger_class_init (GdfLibdryadDebuggerClass *class);
static void gdf_libdryad_debugger_init (BonoboObject *object);
static void connect_signals (GdfLibdryadDebugger *dbg);
static void process_frame_change (GdfLibdryadDebugger *dbg);

static BonoboObjectClass *parent_class;
static POA_GDF_Debugger__epv libdryad_debugger_epv;
static POA_GDF_Debugger__vepv libdryad_debugger_vepv;

#define CORBA_boolean__alloc() (CORBA_boolean*) CORBA_octet_allocbuf (sizeof (CORBA_boolean))

static inline GdfLibdryadDebugger*
libdryad_debugger_from_servant (PortableServer_Servant servant)
{
    return GDF_LIBDRYAD_DEBUGGER (bonobo_object_from_servant (servant));
}

/* public routines */

GdfLibdryadDebugger *
gdf_libdryad_debugger_new (void)
{
    GdfLibdryadDebugger *debugger;
    GDF_Debugger objref;

    debugger = 
	gtk_type_new (gdf_libdryad_debugger_get_type ());
    objref = 
	create_libdryad_debugger (BONOBO_OBJECT (debugger));
    if (objref == CORBA_OBJECT_NIL) {
	gtk_object_destroy (GTK_OBJECT (debugger));
	return NULL;
    }
    
    bonobo_object_construct (BONOBO_OBJECT (debugger),
			     objref);
    
    debugger->gdb_inst = NULL;
    debugger->state = GDF_Debugger_NONE;
    debugger->event_channel = gdf_event_channel_client_new ();

    debugger->pending_exit = -1;

    debugger->stopped_handlers = NULL;
    debugger->locals = gdf_libdryad_symbol_set_new (debugger);
    debugger->symsets = NULL;
    
    debugger->frame_changed = FALSE;
    
    return debugger;
}

GtkType 
gdf_libdryad_debugger_get_type (void)
{
    static GtkType type = 0;
        
    if (!type) {
	GtkTypeInfo info = {
	    "IDL:GDF/Debugger:1.0",
	    sizeof (GdfLibdryadDebugger),
	    sizeof (GdfLibdryadDebuggerClass),
	    (GtkClassInitFunc) gdf_libdryad_debugger_class_init,
	    (GtkObjectInitFunc) gdf_libdryad_debugger_init,
	    NULL,
	    NULL,
	    (GtkClassInitFunc) NULL
	};

	type = gtk_type_unique (bonobo_object_get_type (), &info);
    }

    return type;
}

/* private routines */
CORBA_Object
create_libdryad_debugger (BonoboObject *object) 
{
    POA_GDF_Debugger *servant;
    CORBA_Environment ev;
    CORBA_exception_init (&ev);
    
    servant = 
        (POA_GDF_Debugger*)g_new0(BonoboObjectServant, 1);
    servant->vepv = &libdryad_debugger_vepv;
    
    POA_GDF_Debugger__init((PortableServer_Servant) servant, 
			    &ev);
    if (ev._major != CORBA_NO_EXCEPTION) {
        g_free (servant);
        CORBA_exception_free (&ev);
        return CORBA_OBJECT_NIL;
    }
    CORBA_exception_free (&ev);
    return bonobo_object_activate_servant (object, servant);
}

static void
gdf_libdryad_debugger_destroy (GtkObject *object) 
{
    GdfLibdryadDebugger *dbg = GDF_LIBDRYAD_DEBUGGER (object);
    
    if (dbg->gdb_inst)
	gdb_destroy_instance (dbg->gdb_inst);

    if (dbg->event_channel) {
	gdf_event_channel_client_destroy_channel (dbg->event_channel);
	bonobo_object_unref (BONOBO_OBJECT (dbg->event_channel));
    }
}

static GDF_Debugger_State
impl_get_state (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    return dbg->state;
}

static void
impl_load_binary (PortableServer_Servant servant, 
		  const CORBA_char *file_name,
		  CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;
    gboolean success;
    
    if (dbg->gdb_inst) {
	if (dbg->gdb_inst->file_loaded) 
	    gdb_unload_file (dbg->gdb_inst);
	
	dbg->state = GDF_Debugger_NONE;
	
	event_any = gdf_marshal_event_none ("program_unloaded");
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);

	gdb_destroy_instance (dbg->gdb_inst);
    }
     
    dbg->gdb_inst = gdb_create_instance ();
    connect_signals (dbg);

    dbg->error_condition = 0;

    success = gdb_load_file (dbg->gdb_inst, file_name);

    if (success) {
	event_any = gdf_marshal_event_none ("program_loaded");
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
	if (dbg->tty_name) {
	    gdb_tty_set (dbg->gdb_inst, dbg->tty_name);
	}
    } else {
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
			     ex_GDF_Debugger_InvalidBinary, NULL);
    }
}

static void
impl_unload_binary (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;
    
    if (dbg->gdb_inst) {
	if (dbg->gdb_inst->file_loaded) 
	    gdb_unload_file (dbg->gdb_inst);
	dbg->state = GDF_Debugger_NONE;
	
	event_any = gdf_marshal_event_none ("program_unloaded");
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
	gdb_destroy_instance (dbg->gdb_inst);
    }
    dbg->gdb_inst = NULL;
}

/* "Who's binary?" - Anne Wooten */
static CORBA_boolean
impl__get_binary_loaded (PortableServer_Servant servant, 
			 CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    if (dbg->gdb_inst && dbg->gdb_inst->file_loaded)
	return CORBA_TRUE;
    else
	return CORBA_FALSE;
}

static void
impl_attach (PortableServer_Servant servant, 
	     CORBA_short pid,
	     CORBA_Environment *ev)
{
    /* not implemented */
}

static void
impl_execute (PortableServer_Servant servant, 
	      const GDF_arg_list *args,
	      CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    g_return_if_fail (dbg->gdb_inst != NULL);
    
    gdb_run (dbg->gdb_inst);
}

static void
impl_load_corefile (PortableServer_Servant servant, 
		    const CORBA_char *corefile_name,
		    CORBA_Environment *ev)
{
    /* not implemented */
}

static GDF_source_file_list *
impl_get_sources (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    GDF_source_file_list *list;
    GList *sources;
    GList *i;
    int num;

    sources = gdb_list_sources (dbg->gdb_inst);    

    list = GDF_source_file_list__alloc ();
    list->_length = list->_maximum = g_list_length (sources);
    list->_buffer = CORBA_sequence_CORBA_string_allocbuf (list->_length);
    CORBA_sequence_set_release (list, CORBA_TRUE);
    for (i = sources, num = 0; i != NULL; i = g_list_next (i), num++) {
	list->_buffer[num] = CORBA_string_dup (i->data);
	g_free (i->data);
    }
    g_list_free (sources);
    
    return list;
}

static CORBA_char *
impl_get_absolute_source_path (PortableServer_Servant servant, 
			       const CORBA_char *source_file,
			       CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_char *retval;
    
    if (source_file[0] == '/')
	retval = CORBA_string_dup (source_file);
    else {
	gchar *abs_path;
	abs_path = gdb_get_full_source_path (dbg->gdb_inst, source_file);

	retval = CORBA_string_dup (abs_path);
    }

    return retval;
}

static CORBA_long
impl_set_breakpoint (PortableServer_Servant servant, 
		     const CORBA_char *file_name,
		     CORBA_long line_num,
		     const CORBA_char *condition,
		     CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    gchar *file;
    const GdbBreakpoint *bp;
    CORBA_any *event_any;
    
    g_return_val_if_fail (dbg->gdb_inst != NULL, -1);

    if (strcmp (condition, "") == 0) 
	condition = NULL;
    
    file = strrchr (file_name, G_DIR_SEPARATOR);
    if (!file++)
	file = (char*)file_name;

    bp = gdb_breakpoint_set_linenum (dbg->gdb_inst,
				     file,
				     line_num,
				     condition);

    if (bp) {
	event_any = gdf_marshal_event_long ("breakpoint_set", (glong)bp->num);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
	return bp->num;
    } else {
	/* FIXME: Set exception */
	return -1;
    }
}

static CORBA_long
impl_set_breakpoint_function (PortableServer_Servant servant,
			      const CORBA_char *file_name,
			      const CORBA_char *function_name,
			      const CORBA_char *condition,
			      CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;
    const GdbBreakpoint *bp;

    g_return_val_if_fail (dbg->gdb_inst != NULL, -1);

    if (strcmp (condition, "") == 0) 
	condition = NULL;
    if (strcmp (file_name, "") == 0) 
	file_name = NULL;

    bp = gdb_breakpoint_set_function (dbg->gdb_inst,
				      function_name,
				      condition);

    if (bp) {
	event_any = gdf_marshal_event_long ("breakpoint_set", (glong)bp->num);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
	return bp->num;
    } else {
	/* FIXME: Set exception */
	return -1;
    }
}

static void
impl_enable_breakpoint (PortableServer_Servant servant, 
			CORBA_long bp_num,
			CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;

    g_return_if_fail (dbg->gdb_inst != NULL);
    
    if (gdb_breakpoint_enable (dbg->gdb_inst, bp_num)) {
	event_any = gdf_marshal_event_long ("breakpoint_enabled", 
					    (glong)bp_num);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
    }
}

static void
impl_disable_breakpoint (PortableServer_Servant servant,
			 CORBA_long bp_num,
			 CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;
    
    g_return_if_fail (dbg->gdb_inst != NULL);

    if (gdb_breakpoint_disable (dbg->gdb_inst, bp_num)) {
	event_any = gdf_marshal_event_long ("breakpoint_disabled", 
					    (glong)bp_num);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
    }
}

static void
impl_delete_breakpoint (PortableServer_Servant servant, 
			CORBA_long bp_num,
			CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    CORBA_any *event_any;

    g_return_if_fail (dbg->gdb_inst != NULL);
    
    if (gdb_breakpoint_delete (dbg->gdb_inst, bp_num)) {
	event_any = gdf_marshal_event_long ("breakpoint_deleted", 
					    (glong)bp_num);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
    }
}

static GDF_Breakpoint *
impl_get_breakpoint_info (PortableServer_Servant servant,
			  CORBA_long bp_num,
			  CORBA_Environment *ev)
{
    const GdbBreakpoint *bp;
    GDF_Breakpoint *ret;
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    bp = gdb_breakpoint_get_info (dbg->gdb_inst, bp_num);
    
    ret = GDF_Breakpoint__alloc ();

    if (bp == NULL) {
	/* FIXME: Set an exception */
	return ret;
    }
    
    ret->num = bp_num;
    ret->type = CORBA_string_dup (bp->type);
    ret->enabled = (CORBA_boolean)bp->enabled;
    ret->address = (CORBA_long)bp->address;
    ret->file_name = CORBA_string_dup (bp->file_name);
    ret->line_num = (CORBA_long)bp->line_num;

    return ret;
}

static GDF_RegisterList *
impl_get_registers (PortableServer_Servant servant,
		    CORBA_Environment *ev)
{
    GDF_RegisterList *ret;
    GList *registers;
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    int num_regs = 0;
    GList *r;
    int i = 0;
    
    ret = GDF_RegisterList__alloc ();
    CORBA_sequence_set_release (ret, CORBA_TRUE);

    if (dbg->gdb_inst) {
	registers = gdb_get_registers (dbg->gdb_inst);
	
	if (registers) {
	    num_regs = g_list_length (registers);
	    ret->_length = ret->_maximum = num_regs;
	    ret->_buffer = CORBA_sequence_GDF_Register_allocbuf (num_regs);
	    
	    i = 0;
	    for (r = registers; r != NULL; r = r->next) {
		ret->_buffer[i].name = 
		    CORBA_string_dup (((GdbRegister*)r->data)->name);
		ret->_buffer[i++].value = 
		    CORBA_string_dup (((GdbRegister*)r->data)->value);
	    }
	    
	    gdb_destroy_register_list (dbg->gdb_inst, registers);

	    return ret;
	}
    }
    
    /* Did not get registers; must not be running. */
    CORBA_exception_set (ev, CORBA_USER_EXCEPTION, 
			 ex_GDF_Debugger_NotRunning, NULL);

    return ret;
}

static void
impl_change_frame (PortableServer_Servant servant,
		   CORBA_long new_id,
		   CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    
    if (gdb_change_frame (dbg->gdb_inst, (int) new_id)) {
	CORBA_any *event_any;
	process_frame_change (dbg);
	event_any = gdf_marshal_event_long ("frame_change", 
					    new_id);
	gdf_event_channel_client_push (dbg->event_channel, event_any);
	CORBA_free (event_any);
    }
}

static GDF_StackFrame *
impl_get_frame (PortableServer_Servant servant, 
		CORBA_long id,
		CORBA_Environment *ev)
{
    GdbFrame *frame;
    GDF_StackFrame *gdf_frame;
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    gdf_frame = GDF_StackFrame__alloc ();

    if (dbg->state != GDF_Debugger_NONE) {
	frame = gdb_get_frame (dbg->gdb_inst, id);
	
	gdf_frame->id = frame->id;
	if (frame->function) 
	    gdf_frame->function = CORBA_string_dup (frame->function);
	else
	    gdf_frame->function = CORBA_string_dup ("");
	if (frame->file)
	    gdf_frame->location.file = CORBA_string_dup (frame->file);
	else
	    gdf_frame->location.file = CORBA_string_dup ("");
	gdf_frame->location.line = frame->line_num;
	gdf_frame->virtual_address = (GDF_MemoryAddress)frame->addr;   
	
	gdb_destroy_frame (dbg->gdb_inst, frame);
    } else {
	gdf_frame->id = -1;
	gdf_frame->function = CORBA_string_dup ("");
	gdf_frame->location.file = CORBA_string_dup ("");
	gdf_frame->location.line = -1;
	gdf_frame->virtual_address = 0;
	
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GDF_Debugger_NotRunning, NULL);
    }
    
    return gdf_frame;
}

static GDF_Stack *
impl_get_backtrace (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    GDF_Stack *stack;
    GList *list;
    GList *i;
    int n;

    stack = GDF_Stack__alloc ();
    CORBA_sequence_set_release (stack, CORBA_TRUE);

    if (dbg->state == GDF_Debugger_NONE) {
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GDF_Debugger_NotRunning, NULL);
	return stack;
    }

    list = gdb_get_stack (dbg->gdb_inst);    

    stack->_length = stack->_maximum = g_list_length (list);
    stack->_buffer = CORBA_sequence_GDF_StackFrame_allocbuf (stack->_length);
    for (i = list, n = 0; i != NULL; i = g_list_next (i), n++) {	
	GdbFrame *frame = i->data;
	stack->_buffer[n].id = frame->id;
	stack->_buffer[n].function = 
	    CORBA_string_dup (frame->function ? frame->function : "");
	stack->_buffer[n].location.file = 
	    CORBA_string_dup (frame->file ? frame->file : "");
	stack->_buffer[n].location.line = frame->line_num;
	stack->_buffer[n].virtual_address = (GDF_MemoryAddress)frame->addr;
    }
    gdb_destroy_stack (dbg->gdb_inst, list);
    
    return stack;
}

static void
impl_cont (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_continue (dbg->gdb_inst);    
}

static void
impl_stop (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_stop (dbg->gdb_inst);
}

static void
impl_restart (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_stop (dbg->gdb_inst);
    gdb_run (dbg->gdb_inst);
}

static void
impl_step_over (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_step_over (dbg->gdb_inst);
}

static void
impl_step_into (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_step_into (dbg->gdb_inst);
}

static void
impl_step_out (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    g_return_if_fail (dbg->gdb_inst != NULL);

    gdb_step_out (dbg->gdb_inst);
}

/* Symbol watching */

static void
update_symsets (GdfLibdryadDebugger *dbg, gboolean include_locals)
{
    GList *i;
    
    if (include_locals) {
	gdf_libdryad_symbol_set_update (dbg->locals);
    }

    for (i = dbg->symsets; i != NULL; i = i->next) {
	gdf_libdryad_symbol_set_update (i->data);
    }
}

static void
new_locals (GdfLibdryadDebugger *dbg)
{
    gdb_get_locals (dbg->gdb_inst, dbg->locals);
}

static GDF_SymbolSet
impl_get_locals (PortableServer_Servant servant, CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    return CORBA_Object_duplicate (bonobo_object_corba_objref (BONOBO_OBJECT (dbg->locals)), 
				   ev);
}

static GDF_SymbolSet
impl_allocate_symbol_set (PortableServer_Servant servant, 
			  CORBA_Environment *ev)
{    
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    GdfLibdryadSymbolSet *ss = gdf_libdryad_symbol_set_new (dbg);

    dbg->symsets = g_list_prepend (dbg->symsets, ss);

    return CORBA_Object_duplicate (bonobo_object_corba_objref (BONOBO_OBJECT (ss)), 
				   ev);
}

static CORBA_Object
impl_get_event_channel (PortableServer_Servant servant,
			CORBA_Environment *ev)
{
    CORBA_Object ret;
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);
    GDF_TRACE ();
    ret = bonobo_object_corba_objref (BONOBO_OBJECT (dbg->event_channel));
    return CORBA_Object_duplicate (ret, ev);
}

static void
impl_set_output_tty (PortableServer_Servant servant,
		     const CORBA_char *tty_name,
		     CORBA_Environment *ev)
{
    GdfLibdryadDebugger *dbg = libdryad_debugger_from_servant (servant);

    if (dbg->tty_name)
	g_free (dbg->tty_name);
    
    dbg->tty_name = g_strdup (tty_name);
    
    if (dbg->gdb_inst)
	gdb_tty_set (dbg->gdb_inst, tty_name);
}

static void
init_libdryad_debugger_corba_class (void) 
{
    /* EPV */
    libdryad_debugger_epv.get_state = impl_get_state;
    libdryad_debugger_epv.load_binary = impl_load_binary;
    libdryad_debugger_epv.unload_binary = impl_unload_binary;
    libdryad_debugger_epv._get_binary_loaded = impl__get_binary_loaded;
    libdryad_debugger_epv.attach = impl_attach;
    libdryad_debugger_epv.execute = impl_execute;
    libdryad_debugger_epv.load_corefile = impl_load_corefile;
    libdryad_debugger_epv.get_sources = impl_get_sources;
    libdryad_debugger_epv.get_absolute_source_path = impl_get_absolute_source_path;
    libdryad_debugger_epv.set_breakpoint = impl_set_breakpoint;
    libdryad_debugger_epv.set_breakpoint_function = impl_set_breakpoint_function;
    libdryad_debugger_epv.enable_breakpoint = impl_enable_breakpoint;
    libdryad_debugger_epv.disable_breakpoint = impl_disable_breakpoint;
    libdryad_debugger_epv.delete_breakpoint = impl_delete_breakpoint;
    libdryad_debugger_epv.get_breakpoint_info = impl_get_breakpoint_info;
    libdryad_debugger_epv.get_registers = impl_get_registers;
    libdryad_debugger_epv.change_frame = impl_change_frame;    
    libdryad_debugger_epv.get_frame = impl_get_frame;
    libdryad_debugger_epv.get_backtrace = impl_get_backtrace;
    libdryad_debugger_epv.cont = impl_cont;    
    libdryad_debugger_epv.stop = impl_stop;
    libdryad_debugger_epv.restart = impl_restart;
    libdryad_debugger_epv.step_over = impl_step_over;
    libdryad_debugger_epv.step_into = impl_step_into;
    libdryad_debugger_epv.step_out = impl_step_out;
    libdryad_debugger_epv.get_locals = impl_get_locals;
    libdryad_debugger_epv.allocate_symbol_set = impl_allocate_symbol_set;
    libdryad_debugger_epv.get_event_channel = impl_get_event_channel;
    libdryad_debugger_epv.set_output_tty = impl_set_output_tty;
   
    /* VEPV */
    libdryad_debugger_vepv.Bonobo_Unknown_epv = bonobo_object_get_epv ();
    libdryad_debugger_vepv.GDF_Debugger_epv = &libdryad_debugger_epv;
}
        
static void
gdf_libdryad_debugger_class_init (GdfLibdryadDebuggerClass *class) 
{
    GtkObjectClass *object_class = (GtkObjectClass*) class;
    parent_class = gtk_type_class (bonobo_object_get_type ());
    
    object_class->destroy = gdf_libdryad_debugger_destroy;
    
    init_libdryad_debugger_corba_class ();
}

static void
gdf_libdryad_debugger_init (BonoboObject *object)
{
    GdfLibdryadDebugger *dbg = GDF_LIBDRYAD_DEBUGGER (object);
    
    dbg->tty_name = NULL;
}

/* Perform cleanup tasks associated with a frame change */
static void
process_frame_change (GdfLibdryadDebugger *dbg)
{
    new_locals (dbg);
    update_symsets (dbg, FALSE);
}

static void
execution_starting_cb (GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;

    event_any = gdf_marshal_event_none ("started");
    gdf_event_channel_client_push (dbg->event_channel, event_any);
    CORBA_free (event_any);
}

static void
execution_running_cb (GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;
    dbg->state = GDF_Debugger_RUNNING;

    event_any = gdf_marshal_event_none ("running");
    gdf_event_channel_client_push (dbg->event_channel, event_any);
    CORBA_free (event_any);
}

static void
execution_exited_cb (gint exit_code, GdfLibdryadDebugger *dbg)
{
    /* "exited" should be dealt with after the corresponding "stopped" */ 
    dbg->pending_exit = exit_code;
}

static void
execution_exited_signal_cb (char *sig_id, GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;
    /* "exited" should be dealt with after the corresponding "stopped" */ 
    dbg->pending_exit = 128;  /* FIXME: Map name to signal? */
    
    event_any = gdf_marshal_event_string ("signal_termination", sig_id);
    dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);
}

static void
flush_event_queue (GdfLibdryadDebugger *dbg)
{
    GList *i;
    
    dbg->event_queue = g_list_reverse (dbg->event_queue);
    i = dbg->event_queue;

    while (i) {
	gdf_event_channel_client_push (dbg->event_channel, 
				       (CORBA_any *)i->data);
	CORBA_free (i->data);
	i = g_list_next (i);
    }
    g_list_free (dbg->event_queue);
    dbg->event_queue = NULL;
}

static void
execution_stopped_cb (GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;
    dbg->state = GDF_Debugger_STOPPED;

    event_any = gdf_marshal_event_none ("stopped");
    dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);

    if (dbg->pending_exit != -1) {
	dbg->state = GDF_Debugger_NONE;
	event_any = gdf_marshal_event_long ("exited", 
					    (glong)dbg->pending_exit);
	dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);
	dbg->pending_exit = -1;
    } else {
	/* If the frame changed during the execution, this flag will be set.  
	 * We need to process this here rather than in 
	 * execution_frame_change_cb because any libdryad interaction in 
	 * execution_frame_change_cb will screw up the execution parsing. */
	if (dbg->frame_changed) {
	    process_frame_change (dbg);
	    dbg->frame_changed = FALSE;
	} else {
	    update_symsets (dbg, TRUE);
	}
    }
    
    /* This is the last signal in the sequence, so flush the event queue */
    flush_event_queue (dbg);
}

static void
execution_killed_cb (GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;
    dbg->state = GDF_Debugger_STOPPED;

    g_warning ("libdryad debugger does not send the correct kill signal");

    event_any = gdf_marshal_event_long ("killed", 0);
    gdf_event_channel_client_push (dbg->event_channel, event_any);
    CORBA_free (event_any);
}


static void
execution_source_line_cb (gchar *source_file, gint line_num, 
			  GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;
    
    event_any = gdf_marshal_event_source_location ("source_line", 
						   source_file, line_num);
    dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);
}

static void
execution_frame_change_cb (int frame_id, GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;

    event_any = gdf_marshal_event_long ("frame_change", frame_id);
    dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);
    
    dbg->frame_changed = TRUE;
}

static void
execution_signal_received_cb (char *sig_id, GdfLibdryadDebugger *dbg)
{
    CORBA_any *event_any;

    event_any = gdf_marshal_event_string ("signal", sig_id);
    dbg->event_queue = g_list_prepend (dbg->event_queue, event_any);   
}

static void
error_cb (gchar *error_msg, GdfLibdryadDebugger *dbg)
{
    dbg->error_condition = 1;
}

static void 
connect_signals (GdfLibdryadDebugger *dbg)
{
    gdb_signal_connect (dbg->gdb_inst, "execution-starting",
			execution_starting_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-running",
			execution_running_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-stopped",
			execution_stopped_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-killed",
			execution_killed_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-exited",
			execution_exited_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-exited-signal",
			execution_exited_signal_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-source-line",
			execution_source_line_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-frame-change",
			execution_frame_change_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "execution-signal-received",
			execution_signal_received_cb, (gpointer) dbg);
    gdb_signal_connect (dbg->gdb_inst, "error", 
			error_cb, (gpointer)dbg);
}



