/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-source-viewer.h"

#include "../../pixmaps/current-line.xpm"

struct _GdfSourceViewerPriv {
    GtkWidget *clist;
    GtkWidget *scrolled;

    GHashTable *breakpoints;
    GdfDebuggerClient *dbg;
};

typedef struct {
    gint bp_num;
    gint line_num;
    gboolean enabled;
} SVBreakpoint ;

#define BREAKPOINT_ICON_WIDTH 15
#define BREAKPOINT_ICON_HEIGHT 15

#define GTK_CLIST_CELL_SPACING 1

#define NUM_COLUMNS 3
static const gint sv_column_widths[NUM_COLUMNS - 1] = { 25, 25 };

static void source_viewer_init (GdfSourceViewer *sv);
static void source_viewer_class_init (GdfSourceViewerClass *klass);
static void source_viewer_destroy (GtkObject *object);

static void create_children (GdfSourceViewer *sv);
static gboolean load_source (GdfSourceViewer *sv);
static void draw_line_icons (GdfSourceViewer *sv, gint line_num);
static void draw_current_line (GdfSourceViewer *sv);
static void draw_breakpoint (GdfSourceViewer *sv, SVBreakpoint *bp);
static void draw_empty_line (GdfSourceViewer *sv, gint line_num);
static gboolean line_is_visible (GdfSourceViewer *sv, gint line_num);
static void scroll_to_line (GdfSourceViewer *sv, gint line_num);
static gint button_press_cb (GtkWidget *widget, GdkEventButton *event,
                             GdfSourceViewer *sv);

static gboolean breakpoint_hash_remove_func (gpointer key,
                                             gpointer value,
                                             gpointer user_data);
static guint bp_hash (gconstpointer v);
static gint bp_equal (gconstpointer v, gconstpointer v2);

static GnomeUIInfo popup_menu[] =
{
    {                       /* Will be set/enable/disable breakpoint */
        GNOME_APP_UI_ITEM,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL,
        GNOME_APP_PIXMAP_STOCK,
        NULL,
        0, 0, NULL
    },
    {                       /* Will be remove breakpoint */
        GNOME_APP_UI_ITEM,
        NULL,
        NULL,
        NULL,
        NULL,
        NULL,
        GNOME_APP_PIXMAP_STOCK,
        NULL,
        0, 0, NULL
    },
    GNOMEUIINFO_END
};

static GtkFrameClass *parent_class;

/*
 * Public Interface 
 */

GtkType 
gdf_source_viewer_get_type ()
{
    static GtkType type = 0;
	
    if (!type) {
        static const GtkTypeInfo info = {
            "GdfSourceViewer",
            sizeof (GdfSourceViewer),
            sizeof (GdfSourceViewerClass),
            (GtkClassInitFunc) source_viewer_class_init,
            (GtkObjectInitFunc) source_viewer_init,
            NULL,
            NULL,
            (GtkClassInitFunc)NULL
        };
		
        type = gtk_type_unique (gtk_frame_get_type (), &info);
    }
	
    return type;
}

GtkWidget *
gdf_source_viewer_new (const gchar *source_path)
{
    GdfSourceViewer *sv;
	
    g_return_val_if_fail (source_path != NULL, NULL);

    sv = gtk_type_new (gdf_source_viewer_get_type ());

    sv->source_path = g_strdup (source_path);
	
    if (!load_source (sv)) {
        gtk_object_destroy (GTK_OBJECT (sv));
        return NULL;
    }
	
    return GTK_WIDGET (sv);
}

void
gdf_source_viewer_add_breakpoint (GdfSourceViewer *sv, 
                                  gint bp_num, gint line_num, gboolean enabled)
{
    SVBreakpoint *bp;
	
    g_return_if_fail (sv != NULL);
    g_return_if_fail (GDF_IS_SOURCE_VIEWER (sv));
	
    bp = g_new0 (SVBreakpoint, 1);
	
    bp->bp_num = bp_num;
    bp->line_num = line_num;
    bp->enabled = enabled;
	
    g_hash_table_insert (sv->priv->breakpoints, GINT_TO_POINTER (bp_num),
                         (gpointer)bp);

	
    gtk_clist_set_row_data (GTK_CLIST (sv->priv->clist), line_num - 1, 
                            (gpointer)bp);

    draw_line_icons (sv, line_num);
    scroll_to_line (sv, bp->line_num);
}

void
gdf_source_viewer_enable_breakpoint (GdfSourceViewer *sv, gint bp_num,
                                     gboolean enabled)
{
    SVBreakpoint *bp;
	
    g_return_if_fail (sv != NULL);
    g_return_if_fail (GDF_IS_SOURCE_VIEWER (sv));

    bp = g_hash_table_lookup (sv->priv->breakpoints, GINT_TO_POINTER (bp_num));
    g_return_if_fail (bp != NULL);
	
    bp->enabled = enabled;
    draw_line_icons (sv, bp->line_num);
}

void
gdf_source_viewer_remove_breakpoint (GdfSourceViewer *sv, gint bp_num)
{
    SVBreakpoint *bp;
	
    g_return_if_fail (sv != NULL);
    g_return_if_fail (GDF_IS_SOURCE_VIEWER (sv));
	
    bp = g_hash_table_lookup (sv->priv->breakpoints, GINT_TO_POINTER (bp_num));
    g_return_if_fail (bp != NULL);
	
    g_hash_table_remove (sv->priv->breakpoints, GINT_TO_POINTER (bp_num));
	
    gtk_clist_set_row_data (GTK_CLIST (sv->priv->clist), bp->line_num - 1, 
                            NULL);
	
    draw_line_icons (sv, bp->line_num);

    g_free (bp);
}

gboolean 
gdf_source_viewer_has_breakpoint (GdfSourceViewer *sv, gint bp_num)
{
    SVBreakpoint *bp;
	
    g_return_val_if_fail (sv != NULL, FALSE);
    g_return_val_if_fail (GDF_IS_SOURCE_VIEWER (sv), FALSE);
	
    bp = g_hash_table_lookup (sv->priv->breakpoints, GINT_TO_POINTER (bp_num));
    return (bp != NULL);	
}

void
gdf_source_viewer_view_line (GdfSourceViewer *sv, gint line_num)
{
    scroll_to_line (sv, line_num);
}

void 
gdf_source_viewer_set_current_line (GdfSourceViewer *sv, gint line_num)
{
    if (sv->current_line > -1) {
        gint old_current_line = sv->current_line;
        draw_line_icons (sv, old_current_line);
    }

    sv->current_line = line_num;
    draw_line_icons (sv, line_num);
	
    scroll_to_line (sv, line_num);
}

void 
gdf_source_viewer_clear_current_line (GdfSourceViewer *sv)
{
    gint old_current_line = sv->current_line;
    sv->current_line = 0;
    draw_line_icons (sv, old_current_line);
}

void
gdf_source_viewer_set_debugger (GdfSourceViewer *sv,
                                GdfDebuggerClient *dbg)
{
    g_return_if_fail (sv != NULL);
    g_return_if_fail (GDF_IS_SOURCE_VIEWER (sv));
	
    if (sv->priv->dbg) {
        gtk_object_unref (GTK_OBJECT (sv->priv->dbg));
    }
	
    sv->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
    }
}

/*
 * Class / Object functions
 */

void
source_viewer_class_init (GdfSourceViewerClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;
	
    parent_class = gtk_type_class (gtk_frame_get_type ());
	
    object_class->destroy = source_viewer_destroy;
}

void
source_viewer_init (GdfSourceViewer *sv)
{
    sv->priv = g_new0 (GdfSourceViewerPriv, 1);
	
    sv->source_path = NULL;
    sv->current_line = -1;

    sv->priv->breakpoints = g_hash_table_new (bp_hash, bp_equal);
    sv->priv->dbg = NULL;
	
    create_children (sv);
}

void
source_viewer_destroy (GtkObject *obj)
{
    GdfSourceViewer *sv = GDF_SOURCE_VIEWER (obj);

    if (sv->source_path)
        g_free (sv->source_path);

    g_hash_table_foreach_remove (sv->priv->breakpoints, 
                                 breakpoint_hash_remove_func, NULL);
    g_hash_table_destroy (sv->priv->breakpoints);

    if (sv->priv->dbg)
        gtk_object_unref (GTK_OBJECT (sv->priv->dbg));

    g_free (sv->priv);

    if (GTK_OBJECT_CLASS (parent_class)->destroy)
        (*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}

/* 
 * Helper functions 
 */
void
create_children (GdfSourceViewer *sv)
{
    gchar *titles[] = { N_(""), N_("#"), N_("Source Line") };
    gint i;

    sv->priv->clist = gtk_clist_new_with_titles (NUM_COLUMNS, titles);

    /* Set the width on all but the last column */
    for (i = 0; i < NUM_COLUMNS - 1; i++) {
        gtk_clist_set_column_width (GTK_CLIST (sv->priv->clist), i, 
                                    sv_column_widths[i]);  
    }
	
    gtk_signal_connect (GTK_OBJECT (sv->priv->clist), "button_press_event",
                        GTK_SIGNAL_FUNC (button_press_cb), (gpointer)sv);
	

    sv->priv->scrolled = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sv->priv->scrolled),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
	
    gtk_container_add (GTK_CONTAINER (sv->priv->scrolled), sv->priv->clist);
    gtk_container_add (GTK_CONTAINER (sv), sv->priv->scrolled);
    gtk_widget_show (sv->priv->clist);
    gtk_widget_show (sv->priv->scrolled);
}

gboolean
load_source (GdfSourceViewer *sv)
{
    FILE *in_file;
    /* FIXME: Truncated lines */
    gchar line[1024];
    gint line_num = 0;
    gchar *line_num_str;
    gchar *row[3];
	
    in_file = fopen (sv->source_path, "r");
	
    if (!in_file)
        return FALSE;
	
    gtk_clist_freeze (GTK_CLIST (sv->priv->clist));

    fgets (line, sizeof (line), in_file);
    line_num++;

    while (!feof (in_file)) {
        row[0] = "";
        line_num_str = g_strdup_printf ("%d", line_num);
        row[1] = line_num_str;
        row[2] = line;
        gtk_clist_append (GTK_CLIST (sv->priv->clist), row);
        g_free (line_num_str);
		
        fgets (line, sizeof (line), in_file);
        line_num++;
    }

    gtk_clist_thaw (GTK_CLIST (sv->priv->clist));

    return TRUE;
}

static void 
draw_line_icons (GdfSourceViewer *sv, gint line_num)
{
    SVBreakpoint *bp;
		
    if (sv->current_line == line_num) {
        draw_current_line (sv);
        return;
    }

    bp = gtk_clist_get_row_data (GTK_CLIST (sv->priv->clist), line_num - 1);

    if (bp)
        draw_breakpoint (sv, bp);
    else
        draw_empty_line (sv, line_num);
}

void
draw_current_line (GdfSourceViewer *sv)
{
    GtkWidget *pixmap_widget;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
	
    pixmap_widget = gnome_pixmap_new_from_xpm_d (current_line_xpm);
    pixmap = GNOME_PIXMAP (pixmap_widget)->pixmap;
    mask = GNOME_PIXMAP (pixmap_widget)->mask;
	
    gtk_clist_set_pixmap (GTK_CLIST (sv->priv->clist), sv->current_line - 1, 
                          0, pixmap, mask);
	
    gtk_widget_destroy (pixmap_widget);
}	

void
draw_breakpoint (GdfSourceViewer *sv, SVBreakpoint *bp)
{
    GtkWidget *pixmap_widget = NULL;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
	
    if (bp->enabled) {
        pixmap_widget = 
            gnome_stock_pixmap_widget_at_size (sv->priv->clist,
                                               GNOME_STOCK_PIXMAP_STOP,
                                               BREAKPOINT_ICON_HEIGHT,
                                               BREAKPOINT_ICON_WIDTH);
    } else {
        pixmap_widget = 
            gnome_stock_pixmap_widget_at_size (sv->priv->clist,
                                               GNOME_STOCK_PIXMAP_CLOSE,
                                               BREAKPOINT_ICON_HEIGHT,
                                               BREAKPOINT_ICON_WIDTH);
    }

    pixmap = GNOME_PIXMAP (pixmap_widget)->pixmap;
    mask = GNOME_PIXMAP (pixmap_widget)->mask;
	
    gtk_clist_set_pixmap (GTK_CLIST (sv->priv->clist),  bp->line_num - 1, 0,
                          pixmap, mask);
	
    gtk_widget_destroy (pixmap_widget);
}

void
draw_empty_line (GdfSourceViewer *sv, gint line_num)
{
    gtk_clist_set_text (GTK_CLIST (sv->priv->clist), line_num - 1, 0, "");
}

/* FIXME: This routine has a bit of a bug */
gboolean
line_is_visible (GdfSourceViewer *sv, gint line_num)
{
    gint pixel;
    gint top;
    gint height;
    GtkAdjustment *adjustment;

    pixel =
        (line_num) * GTK_CLIST (sv->priv->clist)->row_height 
        + (line_num) * GTK_CLIST_CELL_SPACING;
	
    adjustment = 
        gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (sv->priv->scrolled));
	
    top = (gint) adjustment->value;
    height = GTK_CLIST (sv->priv->clist)->clist_window_height;

    if (pixel > top && pixel < (top + height))
        return TRUE;
    else
        return FALSE;
}

void
scroll_to_line (GdfSourceViewer *sv, gint line_num)
{
    GtkAdjustment *adjustment;
    gint pixel;
	
    gtk_clist_select_row (sv->priv->clist, line_num - 1, 0);

    if (line_is_visible (sv, line_num))
        return;

    line_num -= 2;  /* The line will be the third one shown */

    pixel = 
        line_num * GTK_CLIST (sv->priv->clist)->row_height
        + line_num * GTK_CLIST_CELL_SPACING;
	
    adjustment = 
        gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (sv->priv->scrolled));
    gtk_adjustment_set_value (adjustment, (gfloat) pixel);
}

static void
set_breakpoint_selected_cb (GtkWidget *widget, GdfSourceViewer *sv)
{
    gint line_num;

    g_return_if_fail (sv->priv->dbg != NULL);
	
    line_num = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (sv), 
                                                     "menu_line"));
	
    gdf_debugger_client_set_breakpoint (sv->priv->dbg,
                                         sv->source_path, line_num, NULL);
}

static void
enable_breakpoint_selected_cb (GtkWidget *widget, GdfSourceViewer *sv)
{
    SVBreakpoint *bp;

    g_return_if_fail (sv->priv->dbg != NULL);
	
    bp = gtk_object_get_data (GTK_OBJECT (sv), "menu_breakpoint");
	
    gdf_debugger_client_enable_breakpoint (sv->priv->dbg, bp->bp_num);
}

static void
disable_breakpoint_selected_cb (GtkWidget *widget, GdfSourceViewer *sv)
{
    SVBreakpoint *bp;

    g_return_if_fail (sv->priv->dbg != NULL);
	
    bp = gtk_object_get_data (GTK_OBJECT (sv), "menu_breakpoint");
	
    gdf_debugger_client_disable_breakpoint (sv->priv->dbg, bp->bp_num);
}

static void
delete_breakpoint_selected_cb (GtkWidget *widget, GdfSourceViewer *sv)
{
    SVBreakpoint *bp;

    g_return_if_fail (sv->priv->dbg != NULL);
	
    bp = gtk_object_get_data (GTK_OBJECT (sv), "menu_breakpoint");
	
    gdf_debugger_client_delete_breakpoint (sv->priv->dbg, bp->bp_num);
}


static void
do_popup (GdfSourceViewer *sv, SVBreakpoint *bp, gint row)
{
    GtkWidget *menu = NULL;
	
    gtk_object_set_data (GTK_OBJECT (sv), "menu_breakpoint", bp);
    gtk_object_set_data (GTK_OBJECT (sv), "menu_line", 
                         GINT_TO_POINTER (row + 1));

    popup_menu[0].type = GNOME_APP_UI_ITEM;
    if (!bp) {
        popup_menu[0].label = _("Set a breakpoint");
        popup_menu[0].hint = _("Set a breakpoint on the current line");
        popup_menu[0].moreinfo = (gpointer)set_breakpoint_selected_cb;
        popup_menu[0].pixmap_type = GNOME_APP_PIXMAP_STOCK;
        popup_menu[0].pixmap_info = GNOME_STOCK_MENU_STOP;
    } else if (bp && bp->enabled) {
        popup_menu[0].label = _("Disable breakpoint");
        popup_menu[0].hint = _("Disable the selected breakpoint");
        popup_menu[0].moreinfo = (gpointer)disable_breakpoint_selected_cb;
        popup_menu[0].pixmap_type = GNOME_APP_PIXMAP_STOCK;
        popup_menu[0].pixmap_info = GNOME_STOCK_MENU_CLOSE;
    } else if (bp && !bp->enabled) {
        popup_menu[0].label = _("Enable breakpoint");
        popup_menu[0].hint = _("Enable the selected breakpoint");
        popup_menu[0].moreinfo = (gpointer)enable_breakpoint_selected_cb;
        popup_menu[0].pixmap_type = GNOME_APP_PIXMAP_STOCK;
        popup_menu[0].pixmap_info = GNOME_STOCK_PIXMAP_STOP;
    } 
	
    if (bp) {
        popup_menu[1].type = GNOME_APP_UI_ITEM;
        popup_menu[1].label = _("Delete breakpoint");
        popup_menu[1].hint = _("Delete the selected breakpoint");
        popup_menu[1].moreinfo = (gpointer)delete_breakpoint_selected_cb;
        popup_menu[1].pixmap_type = GNOME_APP_PIXMAP_STOCK;
        popup_menu[1].pixmap_type = GNOME_APP_PIXMAP_NONE;
        popup_menu[1].pixmap_info = NULL;
    } else {
        popup_menu[1].type = GNOME_APP_UI_ENDOFINFO;
    }
	
    menu = gnome_popup_menu_new (popup_menu);
    gnome_popup_menu_do_popup (menu, NULL, NULL, NULL, sv);
}

gint 
button_press_cb (GtkWidget *widget, GdkEventButton *event, 
                 GdfSourceViewer *sv)
{
    if (event->button == 3) {
        gint row;
        SVBreakpoint *bp;

        if (!gtk_clist_get_selection_info (GTK_CLIST (widget), 
                                           (gint)event->x, (gint)event->y,
                                           &row, NULL))
            return 0;
		

        bp = gtk_clist_get_row_data (GTK_CLIST (widget), row);
	
        do_popup (sv, bp, row);
    }

    return TRUE;
}


/*
 * Hashtable support functions
 */

guint 
bp_hash (gconstpointer v)
{
    return (guint)(GPOINTER_TO_INT (v));
}
	
gint 
bp_equal (gconstpointer v, gconstpointer v2)
{
    return GPOINTER_TO_INT (v) == GPOINTER_TO_INT (v2);
}

/* Called when breakpoint hash is being destroyed */
gboolean
breakpoint_hash_remove_func (gpointer key,
                             gpointer value,
                             gpointer user_data)
{
    g_free (value);
    return TRUE;
}

	





