/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

/* FIXME: Add more marshallers */
#include <config.h>
#include "gdf-event-marshallers.h"

/* Gross macros to get around ORBit incompleteness. Taken from bonobo */
#define CORBA_char__alloc() (CORBA_char *) CORBA_octet_allocbuf (sizeof (CORBA_char))
#define CORBA_short__alloc() (CORBA_short *) CORBA_octet_allocbuf (sizeof (CORBA_short))
#define CORBA_unsigned_short__alloc() (CORBA_unsigned_short *) CORBA_octet_allocbuf (sizeof (CORBA_unsigned_short))
#define CORBA_long__alloc() (CORBA_long *) CORBA_octet_allocbuf (sizeof (CORBA_long))
#define CORBA_unsigned_long__alloc() (CORBA_unsigned_long *) CORBA_octet_allocbuf (sizeof (CORBA_unsigned_long))
#define CORBA_string__alloc() (CORBA_char **)ORBit_alloc(sizeof(gpointer), CORBA_string__free, GUINT_TO_POINTER(1))
#define CORBA_float__alloc() (CORBA_float *) CORBA_octet_allocbuf (sizeof (CORBA_float))
#define CORBA_double__alloc() (CORBA_double *) CORBA_octet_allocbuf (sizeof (CORBA_double))
#define CORBA_boolean__alloc() (CORBA_boolean *) CORBA_octet_allocbuf (sizeof (CORBA_boolean))

static CORBA_any *build_event (const gchar *event_name, 
							   CORBA_TypeCode type,
							   gpointer value);

CORBA_any *
gdf_marshal_event_none (const gchar *event_name)
{
	g_return_val_if_fail (event_name != NULL, NULL);
	
	/* There must be an argument; just use boolean */
	return gdf_marshal_event_boolean (event_name, FALSE);
}

CORBA_any *
gdf_marshal_event_boolean (const gchar *event_name, gboolean value)
{
	CORBA_boolean *b;

	g_return_val_if_fail (event_name != NULL, NULL);
	
	b = CORBA_boolean__alloc ();
	*b = (CORBA_boolean)value;
	
	return build_event (event_name, (CORBA_TypeCode)TC_boolean, (gpointer)b);
}

CORBA_any *
gdf_marshal_event_long (const gchar *event_name, glong value)
{
	CORBA_long *l;

	g_return_val_if_fail (event_name != NULL, NULL);
	
	l = CORBA_long__alloc ();
	*l = (CORBA_long)value;
	
	return build_event (event_name, (CORBA_TypeCode)TC_long, (gpointer)l);
}

CORBA_any *
gdf_marshal_event_short (const gchar *event_name, gshort value)
{
	CORBA_short *s;

	g_return_val_if_fail (event_name != NULL, NULL);
	
	s = CORBA_short__alloc ();
	*s = (CORBA_short)value;
	
	return build_event (event_name, (CORBA_TypeCode)TC_short, (gpointer)s);
}


CORBA_any *
gdf_marshal_event_string (const gchar *event_name, 
						  const gchar* value)
{
	CORBA_char **str;
	const gchar *string_value;

	g_return_val_if_fail (event_name != NULL, NULL);

	if (value == NULL)
		string_value = "";
	else 
		string_value = (const gchar *)value;
	
	str = CORBA_string__alloc ();
	*str = CORBA_string_dup (string_value);

	return build_event (event_name, (CORBA_TypeCode)TC_string, (gpointer)str);
}

CORBA_any *
gdf_marshal_event_char (const gchar *event_name, gchar value)
{
	CORBA_char *c;

	g_return_val_if_fail (event_name != NULL, NULL);

	c = CORBA_char__alloc ();
	*c = (CORBA_char)value;
	
	return build_event (event_name, (CORBA_TypeCode)TC_char, (gpointer)c);
}

CORBA_any *
gdf_marshal_event_double (const gchar *event_name, gdouble value)
{
	CORBA_double *d;

	g_return_val_if_fail (event_name != NULL, NULL);
	
	d = CORBA_double__alloc ();
	*d = (CORBA_double)value;

	return build_event (event_name, (CORBA_TypeCode)TC_double, (gpointer)d);
}

CORBA_any *
gdf_marshal_event_source_location (const gchar *event_name,
								   const gchar *source_path,
								   gint source_line)
{
	GDF_SourceLocation *loc;
	
	g_return_val_if_fail (event_name != NULL, NULL);
	g_return_val_if_fail (source_path != NULL, NULL);
	
	loc = GDF_SourceLocation__alloc ();
	loc->file = CORBA_string_dup (source_path);
	loc->line = source_line;	

	return build_event (event_name, (CORBA_TypeCode)TC_GDF_SourceLocation,
						loc);
}

CORBA_any *
gdf_marshal_event_long_list (const gchar *event_name, GList *list)
{
    GDF_LongSeq *seq;
    int len;
    int i;
    
    g_return_val_if_fail (event_name != NULL, NULL);
    g_return_val_if_fail (list != NULL, NULL);

    len = g_list_length (list);
    seq = GDF_LongSeq__alloc ();
    seq->_length = len;
    seq->_maximum = len;
    seq->_buffer = (CORBA_long *) CORBA_sequence_CORBA_long_allocbuf (len);
    CORBA_sequence_set_release (seq, TRUE);
    
    for (i = 0; list != NULL; list = list->next, i++) {
        seq->_buffer[i] = GPOINTER_TO_INT (list->data);
    }

    return build_event (event_name, (CORBA_TypeCode)TC_GDF_LongSeq, seq);
}

CORBA_any *
build_event (const char *event_name, 
			 CORBA_TypeCode type,
			 gpointer value)
{
	GDF_Event *event;
	CORBA_any *event_any;
	
	event = GDF_Event__alloc ();
	event->event_name = CORBA_string_dup (event_name);
	event->argument._type = type;
	event->argument._value = value;
	CORBA_any_set_release (&event->argument, TRUE);
	
	event_any = CORBA_any__alloc ();
	event_any->_type = (CORBA_TypeCode)TC_GDF_Event;
	event_any->_value = event;
	CORBA_any_set_release (event_any, TRUE);

	return event_any;
}

GdfEvent *
gdf_demarshal_event (CORBA_any *any)
{
	GdfEvent *event;
	GDF_Event *corba_event;

	g_return_val_if_fail (any != NULL, NULL);
	
	corba_event = (GDF_Event*)any->_value;

	event = g_new0 (GdfEvent, 1);
	event->event_name = g_strdup (corba_event->event_name);
	
    /* FIXME: Handle arguments that aren't in the supported arg list using
     * the any_arg field */
	if (corba_event->argument._type->kind == CORBA_tk_boolean) {
		event->arg_type = GDF_EVENT_ARG_BOOLEAN;
		event->argument.bool_arg = *((gboolean*)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_long) {
		event->arg_type = GDF_EVENT_ARG_LONG;
		event->argument.long_arg = *((glong*)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_short) {
		event->arg_type = GDF_EVENT_ARG_SHORT;
		event->argument.short_arg = *((gshort*)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_string) {
		event->arg_type = GDF_EVENT_ARG_STRING;
		event->argument.string_arg = 
			g_strdup (*(gchar **)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_char) {
		event->arg_type = GDF_EVENT_ARG_CHAR;
		event->argument.char_arg = *((gchar*)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_double) {
		event->arg_type = GDF_EVENT_ARG_DOUBLE;
		event->argument.double_arg = *((gdouble*)corba_event->argument._value);
	} else if (corba_event->argument._type->kind == CORBA_tk_struct) {
		if (strcmp (corba_event->argument._type->name, 
					"SourceLocation") == 0) {
			event->arg_type = GDF_EVENT_ARG_SOURCE_LOCATION;
			event->argument.loc_arg = g_malloc (sizeof (GDF_SourceLocation));
			memcpy (event->argument.loc_arg, corba_event->argument._value, 
					sizeof (GDF_SourceLocation));
		} else {
			g_error ("FIXME: Handle arguments that aren't a supported type");
		}
    } else if (corba_event->argument._type->kind == CORBA_tk_alias) {
        if (strcmp (corba_event->argument._type->name, "LongSeq") == 0) {
            GList *ret = NULL;
            GDF_LongSeq *seq;
            int i;
            event->arg_type = GDF_EVENT_ARG_LONG_LIST;
            seq = (GDF_LongSeq*)corba_event->argument._value;
            for (i = 0; i < seq->_length; i++)
                ret = g_list_prepend (ret, GINT_TO_POINTER (seq->_buffer[i]));
            ret = g_list_reverse (ret);
            event->argument.list_arg = ret;
        } else {
			g_error ("FIXME: Handle arguments that aren't a supported type");
		}
	} else {
		g_error ("FIXME: Handle arguments that aren't a supported type");
	}

	return event;
}

void 
gdf_event_destroy (GdfEvent *event)
{
	if (event->arg_type == GDF_EVENT_ARG_STRING) 
		g_free (event->argument.string_arg);
	
	if (event->arg_type == GDF_EVENT_ARG_SOURCE_LOCATION)
		g_free (event->argument.loc_arg);
		
	g_free (event->event_name);
	g_free (event);
}

	
	




