/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-register-viewer.h"

struct _GdfRegisterViewerPriv {
    GtkWidget *vbox;
    GtkWidget *show_fp;
    GtkWidget *clist;
    GtkWidget *scrolled;

    GdfDebuggerClient *dbg;

    int execution_stopped_sig;
    int execution_exited_sig;
    int execution_killed_sig;
    int program_unloaded_sig;
};

enum {
    ARG_0,
    ARG_DEBUGGER
};

static void register_viewer_init (GdfRegisterViewer *rv);
static void register_viewer_class_init (GdfRegisterViewerClass *klass);
static void register_viewer_destroy (GtkObject *object);
static void get_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void set_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void connect_debugger_signals (GdfRegisterViewer *rv);
static void disconnect_debugger_signals (GdfRegisterViewer *rv);
static void load_registers (GdfRegisterViewer *rv);
static void execution_stopped_cb (GdfDebuggerClient *dbg,
                                  GdfRegisterViewer *rv);
static void show_fp_toggled_cb (GtkWidget *toggle_bugger, 
                                GdfRegisterViewer *rv);
static void create_children (GdfRegisterViewer *rv);

static GtkFrameClass *parent_class;

/*
 * Public Interface 
 */

GtkType 
gdf_register_viewer_get_type ()
{
    static GtkType type = 0;
	
    if (!type) {
        static const GtkTypeInfo info = {
            "GdfRegisterViewer",
            sizeof (GdfRegisterViewer),
            sizeof (GdfRegisterViewerClass),
            (GtkClassInitFunc) register_viewer_class_init,
            (GtkObjectInitFunc) register_viewer_init,
            NULL,
            NULL,
            (GtkClassInitFunc)NULL
        };
		
        type = gtk_type_unique (gtk_frame_get_type (), &info);
    }
	
    return type;
}

GtkWidget *
gdf_register_viewer_new ()
{
    GdfRegisterViewer *rv;
    
    rv = gtk_type_new (gdf_register_viewer_get_type ());
	
    return GTK_WIDGET (rv);
}

void
gdf_register_viewer_set_debugger (GdfRegisterViewer *rv,
                                   GdfDebuggerClient *dbg)
{
    g_return_if_fail (rv != NULL);
    g_return_if_fail (GDF_IS_REGISTER_VIEWER (rv));
	
    if (rv->priv->dbg) {
        disconnect_debugger_signals (rv);
        gtk_object_unref (GTK_OBJECT (rv->priv->dbg));
    }
	
    rv->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
        if (GTK_OBJECT_FLOATING (GTK_OBJECT (dbg)))
            gtk_object_sink (GTK_OBJECT (dbg));
        
        connect_debugger_signals (rv);
    }
}


/*
 * Class / Object functions
 */

void
register_viewer_class_init (GdfRegisterViewerClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;

    gtk_object_add_arg_type ("GdfRegisterViewer::debugger",
                             GTK_TYPE_OBJECT,
                             GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
                             ARG_DEBUGGER);
	
    parent_class = gtk_type_class (gtk_frame_get_type ());
	
    object_class->destroy = register_viewer_destroy;
    object_class->get_arg = get_arg;
    object_class->set_arg = set_arg;
}

void
register_viewer_init (GdfRegisterViewer *rv)
{
    rv->priv = g_new0 (GdfRegisterViewerPriv, 1);
    
    rv->priv->dbg = NULL;
	
    create_children (rv);
}

void
register_viewer_destroy (GtkObject *obj)
{
    GdfRegisterViewer *rv = GDF_REGISTER_VIEWER (obj);
    
    if (rv->priv->dbg)
        gtk_object_unref (GTK_OBJECT (rv->priv->dbg));
    
    g_free (rv->priv);
    
    if (GTK_OBJECT_CLASS (parent_class)->destroy)
        (*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}

void 
get_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfRegisterViewer *rv = GDF_REGISTER_VIEWER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (rv->priv->dbg);
        break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfRegisterViewer *rv = GDF_REGISTER_VIEWER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        gdf_register_viewer_set_debugger (rv,
                                          GTK_VALUE_OBJECT (*arg) ? GDF_DEBUGGER_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
        break;
    default :
        break;
    }   
}


/* 
 * Helper functions 
 */

void
connect_debugger_signals (GdfRegisterViewer *rv)
{
    rv->priv->execution_stopped_sig = 
        gtk_signal_connect (GTK_OBJECT (rv->priv->dbg), 
                            "execution_stopped",
                            GTK_SIGNAL_FUNC (execution_stopped_cb),
                            (gpointer)rv);
    rv->priv->execution_exited_sig = 
        gtk_signal_connect_object (GTK_OBJECT (rv->priv->dbg),
                                   "execution_exited",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)rv->priv->clist);
    rv->priv->execution_killed_sig = 
        gtk_signal_connect_object (GTK_OBJECT (rv->priv->dbg),
                                   "execution_killed",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)rv->priv->clist);
    rv->priv->program_unloaded_sig = 
        gtk_signal_connect_object (GTK_OBJECT (rv->priv->dbg),
                                   "program_unloaded",
                                   GTK_SIGNAL_FUNC (gtk_clist_clear),
                                   (gpointer)rv->priv->clist);
}

void
disconnect_debugger_signals (GdfRegisterViewer *rv)
{
    gtk_signal_disconnect (GTK_OBJECT (rv->priv->dbg),
                           rv->priv->execution_stopped_sig);
    gtk_signal_disconnect (GTK_OBJECT (rv->priv->dbg), 
                           rv->priv->execution_exited_sig);
}

void
load_registers (GdfRegisterViewer *rv)
{
    GDF_RegisterList *regs;
    GdfDebuggerClientResult res;
    int i;
    gchar *text[2];
    
    res = gdf_debugger_client_get_registers (rv->priv->dbg,
                                             (gboolean)GTK_TOGGLE_BUTTON (rv->priv->show_fp)->active,
                                             &regs);
    if (res != GDF_DEBUGGER_CLIENT_OK)
        return;
    
    gtk_clist_freeze (GTK_CLIST (rv->priv->clist));
    gtk_clist_clear (GTK_CLIST (rv->priv->clist));

    for (i = 0; i < regs->_length; i++) {
        text[0] = regs->_buffer[i].name;
        text[1] = regs->_buffer[i].value;
        gtk_clist_append (GTK_CLIST (rv->priv->clist), text);
        
    }
    gtk_clist_thaw (GTK_CLIST (rv->priv->clist));
    CORBA_free (regs);
}

void
execution_stopped_cb (GdfDebuggerClient *dbg, 
                      GdfRegisterViewer *rv)
{
    load_registers (rv);
}

void
show_fp_toggled_cb (GtkWidget *toggle_bugger, GdfRegisterViewer *rv)
{
    load_registers (rv);
}

void
create_children (GdfRegisterViewer *rv)
{
    gchar *titles[] = { N_("Reg"), N_("Value") };
    
    rv->priv->show_fp = 
        gtk_check_button_new_with_label (_("Include floating point"));

    gtk_signal_connect (GTK_OBJECT (rv->priv->show_fp), "toggled",
                        GTK_SIGNAL_FUNC (show_fp_toggled_cb), rv);

    rv->priv->clist = gtk_clist_new_with_titles (2, titles);
    
    gtk_clist_set_column_width (GTK_CLIST (rv->priv->clist), 0, 40);
	
    rv->priv->scrolled = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (rv->priv->scrolled),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
	
    rv->priv->vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (rv->priv->vbox), rv->priv->show_fp, 
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (rv->priv->vbox), rv->priv->scrolled,
                        TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (rv->priv->scrolled), rv->priv->clist);
    gtk_container_add (GTK_CONTAINER (rv), rv->priv->vbox);
    gtk_widget_show (rv->priv->show_fp);
    gtk_widget_show (rv->priv->clist);
    gtk_widget_show (rv->priv->scrolled);
    gtk_widget_show (rv->priv->vbox);
}

