/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *                         Martin Baulig <martin@home-of-linux.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <config.h>
#include "gdf-symbol-set-client.h"

#include "gdf-event-marshallers.h"

static void symbol_set_client_class_init (GdfSymbolSetClientClass *class);
static void symbol_set_client_init (GdfSymbolSetClient *prog);
static void symbol_set_client_destroy (GdfSymbolSetClient *prog);

static void event_pushed_cb (GdfEventChannelClient *event_channel,
                             CORBA_any *val,
                             GdfSymbolSetClient *ss);
static GdfSymbolSetClientResult exception_to_result (CORBA_Environment *ev);

static GtkObjectClass *parent_class;

struct _GdfSymbolSetClientPrivate {
    GdfEventChannelClient *event_channel;
};

enum {
    SYMBOL_SET_CHANGED,
    SYMBOL_CHANGED,
    SYMBOL_TYPE_CHANGED,
    LAST_SIGNAL
};

static gint symbol_set_client_signals[LAST_SIGNAL];

static void 
gdf_symbol_set_client_construct (GdfSymbolSetClient *ss, 
                                 CORBA_Object corba_object)
{
    CORBA_Object channel;
    CORBA_Environment ev;

    ss->objref = corba_object;
    ss->priv = g_new0 (GdfSymbolSetClientPrivate, 1);
    
    CORBA_exception_init (&ev);

    channel = GDF_SymbolSet_get_event_channel (corba_object, &ev);
    
    /* FIXME: Check env */

    CORBA_exception_free (&ev);

    ss->priv->event_channel = 
        gdf_event_channel_client_new_from_corba (channel);
    
    gdf_event_channel_client_listen (ss->priv->event_channel);
    
    gtk_signal_connect (GTK_OBJECT (ss->priv->event_channel),
                        "event_pushed", 
                        GTK_SIGNAL_FUNC (event_pushed_cb),
                        (gpointer)ss);
}

/**
 * gdf_symbol_set_client_new_from_objref:
 * @symbol_set: GDF::SymbolSet CORBA object reference.
 *
 * Creates a GdfSymbolSetClient GTK Object which wraps around the
 * GDF::SymbolSet CORBA object.
 *
 * Returns: the initialized GdfSymbolSetClient object.
 */
GdfSymbolSetClient *
gdf_symbol_set_client_new_from_corba (GDF_SymbolSet symbol_set) 
{
    GdfSymbolSetClient *ret;
    CORBA_Environment ev;

    g_return_val_if_fail (symbol_set != CORBA_OBJECT_NIL, NULL);

    ret = gtk_type_new (gdf_symbol_set_client_get_type ());
    gdf_symbol_set_client_construct (ret, symbol_set);
   
    CORBA_exception_init (&ev);

    Bonobo_Unknown_ref (symbol_set, &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);

    return ret;
}

GtkType
gdf_symbol_set_client_get_type (void)
{
    static GtkType type = 0;
    
    if (!type) {
        GtkTypeInfo info = {
            "GdfSymbolSetClient",
            sizeof (GdfSymbolSetClient),
            sizeof (GdfSymbolSetClientClass),
            (GtkClassInitFunc) symbol_set_client_class_init,
            (GtkObjectInitFunc) symbol_set_client_init,
            NULL,
            NULL,
            (GtkClassInitFunc) NULL
        };
	
        type = gtk_type_unique (gtk_object_get_type (), &info);
    }
    
    return type;
}

/**
 * gdf_symbol_set_client_add_expression:
 * @ss: The GdfSymbolSet object.
 * @expr: The expression to add.
 * @handle: An out parameter through which the symbol handle will be returned.
 * 
 * Add an expression to the symbol set.  The symbolset will emit symbol_changed
 * signals when the singal changes.
 * 
 * Return value: 
 **/
GdfSymbolSetClientResult 
gdf_symbol_set_client_add_expression (GdfSymbolSetClient *ss,
                                      const char *expr,
                                      int *handle)
{
    CORBA_Environment ev;
    GdfSymbolSetClientResult res;
    CORBA_long ret;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss),
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);
    ret = GDF_SymbolSet_add_expression (ss->objref, expr, &ev);    

    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);
    
    if (handle) 
        *handle = ret;

    return res;
}

/**
 * gdf_symbol_set_client_get_symbols:
 * @ss: 
 * @ret: 
 * 
 * Get a GDF::SymbolSequence representing all the toplevel symbols in the 
 * symbolset.
 * 
 * Return value: 
 **/
GdfSymbolSetClientResult
gdf_symbol_set_client_get_symbols (GdfSymbolSetClient *ss,
                                   GDF_SymbolSequence **ret)
{  
    GdfSymbolSetClientResult res;
    CORBA_Environment ev;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss),
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (ret != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);
    *ret = GDF_SymbolSet_get_symbols (ss->objref, &ev);    

    res = exception_to_result (&ev);
    if (res != GDF_SYMBOL_SET_CLIENT_OK)
        *ret = NULL;
    CORBA_exception_free (&ev);
    return res;
}

GdfSymbolSetClientResult
gdf_symbol_set_client_get_symbol (GdfSymbolSetClient *ss,
                                  int handle,
                                  GDF_Symbol **ret)
{
    GdfSymbolSetClientResult res;
    CORBA_Environment ev;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss),
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (ret != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);
    *ret = GDF_SymbolSet_get_symbol (ss->objref,
                                     (CORBA_long)handle,
                                     &ev);

    res = exception_to_result (&ev);
    if (res != GDF_SYMBOL_SET_CLIENT_OK)
        *ret = NULL;
    CORBA_exception_free (&ev);
    return res;
}

GdfSymbolSetClientResult
gdf_symbol_set_client_get_symbol_children (GdfSymbolSetClient *ss,
                                           int handle,
                                           GDF_SymbolSequence **ret)
{
    GdfSymbolSetClientResult res;
    CORBA_Environment ev;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss), 
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (ret != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    
    CORBA_exception_init (&ev);

    *ret = GDF_SymbolSet_get_symbol_children (ss->objref, 
                                             (CORBA_long)handle,
                                             &ev);

    res = exception_to_result (&ev);
    if (res != GDF_SYMBOL_SET_CLIENT_OK)
        *ret = NULL;
    CORBA_exception_free (&ev);
    return res;
}

GdfSymbolSetClientResult 
gdf_symbol_set_client_set_symbol_value (GdfSymbolSetClient *ss,
                                        int handle,
                                        const char *value)
{    
    GdfSymbolSetClientResult res;
    CORBA_Environment ev;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss), 
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (value != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GDF_SymbolSet_set_symbol_value (ss->objref, (CORBA_long)handle, 
                                    value, &ev);

    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);
    return res;
}

GdfSymbolSetClientResult 
gdf_symbol_set_client_cast (GdfSymbolSetClient *ss,
                            int handle,
                            const char *type)
{    
    GdfSymbolSetClientResult res;
    CORBA_Environment ev;
    
    g_return_val_if_fail (ss != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (GDF_IS_SYMBOL_SET_CLIENT (ss), 
                          GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);
    g_return_val_if_fail (type != NULL, GDF_SYMBOL_SET_CLIENT_BAD_PARAMS);

    CORBA_exception_init (&ev);

    GDF_SymbolSet_cast (ss->objref, (CORBA_long)handle, type, &ev);

    res = exception_to_result (&ev);
    CORBA_exception_free (&ev);
    return res;
}

/* private routines */
static void
symbol_set_client_class_init (GdfSymbolSetClientClass *klass)
{
    GtkObjectClass *object_class = (GtkObjectClass *)klass;
    
    g_return_if_fail (klass != NULL);
    g_return_if_fail (GDF_IS_SYMBOL_SET_CLIENT_CLASS (klass));
    
    parent_class = gtk_type_class (gtk_object_get_type ());
    
    symbol_set_client_signals[SYMBOL_SET_CHANGED] = 
        gtk_signal_new ("symbol_set_changed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfSymbolSetClientClass,
                                           symbol_set_changed),
                        gtk_marshal_NONE__NONE,
                        GTK_TYPE_NONE, 0,
                        GTK_TYPE_NONE);
    symbol_set_client_signals [SYMBOL_CHANGED] = 
        gtk_signal_new ("symbol_changed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfSymbolSetClientClass,
                                           symbol_changed),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    symbol_set_client_signals [SYMBOL_TYPE_CHANGED] = 
        gtk_signal_new ("symbol_type_changed",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfSymbolSetClientClass,
                                           symbol_type_changed),
                        gtk_marshal_NONE__INT,
                        GTK_TYPE_NONE, 1,
                        GTK_TYPE_INT);
    gtk_object_class_add_signals (object_class,
                                  symbol_set_client_signals,
                                  LAST_SIGNAL);

    klass->symbol_set_changed = NULL;
    klass->symbol_changed = NULL;
    klass->symbol_type_changed = NULL;

    object_class->destroy = (GtkSignalFunc) symbol_set_client_destroy;
}

static void
symbol_set_client_init (GdfSymbolSetClient *client)
{
    client->priv = NULL;
}

static void
symbol_set_client_destroy (GdfSymbolSetClient *client)
{
    CORBA_Environment ev;

    if (client->priv->event_channel) {
        bonobo_object_unref (BONOBO_OBJECT (client->priv->event_channel));
        client->priv->event_channel = NULL;
    }

    if (client->priv) {
        g_free (client->priv);
        client->priv = NULL;
    }

    CORBA_exception_init (&ev);
    
    Bonobo_Unknown_unref (client->objref, &ev);
    
    /* FIXME: Check env */
    
    CORBA_exception_free (&ev);

    if (parent_class->destroy)
        parent_class->destroy (GTK_OBJECT (client));
    
}

static void
emit_event_signal (GdfSymbolSetClient *ss,
                   GdfEvent *event)
{
    /* FIXME: Some table logic would be much better here */

    if (strcmp (event->event_name, "symbol_set_changed") == 0) {
        gtk_signal_emit (GTK_OBJECT (ss), 
                         symbol_set_client_signals[SYMBOL_SET_CHANGED]);
    } else if (strcmp (event->event_name, "symbol_changed") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (ss), 
                         symbol_set_client_signals[SYMBOL_CHANGED],
                         GPOINTER_TO_INT (event->argument.long_arg));
    } else if (strcmp (event->event_name, "symbol_type_changed") == 0) {
        g_assert (event->arg_type == GDF_EVENT_ARG_LONG);
        gtk_signal_emit (GTK_OBJECT (ss), 
                         symbol_set_client_signals[SYMBOL_TYPE_CHANGED],
                         GPOINTER_TO_INT (event->argument.long_arg));
    }
}

void 
event_pushed_cb (GdfEventChannelClient *event_channel,
                 CORBA_any *val,
                 GdfSymbolSetClient *ss)
{
    GdfEvent *event;
    event = gdf_demarshal_event (val);
    
    emit_event_signal (ss, event);

    gdf_event_destroy (event);
}

GdfSymbolSetClientResult 
exception_to_result (CORBA_Environment *ev)
{
    GdfSymbolSetClientResult res;
    static GHashTable *table = NULL;

    if (!table) {
        table = g_hash_table_new (g_str_hash, g_str_equal);      
        g_hash_table_insert (table, ex_GDF_SymbolSet_InvalidType,
                             GINT_TO_POINTER (GDF_SYMBOL_SET_CLIENT_INVALID_TYPE));
    }

    res = GDF_SYMBOL_SET_CLIENT_OK;
    
    if (ev->_major == CORBA_USER_EXCEPTION) {
        res = GPOINTER_TO_INT (g_hash_table_lookup (table, ev->_repo_id));
    }

    return res;
}
