/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/* expense conduit, based on read-expense */

#include <glib.h>
#include <gnome.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-dlp.h>
#include <pi-version.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#include <unistd.h>
#include <pwd.h>
#include <signal.h>
#include <time.h>

#include <gpilotd/gnome-pilot-conduit.h>
#include <gpilotd/gnome-pilot-conduit-standard.h>
#include "expense_conduit.h"

#define CONDUIT_VERSION "0.3"

GnomePilotConduit *conduit_get_gpilot_conduit( guint32 pilotId );
void conduit_destroy_gpilot_conduit( GnomePilotConduit *c );

/* Following depend on the ordering in pi-expense.h ! */
static gchar *ExpenseTypeName[] = { "Airfare", "Breakfast", "Bus", "BusinessMeals", "CarRental", 
                                    "Dinner", "Entertainment", "Fax", "Gas", "Gifts", "Hotel", 
                                    "Incidentals","Laundry", "Limo", "Lodging", "Lunch", "Mileage", 
                                    "Other", "Parking", "Postage", "Snack", "Subway", "Supplies", 
                                    "Taxi", "Telephone", "Tips", "Tolls", "Train" };

static gchar *ExpensePaymentName[] = { "AmEx", "Cash", "Check", "CreditCard", "MasterCard", 
                                       "PrePaid", "VISA", "Unfiled" };

/* these values are hardcoded in the palm expense appl. does it differ for non-us palms??? */
static gchar *ExpenseCurrencyName[] = { "AU$", "S", "BF", "R$", "$CN", "DKK", "Mk", "FRF", "DM", 
                                        "HK$", "ISK", "IEP", "L.", "Y", "Flux", "MXP", "NLG", 
                                        "$NZ", "NOK", "Pts", "SEK", "CHF", "GBP", "$", "EU" };

/* #define EC_DEBUG */
#ifdef EC_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "expense: "##format, ##args)
#else
#define LOG(format,args...)
#endif

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar *tempbuf;

	g_assert(c!=NULL);
	*c = g_new0(ConduitCfg,1);
	(*c)->child = -1;

	prefix = g_strdup_printf("/gnome-pilot.d/expense-conduit/Pilot_%u/",pilotId);
  
	gnome_config_push_prefix(prefix);
	(*c)->dir = gnome_config_get_string( "dir");
	(*c)->dateFormat = gnome_config_get_string( "date_format=%x");
	(*c)->outputFormat = gnome_config_get_int("output_format=0");
	tempbuf = gnome_config_get_string("dir mode=0700");
	(*c)->dirMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);
	tempbuf = gnome_config_get_string("file mode=0600");
	(*c)->fileMode =(mode_t)strtol(tempbuf,NULL,0);
	g_free(tempbuf);

	gnome_config_pop_prefix();

	(*c)->pilotId = pilotId;
	g_free(prefix);
}

/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_assert(c!=NULL);
	g_assert(*c!=NULL);
	g_free( (*c)->dir);
	g_free( (*c)->dateFormat);
	g_free(*c);
	*c = NULL;
}


/* from pilot-xfer */
static void protect_name(char *d, char *s) 
{
	while(*s) {
		switch(*s) {
		case '/': *(d++) = '='; *(d++) = '2'; *(d++) = 'F'; break;
		case '=': *(d++) = '='; *(d++) = '3'; *(d++) = 'D'; break;
		case '\x0A': *(d++) = '='; *(d++) = '0'; *(d++) = 'A'; break;
		case '\x0D': *(d++) = '='; *(d++) = '0'; *(d++) = 'D'; break;
			/*case ' ': *(d++) = '='; *(d++) = '2'; *(d++) = '0'; break;*/
		default: 
			if(*s < ' ') {
				gchar tmp[6];
				g_snprintf(tmp,5,"=%2X",(unsigned char)*s);
				*(d++) = tmp[0]; *(d++) = tmp[1]; *(d++) = tmp[2];
			} else
				*(d++) = *s;
			break;
		}
		++s;
	}
	*d = '\0';
}

/** 
    generates a pathname for a category 
 */
static gchar *category_path(int category, GnomePilotConduit *c) 
{
	static gchar filename[FILENAME_MAX];
	gchar buf[64];
	
	if(category==16) 
		strcpy(buf,"Archived");
	else
		protect_name(buf,GET_CONDUIT_DATA(c)->ai.category.name[category]);
  
	g_snprintf(filename,FILENAME_MAX-1,"%s/%s",
		   GET_CONFIG(c)->dir,
		   buf);
	     
	return filename;
}

static void writeout_record(int fd, struct Expense *record, GnomePilotConduit *c)
{
        char entry[0xffff];
    
        const int kDateStrSize = 30;
        char DateStr[kDateStrSize];
        char *Currency;

        strftime(DateStr, kDateStrSize, GET_CONFIG(c)->dateFormat, &record->date);

        if(record->currency < 24)
                Currency = ExpenseCurrencyName[record->currency];
        /* handle the euro*/ 
        else if(record->currency == 133)
                Currency = ExpenseCurrencyName[24];
        /* handle the custom currencies */
        else if(record->currency >= 128 && record->currency < 133) 
                Currency = GET_CONDUIT_DATA(c)->ai.currencies[record->currency-128].symbol;
        else {
                g_warning(_("Unknown Currency Symbol"));
                Currency = "";
        }

        switch(GET_CONFIG(c)->outputFormat) {
        case eSimpleFormat:
                sprintf(entry, "%s, %s, %s, %s, %s\n", DateStr, ExpenseTypeName[record->type], ExpensePaymentName[record->payment], Currency, record->amount ? record->amount: "<None>");
                break;
        case eComplexFormat:
        default:
                sprintf(entry, "Date: %s, Type: %s, Payment: %s, Currency: %s, Amount: '%s', Vendor: '%s', City: '%s', Attendees: '%s', Note: '%s'\n", DateStr, ExpenseTypeName[record->type], ExpensePaymentName[record->payment], Currency, record->amount ? record->amount: "<None>", record->vendor ? record->vendor: "<None>", record->city ? record->city: "<None>", record->attendees ? record->attendees: "<None>", record->note ? record->note: "<None>");
        }

#ifdef EC_DEBUG
        fprintf(stderr, "%s\n", entry);
#endif
        if(write(fd, entry, strlen(entry)) == -1) {
                perror("writeout_record");
        }
}

/* Parts of this routine is shamelessly stolen from read-expenses.c from the pilot-link 
package, Copyright (c) 1997, Kenneth Albanowski
*/
static gint copy_from_pilot( GnomePilotConduit *c, GnomePilotDBInfo *dbi )
{
        int dbHandle;
        guchar buffer[0xffff];

        struct ExpenseAppInfo *tai;
        struct ExpensePref *tp;

        int i;
        int ret;
        int filehandle[17];

        if(GET_CONFIG(c)->dir==NULL) {
		g_warning(_("No dir specified. Please run expense conduit capplet first."));
		gnome_pilot_conduit_error(c,_("No dir specified. Please run expense conduit capplet first."));
        }

        tai = &(GET_CONDUIT_DATA(c)->ai);
        tp = &(GET_CONDUIT_DATA(c)->pref);

        g_message ("Expense Conduit v.%s", CONDUIT_VERSION);

        if(dlp_OpenDB(dbi->pilot_socket, 0, 0x80|0x40, "ExpenseDB", &dbHandle) < 0) {
                g_warning("Unable to open ExpenseDB");
                return -1;
        }
    
        unpack_ExpensePref(tp, buffer, 0xffff);

#ifdef EC_DEBUG
        fprintf(stderr, "Orig prefs, %d bytes:\n", ret);
        fprintf(stderr, "Expense prefs, current category %d, default category %d\n",
                tp->currentCategory, tp->defaultCategory);
        fprintf(stderr, "  Note font %d, Show all categories %d, Show currency %d, Save backup %d\n",
                tp->noteFont, tp->showAllCategories, tp->showCurrency, tp->saveBackup);
        fprintf(stderr, "  Allow quickfill %d, Distance unit %d, Currencies:\n",
                tp->allowQuickFill, tp->unitOfDistance);
        for(i = 0; i < 7; i++) {
                fprintf(stderr, " %d", tp->currencies[i]);
        }
        fprintf(stderr, "\n");
#endif /* EC_DEBUG */

        ret = dlp_ReadAppBlock(dbi->pilot_socket, dbHandle, 0, buffer, 0xffff);
    
        unpack_ExpenseAppInfo(tai, buffer, 0xffff);

#ifdef EC_DEBUG
        fprintf(stderr, "Orig length %d, new length %d, orig data:\n", ret, i);
        fprintf(stderr, "New data:\n");
    
        fprintf(stderr, "Expense app info, sort order %d\n", tai->sortOrder);
        for(i = 0; i < 4; i++)
                fprintf(stderr, " Currency %d, name '%s', symbol '%s', rate '%s'\n", i, 
                        tai->currencies[i].name, tai->currencies[i].symbol, tai->currencies[i].rate);
#endif /* EC_DEBUG */

        /* make the directory */
        if(mkdir(GET_CONFIG(c)->dir, GET_CONFIG(c)->dirMode) < 0) {
                if(errno != EEXIST) {
                        g_warning ("Unable to create directory:\n\t%s\n", GET_CONFIG(c)->dir);
                        return -1;
                }
        }
        
        /* open one file for every category in Expense */
        for(i = 0; i < 17; i++) {
                if(*tai->category.name[i] != '\0') {
                        LOG("Opening for cat %d: %s\n", i, category_path(i, c));
                        if((filehandle[i] = creat(category_path(i, c), GET_CONFIG(c)->fileMode))== -1) {
                                LOG("copy_from_pilot: error in opening %s", category_path(i, c));
                                perror("");
                                return -1;
                        }
                } else {
                        filehandle[i] = -1;
                }
        }

        /* loop through all the records */
        for (i = 0; ; i++) {
                struct Expense t;
                int attr, category, len;

                len = dlp_ReadRecordByIndex(dbi->pilot_socket, dbHandle, i, buffer, 0, 0, &attr, &category);
                
                /* at the end of all the records? */
                if(len < 0)
                        break;
                /* Skip deleted records */
                if((attr & dlpRecAttrDeleted) || (attr & dlpRecAttrArchived))
                        continue;

                unpack_Expense(&t, buffer, len);
                writeout_record(filehandle[category], &t, c);
                free_Expense(&t);
        }

        /* close all the opened filehandles */
        for(i = 0; i < 17; i++)
                if(filehandle[i] != -1)
                        close(filehandle[i]);
        
        dlp_ResetLastSyncPC( dbi->pilot_socket );
        /* Close the database */
        dlp_CloseDB(dbi->pilot_socket, dbHandle);
        
        return( 0 );
}

static gint synchronize( GnomePilotConduit *c, GnomePilotDBInfo *dbi ) {
        return copy_from_pilot( c, dbi );
}

GnomePilotConduit *conduit_get_gpilot_conduit( guint32 pilotId ) 
{
        GtkObject *retval;
        ConduitCfg *cc;
        ConduitData *cd = g_new0(ConduitData, 1);

        retval = gnome_pilot_conduit_standard_new("ExpenseDB", Expense_Creator);
        gnome_pilot_conduit_construct(GNOME_PILOT_CONDUIT(retval),"Expense");
        g_assert(retval != NULL);
        gtk_signal_connect(retval, "copy_from_pilot", (GtkSignalFunc)copy_from_pilot ,NULL);
        /*
          gtk_signal_connect(retval, "copy_to_pilot", (GtkSignalFunc) ,NULL);
          gtk_signal_connect(retval, "merge_to_pilot", (GtkSignalFunc) ,NULL);
          gtk_signal_connect(retval, "merge_from_pilot", (GtkSignalFunc) ,NULL);
        */
        gtk_signal_connect(retval, "synchronize", (GtkSignalFunc)synchronize ,NULL);
        
        load_configuration(&cc, pilotId );
        gtk_object_set_data(retval,"conduit_config",(gpointer)cc);
        gtk_object_set_data(retval,"conduit_data",(gpointer)cd);
        
        return GNOME_PILOT_CONDUIT(retval); 
}

void conduit_destroy_gpilot_conduit( GnomePilotConduit *c ) 
{
        ConduitCfg *cc;
        ConduitData  *cd;
  
        cc = GET_CONFIG(c);
        cd = GET_CONDUIT_DATA(c);

        destroy_configuration( &cc );
        gtk_object_destroy(GTK_OBJECT(c));
}

