/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescannerselector.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnomescan.h>
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define	GNOME_SCANNER_SELECTOR_PARENT_CLASS(klass)	(GTK_SELECTOR_CLASS (g_type_class_peek_parent ((klass))))
#define GNOME_SCANNER_SELECTOR_ERROR			(g_type_qname (GNOME_TYPE_SCANNER_SELECTOR))
#define GET_PRIVATE(obj)				(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCANNER_SELECTOR, GnomeScannerSelectorPrivate))

enum {
  COLUMN_ICON,
  COLUMN_LABEL,
  N_COLUMNS
};


void				gnome_scanner_selector_probe_done			(GnomeScanContext *context,
											 GnomeScannerSelector *selector);

void				gnome_scanner_selector_fill_store_func			(GnomeScanner *scanner,
											 GtkListStore *store);

void				gnome_scanner_selector_fill_store			(GnomeScannerSelector *selector);

void				gnome_scanner_selector_scanner_choosed			(GtkComboBox *combo,
											 GnomeScannerSelector *selector);

/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scanner_selector_dispose				(GObject *obj);

void				gnome_scanner_selector_set_property 			(GObject *obj,
											 guint property_id,
											 const GValue *value,
											 GParamSpec *pspec);

void				gnome_scanner_selector_get_property 			(GObject *obj,
											 guint property_id,
											 GValue *value,
											 GParamSpec *pspec);

typedef struct _GnomeScannerSelectorPrivate		GnomeScannerSelectorPrivate;

struct _GnomeScannerSelectorPrivate {
  gboolean		dispose_has_run;
  GtkWidget		*combo;
};

enum {
  PROP_0
};


G_DEFINE_TYPE (GnomeScannerSelector, gnome_scanner_selector, GNOME_TYPE_SCAN_OPTION_WIDGET);

void
gnome_scanner_selector_class_init (GnomeScannerSelectorClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScannerSelectorPrivate));
}

void
gnome_scanner_selector_init (GnomeScannerSelector *widget)
{
  GnomeScannerSelectorPrivate *priv = GET_PRIVATE (widget);

  priv->dispose_has_run = FALSE;
  priv->combo = NULL;
}

void
gnome_scanner_selector_dispose (GObject *obj)
{
  GnomeScannerSelector *widget = GNOME_SCANNER_SELECTOR (obj);
  GnomeScannerSelectorPrivate *priv = GET_PRIVATE (widget);
  GnomeScannerSelectorClass *b_klass = GNOME_SCANNER_SELECTOR_GET_CLASS (obj);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  gtk_widget_destroy (priv->combo);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCANNER_SELECTOR_PARENT_CLASS (klass)->dispose (obj); */
}






/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scanner_selector_new:
 * @context: a #GnomeScanContext
 * @show_label: wether to display a label
 * 
 * Create a new #GnomeScannerSelector connected to @context.
 * 
 * Return value: a new #GnomeScannerSelector
 */
GtkWidget*
gnome_scanner_selector_new (GnomeScanContext *context)
{
  GtkWidget *selector, *label, *combo;
  GtkListStore *store;
  GtkCellRenderer *renderer;

  selector = GTK_WIDGET (g_object_new (GNOME_TYPE_SCANNER_SELECTOR,
				       "context", context,
				       NULL));

  /* LABEL */

  gnome_scan_option_widget_set_label (GNOME_SCAN_OPTION_WIDGET (selector), _("Scanner"));
  gnome_scan_option_widget_show_label (GNOME_SCAN_OPTION_WIDGET (selector));

  /* COMBO */
  
  store = gtk_list_store_new (N_COLUMNS,
			      GDK_TYPE_PIXBUF,
			      G_TYPE_STRING);

  combo = gtk_combo_box_new_with_model (GTK_TREE_MODEL (store));

  g_signal_connect (GTK_COMBO_BOX (combo),
		    "changed",
		    (GCallback) gnome_scanner_selector_scanner_choosed,
		    selector);

  GET_PRIVATE (GNOME_SCANNER_SELECTOR (selector))->combo = combo;

  /* layout column ICON */
  renderer = gtk_cell_renderer_pixbuf_new ();

  gtk_cell_renderer_set_fixed_size (GTK_CELL_RENDERER (renderer),
				    32, 32);

  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo),
			      renderer,
			      FALSE);

  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (combo),
				 renderer,
				 "pixbuf",
				 COLUMN_ICON);

  /* layout column LABEL */
  renderer = gtk_cell_renderer_text_new ();

  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combo),
			      renderer,
			      TRUE);

  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (combo),
				 renderer,
				 "markup",
				 COLUMN_LABEL);

  /*   gtk_widget_show (combo); */

  gnome_scan_option_widget_pack_child (GNOME_SCAN_OPTION_WIDGET (selector),
				       combo,
				       FALSE,
				       TRUE);

  /* SIGNALS */

  g_signal_connect (context,
		    "probe-done",
		    (GCallback) gnome_scanner_selector_probe_done,
		    GNOME_SCANNER_SELECTOR (selector));

  gtk_widget_set_no_show_all (selector, TRUE);

  return selector;
}

/* DEVICE PROBE HANDLING */
void
gnome_scanner_selector_probe_done (GnomeScanContext *context,
				   GnomeScannerSelector *selector)
{
  gtk_widget_set_no_show_all (GTK_WIDGET (selector), FALSE);
  gnome_scanner_selector_fill_store (selector);
}

void
gnome_scanner_selector_fill_store_func (GnomeScanner *scanner,
					GtkListStore *store)
{
  GtkTreeIter iter;
  GdkPixbuf *pixbuf;
  GError *error = NULL;
  gchar *label, *icon_name;
  GnomeScannerType type = gnome_scanner_get_scanner_type (scanner);

  static const gchar *prefix = "input-scanner";

  if (type != GNOME_SCANNER_UNKNOWN) {
    icon_name = g_strconcat (prefix, "-", gs_enum_get_nick (GNOME_TYPE_SCANNER_TYPE, type), NULL);
  }
  else {
    icon_name = g_strdup (prefix);
  }

  /* Load ICON */
  pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
				     icon_name,
				     32,
				     0,
				     &error);

  /* Determine LABEL */
  label = g_strconcat ("<b>", gnome_scanner_get_product (scanner), "</b>\n",
		       gnome_scanner_get_vendor (scanner),
		       NULL);

  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter,
		      COLUMN_ICON, pixbuf,
		      COLUMN_LABEL, label,
		      -1);

}

void
gnome_scanner_selector_fill_store (GnomeScannerSelector *selector)
{
  g_return_if_fail (GNOME_IS_SCANNER_SELECTOR (selector));

  GtkListStore *store;
  GSList *devices;
  GnomeScanContext *context;

  context = gnome_scan_option_widget_get_context (GNOME_SCAN_OPTION_WIDGET (selector));
  devices = gnome_scan_context_get_scanners (context);

  store = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (GET_PRIVATE(selector)->combo)));
  gtk_list_store_clear (store);

  /* Fill the store */
  /* TODO: handle store refill */
  g_slist_foreach (devices,
		   (GFunc) gnome_scanner_selector_fill_store_func,
		   store);

  /* Select first */
  gtk_combo_box_set_active (GTK_COMBO_BOX (GET_PRIVATE (selector)->combo),
			    0);

  /* Hide one-option-selector */
  if (g_slist_length (devices) <= 1) {
    gtk_widget_hide (GTK_WIDGET (selector));
  }
  else {
    gtk_widget_show_all (GTK_WIDGET (selector));
  }

}

/* CHOICE HANDLING */
void
gnome_scanner_selector_scanner_choosed (GtkComboBox *combo,
					GnomeScannerSelector *selector)
{
  GSList *scanners;
  GnomeScanner *current;
  GnomeScanContext *context;

  context = gnome_scan_option_widget_get_context (GNOME_SCAN_OPTION_WIDGET (selector));
  scanners = gnome_scan_context_get_scanners (context);
  current = g_slist_nth_data (scanners, gtk_combo_box_get_active (combo));
  gnome_scan_context_select_scanner (context, current);
}
