/* handybrl.c
 *
 * Copyright 2003 HandyTech Elektronik GmbH
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "handybrl.h"
#include "sercomm.h"


#define MAX_BUFFER 90
#define MAX_GET_BUFFER 10
#define MAX_KEYCODE_LENGHT 30


/* Supportet HandyTech devices */
typedef enum {BRW=0, BL2, BS4, BS8, MB2, MB4, MB8} HANDY_DEVICE_TYPE;

/* key-states */
typedef enum {key_idle=0, key_pressed, key_relinquished, key_evaluated} KEY_STATES;

/* keys */
typedef enum
{
	k_f1=0, k_f2, k_f3, k_f4, k_f5, k_f6, k_f7, k_f8,                       /* braille keys */
	k_up, k_down, k_enter, k_esc, k_space, k_rspace,                        /* funktion keys */
	k_b9, k_b10, k_b11, k_b12, k_b13, k_b14,                                /* 16er block */
	k_0, k_1, k_2, k_3, k_4, k_5, k_6, k_7, k_8, k_9,
	k_r01, k_r02, k_03, k_r04, k_r05, k_r06, k_r07, k_r08, k_r09, k_r10,    /* courser routing over textmodules */
	k_r11, k_r12, k_13, k_r14, k_r15, k_r16, k_r17, k_r18, k_r19, k_r20,
	k_r21, k_r22, k_23, k_r24, k_r25, k_r26, k_r27, k_r28, k_r29, k_r30,
	k_r31, k_r32, k_33, k_r34, k_r35, k_r36, k_r37, k_r38, k_r39, k_r40,
	k_r41, k_r42, k_43, k_r44, k_r45, k_r46, k_r47, k_r48, k_r49, k_r50,
	k_r51, k_r52, k_53, k_r54, k_r55, k_r56, k_r57, k_r58, k_r59, k_r60,
	k_r61, k_r62, k_63, k_r64, k_r65, k_r66, k_r67, k_r68, k_r69, k_r70,
	k_r71, k_r72, k_73, k_r74, k_r75, k_r76, k_r77, k_r78, k_r79, k_r80,
	k_s1, k_s2, k_s3, k_s4,                                                 /* courser routing over statusmodules */ 
	MAX_KEYS
} HANDY_KEYS;


/* HandyTech Device Data Type */
typedef struct 
{
	HANDY_DEVICE_TYPE Name;     /* Name of the HandyTech Device */
	unsigned char CellCount;    /* Count of Cells */
	unsigned char IDByte;       /* Identifier Byte of Braille-Device*/
} HANDY_DEVICE_DATA;


static BRL_DEV_CALLBACK ClientCallback = NULL;
static HANDY_DEVICE_DATA HandyDeviceData; 
static unsigned char getbuffer[MAX_GET_BUFFER];
static short gb_index = 0;
static KEY_STATES keyArray[MAX_KEYS];


/* check, is frame complete */
short is_complete_frame()
{
	static int countdown = -1;
	
	if (countdown == 0)  /* frame complete */
	{
		countdown = -1;
		if (getbuffer[getbuffer[2]+3] == 0x16) return 1;
		else gb_index = 0; /* begin new frame */
	}
	else
	{
		if ((getbuffer[2] != 0) && (countdown < 0)) /* set countdown */
		{
			countdown = getbuffer[2];
		}
		else if (countdown >= 0)
		{
			countdown--;
		}
	}
	return 0;
}


/* set evaluated keys to preddes keys */
void refresh_evaluated_to_pressed()
{
	int i;

	for (i=0; i <= MAX_KEYS; i++)
	{
		if (keyArray[i] == key_evaluated)
			keyArray[i] = key_pressed;
	}
}


/* check, which key-combination is pressed */
int OnKeyChanged()
{
	int i;
	unsigned char do_evaluation = 0;

	for (i=1; i < getbuffer[2]; i++)
	{
		switch (getbuffer[i+3])
		{
		
		/* braille-keys*/
		case 0x03:                                     /* B1 bzw. 7 */
			keyArray[k_f7] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x83:
		        if (keyArray[k_f7] != key_evaluated)
		        {
				keyArray[k_f7] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f7] = key_idle;
		        break;
		case 0x07:                                     /* B2 bzw. 3 */
			keyArray[k_f3] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x87:
		        if (keyArray[k_f3] != key_evaluated)
		        {
				keyArray[k_f3] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f3] = key_idle;
		        break;
		case 0x0B:                                     /* B3 bzw. 2 */
			keyArray[k_f2] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x8B:
		        if (keyArray[k_f2] != key_evaluated)
		        {
				keyArray[k_f2] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f2] = key_idle;
		        break;
		case 0x0F:                                     /* B4 bzw. 1 */
			keyArray[k_f1] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x8F:
		        if (keyArray[k_f1] != key_evaluated)
		        {
				keyArray[k_f1] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f1] = key_idle;
		        break;
		case 0x13:                                     /* B5 bzw. 4 */
			keyArray[k_f4] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x93:
		        if (keyArray[k_f4] != key_evaluated)
		        {
				keyArray[k_f4] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f4] = key_idle;
		        break;
		case 0x17:                                     /* B6 bzw. 5 */
			keyArray[k_f5] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x97:
		        if (keyArray[k_f5] != key_evaluated)
		        {
				keyArray[k_f5] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f5] = key_idle;
		        break;
		case 0x1B:                                     /* B7 bzw. 6 */
			keyArray[k_f6] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x9B:
		        if (keyArray[k_f6] != key_evaluated)
		        {
				keyArray[k_f6] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f6] = key_idle;
		        break;
		case 0x1F:                                     /* B8 bzw. 8 */
			keyArray[k_f8] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x9F:
		        if (keyArray[k_f8] != key_evaluated)
		        {
				keyArray[k_f8] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_f8] = key_idle;
		        break;

		/* function keys */
		case 0x04:                                     /* up */
			keyArray[k_up] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x84:
		        if (keyArray[k_up] != key_evaluated)
		        {
				keyArray[k_up] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_up] = key_idle;
		        break;
		case 0x08:                                     /* down */
			keyArray[k_down] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x88:
		        if (keyArray[k_down] != key_evaluated)
		        {
				keyArray[k_down] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_down] = key_idle;
		        break;
		case 0x14:                                     /* enter */
			keyArray[k_enter] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x94:
		        if (keyArray[k_enter] != key_evaluated)
		        {
				keyArray[k_enter] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_enter] = key_idle;
		        break;
		case 0x0C:                                     /* esc */
			keyArray[k_esc] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x8C:
		        if (keyArray[k_esc] != key_evaluated)
		        {
				keyArray[k_esc] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_esc] = key_idle;
		        break;
		case 0x10:                                     /* space */
			keyArray[k_space] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x90:
		        if (keyArray[k_space] != key_evaluated)
		        {
				keyArray[k_space] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_space] = key_idle;
		        break;
		case 0x18:                                     /* space right */
			keyArray[k_rspace] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x98:
		        if (keyArray[k_rspace] != key_evaluated)
		        {
				keyArray[k_rspace] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_rspace] = key_idle;
		        break;

		/* 16er block */
		case 0x12:                                     /* B9 */
			keyArray[k_b9] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x92:
		        if (keyArray[k_b9] != key_evaluated)
		        {
				keyArray[k_b9] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b9] = key_idle;
		        break;
		case 0x02:                                     /* B10 */
			keyArray[k_b10] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x82:
		        if (keyArray[k_b10] != key_evaluated)
		        {
				keyArray[k_b10] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b10] = key_idle;
		        break;
		case 0x11:                                     /* B11 */
			keyArray[k_b11] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x91:
		        if (keyArray[k_b11] != key_evaluated)
		        {
				keyArray[k_b11] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b11] = key_idle;
		        break;
		case 0x01:                                     /* B12 */
			keyArray[k_b12] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x81:
		        if (keyArray[k_b12] != key_evaluated)
		        {
				keyArray[k_b12] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b12] = key_idle;
		        break;
		case 0x09:                                     /* B13 */
			keyArray[k_b13] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x89:
		        if (keyArray[k_b13] != key_evaluated)
		        {
				keyArray[k_b13] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b13] = key_idle;
		        break;
		case 0x0D:                                     /* B14 */
			keyArray[k_b14] = key_pressed;
             		refresh_evaluated_to_pressed();
                        fprintf(stderr, "\naahhh\n");
			break;
		case 0x8D:
		        if (keyArray[k_b14] != key_evaluated)
		        {
				keyArray[k_b14] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_b14] = key_idle;
		        break;
		case 0x05:                                     /* 0 */
			keyArray[k_0] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x85:
		        if (keyArray[k_0] != key_evaluated)
		        {
				keyArray[k_0] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_0] = key_idle;
		        break;
		case 0x15:                                     /* 1 */
			keyArray[k_1] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x95:
		        if (keyArray[k_1] != key_evaluated)
		        {
				keyArray[k_1] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_1] = key_idle;
		        break;
		case 0x19:                                     /* 2 */
			keyArray[k_2] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x99:
		        if (keyArray[k_2] != key_evaluated)
		        {
				keyArray[k_2] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_2] = key_idle;
		        break;
		case 0x1D:                                     /* 3 */
			keyArray[k_3] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x9D:
		        if (keyArray[k_3] != key_evaluated)
		        {
				keyArray[k_3] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_3] = key_idle;
		        break;
		case 0x06:                                     /* 4 */
			keyArray[k_4] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x86:
		        if (keyArray[k_4] != key_evaluated)
		        {
				keyArray[k_4] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_4] = key_idle;
		        break;
		case 0x0A:                                     /* 5 */
			/* keyArray[k_5] = key_pressed; Problem: read maps 0x0d -> 0x0a */
             		refresh_evaluated_to_pressed();
			break;
		case 0x8A:
		        if (keyArray[k_5] != key_evaluated)
		        {
				keyArray[k_5] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_5] = key_idle;
		        break;
		case 0x0E:                                     /* 6 */
			keyArray[k_6] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x8E:
		        if (keyArray[k_6] != key_evaluated)
		        {
				keyArray[k_6] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_6] = key_idle;
		        break;
		case 0x16:                                     /* 7 */
			keyArray[k_7] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x96:
		        if (keyArray[k_7] != key_evaluated)
		        {
				keyArray[k_7] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_7] = key_idle;
		        break;
		case 0x1A:                                     /* 8 */
			keyArray[k_8] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x9A:
		        if (keyArray[k_8] != key_evaluated)
		        {
				keyArray[k_8] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_8] = key_idle;
		        break;
		case 0x1E:                                     /* 9 */
			keyArray[k_9] = key_pressed;
             		refresh_evaluated_to_pressed();
			break;
		case 0x9E:
		        if (keyArray[k_9] != key_evaluated)
		        {
				keyArray[k_9] = key_relinquished;
				do_evaluation = 1;
			} 
		        else keyArray[k_9] = key_idle;
		        break;
			
		/* courser routing */
		default:
		{
			int j;
			for (j=0x20; j <(0x20+HandyDeviceData.CellCount); j++)
			{
				if (getbuffer[i+3] == j) 
				{
					keyArray[k_r01+(j-0x20)] = key_pressed;
					refresh_evaluated_to_pressed();
				}
				else if (getbuffer[i+3] == (j | 0x80))
				{
					if (keyArray[k_r01+(j-0x20)] != key_evaluated)
					{
						keyArray[k_r01+(j-0x20)] = key_relinquished;
						do_evaluation = 1;
					} 
					else keyArray[k_r01+(j-0x20)] = key_idle;	
				}
			}
		}
		}
	}
	return do_evaluation;
}


/* evaluate the pressed key-combination */
short handy_brl_input_parser (int NewVal)
{
 
	static BRAILLE_EVENT_CODE bec;
	static BRAILLE_EVENT_DATA bed;

	if (!ClientCallback) return 0; /* no client callback, no sense to go further */

	bec = bec_raw_byte;
	bed.RawByte = NewVal;
	ClientCallback(bec, &bed);

	/* clear getbuffer */
	if (gb_index == 0) memset(getbuffer, 0x00, MAX_GET_BUFFER);
	
	getbuffer[gb_index++] = NewVal;

	if ((gb_index == 1) && (NewVal != 0x79)) gb_index = 0; /* if not framed -> ignore */

	if (is_complete_frame())
	{
		if (getbuffer[3] == 0x04)   /* key pressed */
		{
			if (OnKeyChanged())
			{
				int i;
				int index = 0;
				char keyCode[MAX_KEYCODE_LENGHT];
				
				/* memset(keyCode, 0, MAX_KEYCODE_LENGHT); */
				for (i=0; i < MAX_KEYCODE_LENGHT; i++)
					keyCode[i] = '\0';
				
				for (i=0; i < MAX_KEYS; i++) /* new state of keys*/
				{
					/* prevent overflowe */
					if (index > (MAX_KEYCODE_LENGHT-6)) index = MAX_KEYCODE_LENGHT-6;

					/* relinquish the keys */
					if (keyArray[i] != key_idle)
					{
						switch (i)
						{
						case k_up:
							index += sprintf(&keyCode[index], "Up");
							break;	
						case k_down:
							index += sprintf(&keyCode[index], "Down");
							break;
						case k_enter:
							index += sprintf(&keyCode[index], "Enter");
							break;
						case k_esc:
							index += sprintf(&keyCode[index], "Esc");
							break;	
						case k_space:
							index += sprintf(&keyCode[index], "Space");
							break;
						case k_rspace:
							index += sprintf(&keyCode[index], "rSpace");
							break;
						default:
						{
							int j;

							/* braille-keys DK01 - DK08 */
							if ((i >= k_f1) && (i <= k_f8))   
								index += sprintf(&keyCode[index], "DK%02d", i-k_f1+1);

							/* 16er Block Bxx & NUMx*/
							if ((i >= k_b9) && (i <= k_b14))
								index += sprintf(&keyCode[index], "B%02d", i-k_b9+9);
							if ((i >= k_0) && (i <= k_9))
								index += sprintf(&keyCode[index], "NUM%d", i-k_0);

							/* courser routing over textmodules HSMxx*/
							 for (j=k_r01; j <= k_r80; j++)
							{
								if (i == j) index += sprintf(&keyCode[index], "HMS%02d", j-k_r01);
							}

							/* courser routing over statusmodules Sx*/
							for (j=k_s1; j <= k_s4; j++)
							{
								if (i == j) index += sprintf(&keyCode[index], "S%1d", j-k_s1);
							}
						}
						}
					}

					/* set new state of keys */
					if (keyArray[i] == key_relinquished) keyArray[i] = key_idle;  
				       	if (keyArray[i] == key_pressed) keyArray[i] = key_evaluated;
				}

				/* adjust key-commands for Braille Star and Braillino */
				if (HandyDeviceData.Name == BL2 ||
				    HandyDeviceData.Name == BS4 || 
				    HandyDeviceData.Name == BS8)
				{
					if (!strcmp(keyCode, "Enter")) sprintf(keyCode, "Down");
					else if (!strcmp(keyCode, "Esc")) sprintf(keyCode, "Up");
					else if (!strcmp(keyCode, "EnterEsc")) sprintf(keyCode, "Esc");
					else if (!strcmp(keyCode, "UpDown")) sprintf(keyCode, "Enter");
				}

				if (!strncmp(keyCode, "HMS", 3))
				{
					bec = bec_sensor;                       /* curser routing */
					bed.Sensor.SensorCodes = keyCode;
					ClientCallback(bec, &bed);
				} else
				{
					bec = bec_key_codes;                        /* other braille key */
					bed.KeyCodes = keyCode;
					ClientCallback(bec, &bed);
				}

				fprintf(stderr, "-- %s --\n", keyCode);

			}
		}
		gb_index = 0; /* begin new frame */
	}

	return 0; /* 0-give next byte, 1-repeat last byte */
}


/* sends line to braille device */
int handy_brl_send_dots (unsigned char *dots, short count, short blocking)
{
	int rv = 0;
	unsigned char sendbuffer[MAX_BUFFER];

	/* clear sendbuffer */
       	memset(sendbuffer, 0x00, MAX_BUFFER);

	/* composing buffer to send */
	sendbuffer[0] = 0x79;
	sendbuffer[1] = HandyDeviceData.IDByte;
	sendbuffer[2] = HandyDeviceData.CellCount+1;
	sendbuffer[3] = 0x01;
	memcpy (&sendbuffer[4], dots, count);
	sendbuffer[HandyDeviceData.CellCount+4] = 0x16;
	
	/* sending buffer to display */
	rv = brl_ser_send_data (sendbuffer, HandyDeviceData.CellCount+5, blocking);

	return rv;
}


/* close braille driver*/
void handy_brl_close_device ()
{
	/* close seial communication */
	brl_ser_set_callback (NULL);
	brl_ser_stop_timer ();
	brl_ser_close_port ();
}


/* open braille driver */
int handy_brl_open_device (char* DeviceName, short Port, BRL_DEV_CALLBACK DeviceCallback, BRL_DEVICE *Device) 
{
	int rv = 0;

	/* Braille Wave */
	if (strcmp ("HTBRW", DeviceName) == 0)
	{
		Device->CellCount = 40;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 13;

		Device->DisplayCount = 1;

		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;

		HandyDeviceData.Name = BRW;
		HandyDeviceData.CellCount = 40;
		HandyDeviceData.IDByte = 0x05;

		rv = 1;
	}

        /* Braillino */
	else if (strcmp ("HTBL2", DeviceName) == 0)
	{
		Device->CellCount = 20;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 14;

		Device->DisplayCount = 1;

		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 20;
		Device->Displays[0].Type = bdt_main;

		HandyDeviceData.Name = BL2;
		HandyDeviceData.CellCount = 20;
		HandyDeviceData.IDByte = 0x72;

		rv = 1;
	}

        /* Braille Star 40 */
	else if (strcmp ("HTBS4", DeviceName) == 0)
	{
		Device->CellCount = 40;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 14;

		Device->DisplayCount = 1;

		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;

		HandyDeviceData.Name = BS4;
		HandyDeviceData.CellCount = 40;
		HandyDeviceData.IDByte = 0x74;		    

		rv = 1;
	}

	/* Braille Star 80 */
	else if (strcmp ("HTBS8", DeviceName) == 0)
	{
		Device->CellCount = 80;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 30;

		Device->DisplayCount = 1;

		Device->Displays[0].StartCell = 0;
		Device->Displays[0].Width = 80;
		Device->Displays[0].Type = bdt_main;
		
		HandyDeviceData.Name = BS8;
		HandyDeviceData.CellCount = 80;
		HandyDeviceData.IDByte = 0x78;
		
		rv = 1;
	}

	/* Modular 44 */
	else if (strcmp ("HTMB4", DeviceName) == 0)
	{
		Device->CellCount = 44;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 26;

		Device->DisplayCount = 2;

		Device->Displays[0].StartCell = 4;
		Device->Displays[0].Width = 40;
		Device->Displays[0].Type = bdt_main;
		
		Device->Displays[1].StartCell = 0;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;

		HandyDeviceData.Name = MB4;
		HandyDeviceData.CellCount = 44;
		HandyDeviceData.IDByte = 0x89;

		rv = 1;
	}

	/* Modular 84 */
	else if (strcmp ("HTMB8", DeviceName) == 0)
	{
		Device->CellCount = 84;
		Device->InputType = bit_make_break_codes;
		Device->KeyCount = 26;

		Device->DisplayCount = 2;

		Device->Displays[0].StartCell = 4;
		Device->Displays[0].Width = 80;
		Device->Displays[0].Type = bdt_main;
		
		Device->Displays[1].StartCell = 0;
		Device->Displays[1].Width = 4;
		Device->Displays[1].Type = bdt_status;

		HandyDeviceData.Name = MB8;
		HandyDeviceData.CellCount = 84;
		HandyDeviceData.IDByte = 0x88;

		rv = 1;
	}

	else
	{
		/* Error: unknown device */
		rv = 0;
	}

	if (rv) /* device found */
	{
		/* fill device function with callback-functions */
		Device->send_dots = handy_brl_send_dots;
		Device->close_device = handy_brl_close_device;

		/* open serial communication */
		if (brl_ser_open_port (Port))
		{
			/* set serial parameter */
			rv = handy_set_comm_param ();

			brl_ser_set_callback (handy_brl_input_parser);

			/* reset braille-device */
			rv = brl_ser_send_data ("\xFF", 1, 0);

			/* set all keys to key_idle */
			

			brl_ser_init_glib_poll ();
			ClientCallback = DeviceCallback;
		}
		else 
		{
			/* Error: Cannot open port */
			rv = 0;
		}
	}

	return rv;
}
