// -*- C++ -*-

/* 
 * pieplotstate.h 
 *
 * Copyright (C) 1999 Frank Koormann, Bernhard Reiter & Jan-Oliver Wagner
 *
 * Developed by Frank Koormann <fkoorman@usf.uos.de>,
 * Bernhard Reiter <breiter@usf.uos.de> and
 * Jan-Oliver Wagner <jwagner@usf.uos.de>.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef GUPPI_PIEPLOTSTATE_H
#define GUPPI_PIEPLOTSTATE_H

#include "data.h"
#include "datastore.h"
#include "plotutils.h"
#include "plotstate.h"

class PiePlotState;

class Slice {
public:

  Slice(PiePlotState * s) :
    state_(s), index_(0), posX_(50.0), posY_(50.0),
    radius_(20.0), width_(10.0), start_(45.0), color_(250,0,0)
    { }

  ~Slice()
    { }

  // Position of slice center 
  double posX() const { return posX_; }
  double posY() const { return posY_; }

  double start() const { return start_; }
  double width() const { return width_; }
  double radius() const { return radius_; }

  const Color & color() const { return color_; }

  // Setting should only be done by PiePlotState
  void set_color(const Color & c) { color_ = c; }
  void set_index(guint i) { index_ = i; }
  void set_posX(double x) { posX_ = x; }
  void set_posY(double y) { posY_ = y; }
  void set_width(double w) { width_ = w; }
  void set_start(double s) { start_ = s; }
  void set_radius(double r) { radius_ = r; }


private:

  PiePlotState * state_;

  guint index_;
  double posX_, posY_;
  double radius_;
  double width_;	// angle
  double start_;	// angle
  Color color_;
};

class PiePlotState : public PlotState, public Data::View {
public:

  PiePlotState(DataStore * ds);
  virtual ~PiePlotState();

  void set_data(Data * d);
  ScalarData* data();
  
  DataStore* store() { return store_; }

  typedef vector<Slice>::iterator iterator;
  typedef vector<Slice>::const_iterator const_iterator;

  iterator begin() { return slices_.begin(); }
  iterator end() { return slices_.end(); }
  const_iterator begin() const { return slices_.begin(); }
  const_iterator end() const { return slices_.end(); }

  gsize size() { return slices_.size(); }

  const Color & background() const { return background_; }

  void set_background(const Color & c) { 
    background_ = c; 
    state_model.background_changed(this, background_); 
  }

  const string& name() const;
  void set_name(const string & name);

  // Whether to automatically set the name when the data changes
  bool name_from_data() const;
  void set_name_from_data(bool setting);

  // Width/Height are in display coords (e.g. canvas item coords)
  double width() const { return width_; }
  double height() const { return height_; }

  void set_width(double w); 
  void set_height(double h);

  void size_request(double* w, double* h);

  const Rectangle & plot_rect() const;

  virtual void change_values(Data* d) {
    g_return_if_fail((ScalarData*)d == data_);
    rebuild_slices();
  }

  virtual void change_values(Data* d, const vector<guint> & which) {
    change_values(d);
  }

  virtual void change_name(Data* d, const string & name) {
    g_return_if_fail((ScalarData*)d == data_);
  }

  // This is a broken way to determine the data is gone; 
  //  we need to use remove_data from the datastore. FIXME
  virtual void destroy_model() {
    data_ = 0; // so we don't reference it in set_data()
    rebuild_slices();
  }

  class View : public ::View {
  public:

    virtual void change_slices(PiePlotState* state) = 0;
    virtual void change_data(PiePlotState* state, Data* data) = 0;
    virtual void change_background(PiePlotState* state, const Color & bg) = 0;
    virtual void change_size(PiePlotState* state, double width, double height) = 0;
    virtual void change_name(PiePlotState* state, const string& name) = 0;
  };

  class Model : public ::Model<PiePlotState::View*> {
  public:

    void slices_changed(PiePlotState* state) {
      lock_views();
      iterator i = begin();
      while (i != end()) {
        (*i)->change_slices(state);
        ++i;
      }
      unlock_views();
    }

    void data_changed(PiePlotState* state, Data* d) {
      lock_views();
      iterator i = begin();
      while (i != end()) {
        (*i)->change_data(state,d);
        ++i;
      }
      unlock_views();
    }

    void background_changed(PiePlotState* state, const Color & bg) {
      lock_views();
      iterator i = begin();
      while (i != end()) {
        (*i)->change_background(state,bg);
        ++i;
      }
      unlock_views();
    }

    void size_changed(PiePlotState* state, double width, double height) {
      lock_views();
      iterator i = begin();
      while (i != end()) {
        (*i)->change_size(state,width,height);
        ++i;
      }
      unlock_views();
    }

    void name_changed(PiePlotState* state, const string & name) {
      lock_views();
      iterator i = begin();
      while (i != end()) {
        (*i)->change_name(state,name);
        ++i;
      }
      unlock_views();
    }
  };

  Model state_model;  

private:

  ScalarData * data_;
  DataStore * store_;
  vector<Slice> slices_;

  double width_;
  double height_;

  Color background_;

  Rectangle plot_rect_;

  string name_;

  bool set_name_from_data_;

  void rebuild_slices();
  void grab_data(Data* d);
  void release_data();
  void recalc_layout();
  void update_name_from_data();
};

#endif
