/* testhashset.vala
 *
 * Copyright (C) 2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>




void test_hashset_add (void);
void test_hashset_clear (void);
void test_hashset_contains (void);
void test_hashset_remove (void);
void test_hashset_size (void);
void test_hashset_iterator (void);
static void _test_hashset_add_gcallback (void);
static void _test_hashset_clear_gcallback (void);
static void _test_hashset_contains_gcallback (void);
static void _test_hashset_remove_gcallback (void);
static void _test_hashset_size_gcallback (void);
static void _test_hashset_iterator_gcallback (void);
void _main (char** args, int args_length1);



void test_hashset_add (void) {
	GeeHashSet* hashset;
	GeeHashSet* hashsetInt;
	GeeHashSet* hashsetObject;
	GObject* fooObject;
	GObject* fooObject2;
	/* Check adding of strings*/
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	gee_collection_add ((GeeCollection*) hashset, "42");
	g_assert (gee_collection_contains ((GeeCollection*) hashset, "42"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	gee_collection_add ((GeeCollection*) hashset, "43");
	g_assert (gee_collection_contains ((GeeCollection*) hashset, "42"));
	g_assert (gee_collection_contains ((GeeCollection*) hashset, "43"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	gee_collection_add ((GeeCollection*) hashset, "43");
	g_assert (gee_collection_contains ((GeeCollection*) hashset, "42"));
	g_assert (gee_collection_contains ((GeeCollection*) hashset, "43"));
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check adding of ints*/
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 1);
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	/* Check adding of objects*/
	hashsetObject = gee_hash_set_new (G_TYPE_OBJECT, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_hash, g_direct_equal);
	fooObject = g_object_new (G_TYPE_OBJECT, NULL);
	gee_collection_add ((GeeCollection*) hashsetObject, fooObject);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetObject, fooObject));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 1);
	fooObject2 = g_object_new (G_TYPE_OBJECT, NULL);
	gee_collection_add ((GeeCollection*) hashsetObject, fooObject2);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetObject, fooObject));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetObject, fooObject2));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	/* Check add same element*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	gee_collection_add ((GeeCollection*) hashsetObject, fooObject2);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetObject, fooObject));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetObject, fooObject2));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetObject) == 2);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
	(hashsetObject == NULL) ? NULL : (hashsetObject = (g_object_unref (hashsetObject), NULL));
	(fooObject == NULL) ? NULL : (fooObject = (g_object_unref (fooObject), NULL));
	(fooObject2 == NULL) ? NULL : (fooObject2 = (g_object_unref (fooObject2), NULL));
}


void test_hashset_clear (void) {
	GeeHashSet* hashset;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear on empty set*/
	gee_collection_clear ((GeeCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear one item*/
	gee_collection_add ((GeeCollection*) hashset, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	gee_collection_clear ((GeeCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check clear multiple items*/
	gee_collection_add ((GeeCollection*) hashset, "1");
	gee_collection_add ((GeeCollection*) hashset, "2");
	gee_collection_add ((GeeCollection*) hashset, "3");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 3);
	gee_collection_clear ((GeeCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
}


void test_hashset_contains (void) {
	GeeHashSet* hashsetString;
	GeeHashSet* hashsetInt;
	hashsetString = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check on empty set*/
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "1"));
	/* Check items*/
	gee_collection_add ((GeeCollection*) hashsetString, "10");
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "10"));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "30"));
	gee_collection_add ((GeeCollection*) hashsetString, "20");
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "10"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "30"));
	gee_collection_add ((GeeCollection*) hashsetString, "30");
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "10"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "20"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "30"));
	/* Clear and recheck*/
	gee_collection_clear ((GeeCollection*) hashsetString);
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "10"));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "20"));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetString, "30"));
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* Check items*/
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (10));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (30)));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (20));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (30)));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (30));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (30)));
	/* Clear and recheck*/
	gee_collection_clear ((GeeCollection*) hashsetInt);
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (10)));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (20)));
	g_assert (!gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (30)));
	(hashsetString == NULL) ? NULL : (hashsetString = (g_object_unref (hashsetString), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
}


void test_hashset_remove (void) {
	GeeHashSet* hashsetString;
	GeeHashSet* hashsetInt;
	hashsetString = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check remove if list is empty*/
	gee_collection_remove ((GeeCollection*) hashsetString, "42");
	/* Add 5 different elements*/
	gee_collection_add ((GeeCollection*) hashsetString, "42");
	gee_collection_add ((GeeCollection*) hashsetString, "43");
	gee_collection_add ((GeeCollection*) hashsetString, "44");
	gee_collection_add ((GeeCollection*) hashsetString, "45");
	gee_collection_add ((GeeCollection*) hashsetString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 5);
	/* Check remove first*/
	gee_collection_remove ((GeeCollection*) hashsetString, "42");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 4);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "43"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "44"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "45"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "46"));
	/* Check remove last*/
	gee_collection_remove ((GeeCollection*) hashsetString, "46");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 3);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "43"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "44"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "45"));
	/* Check remove in between*/
	gee_collection_remove ((GeeCollection*) hashsetString, "44");
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetString) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "43"));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetString, "45"));
	/* Check removing of int element*/
	hashsetInt = gee_hash_set_new (G_TYPE_INT, NULL, NULL, g_direct_hash, g_direct_equal);
	/* Add 5 different elements*/
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (44));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (45));
	gee_collection_add ((GeeCollection*) hashsetInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 5);
	/* Remove first*/
	gee_collection_remove ((GeeCollection*) hashsetInt, GINT_TO_POINTER (42));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 4);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (44)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (45)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (46)));
	/* Remove last*/
	gee_collection_remove ((GeeCollection*) hashsetInt, GINT_TO_POINTER (46));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 3);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (44)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (45)));
	/* Remove in between*/
	gee_collection_remove ((GeeCollection*) hashsetInt, GINT_TO_POINTER (44));
	g_assert (gee_collection_get_size ((GeeCollection*) hashsetInt) == 2);
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (43)));
	g_assert (gee_collection_contains ((GeeCollection*) hashsetInt, GINT_TO_POINTER (45)));
	(hashsetString == NULL) ? NULL : (hashsetString = (g_object_unref (hashsetString), NULL));
	(hashsetInt == NULL) ? NULL : (hashsetInt = (g_object_unref (hashsetInt), NULL));
}


void test_hashset_size (void) {
	GeeHashSet* hashset;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check empty list*/
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	/* Check when one item*/
	gee_collection_add ((GeeCollection*) hashset, "1");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 1);
	/* Check when more items*/
	gee_collection_add ((GeeCollection*) hashset, "2");
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 2);
	/* Check when items cleared*/
	gee_collection_clear ((GeeCollection*) hashset);
	g_assert (gee_collection_get_size ((GeeCollection*) hashset) == 0);
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
}


void test_hashset_iterator (void) {
	GeeHashSet* hashset;
	GeeIterator* iterator;
	GeeIterator* _tmp0_;
	char* firstString;
	char* secondString;
	hashset = gee_hash_set_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal);
	/* Check iterate empty list*/
	iterator = gee_iterable_iterator ((GeeIterable*) hashset);
	g_assert (!gee_iterator_next (iterator));
	/* Check iterate list*/
	gee_collection_add ((GeeCollection*) hashset, "42");
	gee_collection_add ((GeeCollection*) hashset, "43");
	_tmp0_ = NULL;
	iterator = (_tmp0_ = gee_iterable_iterator ((GeeIterable*) hashset), (iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL)), _tmp0_);
	/* A set is usually not ordered, so this is not a requirement */
	g_assert (gee_iterator_next (iterator));
	firstString = (char*) gee_iterator_get (iterator);
	g_assert (gee_collection_contains ((GeeCollection*) hashset, firstString));
	g_assert (gee_iterator_next (iterator));
	secondString = (char*) gee_iterator_get (iterator);
	g_assert (gee_collection_contains ((GeeCollection*) hashset, secondString));
	g_assert (!g_str_equal (firstString, secondString));
	/* they can not be identical neither equal*/
	g_assert (!gee_iterator_next (iterator));
	(hashset == NULL) ? NULL : (hashset = (g_object_unref (hashset), NULL));
	(iterator == NULL) ? NULL : (iterator = (g_object_unref (iterator), NULL));
	firstString = (g_free (firstString), NULL);
	secondString = (g_free (secondString), NULL);
}


static void _test_hashset_add_gcallback (void) {
	test_hashset_add ();
}


static void _test_hashset_clear_gcallback (void) {
	test_hashset_clear ();
}


static void _test_hashset_contains_gcallback (void) {
	test_hashset_contains ();
}


static void _test_hashset_remove_gcallback (void) {
	test_hashset_remove ();
}


static void _test_hashset_size_gcallback (void) {
	test_hashset_size ();
}


static void _test_hashset_iterator_gcallback (void) {
	test_hashset_iterator ();
}


void _main (char** args, int args_length1) {
	g_test_init (&args_length1, &args, NULL);
	g_test_add_func ("/HashSet/Collection/add", _test_hashset_add_gcallback);
	g_test_add_func ("/HashSet/Collection/clear", _test_hashset_clear_gcallback);
	g_test_add_func ("/HashSet/Collection/contains", _test_hashset_contains_gcallback);
	g_test_add_func ("/HashSet/Collection/remove", _test_hashset_remove_gcallback);
	g_test_add_func ("/HashSet/Collection/size", _test_hashset_size_gcallback);
	g_test_add_func ("/HashSet/Iterable/iterator", _test_hashset_iterator_gcallback);
	g_test_run ();
}


int main (int argc, char ** argv) {
	g_type_init ();
	_main (argv, argc);
	return 0;
}




