

/* traversable.vala
 *
 * Copyright (C) 2011  Maciej Piechotka
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Maciej Piechotka <uzytkownik2@gmail.com>
 */



namespace Gee {
	public delegate A FoldFunc<A, G> (owned G g, owned A a);
	public delegate void ForallFunc<G> (owned G g);
	public delegate Lazy<A>? UnfoldFunc<A> ();
	public delegate Traversable.Stream StreamFunc<G, A> (Traversable.Stream state, owned Lazy<G>? g, out Lazy<A>? lazy);
	public delegate A MapFunc<A, G> (owned G g);
	public delegate bool Predicate<G> (G g);
}

/**
 * It's a common interface for {@link Iterator} and {@link Iterable}. It
 * provides a fast, high level functions.
 *
 * ''{@link Iterator} implementation:'' Please note that most of the functions
 * affect the state of the iterator by moving it forward.
 * Even if the iterator is {@link BidirIterator bidirectional} it ''must not''
 * rewind the state.
 *
 * ''{@link Iterable} implementation:'' {@link Iterator.valid validity}
 * of returned iterator is the same as for {@link Iterator.valid invalid}
 * iterator. In other words the following code is semantically equivalent:
 *
 * {{
 *     var x = iterable.function (args);
 *     var x = iterable.iterator ().function(args);
 * }}
 *
 * @since 0.7.0
 */
public interface Gee.Traversable<G> : Object {
	/**
	 * Apply function to each element returned by iterator. 
	 *
	 * ''{@link Iterator} implementation:'' Operation moves the iterator
	 * to last element in iteration. If iterator points at some element it
	 * will be included in iteration.
	 */
	public new abstract void foreach (ForallFunc<G> f);

	/**
	 * Stream function is an abstract function allowing writing many
	 * operations.
	 *
	 * The stream function accepts three parameter:
	 *
	 *   1. state. It is usually the last returned value from function but
	 *      it may be {@link Stream.END} when {@link Stream.CONTINUE} was
	 *      returned and there was no more elements.
	 *   2. input. It is valid only if first argument is
	 *      {@link Stream.CONTINUE}
	 *   3. output. It is valid only if result is Stream.YIELD
	 *
	 * It may return one of 3 results:
	 *
	 *   1. {@link Stream.YIELD}. It means that value was yielded and can
	 *      be passed to outgoing iterator.
	 *   2. {@link Stream.CONTINUE}. It means that the function needs to be
	 *      called with next element or with {@link Stream.END} if it is
	 *      end of stream). If the state element was Stream.END during the
	 *      current iteration function ''must not'' return {@link Stream.CONTINUE}
	 *   3. Stream.END. It means that the last argument was yielded.
	 *
	 * If the function yields the value immediately then the returning iterator
	 * is {@link valid} and points to this value as well as in case when the
	 * parent iterator is {@link Iterator.valid valid} and function yields
	 * after consuming 1 input. In other case returned iterator is invalid.
	 *
	 * ''{@link Iterator} implementation:'' If iterator is
	 * {@link Iterator.valid valid} the current value should be fed
	 * immediately to function if during initial call function returns
	 * {@link Stream.CONTINUE}. The parent iterator cannot be used before
	 * the functions return {@link Stream.END} afterwards it points on the
	 * last element consumed.
	 *
	 * @param f function generating stream
	 * @return iterator containing values yielded by stream
	 */
	public abstract Iterator<A> stream<A> (owned StreamFunc<G, A> f);

	/**
	 * Standard aggregation function.
	 *
	 * It takes a function, seed and first element, returns the new seed and
	 * progress to next element when the operation repeats.
	 *
	 * ''Note:'' Default implementation uses {@link foreach}.
	 *
	 * ''{@link Iterator} implementation:'' Operation moves the iterator to
	 * last element in iteration. If iterator is
	 * {@link Iterator.valid valid} the current element will be considered
	 * as well.
	 *
	 */
	public virtual A fold<A> (FoldFunc<A, G> f, owned A seed)
	{
		this.foreach ((item) => {seed = f ((owned) item, (owned) seed);});
		return (owned) seed;
	}

	/**
	 * Produces an iterator pointing at elements generated by function passed.
	 *
	 * Iterator is lazy evaluated but value is force-evaluated when
	 * iterator {@link Iterator.next moves to next element}.
	 *
	 * ''Note:'' Default implementation uses {@link stream}.
	 *
	 * ''{@link Iterator} implementation:'' If the parent iterator is
	 * {@link Iterator.valid valid} so is the returned one. Using the parent
	 * iterator is not allowed before the inner iterator {@link Iterator.next
	 * next} return false and then it points on its last element.
	 *
	 * @param f Mapping function
	 * @return Iterator listing mapped value
	 */
	public virtual Iterator<A> map<A> (MapFunc<A, G> f) {
		return stream<A>((state, item, out val) => {
			switch (state) {
			case Stream.YIELD:
				val = null;
				return Stream.CONTINUE;
			case Stream.CONTINUE:
				val = new Lazy<A>(() => {
					A tmp = item.get ();
					item = null;
					return (f ((owned)tmp));
				});
				return Stream.YIELD;
			case Stream.END:
				val = null;
				return Stream.END;
			default:
				assert_not_reached ();
			}
		});
	}

	/**
	 * Creates a new iterator that is initially pointing to seed. Then
	 * subsequent values are obtained after applying the function to previous
	 * value and the subsequent items.
	 *
	 * The resulting iterator is always valid and it contains the seed value.
	 *
	 * ''Note:'' Default implementation uses {@link stream}.
	 *
	 * ''{@link Iterator} implementation:'' Using the parent
	 * iterator is not allowed befor the inner iterator {@link Iterator.next
	 * next} return false and then it points on its last element.
	 *
	 * @param f Folding function
	 * @param seed original seed value
	 * @return Iterator containing values of subsequent values of seed
	 */
	public virtual Iterator<A> scan<A> (FoldFunc<A, G> f, owned A seed) {
		bool seed_emitted = false;
		return stream<A>((state, item, out val) => {
			switch (state) {
			case Stream.YIELD:
				if (seed_emitted) {
					val = null;
					return Stream.CONTINUE;
				} else {
					val = new Lazy<A>.from_value (seed);
					seed_emitted = true;
					return Stream.YIELD;
				}
			case Stream.CONTINUE:
				val = new Lazy<A> (() => {
					A tmp = item.get ();
					item = null;
					seed = f ((owned) tmp, (owned) seed);
					return seed;
				});
				return Stream.YIELD;
			case Stream.END:
				val = null;
				return Stream.END;
			default:
				assert_not_reached ();
			}
		});
	}

	/**
	 * Creates a new iterator that contains only values that fullfills the
	 * predicate.
	 *
	 * ''Note:'' There is implementation {@link filter_impl}.
	 *
	 * ''{@link Iterator} implementation:'' Resulting iterator is valid. Using the parent
	 * iterator is not allowed before the inner iterator {@link Iterator.next
	 * next} return false and then it points on its last element.
	 *
	 * @param f Folding function
	 * @param seed original seed value
	 * @return Iterator containing values of subsequent values of seed
	 */
	public abstract Iterator<G> filter (owned Predicate<G> f);

	/**
	 * Creates a new iterator which contains elements from iterable. The
	 * first argument states the offset i.e. number of elements the iterator
	 * skips by default.
	 *
	 * ''Note:'' There is implementation {@link chop_impl}.
	 *
	 * ''{@link Iterator} implementation:'' Resulting iterator is valid when
	 * parent iterator is valid and the offset is 0. Using the parent
	 * iterator is not allowed before the inner iterator {@link Iterator.next
	 * next} return false and then it points on its last element.
	 *
	 * ''{@link Iterable} implementation:'' Resulting iterator is invalid.
	 *
	 * @param offset the offset to first element the iterator is pointing to
	 * @param length maximum number of elements iterator may return. Negative
	 *        value means that the number is unbounded
	 */
	public abstract Iterator<G> chop (int offset, int length = -1);


	/**
	 * Implementation based on {@link stream} for {@link filter}.
	 *
	 * @param input The current Traversable
	 * @param pred Predicate
	 * @returns Filtered iterator
	 * @see filter
	 * @see stream
	 */
	public static Iterator<G> filter_impl<G> (Traversable<G> input, owned Predicate<G> pred) {
		return input.stream<G> ((state, item, out val) => {
			switch (state) {
			case Stream.YIELD:
				val = null;
				return Stream.CONTINUE;
			case Stream.CONTINUE:
				G g = item.get ();
				if (pred (g)) {
					val = item;
					return Stream.YIELD;
				} else {
					val = null;
					return Stream.CONTINUE;
				}
			case Stream.END:
				val = null;
				return Stream.END;
			default:
				assert_not_reached ();
			};
		});
	}

	/**
	 * Implementation based on {@link stream} for {@link filter}.
	 *
	 * @param input The current Traversable
	 * @param offset The offset
	 * @param length The length
	 */
	public static Iterator<G> chop_impl<G> (Traversable<G> input, int offset, int length) {
		assert (offset >= 0);
		return input.stream<G> ((state, item, out val) => {
			switch (state) {
			case Stream.YIELD:
				val = null;
				if (offset > 0) {
					return Stream.CONTINUE;
				} else if (length > 0) {
					length--;
					return length != 0 ? Stream.CONTINUE : Stream.END;
				} else if (length == 0) {
					return Stream.END;
				} else {
					return Stream.CONTINUE;
				}
			case Stream.CONTINUE:
				if (offset == 0) {
					val = item;
					return Stream.YIELD;
				} else {
					val = null;
					offset--;
					return Stream.CONTINUE;
				}
			case Stream.END:
				val = null;
				return Stream.END;
			default:
				assert_not_reached ();
			};
		});
	}

	public enum Stream {
		YIELD,
		CONTINUE,
		END
	}

}

