/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gpa-printer-selector.c: A simple Optionmenu for selecting printers
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors :
 *    Lauris Kaplinski <lauris@ximian.com>
 *
 *  Copyright (C) 2000-2001 Ximian, Inc. and Jose M. Celorio
 *
 */

#include <config.h>

#include <string.h>
#include <gtk/gtk.h>

#include "gnome-print-i18n.h"
#include "gpa-printer-selector.h"

static void gpa_printer_selector_class_init (GPAPrinterSelectorClass *klass);
static void gpa_printer_selector_init (GPAPrinterSelector *selector);
static void gpa_printer_selector_finalize (GObject *object);
static gint gpa_printer_selector_construct (GPAWidget *widget);

static void gpa_printer_selector_rebuild_menu (GPAPrinterSelector *ps);

static GPAWidgetClass *parent_class;

GtkType
gpa_printer_selector_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GPAPrinterSelectorClass),
			NULL, NULL,
			(GClassInitFunc) gpa_printer_selector_class_init,
			NULL, NULL,
			sizeof (GPAPrinterSelector),
			0,
			(GInstanceInitFunc) gpa_printer_selector_init
		};
		type = g_type_register_static (GPA_TYPE_WIDGET, "GPAPrinterSelector", &info, 0);
	}
	return type;
}

static void
gpa_printer_selector_class_init (GPAPrinterSelectorClass *klass)
{
	GObjectClass *object_class;
	GPAWidgetClass *gpa_class;

	object_class = (GObjectClass *) klass;
	gpa_class = (GPAWidgetClass *) klass;

	parent_class = gtk_type_class (GPA_TYPE_WIDGET);

	object_class->finalize = gpa_printer_selector_finalize;

	gpa_class->construct = gpa_printer_selector_construct;
}

static void
gpa_printer_selector_init (GPAPrinterSelector *selector)
{
	selector->menu = gtk_option_menu_new ();
	gtk_container_add (GTK_CONTAINER (selector), selector->menu);
	gtk_widget_show (selector->menu);
	selector->printers = NULL;
	selector->printerlist = NULL;
	selector->updating = FALSE;
	selector->printer_handler = 0;
}

static void
gpa_printer_selector_finalize (GObject *object)
{
	GPAPrinterSelector *ps;

	ps = (GPAPrinterSelector *) object;

	while (ps->printerlist) {
		gpa_node_unref (GPA_NODE (ps->printerlist->data));
		ps->printerlist = g_slist_remove (ps->printerlist, ps->printerlist->data);
	}

	if (ps->printers) {
		gpa_node_unref (ps->printers);
		ps->printers = NULL;
	}

	if (ps->printer_handler) {
		g_signal_handler_disconnect (ps->printer_node, ps->printer_handler);
		gpa_node_unref (ps->printer_node);
		ps->printer_handler = 0;
		ps->printer_node = NULL;		
	};
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gint
gpa_printer_selector_node_modified_cb (GPANode *node, guint flags, GPAPrinterSelector *ps)
{
	if (ps->updating) {
		/* This is not working because gpa-node emits the modified signal on a g_idle
		 * I still want this code here in case something changes in the future (Chema)
		 */
		return TRUE;
	}

	ps->updating = TRUE;
	gpa_printer_selector_rebuild_menu (ps);
	ps->updating = FALSE;

	return TRUE;
}

static gint
gpa_printer_selector_construct (GPAWidget *widget)
{
	GPAPrinterSelector *selector;
	GPANode *node;

	selector = GPA_PRINTER_SELECTOR (widget);
	node = GNOME_PRINT_CONFIG_NODE (widget->config);

	selector->printers = gpa_node_get_child_from_path (node, "Globals.Printers");
	g_return_val_if_fail (selector->printers != NULL, FALSE);

	selector->printer_node = gpa_node_get_child_from_path (node, "Printer");
	selector->printer_handler = g_signal_connect (G_OBJECT (selector->printer_node), "modified",
						      (GCallback) gpa_printer_selector_node_modified_cb, selector);
	
	gpa_printer_selector_rebuild_menu (selector);

	return TRUE;
}

static void
gpa_printer_selector_menuitem_activate (GtkWidget *widget, gint index)
{
	GPAWidget *gpaw;
	GPANode *printer;
	gchar *value;

	gpaw = g_object_get_data (G_OBJECT (widget), "GPAWidget");
	g_return_if_fail (gpaw != NULL);
	g_return_if_fail (GPA_IS_WIDGET (gpaw));

	if (((GPAPrinterSelector *)gpaw)->updating) {
		/* This is not working because gpa-node emits the modified signal on a g_idle
		 * I still want this code here in case something changes in the future (Chema)
		 */
		return;
	}

	printer = g_slist_nth_data (GPA_PRINTER_SELECTOR (gpaw)->printerlist, index);
	g_return_if_fail (printer != NULL);
	g_return_if_fail (GPA_IS_NODE (printer));

	value = gpa_node_get_value (printer);
	g_return_if_fail (value != NULL);

	gnome_print_config_set (gpaw->config, "Printer", value);

	g_free (value);
}

static void
gpa_printer_selector_add_printer_activate (GtkWidget *widget, GPAWidget *gpaw)
{
	g_print ("Add printer activated\n");
}

static void
gpa_printer_selector_manage_printers_activate (GtkWidget *widget, GPAWidget *gpaw)
{
	g_print ("Manage printers activated\n");
}

static void
gpa_printer_selector_rebuild_menu (GPAPrinterSelector *ps)
{
	GPANode *printer;
	GtkWidget *menu, *item;
	GSList *l;
	guchar *defid;
	gint idx, def;

	g_return_if_fail (ps->printers != NULL);

	/* selector->printerlist */
	while (ps->printerlist) {
		printer = GPA_NODE (ps->printerlist->data);
		gpa_node_unref (printer);
		ps->printerlist = g_slist_remove (ps->printerlist, printer);
	}
	/* Clear menu */
	gtk_option_menu_remove_menu (GTK_OPTION_MENU (ps->menu));

	/* Construct local printer list */
	for (printer = gpa_node_get_child (ps->printers, NULL); printer != NULL; printer = gpa_node_get_child (ps->printers, printer)) {
		/* printer IS referenced */
		ps->printerlist = g_slist_prepend (ps->printerlist, printer);
	}

	/* Get id of default printer */
	defid = gpa_node_get_path_value (GNOME_PRINT_CONFIG_NODE (((GPAWidget *)ps)->config), "Printer");

	menu = gtk_menu_new ();

	ps->printerlist = g_slist_reverse (ps->printerlist);

	/* Construct menu */
	idx = 0;
	def = 0;
	for (l = ps->printerlist; l != NULL; l = l->next) {
		gchar *val;
		printer = GPA_NODE (l->data);
		val = gpa_node_get_path_value (printer, "Name");
		if (!val) {
			g_warning ("Printer does not have 'Name' attribute");
		} else {
			item = gtk_menu_item_new_with_label (val);
			g_object_set_data (G_OBJECT (item), "GPAWidget", ps);
			g_signal_connect (G_OBJECT (item), "activate",
					  (GCallback) gpa_printer_selector_menuitem_activate, GINT_TO_POINTER (idx));
			gtk_widget_show (item);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
			g_free (val);
			if (defid) {
				guchar *id;
				id = gpa_node_get_value (printer);
				if (!strcmp (id, defid))
					def = idx;
				g_free (id);
			}
			idx += 1;
		}
	}
	if (defid)
		g_free (defid);

	/* Append special menuitems */
	if (idx > 0) {
		/* At least one printer */
		item = gtk_menu_item_new ();
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
		item = gtk_menu_item_new_with_label (_("Manage printers"));
		g_signal_connect (G_OBJECT (item), "activate",
				  (GCallback) gpa_printer_selector_manage_printers_activate, ps);
		gtk_widget_set_sensitive (item, FALSE);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	} else {
		/* No printers */
		item = gtk_menu_item_new_with_label (_("Add new printer"));
		g_signal_connect (G_OBJECT (item), "activate",
				  (GCallback) gpa_printer_selector_add_printer_activate, ps);
		gtk_widget_set_sensitive (item, FALSE);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	}

	gtk_widget_show (menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (ps->menu), menu);

	gtk_option_menu_set_history (GTK_OPTION_MENU (ps->menu), def);
}

