/* interface_cb.c
 *
 * Copyright (C) 1999 - 2002 Vivien Malerba
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "interface_cb.h"
#include "server-rs.h"
#include "mainpagequery.h"
#include "relship.h"
#include <bonobo.h>
#include <libgnomedb/libgnomedb.h>
#include "item-selector.h"
#include "object-selector.h"
#include "settings.h"

#include "passwd.xpm"

#define XML_MERGEANT_DTD_FILE DTDINSTALLDIR"/mergeant.dtd"
/* DTD validation */
extern int xmlDoValidityCheckingDefaultValue;

/* Password dialog pixmap */
static GdkPixmap *passwd_icon = NULL;
static GdkBitmap *passwd_mask = NULL;


/*
 * Display a dialog to help the contributer
 */
static void
todo (ConfManager * conf, const gchar *file, int line)
{
	gchar *str;
	str = g_strdup_printf (_("This function needs to be implemented\n"
				 "see file %s line %d if you want to contribute."),
			       file, line);
	gnome_ok_dialog_parented (str, GTK_WINDOW (conf->app));
	g_free (str);
}


/********************************************************************
 *
 * User Interface actions CBs
 *
 ********************************************************************/



/* Cb to open the connexion to the SQL server */
void
sql_conn_open_cb (GtkWidget * widget, ConfManager * conf)
{
	gchar *str;

	if (!conf->last_failed && *(conf->srv->gda_datasource->str)) {
		/* we check if we have a password or not; if not, ask for one, except if we 
		   have already entered it into the gnome-db login dialog */
		if (conf->user_name && !conf->conn_open_requested &&
		    (!conf->user_passwd || (*conf->user_passwd == 0))) {
			GtkWidget *passdlg, *label, *entry, *pixmap, *hb, *vb;
			gint button;

			passdlg = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app), 0,
							       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							       GTK_STOCK_OK, GTK_RESPONSE_OK,
							       NULL);
			gtk_dialog_set_default_response (GTK_DIALOG (passdlg), GTK_RESPONSE_OK);


			hb = gtk_hbox_new (FALSE, GNOME_PAD);
			gtk_box_pack_start (GTK_BOX (GTK_DIALOG (passdlg)->vbox), hb, TRUE, TRUE, GNOME_PAD);
			gtk_widget_show (hb);

			vb = gtk_vbox_new (FALSE, 0);
			gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, FALSE, GNOME_PAD/2.);
			gtk_widget_show (vb);
			
			if (!passwd_icon)
				passwd_icon = gdk_pixmap_create_from_xpm_d (GTK_WIDGET (conf->app)->window, 
									    &passwd_mask,
									    NULL, passwd_xpm);
			pixmap = gtk_pixmap_new (passwd_icon, passwd_mask);
			gtk_box_pack_start (GTK_BOX (vb), pixmap, FALSE, FALSE, GNOME_PAD/2.);
			gtk_widget_show (pixmap);

			vb = gtk_vbox_new (FALSE, 0);
			gtk_box_pack_start (GTK_BOX (hb), vb, TRUE, TRUE, GNOME_PAD/2.);
			gtk_widget_show (vb);

			label = gtk_label_new (NULL);
			str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
					       _("Password requested"),
					       _("Enter password to open the connection,\n"
						 "or leave it empty if no password is required.\n\n"));
			gtk_label_set_markup (GTK_LABEL (label), str);
			g_free (str);
			gtk_box_pack_start (GTK_BOX (vb), label, TRUE, TRUE, GNOME_PAD/2.);
			gtk_widget_show (label);


			hb = gtk_hbox_new (FALSE, GNOME_PAD);
			gtk_box_pack_start (GTK_BOX (vb), hb, TRUE, TRUE, GNOME_PAD);
			gtk_widget_show (hb);

			label = gtk_label_new (_("Password:"));
			gtk_box_pack_start (GTK_BOX (hb), label, FALSE, TRUE, GNOME_PAD/2.);

			entry = gtk_entry_new ();
			gtk_entry_set_visibility (GTK_ENTRY (entry), FALSE);
			gtk_box_pack_start (GTK_BOX (hb), entry, TRUE, TRUE, GNOME_PAD/2.);

			gtk_widget_show_all (GTK_DIALOG (passdlg)->vbox);
			button = gtk_dialog_run (GTK_DIALOG (passdlg));
			switch (button) {
			case GTK_RESPONSE_DELETE_EVENT:  
			case GTK_RESPONSE_CANCEL:
			default: 
				gtk_widget_destroy (passdlg);
				return;
				break;
			case GTK_RESPONSE_OK: 
				/* fetch the password */
				if (conf->user_passwd)
					g_free (conf->user_passwd);
				conf->user_passwd = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
				g_string_assign (conf->srv->password, conf->user_passwd);
				gtk_widget_destroy (passdlg);
				break;
			}

		}
		server_access_open_connect (conf->srv);
		conf->last_failed = (conf->srv->cnc == NULL);
	}
	else {
		conf->last_failed = FALSE;
		conf->conn_open_requested = TRUE;
		options_config_cb (NULL, conf);
	}
}

/* Cb to close the connexion for the SQL server */
void
sql_conn_close_cb (GtkWidget * widget, ConfManager * conf)
{
	gint answer;
	gchar *str;
	GtkWidget *dialog, *label, *hb, *pixmap, *vb;

	/* ask if the user really wants to quit */
	if (!conf->save_up_to_date) {
		dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app),
						      0,
						      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
						      _("Disconnect without saving"), 2,
						      _("Save and Disconnect"), 1,
						      NULL);
		
		label = gtk_label_new (NULL);
		str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
				       _("Save changes before disconnecting?"),
				       _("Some data have not yet been saved and will be lost\n"
					 "if not saved before "
					 "closing the connection."));
		gtk_label_set_markup (GTK_LABEL (label), str);
		g_free (str);
	}
	else {
		dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app), 
						      0,
						      GTK_STOCK_NO, GTK_RESPONSE_CANCEL,
						      GTK_STOCK_YES, 2, NULL);

		label = gtk_label_new (NULL);
		str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>",
				       _("Do you really want to disconnect?"));
		gtk_label_set_markup (GTK_LABEL (label), str);
		g_free (str);
	}
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);

	hb = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hb, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (hb);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, FALSE, GNOME_PAD/2.);
	gtk_widget_show (vb);
	
	pixmap = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_DIALOG);
	gtk_box_pack_start (GTK_BOX (vb), pixmap, FALSE, FALSE, GNOME_PAD/2.);
	gtk_widget_show (pixmap);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (vb);

	gtk_box_pack_start (GTK_BOX (vb), label, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (label);
	answer = gtk_dialog_run (GTK_DIALOG (dialog));

	switch (answer) {
	case 1: /* Save and disconnect asked */
		file_save_cb (NULL, conf);
		if (!conf->working_file)
			conf->close_after_saving = TRUE;
	case 2: /* disconnect, no saving */
		/* if we've got a pending transaction, ask the user what to do */
		if (conf->current_transaction) {
			if (gnome_db_yes_no_dialog (dialog,
						    _("A transaction is pending. Would you like to "
						      "commit the changes you did (select Yes) or "
						      "discard them (select No)?"))) {
				gda_connection_commit_transaction (conf->srv->cnc, conf->current_transaction);
			} else
				gda_connection_rollback_transaction (conf->srv->cnc, conf->current_transaction);

			g_object_unref (G_OBJECT (conf->current_transaction));
			conf->current_transaction = NULL;
		}

		conf->save_up_to_date = TRUE;
		/* here we delay the closing of the connection because the user need to choose
		   a filename to save data to */
		if (! conf->close_after_saving)
			server_access_close_connect (conf->srv);
		break;
	default:
		break;
	}
	gtk_widget_destroy (dialog);
}

/* transaction-related callbacks */
void
sql_begin_trans_cb (GtkWidget * widget, ConfManager * conf)
{
	gchar *xname;
	static guint counter = 0;

	if (!gda_connection_supports (conf->srv->cnc, GDA_CONNECTION_FEATURE_TRANSACTIONS)) {
		gnome_db_show_error (_("Underlying data source does not support transactions"));
		return;
	}

	if (conf->current_transaction) {
		gnome_db_show_error (_("A transaction is already running"));
		return;
	}

	xname = g_strdup_printf ("mergeant_transaction_%d", counter++);
	conf->current_transaction = gda_transaction_new (xname);

	/* start the transaction on the server */
	if (gda_connection_begin_transaction (conf->srv->cnc, conf->current_transaction)) {
		gtk_image_set_from_stock (GTK_IMAGE (conf->icon_transaction),
					  GNOME_DB_STOCK_WITHIN_TRANSACTION,
					  GTK_ICON_SIZE_MENU);
		gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_transaction,
				      _("Transaction in progress"), _("Transaction in progress"));
		gnome_appbar_set_status (GNOME_APPBAR (conf->appbar), _("Transaction started"));
	} else {
		g_object_unref (G_OBJECT (conf->current_transaction));
		conf->current_transaction = NULL;
	}

	g_free (xname);
}

void
sql_commit_trans_cb (GtkWidget * widget, ConfManager * conf)
{
	if (!gda_connection_supports (conf->srv->cnc, GDA_CONNECTION_FEATURE_TRANSACTIONS)) {
		gnome_db_show_error (_("Underlying data source does not support transactions"));
		return;
	}

	if (!conf->current_transaction) {
		gnome_db_show_error (_("A transaction must be started before committing"));
		return;
	}

	if (gda_connection_commit_transaction (conf->srv->cnc, conf->current_transaction)) {
		g_object_unref (G_OBJECT (conf->current_transaction));
		conf->current_transaction = NULL;

		gtk_image_set_from_stock (GTK_IMAGE (conf->icon_transaction),
					  GNOME_DB_STOCK_NO_TRANSACTION,
					  GTK_ICON_SIZE_MENU);
		gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_transaction,
				      _("No transaction"), _("No transaction"));
		gnome_appbar_set_status (GNOME_APPBAR (conf->appbar), _("Transaction saved"));
	}
}

void
sql_rollback_trans_cb (GtkWidget *widget, ConfManager * conf)
{
	if (!gda_connection_supports (conf->srv->cnc, GDA_CONNECTION_FEATURE_TRANSACTIONS)) {
		gnome_db_show_error (_("Underlying data source does not support transactions"));
		return;
	}

	if (!conf->current_transaction) {
		gnome_db_show_error (_("A transaction must be started before cancelling"));
		return;
	}

	if (gda_connection_rollback_transaction (conf->srv->cnc, conf->current_transaction))
		gnome_appbar_set_status (GNOME_APPBAR (conf->appbar), _("Transaction aborted"));

	g_object_unref (G_OBJECT (conf->current_transaction));
	conf->current_transaction = NULL;

	gtk_image_set_from_stock (GTK_IMAGE (conf->icon_transaction),
				  GNOME_DB_STOCK_NO_TRANSACTION,
				  GTK_ICON_SIZE_MENU);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_transaction,
			      _("No transaction"), _("No transaction"));
}

/*
 * options_config_cb
 */

static gint open_conn_idle_func (ConfManager * conf);

void
options_config_cb (GtkWidget * widget, ConfManager * conf)
{
        GtkWidget *props;

        props = gnome_db_login_dialog_new (_("Connection's configuration"));
        if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (props))) {
                conf->save_up_to_date = FALSE;

                /* fetching DSN and provider */
                g_string_assign (conf->srv->gda_datasource,
                                 gnome_db_login_dialog_get_dsn(GNOME_DB_LOGIN_DIALOG (props)));
                g_string_assign (conf->srv->user_name,
                                 gnome_db_login_dialog_get_username(GNOME_DB_LOGIN_DIALOG (props)));
                g_string_assign (conf->srv->password,
                                 gnome_db_login_dialog_get_password(GNOME_DB_LOGIN_DIALOG (props)));

                if (conf->user_name)
                        g_free (conf->user_name);
                conf->user_name = g_strdup (conf->srv->user_name->str);
                if (conf->user_passwd)
                        g_free (conf->user_passwd);
                conf->user_passwd = g_strdup (conf->srv->password->str);

                /* if the connection was to be opened, we open it now */
                if (conf->conn_open_requested)
                        gtk_idle_add ((GtkFunction) open_conn_idle_func, conf);
        }
        gtk_widget_destroy (props);
}


/* used as a gtk_idle_func! */
static gint
open_conn_idle_func (ConfManager * conf)
{
        sql_conn_open_cb (NULL, conf);
        return FALSE;
}


/*
 * sql_show_relations_cb
 */

static void show_relations_dialog_response_cb   (GtkDialog *dlg, gint button_number,
						 ConfManager *conf);
static void show_relations_dialog_destroy_cb    (GtkDialog *dlg, ConfManager *conf);
static void show_relations_dialog_conn_close_cb (ServerAccess *srv, ConfManager *conf);

void
sql_show_relations_cb (GObject * obj, ConfManager * conf)
{
	/* to hide the DLG */
	if (!conf) 
		return;

	if (!conf->relations_dialog) {
		GtkWidget *dlg, *rels;
		RelShip *rs;

		dlg = gtk_dialog_new_with_buttons (_("Database Relations"), NULL, 0,
						   GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);

		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, FALSE);
		g_signal_connect (G_OBJECT (dlg), "response",
				  G_CALLBACK (show_relations_dialog_response_cb), conf);
		g_signal_connect (G_OBJECT (dlg), "destroy",
				  G_CALLBACK (show_relations_dialog_destroy_cb), conf);
		g_signal_connect (G_OBJECT (conf->srv), "conn_to_close",
				  G_CALLBACK (show_relations_dialog_conn_close_cb), conf);

		/* dialog contents */
		rs = RELSHIP (relship_find (QUERY (conf->top_query)));
		rels = relship_get_sw_view (rs);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), rels, TRUE, TRUE, 0);

		gtk_widget_set_usize (dlg, 825, 570);
		gtk_widget_show_all (dlg);

		conf->relations_dialog = dlg;
	}
	else
		gdk_window_raise (conf->relations_dialog->window);
}


static void 
show_relations_dialog_destroy_cb (GtkDialog *dlg, ConfManager *conf)
{
	conf->relations_dialog = NULL;
	g_signal_handlers_disconnect_by_func (G_OBJECT (conf->srv), 
					      G_CALLBACK (show_relations_dialog_conn_close_cb), conf);
}

static void
show_relations_dialog_response_cb (GtkDialog *dlg, gint button_number, ConfManager *conf)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}


static void 
show_relations_dialog_conn_close_cb (ServerAccess *srv, ConfManager *conf)
{
	gtk_widget_destroy (GTK_WIDGET (conf->relations_dialog));
}








/*
 * sql_data_view_cb
 */

void
sql_data_view_cb (GtkWidget * widget, ConfManager * conf)
{
	GtkWidget *dlg;

	/* PORTING FIXME with Rodrigo's magic widget */
	/*dlg = sql_data_wid_new_dlg (conf->srv);
	  gtk_widget_show (dlg);*/
}


/*
 * sql_mem_update_cb
 */

static void append_refresh_element_table (GtkWidget *table, guint x, guint y, 
					  const gchar * key, const gchar *descr);
static gboolean clean_empty_shell_queries (Query *q);
static gboolean actual_sql_mem_update_cb (ConfManager * conf);
void
sql_mem_update_cb (GtkWidget * widget, ConfManager * conf)
{
	if (!conf->refresh_status_dlg) {
		GtkWidget *dlg, *table, *pbar, *wid;

		/* building the status dialog */
		dlg = gtk_dialog_new_with_buttons (_("Database structure synchronisation"),
						   GTK_WINDOW (conf->app), GTK_DIALOG_MODAL,
						   GTK_STOCK_STOP, GTK_RESPONSE_CANCEL, NULL);

		/* FIXME: TODO */
		gtk_dialog_set_response_sensitive (GTK_DIALOG (dlg), GTK_RESPONSE_CANCEL, FALSE);

		/* table and status items */
		table = gtk_table_new (7, 2, FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (table), 0);
		gtk_table_set_col_spacings (GTK_TABLE (table), 6);
		gtk_table_set_row_spacings (GTK_TABLE (table), 6);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), table, TRUE, TRUE, 6);
		
		append_refresh_element_table (table, 0, 0, "DATA_TYPES", _("Data types analysis"));
		append_refresh_element_table (table, 0, 1, "FUNCTIONS", _("Functions analysis"));
		append_refresh_element_table (table, 0, 2, "AGGREGATES", _("Aggregates analysis"));
		append_refresh_element_table (table, 0, 3, "TABLES", _("Tables analysis"));
		append_refresh_element_table (table, 0, 4, "VIEWS", _("Views analysis"));
		append_refresh_element_table (table, 0, 5, "SEQUENCES", _("Sequences analysis"));
		gtk_widget_show (table);
		g_object_set_data (G_OBJECT (dlg), "table", table);

		/* show the first active item */
		wid = g_object_get_data (G_OBJECT (table), "DATA_TYPESN");
		gtk_widget_show (wid);

		/* progress bar */
		pbar = gtk_progress_bar_new ();
		gtk_progress_bar_set_orientation (GTK_PROGRESS_BAR (pbar), GTK_PROGRESS_LEFT_TO_RIGHT);
		gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (pbar), .0);
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), pbar, TRUE, TRUE, 6);
		gtk_widget_show (pbar);
		g_object_set_data (G_OBJECT (dlg), "pbar", pbar);

		gtk_widget_show (dlg);
		conf->refresh_status_dlg = dlg;
		gtk_idle_add ((GtkFunction) actual_sql_mem_update_cb, (gpointer) conf);
	}
}

static gboolean 
actual_sql_mem_update_cb (ConfManager * conf)
{

	/* update of the data types known by the provider */
	server_access_refresh_datas (conf->srv);
	
	/* let's refresh the memory's representation */
	database_refresh (conf->db, conf->srv);
	
	/* clean the Queries which are just empty shells */
	clean_empty_shell_queries (QUERY (conf->top_query));
		
	if (conf->refresh_status_dlg) {
		/* close the dialog */
		gtk_widget_destroy (conf->refresh_status_dlg);
		conf->refresh_status_dlg = NULL;
	}

	return FALSE;
}

static void 
append_refresh_element_table (GtkWidget *table, guint x, guint y, 
			      const gchar * key, const gchar *descr)
{
	gchar *str;
	GtkWidget *vbox, *label, *wid;
	
	/* icons */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox);
	gtk_table_attach (GTK_TABLE (table), vbox, x, x+1, y, y+1, 0, 0, 0, 0);

	wid = gtk_image_new_from_stock (GTK_STOCK_APPLY, GTK_ICON_SIZE_MENU);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	str = g_strdup_printf ("%sD", key);
	g_object_set_data (G_OBJECT (table), str, wid);
	g_free (str);
	gtk_widget_hide (wid);

	wid = gtk_image_new_from_stock (GTK_STOCK_GO_FORWARD, GTK_ICON_SIZE_MENU);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, FALSE, 0);
	str = g_strdup_printf ("%sN", key);
	g_object_set_data (G_OBJECT (table), str, wid);
	g_free (str);
	gtk_widget_hide (wid);

	/* description */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox);
	gtk_table_attach_defaults (GTK_TABLE (table), vbox, x+1, x+2, y, y+1);

	label = gtk_label_new (descr);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	str = g_strdup_printf ("%sL", key);
	g_object_set_data (G_OBJECT (table), str, label);
	gtk_widget_show (label);
	g_free (str);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", descr);
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	str = g_strdup_printf ("%sB", key);
	g_object_set_data (G_OBJECT (table), str, label);
	g_free (str);
}


/* This function destroys any Query which is only an empty shell (no QueryFields and
   no sub queries), it returns TRUE if the query q was cleaned (removed) */
static gboolean
clean_empty_shell_queries (Query *q)
{
	GSList *list;

	g_assert (q); 

	/* clean the sub queries first */
	list = q->sub_queries;
	while (list) {
		if (clean_empty_shell_queries (QUERY (list->data)))
			list = q->sub_queries;
		else
			list = g_slist_next (list);
	}

	/* do we remove that query ? */
	if (!q->fields && !q->sub_queries && q->parent) {
		query_del_sub_query (q->parent, q);
		return TRUE;
	}
	else
		return FALSE;
}

/*
 * rescan_display_plugins_cb
 */

static void refresh_plugins_table_cb (GObject * obj, ConfManager * conf);
static gboolean is_plugin_used (ConfManager * conf, DataHandler *dh);
struct foreach_hash_struct
{
	gpointer find_value;
	gboolean found;
};
static void foreach_hash_cb (gpointer key, gpointer value,
			     struct foreach_hash_struct *user_data);


void
rescan_display_plugins_cb (GObject * obj, ConfManager * conf)
{
	server_access_rescan_display_plugins (conf->srv, conf->plugins_dir);
	if (conf->config_plugins_dlg)
		refresh_plugins_table_cb (NULL, conf);
}


/* 
 * refreah the table showing information about which plugin is used
 */
static void
refresh_plugins_table_cb (GObject * obj, ConfManager * conf)
{
	GSList *list;
	const gchar *text[5];
	DataHandler *dh;
	GtkCList *clist;
	gint row;

	clist = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "clist");
	list = conf->srv->data_types_handlers;
	gtk_clist_freeze (clist);
	while (list) {
		dh = DATA_HANDLER (list->data);
		if (data_handler_is_plugin (dh)) {
			row = gtk_clist_find_row_from_data (clist, list->data);
			text[0] = data_handler_get_plugin_name (dh);
			text[1] = data_handler_get_version (dh);

			if (is_plugin_used (conf, dh))
				text[2] = _("Yes");
			else
				text[2] = _("No");
			text[3] = data_handler_get_descr (dh);
			text[4] = data_handler_get_plugin_file (dh);
			if (row >= 0) {
				gtk_clist_remove (clist, row);
				gtk_clist_insert (clist, row, text);
			}
			else
				row = gtk_clist_append (clist, text);
			gtk_clist_set_row_data (clist, row, dh);
		}
		list = g_slist_next (list);
	}

	/* removing any row not related to a plugin anymore */
	row = 0;
	while (row < clist->rows) {
		dh = gtk_clist_get_row_data (clist, row);
		if (!g_slist_find (conf->srv->data_types_handlers, dh))
			gtk_clist_remove (clist, row);
		else
			row++;
	}

	gtk_clist_thaw (clist);
}

static gboolean
is_plugin_used (ConfManager * conf, DataHandler *dh)
{
	gboolean found;
	struct foreach_hash_struct *fhs;
	fhs = g_new (struct foreach_hash_struct, 1);
	fhs->find_value = dh;
	fhs->found = FALSE;

	g_hash_table_foreach (conf->srv->types_objects_hash,
			      (GHFunc) (foreach_hash_cb), fhs);

	found = fhs->found;
	g_free (fhs);

	return found;
}

static void
foreach_hash_cb (gpointer key, gpointer value,
		 struct foreach_hash_struct *user_data)
{
	if (value == user_data->find_value)
		user_data->found = TRUE;
}



/*
 * config_display_plugins_cb
 */
/* dialog itself */
static void config_plugins_dlg_destroy_cb (GtkWidget * wid,
					   ConfManager * conf);

/* 1st page */
static void change_location_cb (GObject * obj, ConfManager * conf);

/* 2nd page */
static void data_types_os_select_cb (ObjectSelector * os, ServerDataType *dt, ConfManager * conf);
static void plugins_ch_combo_sel_changed_cb (GObject * obj, gpointer newsel, ConfManager * conf);
static void plugin_update_description (GtkWidget * Bdescr, GtkWidget * Ddescr, DataHandler *dh);

/* 3rd page */
static void table_field_os_select_cb (ObjectSelector * os, GObject *obj, ConfManager * conf);
static void plugins_ch_combo_sel_changed_cb2 (GObject * object, gpointer * newsel, ConfManager * conf);

void
config_display_plugins_cb (GObject * obj, ConfManager * conf)
{
	GtkWidget *props, *label, *page, *wid, *sw, *box, *table, *box2, *clist, *nb;
	GtkWidget *frame, *tmpwid;
	
	if (conf->config_plugins_dlg) {
		gdk_window_raise (conf->config_plugins_dlg->window);
		return;
	}

	props = gtk_dialog_new_with_buttons (_("Plugins preferences"), GTK_WINDOW (conf->app), 0,
					     GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	conf->config_plugins_dlg = props;
	nb = gtk_notebook_new ();
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (props)->vbox), nb);

	/* 
	 * known plugins and their usage: 1st page
	 */
	label = gtk_label_new (_("Plugins list"));
	page = gtk_hbox_new (FALSE, GNOME_PAD);
	g_object_set_data (G_OBJECT (props), "pluglist", page);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), page, label);

	/* on the left is a clist in its sw and the location of plugins */
	box = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (page), box, TRUE, TRUE, GNOME_PAD);

	box2 = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), box2, FALSE, TRUE, GNOME_PAD);
	wid = gtk_label_new (_("Plugins location:"));
	gtk_box_pack_start (GTK_BOX (box2), wid, FALSE, TRUE, GNOME_PAD);
	wid = gtk_label_new (conf->plugins_dir);
	g_object_set_data (G_OBJECT (props), "loclabel", wid);
	gtk_box_pack_start (GTK_BOX (box2), wid, FALSE, TRUE, GNOME_PAD);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (box), sw, TRUE, TRUE, GNOME_PAD);
	clist = gtk_clist_new (5);
	g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "clist",
			   clist);
	gtk_clist_set_column_title (GTK_CLIST (clist), 0, _("Plugin"));
	gtk_clist_set_column_title (GTK_CLIST (clist), 1, _("Version"));
	gtk_clist_set_column_title (GTK_CLIST (clist), 2, _("Used"));
	gtk_clist_set_column_title (GTK_CLIST (clist), 3, _("Description"));
	gtk_clist_set_column_title (GTK_CLIST (clist), 4, _("File"));
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 1, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 2, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 3, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (clist), 4, TRUE);
	gtk_clist_set_selection_mode (GTK_CLIST (clist),
				      GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_show (GTK_CLIST (clist));
	gtk_clist_column_titles_passive (GTK_CLIST (clist));
	gtk_container_add (GTK_CONTAINER (sw), clist);

	/* on the right are some buttons and the dir. location of the plugins */
	box = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (box),
				   GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (page), box, FALSE, FALSE, GNOME_PAD);
	wid = gtk_button_new_with_label (_("Change location"));
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (change_location_cb), conf);
	gtk_container_add (GTK_CONTAINER (box), wid);
	wid = gtk_button_new_with_label (_("Rescan plugins"));
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (rescan_display_plugins_cb),
			  conf);
	gtk_container_add (GTK_CONTAINER (box), wid);
	wid = gtk_button_new_with_label (_("Refresh"));
	gtk_container_add (GTK_CONTAINER (box), wid);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (refresh_plugins_table_cb), conf);

	g_signal_connect (G_OBJECT (conf->srv), "objects_bindings_updated",
			  G_CALLBACK (refresh_plugins_table_cb), conf);

	gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);
	refresh_plugins_table_cb (NULL, conf);


	if (server_access_is_open (conf->srv)) {
		/* 
		 * Data types bindings: 2nd page
		 */
		label = gtk_label_new (_("Data Type bindings"));
		page = gtk_hpaned_new ();
		gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);
		g_object_set_data (G_OBJECT (props), "dtbinding", page);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), page, label);
	       		
		/* on the left side is an ObjectSelector */
		wid = object_selector_new (conf, OBJECT_SELECTOR_DATA_TYPES, -1);
		gtk_paned_add1 (GTK_PANED (page), wid);
		gtk_widget_set_size_request (wid, 150, -1);
		object_selector_set_comments_visible (OBJECT_SELECTOR (wid), FALSE);
		object_selector_set_column_label (OBJECT_SELECTOR (wid), 0, _("Data types"));
		g_signal_connect (G_OBJECT (wid), "selection_changed",
				  G_CALLBACK (data_types_os_select_cb), conf);
		
		/* on the right side is the action area... */
		box = gtk_vbox_new (FALSE, 0);
		gtk_paned_add2 (GTK_PANED (page), box);

		table = gtk_table_new (3, 2, FALSE);
		gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD);
		gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
		wid = gtk_label_new (_("Data Type:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = gtk_label_new (_("<Select one>"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "seltname", wid);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
		wid = gtk_label_new (_("Description:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_label_set_line_wrap (GTK_LABEL (wid), TRUE);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = gtk_label_new (_("<NONE>"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "seltdescr", wid);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
		wid = gtk_label_new (_("Display Plugin:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 2, 3,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = item_selector_new (ITEM_SELECTOR_MENU);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "selcc", wid);
		item_selector_set_static_content_offset (ITEM_SELECTOR (wid), NULL, 0);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
		g_signal_connect (G_OBJECT (wid), "selection_changed",
				  G_CALLBACK (plugins_ch_combo_sel_changed_cb), conf);
		
		/* ...and a frame and label to put detailled description of the plugin */
		frame = gtk_frame_new (_("Plugin description"));
		gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, GNOME_PAD);
		wid = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_add (GTK_CONTAINER (frame), wid);
		
		tmpwid = gtk_label_new ("");
		gtk_box_pack_start (GTK_BOX (wid), tmpwid, FALSE, FALSE, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr", tmpwid);
		
		tmpwid = gtk_label_new ("");
		gtk_box_pack_start (GTK_BOX (wid), tmpwid, FALSE, FALSE, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr", tmpwid);
		
		gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);
		
		/* 
		 * Fine tuned plugins usage: 3rd page
		 */
		label = gtk_label_new (_("Individual objects bindings"));
		page = gtk_hpaned_new ();
		g_object_set_data (G_OBJECT (props), "objbinding", page);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), page, label);		
		
		/* on the left side is an Objest Selector */
		wid = object_selector_new (conf, OBJECT_SELECTOR_TABLES | OBJECT_SELECTOR_VIEWS |
					   OBJECT_SELECTOR_DBFIELDS, -1);
		gtk_paned_add1 (GTK_PANED (page), wid);
		gtk_widget_set_size_request (wid, 150, -1);
		object_selector_set_comments_visible (OBJECT_SELECTOR (wid), TRUE);
		object_selector_set_column_label (OBJECT_SELECTOR (wid), 0, _("Tables' fields"));
		object_selector_set_column_label (OBJECT_SELECTOR (wid), 1, _("Field type"));
		g_signal_connect (G_OBJECT (wid), "selection_changed",
				  G_CALLBACK (table_field_os_select_cb), conf);

				
		/* on the right side is the action area */
		box = gtk_vbox_new (FALSE, 0);
		gtk_paned_add2 (GTK_PANED (page), box);
		
		table = gtk_table_new (3, 2, FALSE);
		gtk_table_set_col_spacing (GTK_TABLE (table), 0, GNOME_PAD);
		gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, 0);
		wid = gtk_label_new (_("Object:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = gtk_label_new (_("<Select one>"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "seltab", wid);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
		wid = gtk_label_new (_("Data type:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = gtk_label_new ("-");
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "tabtype", wid);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
		wid = gtk_label_new (_("Display Plugin:"));
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 2, 3,
				  GTK_FILL, 0, GNOME_PAD/2., GNOME_PAD/2.);
		wid = item_selector_new (ITEM_SELECTOR_MENU);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "seltabplug", wid);
		item_selector_set_static_content_offset (ITEM_SELECTOR (wid), NULL, 0);
		gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
		g_signal_connect (G_OBJECT (wid), "selection_changed",
				  G_CALLBACK (plugins_ch_combo_sel_changed_cb2), conf);
		
		/* below is a frame and label to put detailled description of the plugin */
		frame = gtk_frame_new (_("Plugin description"));
		gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, GNOME_PAD);
		wid = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_add (GTK_CONTAINER (frame), wid);
		
		tmpwid = gtk_label_new ("");
		gtk_box_pack_start (GTK_BOX (wid), tmpwid, FALSE, FALSE, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr2", tmpwid);
		
		tmpwid = gtk_label_new ("");
		gtk_box_pack_start (GTK_BOX (wid), tmpwid, FALSE, FALSE, 0);
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr2", tmpwid);
		gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);
	}

	gtk_widget_show_all (nb);
	gtk_widget_set_usize (props, 600, 300);

	g_signal_connect (G_OBJECT (props), "destroy",
			  G_CALLBACK (config_plugins_dlg_destroy_cb), conf);

	g_signal_connect_swapped (GTK_OBJECT (props), "response", 
				  G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (props));

	gtk_widget_show (props);
}

static void
config_plugins_dlg_destroy_cb (GtkWidget * wid, ConfManager * conf)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (conf->srv),
					      G_CALLBACK (refresh_plugins_table_cb), conf);

	conf->config_plugins_dlg = NULL;
}


static void
change_location_cb (GObject * obj, ConfManager * conf)
{
	GtkWidget *fs;
	gint result;
	gchar *str, *ptr;
	GtkWidget *wid;
		

	fs = gtk_file_selection_new ("Select plugins directory");
	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (fs));

	result = gtk_dialog_run (GTK_DIALOG (fs));
	switch (result) {
	case GTK_RESPONSE_OK:
		str = g_strdup (gtk_file_selection_get_filename	(GTK_FILE_SELECTION (fs)));
		/* removing trailing caracters in case we have a file instead of a dir */
		ptr = str + strlen (str) - 1;
		while (*ptr != '/') {
			*ptr = '\0';
			ptr = str + strlen (str) - 1;
		}
		*ptr = '\0';
		if (conf->plugins_dir)
			g_free (conf->plugins_dir);
		conf->plugins_dir = g_strdup (str);
		g_free (str);
		/* refreshing the display */
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "loclabel");
		gtk_label_set_text (GTK_LABEL (wid), conf->plugins_dir);
		rescan_display_plugins_cb (NULL, conf);
		break;
	default:
		break;
	}
	gtk_widget_destroy (fs);
}


static gchar *data_handler_get_plugin_name_proxy (DataHandler *dh);
static void
data_types_os_select_cb (ObjectSelector * os, ServerDataType *dt, ConfManager * conf)
{
	if (dt) {
		GtkWidget *wid;
		DataHandler *dh = NULL;
		GSList *blist, *list;
		gint i;
	
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "dt", dt);
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltname");
		gtk_label_set_text (GTK_LABEL (wid), dt->sqlname);
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltdescr");
		gtk_label_set_text (GTK_LABEL (wid), dt->descr);
		/* computing the list for choice combo from the gda types accepted
		   by each plugin */
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "selcc");
		blist = g_slist_append (NULL, NULL);
		list = conf->srv->data_types_handlers;
		while (list) {
			dh = DATA_HANDLER (list->data);
			if (data_handler_is_plugin (dh)) {
				gboolean found;
				found = FALSE;
				i = 0;
				if (data_handler_get_nb_gda_type (dh) == 0)	/* any Gda Type is OK */
					found = TRUE;

				while ((i < data_handler_get_nb_gda_type (dh)) && !found) {
					if (data_handler_get_gda_type (dh, i) == dt->gda_type)
						found = TRUE;
					i++;
				}
				if (found) 
					blist = g_slist_append (blist, dh);
			}
			list = g_slist_next (list);
		}

		g_signal_handlers_block_by_func (G_OBJECT (wid), 
						 G_CALLBACK (plugins_ch_combo_sel_changed_cb), 
						 conf);

		item_selector_set_static_content_func (ITEM_SELECTOR (wid), blist, 
						       (ItemSelectorFunc) data_handler_get_plugin_name_proxy);
		/* if a plugin is already bound, set it as default */
		dh = server_access_get_object_handler (conf->srv, G_OBJECT (dt));
		if (dh) {
			GtkWidget *w1, *w2;

			w1 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr");
			w2 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr");

			i = g_slist_index (blist, dh);
			if (i >= 0) {
				item_selector_set_selection_num (ITEM_SELECTOR (wid), i);
				/* updating the plugin description */
				plugin_update_description (w1, w2, dh);
			}
			else {
				item_selector_set_selection_num (ITEM_SELECTOR (wid), 0);
				plugin_update_description (w1, w2, NULL);
			}
		}

		g_signal_handlers_unblock_by_func (G_OBJECT (wid), 
						   G_CALLBACK (plugins_ch_combo_sel_changed_cb), 
						   conf);

		g_slist_free (blist);
	}
	else {
		GtkWidget *wid;

		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltname");
		gtk_label_set_text (GTK_LABEL (wid), _("<Select one>"));
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltdescr");
		gtk_label_set_text (GTK_LABEL (wid), _("<NONE>"));
		
		/* removing any comment in the plugin detail frame */
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr");
		gtk_label_set_text (GTK_LABEL (wid), "");
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr");
		gtk_label_set_text (GTK_LABEL (wid), "");
		
		/* choice combo */
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "dt", NULL);	
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "selcc");
		item_selector_set_static_content_offset (ITEM_SELECTOR (wid), NULL, 0);
	}
}


static gchar *
data_handler_get_plugin_name_proxy (DataHandler *dh)
{
	if (dh)
		return g_strdup (data_handler_get_plugin_name (dh));
	else 
		return g_strdup (_("None"));
}


static void
plugins_ch_combo_sel_changed_cb (GObject * obj, gpointer newsel, ConfManager * conf)
{
	gpointer ptr;
	ServerDataType *dt;

	dt = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "dt");
	ptr = g_hash_table_lookup (conf->srv->types_objects_hash, dt);
	if (ptr)
		server_access_unbind_object_handler (conf->srv, G_OBJECT (dt));

	if (newsel) {
		server_access_bind_object_handler (conf->srv, G_OBJECT (dt), DATA_HANDLER (newsel));
		/* tell that we have modified the working environment */
		conf->save_up_to_date = FALSE;
	}

	refresh_plugins_table_cb (NULL, conf);

	if (newsel) {
		GtkWidget *w1, *w2;
		w1 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr");
		w2 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr");
		plugin_update_description (w1, w2, DATA_HANDLER (newsel));
	}
}



static void
plugin_update_description (GtkWidget * Bdescr, GtkWidget * Ddescr, DataHandler *dh)
{
	GtkWidget *wid;

	wid = Bdescr;
	if (dh) 
		gtk_label_set_text (GTK_LABEL (wid), data_handler_get_descr (dh));
	else
		gtk_label_set_text (GTK_LABEL (wid), "");
	wid = Ddescr;
	if (dh) {
		if (data_handler_get_descr_detail (dh)) 
			gtk_label_set_text (GTK_LABEL (wid), data_handler_get_descr_detail (dh));
		else
			gtk_label_set_text (GTK_LABEL (wid), "");
	}
	else
		gtk_label_set_text (GTK_LABEL (wid), "");
}



static void
table_field_os_select_cb (ObjectSelector * os, GObject *obj, ConfManager * conf)
{
	GtkWidget *wid;
	ServerDataType *dt;
	GSList *blist, *list;
	DataHandler *dh;
	gint i;
	DbField *field;
	DbTable *table;
	gchar *str;

	/* we may have pointers to DbTable objects, which we treat the same as NULL */
	if (!IS_DB_FIELD (obj)) {
		field = NULL;
		table = NULL;
		dt = NULL;
	}
	else {
		field = DB_FIELD (obj);
		table = database_find_table_from_field (conf->db, field);
		dt = field->type;
	}

	if (dt) {
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "obj", field);
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltab");
		str = g_strdup_printf ("%s.%s", table->name, field->name);
		gtk_label_set_text (GTK_LABEL (wid), str);
		g_free (str);
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "tabtype");
		gtk_label_set_text (GTK_LABEL (wid), dt->sqlname);
		
		/* computing the list for choice combo from the gda types accepted
		   by each plugin */
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltabplug");
		blist = g_slist_append (NULL, NULL);
		list = conf->srv->data_types_handlers;
		while (list) {
			dh = DATA_HANDLER (list->data);
			if (data_handler_is_plugin (dh)) {
				gboolean found;
				found = FALSE;
				i = 0;
				if (data_handler_get_nb_gda_type (dh) == 0)	/* any Gda Type is OK */
					found = TRUE;
				while ((i < data_handler_get_nb_gda_type (dh)) && !found) {
					if (data_handler_get_gda_type (dh, i) == dt->gda_type)
						found = TRUE;
					i++;
				}
				if (found)
					blist = g_slist_append (blist, dh);
			}
			list = g_slist_next (list);
		}
		
		g_signal_handlers_block_by_func (G_OBJECT (wid), 
						 G_CALLBACK (plugins_ch_combo_sel_changed_cb2), 
						 conf);
		
		item_selector_set_static_content_func (ITEM_SELECTOR (wid), blist,
						       (ItemSelectorFunc) data_handler_get_plugin_name_proxy);

		/* if a plugin is already bound, set it as default */
		dh = server_access_get_object_handler (conf->srv, G_OBJECT (field));
		if (dh) {
			GtkWidget *w1, *w2;
			w1 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr2");
			w2 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr2");

			i = g_slist_index (blist, dh);
			if (i >= 0) {
				item_selector_set_selection_num (ITEM_SELECTOR (wid), i);
				/* updating the plugin description */
				plugin_update_description (w1, w2, dh);
			}
			else {
				item_selector_set_selection_num (ITEM_SELECTOR (wid), 0);
				plugin_update_description (w1, w2, NULL);
			}
		}

		g_signal_handlers_unblock_by_func (G_OBJECT (wid), 
						   G_CALLBACK (plugins_ch_combo_sel_changed_cb2), 
						   conf);

		g_slist_free (blist);

	}
	else {
		GtkWidget *wid;

		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltab");
		gtk_label_set_text (GTK_LABEL (wid), _("<Select one>"));
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "tabtype");
		gtk_label_set_text (GTK_LABEL (wid), "-");

		/* removing any comment in the plugin detail frame */
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr2");
		gtk_label_set_text (GTK_LABEL (wid), "");
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr2");
		gtk_label_set_text (GTK_LABEL (wid), "");

		/* choice combo */
		g_object_set_data (G_OBJECT (conf->config_plugins_dlg), "obj", NULL);
		wid = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "seltabplug");
		item_selector_set_static_content_offset (ITEM_SELECTOR (wid), NULL, 0);
	}
}

/* build the contents of the tables clist contents */
static void
plugins_ch_combo_sel_changed_cb2 (GObject * object,
				  gpointer * newsel, ConfManager * conf)
{
	gpointer ptr;
	gpointer obj;

	obj = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "obj");
	ptr = g_hash_table_lookup (conf->srv->types_objects_hash, obj);
	if (ptr) 
		server_access_unbind_object_handler (conf->srv, G_OBJECT (obj));

	if (newsel) {
		server_access_bind_object_handler (conf->srv, G_OBJECT (obj), DATA_HANDLER (newsel));

		/* tell that we have modified the working environment */
		conf->save_up_to_date = FALSE;
	}
	refresh_plugins_table_cb (NULL, conf);

	if (newsel) {
		GtkWidget *w1, *w2;
		w1 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Bdescr2");
		w2 = g_object_get_data (G_OBJECT (conf->config_plugins_dlg), "Ddescr2");
		plugin_update_description (w1, w2, DATA_HANDLER (newsel));
	}
}


/*
 * CBs for the View menu
 */

/* activate the tables page */
void
view_tables_tab_cb (GtkWidget * widget, ConfManager * conf)
{
	show_tables_page_cb (widget, NULL, NULL, conf);
}

/* activate the sequences page */
void
view_sequences_tab_cb (GtkWidget * widget, ConfManager * conf)
{
	if (conf->srv->features.sequences)
		show_seqs_page_cb (widget, NULL, NULL, conf);
}

/* activate the queries page */
void
view_queries_tab_cb (GtkWidget * widget, ConfManager * conf)
{
	show_queries_page_cb (widget, NULL, NULL, conf);
}

/* activate the forms page */
void
view_forms_tab_cb (GtkWidget * widget, ConfManager * conf)
{
	show_forms_page_cb (widget, NULL, NULL, conf);
}

/* activate the SQL page */
void
view_sql_tab_cb (GtkWidget * widget, ConfManager * conf)
{
	show_sql_page_cb (widget, NULL, NULL, conf);
}

/*
 * printer_setup_cb
 */

void
printer_setup_cb (GObject * obj, ConfManager * conf)
{
	todo (conf, __FILE__, __LINE__);
}


/*
 * CBs to set the tab of the notebook to display the right page
 */

void
show_tables_page_cb (GtkWidget * shortcut, const gchar *label, const gchar *tooltip, ConfManager * conf)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (conf->nb), 0);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (conf->title_bar), _("Tables and Views"));

	gtk_image_set_from_stock (GTK_IMAGE (conf->icon_current_page), GNOME_DB_STOCK_TABLES, GTK_ICON_SIZE_MENU);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_current_page,
			      _("Tables and Views page"), _("Tables and Views page"));
}

void
show_seqs_page_cb (GtkWidget * wid, const gchar *label, const gchar *tooltip, ConfManager * conf)
{
	if (conf->srv->features.sequences) {
		gtk_notebook_set_page (GTK_NOTEBOOK (conf->nb), 1);
		gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (conf->title_bar), _("Sequences"));

		gtk_image_set_from_stock (GTK_IMAGE (conf->icon_current_page), "mergeant_sequences", GTK_ICON_SIZE_MENU);
		gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_current_page,
				      _("Sequences page"), _("Sequences page"));
	}
}

void
show_queries_page_cb (GtkWidget * shortcut, const gchar *label, const gchar *tooltip, ConfManager * conf)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (conf->nb), 2);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (conf->title_bar), _("Queries"));

	gtk_image_set_from_stock (GTK_IMAGE (conf->icon_current_page), "mergeant_queries", GTK_ICON_SIZE_MENU);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_current_page,
			      _("Queries page"), _("Queries page"));
}

void
show_forms_page_cb (GtkWidget * shortcut, const gchar *label, const gchar *tooltip, ConfManager * conf)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (conf->nb), 3);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (conf->title_bar), _("Forms"));

	gtk_image_set_from_stock (GTK_IMAGE (conf->icon_current_page), "mergeant_forms", GTK_ICON_SIZE_MENU);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_current_page,
			      _("Forms page"), _("Forms page"));
}

void
show_sql_page_cb (GtkWidget * shortcut, const gchar *label, const gchar *tooltip, ConfManager * conf)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (conf->nb), 4);
	gnome_db_gray_bar_set_text (GNOME_DB_GRAY_BAR (conf->title_bar), _("SQL"));

	gtk_image_set_from_stock (GTK_IMAGE (conf->icon_current_page), GNOME_DB_STOCK_SQL, GTK_ICON_SIZE_MENU);
	gtk_tooltips_set_tip (GTK_TOOLTIPS (conf->icon_area_tooltips), conf->icon_current_page,
			      _("SQL page"), _("SQL page"));
}


/*
 * run_preferences_cb
 */
void
run_preferences_cb (GtkWidget *w, ConfManager *conf)
{
	settings_dialog_display (conf->app, conf);
}

/*
 * run_gnomedb_manager_cb
 */
void
run_gnomedb_manager_cb (GtkWidget * w, ConfManager * conf)
{
	if (gnome_execute_shell (NULL, "gnome-database-properties") == -1)
		gnome_db_show_error (_("Could not execute database properties applet"));
}


/* 
 * set_opened_file
 */

gint
set_opened_file (ConfManager * conf, gchar * filetxt)
{
	gchar *txt;
	gint retval = 0;
	xmlDocPtr doc;
	xmlNodePtr tree;

	if (!g_file_test (filetxt, G_FILE_TEST_EXISTS)) {
		txt = g_strdup_printf (_("Cannot open file '%s'\n(file does not exist "
					 "or is not readable)."), filetxt);
		gnome_error_dialog (txt);
		g_free (txt);
		return -1;
	}
	else {
		doc = xmlParseFile (filetxt);
		if (doc) {
			txt = xmlGetProp (xmlDocGetRootElement(doc), "id_serial");
			if (txt) {
				conf->id_serial = atoi (txt);
			}
			tree = xmlDocGetRootElement(doc)->xmlChildrenNode;
			while (tree) {
				txt = mergeant_xml_clean_string_ends
					(xmlNodeGetContent (tree));
				if (!strcmp (tree->name, "gda_datasource")) {
					if (txt)
						g_string_assign (conf->srv->gda_datasource, txt);
					else
						g_string_assign (conf->srv->gda_datasource, "");
				}

				if (!strcmp (tree->name, "username")) {
					if (txt)
						g_string_assign (conf->srv->user_name, txt);
					else
						g_string_assign (conf->srv->user_name, "");
					if (conf->user_name)
						g_free (conf->user_name);
					conf->user_name = g_strdup (conf->srv->user_name->str);
				}

				tree = tree->next;
				if (txt)
					g_free (txt);
			}
			xmlFreeDoc (doc);
			g_string_assign (conf->srv->password, "");

			if (conf->working_file)
				g_free (conf->working_file);
			conf->working_file = filetxt;
			txt = conf_manager_get_title (conf);
			gtk_window_set_title (GTK_WINDOW (conf->app), txt);
			g_free (txt);
		}
		else {
			gnome_warning_dialog (_("An error has occured while loading the "
						"connexion parameters.\n"
						"Set them manually."));
			retval = -1;	/* ERROR backpropagation */
		}
	}

	return retval;
}


/*
 * file_new_cb
 */

static void file_new_dlg_cancel_cb (GtkWidget * widget, GtkWidget *data);
static void file_new_dlg_ok_cb (GtkWidget * widget, GtkWidget *data);

void
file_new_cb (GtkWidget * widget, ConfManager * conf)
{
	GtkWidget *dlg;

	dlg = gtk_file_selection_new (_("New file"));
	gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
	g_object_set_data (G_OBJECT (dlg), "conf", conf);
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->cancel_button), 
			  "clicked", G_CALLBACK (file_new_dlg_cancel_cb), dlg);	/* ! */
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button),
			  "clicked",
			  G_CALLBACK (file_new_dlg_ok_cb), dlg);
	gtk_widget_show (dlg);
}

static void
file_new_dlg_cancel_cb (GtkWidget * widget, GtkWidget *data)
{
	gtk_widget_destroy (GTK_WIDGET (data));
}

static void
file_new_dlg_ok_cb (GtkWidget * widget, GtkWidget *data)
{
	ConfManager *conf;
	gchar *filetxt;

	conf = (ConfManager *) g_object_get_data (G_OBJECT (data), "conf");
	filetxt = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (data)));
	gtk_widget_destroy (GTK_WIDGET (data));

	if (conf->working_file && !conf->save_up_to_date) {
		GtkWidget *wid;
		guint answer;
		
		wid = gtk_message_dialog_new (GTK_WINDOW (conf->app),
					      GTK_DIALOG_MODAL,
					      GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      _("Some work has not been "
						"saved.\nDo you want to save it now?"));
		answer = gtk_dialog_run (GTK_DIALOG (wid));
		if (answer == GTK_RESPONSE_YES)
			file_save_cb (NULL, conf);
		gtk_widget_destroy (wid);
	}

	if (conf->working_file)
		g_free (conf->working_file);
	conf->working_file = filetxt;
	filetxt = conf_manager_get_title (conf);
	gtk_window_set_title (GTK_WINDOW (conf->app), filetxt);
	g_free (filetxt);
}


/*
 * file_open_cb
 */

static void file_open_dlg_ok_cb (GtkWidget * widget, GtkWidget *data);
static void file_open_dlg_cancel_cb (GtkWidget * widget, GtkWidget *data);

void
file_open_cb (GtkWidget * widget, ConfManager * conf)
{
	GtkWidget *dlg;

	/* first test if the connection is established, ... */
	if (server_access_is_open (conf->srv)) {
		gnome_warning_dialog (_
				      ("The connection to an SQL server is already "
				       "established.\nDisconnect before trying to open "
				       "another file."));
		return;
	}

	/* let's now open a dialog... */
	dlg = gtk_file_selection_new (_("File to open"));
	gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (dlg));
	gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
	g_object_set_data (G_OBJECT (dlg), "conf", conf);
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->cancel_button),
			  "clicked", G_CALLBACK (file_open_dlg_cancel_cb), dlg);
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button),
			  "clicked", G_CALLBACK (file_open_dlg_ok_cb), dlg);
	gtk_widget_show (dlg);
}

static void
file_open_dlg_ok_cb (GtkWidget * widget, GtkWidget *data)
{
	ConfManager *conf;
	gchar *filetxt;

	conf = (ConfManager *) g_object_get_data (G_OBJECT (data),
						  "conf");
	filetxt = g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (data)));
	gtk_widget_destroy (GTK_WIDGET (data));

	if (set_opened_file (conf, filetxt) == -1)
		g_free (filetxt);
}

static void
file_open_dlg_cancel_cb (GtkWidget * widget, GtkWidget *data)
{
	gtk_widget_destroy (GTK_WIDGET (data));
}


/*
 * file_close_cb
 */

void
file_close_cb (GtkWidget * widget, ConfManager * conf)
{
	gchar *str;

	if (conf->working_file) {
		g_free (conf->working_file);
		conf->working_file = NULL;
		str = conf_manager_get_title (conf);
		gtk_window_set_title (GTK_WINDOW (conf->app), str);
		g_free (str);
	}
}


/*
 * file_save_cb
 */
static void quit_real (ConfManager * conf);
void
file_save_cb (GtkWidget * widget, ConfManager * conf)
{
	if (conf->working_file) {
		xmlDocPtr doc;
		xmlNodePtr root, tree;
		gchar *str;

		doc = xmlNewDoc ("1.0");
		/* DTD insertion */
		xmlCreateIntSubset(doc, "CONNECTION", NULL, XML_MERGEANT_DTD_FILE);

		root = xmlNewDocNode (doc, NULL, "CONNECTION", NULL);
		xmlDocSetRootElement (doc, root);


		/* Building the tree */
		str = g_strdup_printf ("%d", conf->id_serial);
		xmlSetProp (root, "id_serial", str);
		g_free (str);
		tree = xmlNewChild (root, NULL, "gda_datasource", conf->srv->gda_datasource->str);
		tree = xmlNewChild (root, NULL, "username", conf->srv->user_name->str);

		/* the ServerAccess object */
		server_access_build_xml_tree (conf->srv, doc);

		/* the Database object */
		database_build_xml_tree (conf->db, doc);

		/* all the Query objects, from the top level one */
		g_assert (conf->top_query);
		query_build_xml_tree (QUERY (conf->top_query), root, conf);
		
		/* ALL the GUI goes here */
		tree = xmlNewChild (root, NULL, "GUI", NULL);
		relship_build_xml_tree (NULL, tree, conf);


		xmlSaveFormatFile (conf->working_file, doc, TRUE);
		xmlFreeDoc (doc);
		conf->save_up_to_date = TRUE;

		/* check if we were waiting after saving to close the connection */
		if (conf->close_after_saving) {
			conf->close_after_saving = FALSE;
			server_access_close_connect (conf->srv);
		}
		/* check if we were waiting after saving to quit the application */
		if (conf->quit_after_saving) {
			conf->quit_after_saving = FALSE;
			quit_real (conf);
		}
	}
	else if (!conf->save_up_to_date)
		file_save_as_cb (NULL, conf);
}

/*
 * file_save_as_cb
 */

static void file_saveas_dlg_ok_cb (GtkWidget * widget, GtkWidget *data);

void
file_save_as_cb (GtkWidget * widget, ConfManager * conf)
{
	GtkWidget *dlg;

	/* let's now open a dialog... */
	dlg = gtk_file_selection_new (_("File to save to"));
	gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
	g_object_set_data (G_OBJECT (dlg), "conf", conf);
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->cancel_button), "clicked", 
			  G_CALLBACK (file_open_dlg_cancel_cb), dlg);	/* ! */
	g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button), "clicked",
			  G_CALLBACK (file_saveas_dlg_ok_cb), dlg);
	gtk_widget_show (dlg);
}


static void
file_saveas_dlg_ok_cb (GtkWidget * widget, GtkWidget *data)
{
	ConfManager *conf;
	gchar *txt, *filetxt;

	conf = (ConfManager *) g_object_get_data (G_OBJECT (data),
						  "conf");
	filetxt =
		g_strdup (gtk_file_selection_get_filename
			  (GTK_FILE_SELECTION (data)));
	gtk_widget_destroy (GTK_WIDGET (data));

	if (g_file_test (filetxt, G_FILE_TEST_IS_DIR)) {
		txt = g_strdup_printf (_("File '%s' is a directory!\n"),
				       filetxt);
		gnome_error_dialog (txt);
		g_free (txt);
		return;
	}
	else {
		if (conf->working_file)
			g_free (conf->working_file);
		conf->working_file = filetxt;
		txt = conf_manager_get_title (conf);
		gtk_window_set_title (GTK_WINDOW (conf->app), txt);
		g_free (txt);
		file_save_cb (NULL, conf);
	}
	
	/* we set the following flag back to FALSE */
	conf->close_after_saving = FALSE;
}

void
quit_cb (GtkWidget * widget, ConfManager * conf)
{
	gint answer;
	gchar *str;
	GtkWidget *dialog, *label, *hb, *vb, *pixmap;

	/* ask if the user really wants to quit */
	if (server_access_is_open (conf->srv) && !conf->save_up_to_date) {
		dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app), 0,
						      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
						      _("Quit without saving"), 2,
						      _("Save and Quit"), 1, 
						      NULL);

		label = gtk_label_new (NULL);
		str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
				       _("Do you want to save before quitting?"),
				       _("Some data have not yet been saved and will be lost\n"
					 "if not saved before quitting."));
		gtk_label_set_markup (GTK_LABEL (label), str);
		g_free (str);

	}
	else {
		dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app), 0,
						      GTK_STOCK_NO, GTK_RESPONSE_CANCEL,
						      GTK_STOCK_YES, 2, NULL);

		label = gtk_label_new (NULL);
		str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>",
				       _("Do you really want to quit?"));
		gtk_label_set_markup (GTK_LABEL (label), str);
		g_free (str);
	}
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);

	hb = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hb, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (hb);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, FALSE, FALSE, GNOME_PAD/2.);
	gtk_widget_show (vb);

	pixmap = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_DIALOG);
	gtk_box_pack_start (GTK_BOX (vb), pixmap, FALSE, FALSE, GNOME_PAD/2.);
	gtk_widget_show (pixmap);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hb), vb, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (vb);

	gtk_box_pack_start (GTK_BOX (vb), label, TRUE, TRUE, GNOME_PAD/2.);
	gtk_widget_show (label);

	answer = gtk_dialog_run (GTK_DIALOG (dialog));

	switch (answer) {
	case 1: /* Save and quit */
		file_save_cb (NULL, conf);
		if (!conf->working_file)
			conf->quit_after_saving = TRUE;
	case 2: /* just quit */
		/* if we've got a pending transaction, ask the user what to do */
		if (conf->current_transaction) {
			if (gnome_db_yes_no_dialog (dialog,
						    _("A transaction is pending. Would you like to "
						      "commit the changes you did (select Yes) or "
						      "discard them (select No)?"))) {
				gda_connection_commit_transaction (conf->srv->cnc, conf->current_transaction);
			} else
				gda_connection_rollback_transaction (conf->srv->cnc, conf->current_transaction);

			g_object_unref (G_OBJECT (conf->current_transaction));
			conf->current_transaction = NULL;
		}

		conf->save_up_to_date = TRUE;
		if (!conf->quit_after_saving)
			quit_real (conf);
		break;
	default:
		break;
	}
	gtk_widget_destroy (dialog);
}

static void 
quit_real (ConfManager * conf)
{
	/* saving state of the configuration */
	settings_set_plugins_dir (conf->plugins_dir);

	if (!conf->save_up_to_date) /* ALL data will be lost */
		conf->save_up_to_date = TRUE;
	
	/* clears everything */
	g_object_unref (G_OBJECT (conf));

	/* and then quit */
	gtk_main_quit ();
}


/********************************************************************
 *
 * CBs to signals emitted by Database objects
 *
 ********************************************************************/


/*
 * sql_server_conn_open_cb
 */

static void xml_parser_error_func (void *ctx, const char *msg, ...);
static void
xml_parser_error_func (void *ctx, const char *msg, ...)
{
	static gboolean has_error = FALSE;
	xmlValidCtxtPtr ctxt = (xmlValidCtxtPtr) ctx;
	ConfManager *conf;
	va_list args;
	gchar *str;

	va_start (args, msg);
	str = g_strdup_vprintf (msg, args);
	va_end (args);
	g_print ("XML ERROR: %s\n", str);
	g_free (str);

	if (!has_error) {
		GtkWidget *dlg;
		conf = CONF_MANAGER (ctxt->userData);
		va_start (args, msg);
		str = g_strdup_vprintf (msg, args);
		va_end (args);

		dlg = gtk_message_dialog_new (GTK_WINDOW (conf->app),
					      GTK_DIALOG_MODAL,
					      GTK_MESSAGE_ERROR,
					      GTK_BUTTONS_CLOSE,
					      _("The XML file has the following error:\n%s"),
					      str);
		g_free (str);

		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
		has_error = TRUE;
	}
}

/* from the server to tell the connection is UP */
void
sql_server_conn_open_cb (GObject * wid, ConfManager * conf)
{
	xmlDocPtr doc;
	xmlNodePtr tree;
	gboolean ok, csud, res;
	gchar *txt;

	conf->loading_in_process = TRUE;
	gnome_app_flash (GNOME_APP (conf->app),
			 _("Connection to the SQL server opened"));
	csud = conf->save_up_to_date;

	/* try to load the structure from an XML file if it exists */
	if (conf->working_file && g_file_test (conf->working_file, G_FILE_TEST_EXISTS)) {
		/* 
		 * load the XML file 
		 */

		ok = TRUE;
		xmlDoValidityCheckingDefaultValue = 0;
		doc = xmlParseFile (conf->working_file);
		if (doc) {
			xmlValidCtxtPtr validc;
			gint i;
			validc = g_new0 (xmlValidCtxt, 1);
			validc->userData = conf;
			validc->error = xml_parser_error_func;
			validc->warning = xml_parser_error_func;
			xmlDoValidityCheckingDefaultValue = 1;
			i = xmlValidateDocument (validc, doc);
			if (!i) {
				xmlFreeDoc (doc);
				doc = NULL;
			}
		}
		
		if (doc) {
			tree = xmlDocGetRootElement(doc)->xmlChildrenNode;
			while (tree) {
				txt = mergeant_xml_clean_string (xmlNodeGetContent (tree));
				/*
				 * ServerAccess object
				 */
				if (!strcmp (tree->name, "SERVER")) {
					res = server_access_build_from_xml_tree (conf->srv, tree);
					ok = res ? ok : FALSE;
				}

				/*
				 * Database object
				 */
				if (!strcmp (tree->name, "DATABASE")) {
					res = database_build_db_from_xml_tree (conf->db, tree);
					ok = res ? ok : FALSE;
				}


				/*
				 * Query Objects
				 */
				if (!strcmp (tree->name, "Query")) {
					/* conf->top_query already exists because the
					   connection is opened */
					Query *q;
					q = QUERY (query_build_from_xml_tree (conf, tree, NULL));
					ok = q ? ok : FALSE;
				}

				/*
				 * QueryEnv Objects
				 */
				if (!strcmp (tree->name, "FORMS")) {
					xmlNodePtr node;

					node = tree->xmlChildrenNode;
					while (node) {
						if (!strcmp (node->name, "QueryEnv"))
							query_env_build_from_xml_tree (conf, node);
						node = node->next;
					}
				}

				/*
				 *  GUI parts
				 */
				if (!strcmp (tree->name, "GUI")) {
					xmlNodePtr node;

					node = tree->xmlChildrenNode;
					while (node) {
						if (!strcmp (node->name, "RelShip"))
							relship_build_from_xml_tree (conf, node);
						node = node->next;
					}
				}


				tree = tree->next;
				if (txt)
					g_free (txt);
			}
			xmlFreeDoc (doc);
			conf->save_up_to_date = csud;
		}
		else
			ok = FALSE;

		if (!ok)
			gnome_warning_dialog (_
					      ("An error has occured while loading the "
					       "selected document. The file (XML format)\n"
					       "is probably corrupted. Try to repair it."));
		else {
			/* ask to refresh the structure */
			GtkWidget *dlg, *label, *hb, *pixmap;
			gint answer;
			gchar *str;

			dlg = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW (conf->app),
							   0,
							   GTK_STOCK_NO, GTK_RESPONSE_NO,
							   GTK_STOCK_YES, GTK_RESPONSE_YES, NULL);
			

			label = gtk_label_new (NULL);
			str = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
					       _("Synchronize with the database structure?"),
					       _("To update the list of data types,"
						 " tables, functions, "
						 "sequences, ...\nfrom the server (recommended "
						 "if the structure of the DB has been "
						 "modified).\n"
						 "Say Yes if unsure."));
			gtk_label_set_markup (GTK_LABEL (label), str);
			g_free (str);

			hb = gtk_hbox_new (FALSE, GNOME_PAD);
			gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hb, TRUE, TRUE, GNOME_PAD);
			gtk_widget_show (hb);

			pixmap = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION, GTK_ICON_SIZE_DIALOG);
			gtk_box_pack_start (GTK_BOX (hb), pixmap, TRUE, TRUE, GNOME_PAD/2.);
			gtk_widget_show (pixmap);

			gtk_box_pack_start (GTK_BOX (hb), label, TRUE, TRUE, GNOME_PAD/2.);
			gtk_widget_show (label);

			answer = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			if (answer == GTK_RESPONSE_YES) 
				sql_mem_update_cb (NULL, conf);
		}
	}
	else {
		/* automatically update the database structure */
		sql_mem_update_cb (NULL, conf);
	}


	/* sets the notebook to be displayed */
	if (conf->srv->features.sequences)
		gtk_widget_show (conf->sequences_page);
	else
		gtk_widget_hide (conf->sequences_page);

	gtk_widget_hide (conf->welcome);
	gtk_widget_show (conf->working_box);

	conf->loading_in_process = FALSE;
}



/*
 * sql_server_conn_to_close_cb
 */

/* from the server to tell the connection is going to go DOWN */
void
sql_server_conn_to_close_cb (GObject * wid, ConfManager * conf)
{
	if (conf->check_dlg) {
		conf->check_perform = FALSE;
		gtk_widget_destroy (GTK_WIDGET (conf->check_dlg));
		conf->check_dlg = NULL;
		conf->check_pbar = NULL;
		conf->check_link_name = NULL;
		conf->check_errors = NULL;
	}
}


/*
 * sql_server_conn_close_cb
 */

/* from the server to tell the connection is DOWN */
void
sql_server_conn_close_cb (GObject * wid, ConfManager * conf)
{
	gnome_app_flash (GNOME_APP (conf->app),
			 _("Connection to the SQL server closed"));

	/* sets the welcome message to be displayed */
	gtk_widget_hide (conf->working_box);
	gtk_widget_show (conf->welcome);

	/* Tell that the work has not been modified (like when starting Mergeant) */
	conf->save_up_to_date = TRUE;
}




/*
 * sql_server_event_cb
 */
void sql_server_event_cb (ServerAccess *srv, GdaConnection *cnc,
			  GdaClientEvent event, GdaParameterList *params,
			  ConfManager * conf)
{
	switch (event) {
	case GDA_CLIENT_EVENT_ERROR :
		if (!conf->error_dlg) {
			conf->error_dlg = gnome_db_error_dialog_new (_("Error Viewer"));
			g_object_add_weak_pointer (G_OBJECT (conf->error_dlg), (gpointer) & (conf->error_dlg));
		}
		else 
			gdk_window_raise (GTK_WIDGET (conf->error_dlg)->window);

		gnome_db_error_dialog_show_errors (GNOME_DB_ERROR_DIALOG (conf->error_dlg),
						   gda_connection_get_errors (cnc));
		break;
	default :
	}
}
