/* settings.c
 *
 * Copyright (C) 2002 Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gconf/gconf.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtktable.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomedb/gnome-db-config.h>
#include <libgnomedb/gnome-db-gray-bar.h>
#include <libgnomedb/gnome-db-stock.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-shortcut.h"
#include "settings.h"
#include "settings-utils.h"

static GtkWidget *settings_dialog = NULL;
static GtkWidget *settings_notebook = NULL;

/*
 * Interface settings tab
 */

static GtkWidget *
interface_settings_new (void)
{
	gchar *str;
	GtkWidget *table, *box, *button, *label;

	table = gnome_db_new_table_widget (3, 10, FALSE);

	/*
	 * Main window options
	 */
	label = gnome_db_new_label_widget ("");
	str = g_strdup_printf ("<b>%s</b>", _("Main window"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_table_attach (GTK_TABLE (table), label, 0, 3, 0, 1, GTK_FILL, GTK_FILL, 2, 6);

	button = settings_utils_new_check_button (_("Remember window size"), ENTRY_REMEMBER_WINDOW_SIZE);
	gtk_table_attach (GTK_TABLE (table), button, 1, 3, 1, 2, GTK_FILL, GTK_FILL, 15, 2);

	button = settings_utils_new_check_button (_("Show shortcut bar"), ENTRY_SHOW_SHORTCUT_BAR);
	gtk_table_attach (GTK_TABLE (table), button, 1, 3, 2, 3, GTK_FILL, GTK_FILL, 15, 2);

	/*
	 * Source editor options
	 */
	label = gnome_db_new_label_widget ("");
	str = g_strdup_printf ("<b>%s</b>", _("Source editor"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_table_attach (GTK_TABLE (table), label, 0, 3, 3, 4, GTK_FILL, GTK_FILL, 2, 6);

	button = settings_utils_new_check_button (_("Highlight syntax"), GNOME_DB_CONFIG_KEY_EDITOR_HIGHLIGHT);
	gtk_table_attach (GTK_TABLE (table), button, 1, 3, 4, 5, GTK_FILL, GTK_FILL, 15, 2);

	button = settings_utils_new_check_button (_("Show line numbers"), GNOME_DB_CONFIG_KEY_EDITOR_SHOW_LINE_NUMBERS);
	gtk_table_attach (GTK_TABLE (table), button, 1, 3, 5, 6, GTK_FILL, GTK_FILL, 15, 2);

	label = gnome_db_new_label_widget (_("Tab stops"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 6, 7, GTK_FILL, GTK_FILL, 15, 2);
	button = settings_utils_new_integer_entry (GNOME_DB_CONFIG_KEY_EDITOR_TAB_STOP);
	gtk_table_attach (GTK_TABLE (table), button, 2, 3, 6, 7, GTK_FILL, GTK_FILL, 15, 2);

	return table;
}

/*
 * Database settings tab
 */

static GtkWidget *
database_settings_new (ConfManager *conf)
{
	gchar *str;
	GtkWidget *table, *cb, *label;

	table = gnome_db_new_table_widget (9, 3, FALSE);

	/*
	 * Connection preferences
	 */
	label = gnome_db_new_label_widget ("");
	str = g_strdup_printf ("<b>%s</b>", _("Connection preferences"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_table_attach (GTK_TABLE (table), label, 0, 3, 0, 1, GTK_FILL, GTK_FILL, 2, 6);

	label = gnome_db_new_label_widget (_("Datasource"));
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 15, 2);
	cb = settings_utils_new_dsn_selector (ENTRY_DEFAULT_DATASOURCE);
	gtk_table_attach (GTK_TABLE (table), cb, 2, 3, 1, 2, GTK_FILL, GTK_FILL, 15, 2);
	
	label = gnome_db_new_label_widget (_("Username"));
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 15, 2);
	cb = gnome_db_new_entry_widget (0, TRUE);
	if (conf->srv->user_name != NULL)
		gtk_entry_set_text (GTK_ENTRY (cb), conf->srv->user_name->str);
	gtk_table_attach (GTK_TABLE (table), cb, 2, 3, 2, 3, GTK_FILL, GTK_FILL, 15, 2);

	label = gnome_db_new_label_widget (_("Password"));
	gtk_table_attach (GTK_TABLE (table), label, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 15, 2);
	cb = gnome_db_new_entry_widget (0, TRUE);
	gtk_entry_set_visibility (GTK_ENTRY (cb), FALSE);
	if (conf->srv->password != NULL)
		gtk_entry_set_text (GTK_ENTRY (cb), conf->srv->password->str);
	gtk_table_attach (GTK_TABLE (table), cb, 2, 3, 3, 4, GTK_FILL, GTK_FILL, 15, 2);

	/*
	 * Database confirmation options
	 */
	label = gnome_db_new_label_widget ("");
	str = g_strdup_printf ("<b>%s</b>", _("Confirmation"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_table_attach (GTK_TABLE (table), label, 0, 3, 4, 5, GTK_FILL, GTK_FILL, 2, 6);

	cb = settings_utils_new_check_button (_("Ask for confirmation on SELECT statements"),
					      ENTRY_SELECT_CONFIRMATION_DIALOG);
	gtk_table_attach (GTK_TABLE (table), cb, 1, 3, 5, 6, GTK_FILL, GTK_FILL, 15, 2);

	cb = settings_utils_new_check_button (_("Ask for confirmation on INSERT statements"),
					      ENTRY_INSERT_CONFIRMATION_DIALOG);
	gtk_table_attach (GTK_TABLE (table), cb, 1, 3, 6, 7, GTK_FILL, GTK_FILL, 15, 2);

	cb = settings_utils_new_check_button (_("Ask for confirmation on UPDATE statements"),
					      ENTRY_UPDATE_CONFIRMATION_DIALOG);
	gtk_table_attach (GTK_TABLE (table), cb, 1, 3, 7, 8, GTK_FILL, GTK_FILL, 15, 2);

	cb = settings_utils_new_check_button (_("Ask for confirmation on DELETE statements"),
					      ENTRY_DELETE_CONFIRMATION_DIALOG);
	gtk_table_attach (GTK_TABLE (table), cb, 1, 3, 8, 9, GTK_FILL, GTK_FILL, 15, 2);

	return table;
}

/*
 * Settings dialog implementation
 */

static void
dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data)
{
	gtk_widget_hide (settings_dialog);
}

void
settings_dialog_display (GtkWidget *parent, ConfManager *conf)
{
	GtkWidget *box;

	if (!settings_dialog) {
		/* create the dialog */
		settings_dialog = gtk_dialog_new_with_buttons (
			_("Mergeant Preferences"), parent, 0,
			GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
			NULL);
		gtk_dialog_set_has_separator (GTK_DIALOG (settings_dialog), FALSE);
		gnome_window_icon_set_from_file (GTK_WINDOW (settings_dialog),
						 PIXMAPDIR "/mergeant.png");
		g_signal_connect (G_OBJECT (settings_dialog), "response",
				  G_CALLBACK (dialog_response_cb), NULL);

		settings_notebook = gnome_db_new_notebook_widget ();
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (settings_notebook), TRUE);
		gtk_notebook_set_show_border (GTK_NOTEBOOK (settings_notebook), FALSE);

		gtk_notebook_append_page (GTK_NOTEBOOK (settings_notebook),
					  interface_settings_new (),
					  gtk_label_new (_("Interface")));
		gtk_notebook_append_page (GTK_NOTEBOOK (settings_notebook),
					  database_settings_new (conf),
					  gtk_label_new (_("Database")));
		
		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (settings_dialog)->vbox), settings_notebook, TRUE, TRUE, GNOME_PAD);
	}

	gtk_widget_show (settings_dialog);
}

/*
 * Accessor functions for configuration settings
 */

gchar *
settings_get_default_datasource (void)
{
	return gnome_db_config_get_string (ENTRY_DEFAULT_DATASOURCE);
}

void
settings_set_default_datasource (const gchar *ds)
{
	gnome_db_config_set_string (ENTRY_DEFAULT_DATASOURCE, ds);
}

gchar *
settings_get_plugins_dir (void)
{
	return gnome_db_config_get_string (ENTRY_PLUGINS_DIR);
}

void
settings_set_plugins_dir (const gchar *dir)
{
	gnome_db_config_set_string (ENTRY_PLUGINS_DIR, dir);
}

gboolean
settings_get_delete_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_DELETE_CONFIRMATION_DIALOG);
}

void
settings_set_delete_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_DELETE_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_insert_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_INSERT_CONFIRMATION_DIALOG);
}

void
settings_set_insert_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_INSERT_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_select_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_SELECT_CONFIRMATION_DIALOG);
}

void
settings_set_select_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_SELECT_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_update_confirmation_dialog (void)
{
	return gnome_db_config_get_boolean (ENTRY_UPDATE_CONFIRMATION_DIALOG);
}

void
settings_set_update_confirmation_dialog (gboolean confirm)
{
	gnome_db_config_set_boolean (ENTRY_UPDATE_CONFIRMATION_DIALOG, confirm);
}

gboolean
settings_get_remember_window_size (void)
{
	return gnome_db_config_get_boolean (ENTRY_REMEMBER_WINDOW_SIZE);
}

void
settings_set_remember_window_size (gboolean remember)
{
	gnome_db_config_set_boolean (ENTRY_REMEMBER_WINDOW_SIZE, remember);
}

gint
settings_get_window_width (void)
{
	return gnome_db_config_get_int (ENTRY_WINDOW_WIDTH);
}

void
settings_set_window_width (gint width)
{
	gnome_db_config_set_int (ENTRY_WINDOW_WIDTH, width);
}

gint
settings_get_window_height (void)
{
	return gnome_db_config_get_int (ENTRY_WINDOW_HEIGHT);
}

void
settings_set_window_height (gint height)
{
	gnome_db_config_set_int (ENTRY_WINDOW_HEIGHT, height);
}

gboolean
settings_get_show_shortcut_bar (void)
{
	return gnome_db_config_get_boolean (ENTRY_SHOW_SHORTCUT_BAR);
}
