/* mg-work-widget.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-base.h"
#include "mg-conf.h"
#include "mg-context.h"
#include "mg-qfield.h"
#include "mg-entity.h"
#include "mg-parameter.h"
#include "mg-work-widget.h"


/* signals */
enum
{
	LAST_SIGNAL
};

static gint mg_work_widget_signals[LAST_SIGNAL] = { };

static void mg_work_widget_iface_init (gpointer g_class);

GType
mg_work_widget_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgWorkWidgetIface),
			(GBaseInitFunc) mg_work_widget_iface_init,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) NULL,
			NULL,
			NULL,
			0,
			0,
			(GInstanceInitFunc) NULL
		};
		
		type = g_type_register_static (G_TYPE_INTERFACE, "MgWorkWidget", &info, 0);
		g_type_interface_add_prerequisite (type, G_TYPE_OBJECT);
	}
	return type;
}


static void
mg_work_widget_iface_init (gpointer g_class)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		initialized = TRUE;
	}
}


/**
 * mg_work_widget_run
 * @iface: an object which implements the #MgWorkWidget interface
 * @mode: an OR'ed collection of flags from #MgEnumsMode
 * 
 * Makes the widget run the queries it has to run in order to display
 * usefull data. Havin a separate step to "run" the widget allows to
 * have the execution context related to other events.
 */
void
mg_work_widget_run (MgWorkWidget *iface, guint mode)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->run)
		(MG_WORK_WIDGET_GET_IFACE (iface)->run) (iface, mode);
}

/**
 * mg_work_widget_set_mode
 * @iface: an object which implements the #MgWorkWidget interface
 * @mode: an OR'ed collection of flags from #MgEnumsMode
 * 
 * Changes the mode that was set when mg_work_widget_run() was called.
 */
void
mg_work_widget_set_mode (MgWorkWidget *iface, guint mode)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->set_mode)
		(MG_WORK_WIDGET_GET_IFACE (iface)->set_mode) (iface, mode);
}

/**
 * mg_work_widget_set_entry_editable
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * @editable:
 *
 * Sets if the data entry in the @iface widget corresponding to @field can be edited or not.
 */
void 
mg_work_widget_set_entry_editable (MgWorkWidget *iface, MgQfield *field, gboolean editable)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->set_entry_editable)
		(MG_WORK_WIDGET_GET_IFACE (iface)->set_entry_editable) (iface, field, editable);	
}

/**
 * mg_work_widget_show_entry_actions
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * @show_actions
 * 
 * Control if the actions buttons the @iface widget offers are displayed or not.
 */
void
mg_work_widget_show_entry_actions (MgWorkWidget *iface, MgQfield *field, gboolean show_actions)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->show_entry_actions)
		(MG_WORK_WIDGET_GET_IFACE (iface)->show_entry_actions) (iface, field, show_actions);
}


/**
 * mg_work_widget_show_global_actions
 * @iface: an object which implements the #MgWorkWidget interface
 * @show_actions
 * 
 * Control if the actions buttons the @iface widget offers are displayed or not.
 */
void
mg_work_widget_show_global_actions (MgWorkWidget *iface, gboolean show_actions)
{
	g_return_if_fail (iface && IS_MG_WORK_WIDGET (iface));

	if (MG_WORK_WIDGET_GET_IFACE (iface)->show_global_actions)
		(MG_WORK_WIDGET_GET_IFACE (iface)->show_global_actions) (iface, show_actions);
}


/**
 * mg_work_widget_get_param_for_field
 * @iface: an object which implements the #MgWorkWidget interface
 * @field: a #MgQfield object which belongs to the #MgQuery which @iface uses
 * 
 * Get the #MgParameter object corresponding to @field, as created by the @iface
 * object.
 */
MgParameter *
mg_work_widget_get_param_for_field (MgWorkWidget *iface, MgQfield *field)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_param_for_field) (iface, field);
	return NULL;
}

/**
 * mg_work_widget_has_been_changed
 * @iface: an object which implements the #MgWorkWidget interface
 * 
 * Tells if the user has changed any of the data displayed by @iface since
 * the last refresh.
 */
gboolean
mg_work_widget_has_been_changed (MgWorkWidget *iface)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), FALSE);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->has_been_changed)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->has_been_changed) (iface);
	return FALSE;
}

/**
 * mg_work_widget_get_exec_context
 * @iface: an object which implements the #MgWorkWidget interface
 * 
 * Get the #MgContext which represents the necessary parameters
 * to be given a value before the #MgQuery used by @iface can be run.
 */
MgContext *
mg_work_widget_get_exec_context (MgWorkWidget *iface)
{
	g_return_val_if_fail (iface && IS_MG_WORK_WIDGET (iface), NULL);

	if (MG_WORK_WIDGET_GET_IFACE (iface)->get_exec_context)
		return (MG_WORK_WIDGET_GET_IFACE (iface)->get_exec_context) (iface);

	return NULL;
}


/**
 * mg_work_widget_bind_to_work_widget
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @dest_field_xml_id:
 * @source_iface: an object which implements the #MgWorkWidget interface
 * @source_field_xml_id:
 *
 * Retreives the #MgParameter objects for the @dest_field_xml_id and @source_field_xml_id
 * and bind them together: when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_work_widget (MgWorkWidget *dest_iface, const gchar *dest_field_xml_id, 
				    MgWorkWidget *source_iface, const gchar *source_field_xml_id)
{
	gchar *str, *ptr, *tok;
	MgQuery  *query_dest, *query_src;
	MgParameter *param_dest, *param_src;
	MgContext *context;
	MgConf *conf;
	MgQfield *field;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_iface && IS_MG_WORK_WIDGET (source_iface));
	g_return_if_fail (dest_field_xml_id && *dest_field_xml_id);
	g_return_if_fail (source_field_xml_id && *source_field_xml_id);
	
	/* find the destination parameter:
	 * it MUST be in the Exec context of the SELECT query used by the widget, 
	 * otherwise it's stupid and dangerous.
	 */
	context = mg_work_widget_get_exec_context (dest_iface);
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (dest_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_dest = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_dest);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_dest), dest_field_xml_id));
	g_return_if_fail (field);
	param_dest = mg_work_widget_get_param_for_field (dest_iface, field);
	g_return_if_fail (param_dest);
	g_return_if_fail (g_slist_find (context->parameters, param_dest));

	/* find the source parameter */
	context = mg_work_widget_get_exec_context (source_iface);
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (source_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_src = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_src);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_src), source_field_xml_id));
	g_return_if_fail (field);
	param_src = mg_work_widget_get_param_for_field (source_iface, field);
	g_return_if_fail (param_src);

	/* bind the two parameters */
	mg_parameter_bind_to_param (param_dest, param_src);
}

/**
 * mg_work_widget_bind_to_context
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @dest_field_xml_id:
 * @source_context: a #MgContext object
 * @source_field_xml_id:
 *
 * Retreives the #MgParameter objects for the @dest_field_xml_id and @source_field_xml_id
 * and bind them together: when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_context (MgWorkWidget *dest_iface, const gchar *dest_field_xml_id,
				MgContext *source_context, const gchar *source_field_xml_id)
{
	gchar *str, *ptr, *tok;
	MgQuery  *query_dest, *query_src;
	MgParameter *param_dest, *param_src;
	MgContext *context;
	MgConf *conf;
	MgQfield *field;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_context && IS_MG_CONTEXT (source_context));
	g_return_if_fail (dest_field_xml_id && *dest_field_xml_id);
	g_return_if_fail (source_field_xml_id && *source_field_xml_id);
	
	/* find the destination parameter:
	 * it MUST be in the Exec context of the SELECT query used by the widget, 
	 * otherwise it's stupid and dangerous.
	 */
	context = mg_work_widget_get_exec_context (dest_iface);
	conf = mg_base_get_conf (MG_BASE (context));
	str = g_strdup (dest_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_dest = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_dest);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_dest), dest_field_xml_id));
	g_return_if_fail (field);
	param_dest = mg_work_widget_get_param_for_field (dest_iface, field);
	g_return_if_fail (param_dest);
	g_return_if_fail (g_slist_find (context->parameters, param_dest));

	/* find the source parameter */
	conf = mg_base_get_conf (MG_BASE (source_context));
	str = g_strdup (source_field_xml_id);
	ptr = strtok_r (str, ":", &tok);
	g_return_if_fail (ptr);
	query_src = mg_conf_get_query_by_xml_id (conf, ptr);
	g_free (str);
	g_return_if_fail (query_src);
	field = MG_QFIELD (mg_entity_get_field_by_xml_id (MG_ENTITY (query_src), source_field_xml_id));
	g_return_if_fail (field);
	param_src = mg_context_find_parameter_for_field (source_context, field);
	g_return_if_fail (param_src);

	/* bind the two parameters */
	mg_parameter_bind_to_param (param_dest, param_src);
}

/**
 * mg_work_widget_bind_to_context_all
 * @dest_iface: an object which implements the #MgWorkWidget interface
 * @source_context: a #MgContext object
 *
 * For each #MgParameter of @source_context, retreives the corresponding #MgParameter 
 * in @dest_iface's exec. context, and bind the two together:
 * when the 'source' parameter changes, the 'dest' one takes
 * the value of the 'source' one.
 */
void
mg_work_widget_bind_to_context_all (MgWorkWidget *dest_iface, MgContext *source_context)
{
	GSList *list, *dest_fields;
	MgParameter *param_dest, *param_src;

	g_return_if_fail (dest_iface && IS_MG_WORK_WIDGET (dest_iface));
	g_return_if_fail (source_context && IS_MG_CONTEXT (source_context));

	list = source_context->parameters;
	while (list) {
		param_src = MG_PARAMETER (list->data);
		dest_fields = mg_parameter_get_dest_fields (param_src);
		param_dest = NULL;
		while (dest_fields && !param_dest) {
			param_dest = mg_work_widget_get_param_for_field (dest_iface, MG_QFIELD (dest_fields->data));
			dest_fields = g_slist_next (dest_fields);
		}

		if (param_dest) 
			/* bind the two parameters */
			mg_parameter_bind_to_param (param_dest, param_src);
		
		list = g_slist_next (list);
	}
}
