/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *Copyright 2001-2002 dodji SEKETELI, Gal CHAMOULAUD.
 *http://www.freespiders.org
 */


/**
 *@file
 *This widget is the visual tree editor.
 *It can load an xml tree, 
 *visualize it as a tree and allow 
 *the user to edit the tree.
 *Each time one of the nodes of the 
 *tree is selected, cut, copied etc, 
 *this widget emits signals
 *that can be caught and handler by the caller. 
 *See mlview-tree-editor.h for the declaration of the
 *pulic services offered by this widget.
 *
 */

#include <string.h>
#include <stdlib.h>

#include "mlview-tree-editor.h"
#include "mlview-node-type-picker.h"
#include "mlview-global-settings.h"
#include "mlview-utils.h"



#define SEARCH_IN_NAMES_CHECK_BUTTON "search-in-names-check-box"
#define SEARCH_IN_ATTRIBUTES_NAMES_CHECK_BUTTON "search-in-attributes-names-check-box"
#define SEARCH_IN_ATTRIBUTES_VALUES_CHECK_BUTTON "search-in-attributes-values-check-box"
#define SEARCH_IN_CONTENT_CHECK_BUTTON "search-in-content-check-box"
#define SEARCHED_TEXT_ENTRY "searched-text-entry"

#define IS_THE_FIND_DIALOG "is-the-find-dialog"

/*================================================================
 *The struct defining the private attributes of 
 *MlViewTreeEditor and it associated macros
 *=================================================================*/

#define PRIVATE(mlview_tree_editor) ((mlview_tree_editor)->priv)


struct _MlViewTreeEditorPrivate {
        /*the xml document being edited. FIXME => handle it lifetime !!! */
        xmlDocPtr xml_doc;

        MlViewXMLDocument *mlview_xml_doc;

        /*the viewable tree that matches xml_doc */
        GtkCTree *viewable_doc;

        /*the menu poped up when the user right clicks on an xml_node */
        GtkWidget *xml_node_popup_menu;

        GtkCTreeNode *current_selected_node;

        /*the node type picker used when creating a new node. 
         *MlViewTreeEditor does not have to handle it lifetime
         */
        MlViewNodeTypePicker *node_type_picker;

        MlViewAppContext *app_context;
};


enum {
        TREE_CHANGED = 1,
        NODE_CUT,
        NODE_PASTED,
        NODE_ADDED,
        NODE_SELECTED,
        NUMBER_OF_SIGNALS
};

static const gchar *p_tree_editor_titles[] = {
        N_("the xml document"),
        N_("the xml document tags")
};

#define MLVIEW_TREE_EDITOR_NODE_CLIPBOARD_SIZE 128
static const gint TREE_EDITOR_NUMBER_OF_COLUMNS = 1;
static const gint TREE_EDITOR_TREE_COLUMN = 0;
static const gint TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT = 2;
static const gint MLVIEW_CONTENT_MAX_LEN = 512;
static const gint MLVIEW_TREE_EDITOR_DEFAULT_EXPANSION_DEPTH = 2;

static GtkDialog *p_find_dialog = NULL;


/*===================================================================
 *The edition popup menu definition array.
 *This menu is poped up on an xml visual node.
 *Note that this pop up menu must be dynamicaly build when this widget supports
 *Validation.
 *
 *FIXME: do not use this popup menu when validation is on.
 *Maybe build a dynamic menu using GtkMenu widgetware.
 *==================================================================*/


/*===================================================================
 *declaration of the private functions used and defined by this object.
 *===================================================================*/

static void
 mlview_tree_editor_init_class (MlViewTreeEditorClass * a_klass);

static void
 mlview_tree_editor_init (MlViewTreeEditor * a_editor);


/*
 *helper functions that build the visual 
 *tree from the xml document object model 
 */
static GtkCTreeNode *build_ctree_from_xml_tree (MlViewAppContext
                                                * a_app_context,
                                                xmlNodePtr
                                                a_node,
                                                GtkCTreeNode *
                                                a_parent_node,
                                                GtkCTree **
                                                a_visual_tree_ptr);

static void
 build_ctree_from_xml_doc (MlViewAppContext * a_app_context,
                           xmlDocPtr a_doc,
                           GtkCTree ** a_visual_tree_ptr,
                           gchar * a_doc_name,
                           guint a_expansion_level);



/*
 *helper functions that 
 *build the strings that you see on 
 *the right side of a visual xml node.
 */
static void
 attributes_list_to_string (void *a_attr_node, gchar ** a_result);

static gchar *node_to_start_tag (const xmlNodePtr a_node);


/*
 *helper functions of the 
 *search (find) api
 */
static gint
xml_node_name_contains_string (const xmlNode * a_xml_node,
                               const gchar * a_str,
                               gchar ** a_found_str);

static gint
xml_node_attributes_name_contains_string (const xmlNode *
                                          a_xml_node,
                                          const gchar * a_str,
                                          gchar ** a_found_str);

static gint
xml_node_attributes_value_contains_string (const xmlNode *
                                           a_xml_node,
                                           const gchar * a_str,
                                           gchar ** a_found_str);

static gint
xml_node_content_contains_string (const xmlNode * a_xml_node,
                                  const gchar * a_str,
                                  gchar ** a_found_str);

static enum WhereInTheNodeBitmap
 xml_node_contains_string
        (const xmlNode * a_xml_node,
         const gchar * a_str,
         const enum WhereInTheNodeBitmap a_where_in_node_mask);

static GtkCTreeNode *downward_find_visual_node_that_contains
        (GtkCTree * a_visual_tree,
         GtkCTreeNode * a_starting_from,
         const gchar * a_str,
         const enum WhereInTheNodeBitmap a_where_in_node_bitmap,
         gboolean a_start_after);

static gboolean
mlview_tree_editor_find_dialog_search_in_names_is_on (GtkDialog * a_find_dialog);

static gboolean mlview_tree_editor_find_dialog_search_in_attributes_names_is_on (GtkDialog * a_find_dialog);

static gboolean mlview_tree_editor_find_dialog_search_in_attributes_values_is_on (GtkDialog * a_find_dialog);

static gboolean
mlview_tree_editor_find_dialog_search_in_content_is_on (GtkDialog * a_find_dialog);

static GtkCTreeNode * mlview_tree_editor_get_data_from_find_dialog_and_perform_search (MlViewTreeEditor * a_tree_editor, GtkDialog * a_dialog,
                                                                                       GtkCTreeNode * a_start_from, gboolean a_start_after);


/*
 *callbacks and signal handlers 
 *used to update the state of the 
 *widget when user cliks on a node
 */
static void
mlview_tree_editor_node_selected_default_signal_handler (MlViewTreeEditor * a_editor,
                                                         GtkCTreeNode * a_visual_node, gpointer a_user_data);

static void mlview_tree_editor_tree_select_row_cb (GtkCTree * a_visual_tree,
                                                   GList * a_node, gint a_column, 
                                                   gpointer a_tree_editor);


static gboolean
mlview_tree_editor_event_cb (GtkWidget * a_widget,
                             GdkEvent * a_event,
                             gpointer a_user_data);

static gint
mlview_tree_editor_build_node_insertion_submenu (MlViewTreeEditor * a_editor,
                                                 enum NODE_INSERTION_SCHEME a_insertion_scheme,
                                                 GtkMenu ** a_submenu);

static void
menu_add_child_node_activate_cb (GtkMenuItem * a_menu_item,
                                 gpointer a_user_data);

static void
menu_insert_prev_sibling_node_activate_cb (GtkMenuItem * a_menu_item,
                                            gpointer a_user_data);

static void
menu_insert_next_sibling_node_activate_cb (GtkMenuItem * a_menu_item,
                                            gpointer a_user_data);

/*
 *helper functions called by the 
 *callbacks (controllers) when the user 
 *selects a type of node during node adding as child/sibling
 */
static void
mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_add_child (MlViewTreeEditor * a_tree_editor);

static void mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_insert_sibling_node (MlViewTreeEditor * a_tree_editor);


/*
 *helper function to build, cache the find
 * (search) dialog box and to retrieve info from it
 */
static GtkDialog * mlview_tree_editor_find_dialog_box_retrieve (MlViewAppContext *
                                                                a_context);

static gchar * mlview_tree_editor_find_dialog_get_searched_text (GtkDialog *
                                                                 a_find_dialog);

static guint mlview_tree_editor_signals[NUMBER_OF_SIGNALS] =
        { 0 };
static GtkVBoxClass *parent_class = NULL;


/*==================================================
 *private methods required by GTK typing system
 *
 *==================================================
 */

/**
 *The Gtk standard class initialyzer of the MlViewTreeEditorClass class. 
 *
 */
static void
mlview_tree_editor_init_class (MlViewTreeEditorClass * a_klass)
{
        GtkObjectClass *object_class;

        g_return_if_fail (a_klass != NULL);

        parent_class = g_type_class_peek_parent (a_klass);
        object_class = GTK_OBJECT_CLASS (a_klass);
        g_return_if_fail (object_class);
        object_class->destroy = mlview_tree_editor_destroy;

        /*FIXME:add signals creation */
        mlview_tree_editor_signals[TREE_CHANGED] =
                g_signal_new ("tree-changed",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeEditorClass,
                               tree_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);

        mlview_tree_editor_signals[NODE_CUT] =
                g_signal_new ("node-cut",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeEditorClass, node_cut),
                              NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1,
                              GTK_TYPE_POINTER);

        mlview_tree_editor_signals[NODE_PASTED] =
                g_signal_new ("node-pasted",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeEditorClass,
                               node_pasted), NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1,
                              GTK_TYPE_POINTER);

        mlview_tree_editor_signals[NODE_ADDED] =
                g_signal_new ("node-added",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeEditorClass,
                               node_added), NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1,
                              GTK_TYPE_POINTER);

        mlview_tree_editor_signals[NODE_SELECTED] =
                g_signal_new ("node-selected",
                              G_TYPE_FROM_CLASS (object_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeEditorClass,
                               node_selected), NULL, NULL,
                              gtk_marshal_NONE__POINTER,
                              GTK_TYPE_NONE, 1,
                              GTK_TYPE_POINTER);

        a_klass->tree_changed = NULL;
        a_klass->node_cut = NULL;
        a_klass->node_added = NULL;
        a_klass->node_pasted = NULL;
        a_klass->node_selected =
                mlview_tree_editor_node_selected_default_signal_handler;
}


/**
 *The instance initialyzer of the MlViewTreeEditor. 
 *
 */
static void
mlview_tree_editor_init (MlViewTreeEditor * a_editor)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (PRIVATE (a_editor) == NULL);

        PRIVATE (a_editor) =
                g_malloc0 (sizeof (MlViewTreeEditorPrivate));
}


/*==============================================
 *private helper functions
 *==============================================
 */

/**
 *Helper function. Given an attributes node,
 * builds the string "attr1=val1 ... attrN=valN"
 *Note that this function is recursive. 
 *
 */
static void
attributes_list_to_string (void *a_attr_node, gchar ** a_result)
{
        xmlAttrPtr xml_attr = (xmlAttrPtr) a_attr_node;
        xmlNodePtr xml_node = (xmlNodePtr) a_attr_node;

        static int num_of_use = 0;

        if (num_of_use++ == 0)
                *a_result = NULL;

        if (a_attr_node == NULL)
                return;

        if (xml_attr->type == XML_ATTRIBUTE_NODE) {
                gchar *tmp_str = *a_result,
                        *name;

                if (xml_attr->ns != NULL
                    && xml_attr->ns->prefix != NULL) {
                        name = g_strconcat (xml_attr->ns->prefix,
                                            ":", xml_attr->name,
                                            NULL);
                } else {
                        name = g_strdup (xml_attr->name);
                }
                if (tmp_str == NULL)
                        *a_result = g_strdup (name);
                else
                        *a_result =
                                g_strconcat (tmp_str, " ", name,
                                             NULL);
                if (tmp_str) {
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
                if (name) {
                        g_free (name);
                        name = NULL;
                }
                if (xml_attr->children)
                        attributes_list_to_string (xml_attr->
                                                   children,
                                                   a_result);

                if (xml_attr->next)
                        attributes_list_to_string (xml_attr->
                                                   next,
                                                   a_result);
        } else if (xml_node->type == XML_TEXT_NODE) {

                gchar *tmp_str = *a_result;

                if (tmp_str) {

                        *a_result =
                                g_strconcat (tmp_str, "=\"",
                                             xml_node->content,
                                             "\"", NULL);
                        g_free (tmp_str);
                        tmp_str = NULL;
                }
        }
}

/**
 *Helper function. Given an xml node, 
 *builds the string "<tagname attr1=val1, attr2=val2 ... attrN=valN>" or 
 *
 *@return the content if it a xml text node. 
 */
static gchar *
node_to_start_tag (const xmlNodePtr a_node)
{
        guchar *result = NULL,
                *content = NULL;

        g_return_val_if_fail (a_node != NULL, NULL);

        if (a_node->type == XML_ELEMENT_NODE) {

                guchar *utf8_ns_prefix = NULL,
                        *ns_prefix = NULL,
                        *name = NULL,
                        *tmp_str = NULL;

                gchar *attr_str = NULL;

                attributes_list_to_string (a_node->properties,
                                           &attr_str);

                if (a_node->ns != NULL && a_node->ns->prefix) {
                        enum MlViewStatus status = MLVIEW_OK;

                        utf8_ns_prefix =
                                g_strconcat (a_node->ns->prefix,
                                             ":", NULL);

                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_ns_prefix, &ns_prefix);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);

                } else {
                        ns_prefix = NULL;
                }

                if (ns_prefix) {
                        enum MlViewStatus status = MLVIEW_OK;

                        status = mlview_utils_utf8_str_to_isolat1
                                ((guchar *) a_node->name,
                                 &tmp_str);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);
                        g_return_val_if_fail (tmp_str != NULL,
                                              NULL);

                        name = g_strconcat (ns_prefix, tmp_str,
                                            NULL);
                } else {
                        enum MlViewStatus status = MLVIEW_OK;

                        status = mlview_utils_utf8_str_to_isolat1
                                ((guchar *) a_node->name, &name);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);
                }

                if (ns_prefix) {
                        g_free (ns_prefix);
                        ns_prefix = NULL;
                }

                if (utf8_ns_prefix) {
                        g_free (utf8_ns_prefix);
                        utf8_ns_prefix = NULL;
                }

                if (tmp_str) {
                        g_free (tmp_str);
                        tmp_str = NULL;
                }

                if (a_node->children != NULL) {
                        if (attr_str)
                                result = g_strconcat ("<", name,
                                                      " ",
                                                      attr_str,
                                                      ">", NULL);
                        else
                                result = g_strconcat ("<", name,
                                                      ">", NULL);
                } else {        /*empty tag */
                        if (attr_str)
                                result = g_strconcat ("<", name,
                                                      " ",
                                                      attr_str,
                                                      "/>",
                                                      NULL);
                        else
                                result = g_strconcat ("<", name,
                                                      "/>",
                                                      NULL);
                }

                if (name) {
                        g_free (name);
                        name = NULL;
                }

        } else if (xmlNodeIsText (a_node)) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL;

                utf8_content = xmlNodeGetContent (a_node);

                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);
                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }
                if (content == NULL) {
                        xmlNodeSetContent (a_node, "text");
                        content = xmlNodeGetContent (a_node);
                }

                if (content
                    && strlen (content) > MLVIEW_CONTENT_MAX_LEN)
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';

                if (content != NULL) {
                        result = g_strdup (content);
                        g_free (content);
                }

        } else if (a_node->type == XML_COMMENT_NODE) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL;

                utf8_content = xmlNodeGetContent (a_node);

                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);

                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }

                if (content == NULL) {
                        xmlNodeSetContent (a_node,
                                           "<!--comment-->");
                        content = xmlNodeGetContent (a_node);
                }

                if (strlen (content) > MLVIEW_CONTENT_MAX_LEN)
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';

                result = g_strconcat ("<!--", content, "-->",
                                      NULL);

                if (content != NULL) {
                        g_free (content);
                        content = NULL;
                }

        } else if (a_node->type == XML_PI_NODE) {
                enum MlViewStatus status = MLVIEW_OK;
                guchar *utf8_content = NULL,
                        *name = NULL;

                utf8_content = xmlNodeGetContent (a_node);

                if (utf8_content) {
                        status = mlview_utils_utf8_str_to_isolat1
                                (utf8_content, &content);
                        g_return_val_if_fail (status ==
                                              MLVIEW_OK, NULL);

                        xmlFree (utf8_content);
                        utf8_content = NULL;
                }

                if (content == NULL) {
                        xmlNodeSetContent (a_node,
                                           "<?processing instruction node>");
                }

                status = mlview_utils_utf8_str_to_isolat1 ((guchar *)
                                                           a_node->
                                                           name,
                                                           &name);
                g_return_val_if_fail (status == MLVIEW_OK, NULL);


                if (strlen (content) > MLVIEW_CONTENT_MAX_LEN)
                        content[MLVIEW_CONTENT_MAX_LEN] = '\0';

                result = g_strconcat ("<?", name, " ", content,
                                      ">", NULL);

                if (content != NULL) {
                        g_free (content);
                        content = NULL;
                }

                if (name != NULL) {
                        g_free (name);
                        name = NULL;
                }

        }
        return result;
}


/**
 *Helper function. Takes an xml tree in 
 *argument and builds the corresponding 
 *visual tree which is an instance of GtkCTree. 
 *The newly build tree is added
 *to the GtkCTree given in argument 
 *(a_visual_tree) as a subtree of the tree node given 
 *in argument (a_parent_node). 
 */
static GtkCTreeNode *
build_ctree_from_xml_tree (MlViewAppContext * a_app_context,
                           xmlNodePtr a_node,
                           GtkCTreeNode * a_parent_node,
                           GtkCTree ** a_visual_tree_ptr)
{
        xmlNode *node = NULL;
        GtkCTreeNode *visual_node = NULL;
        GtkCTreeNode *result = NULL;

        GdkPixmap *open_element_node_xpm = NULL;
        GdkBitmap *open_element_node_bmp = NULL;
        GdkPixmap *close_element_node_xpm = NULL;
        GdkBitmap *close_element_node_bmp = NULL;
        GdkPixmap *text_node_xpm = NULL;
        GdkBitmap *text_node_bmp = NULL;

        g_return_val_if_fail (a_node != NULL, NULL);

        mlview_app_context_get_xpm
                (a_app_context,
                 (char *) MLVIEW_STG_K_OPEN_ELEMENT_NODE_XPM,
                 &open_element_node_xpm, &open_element_node_bmp);

        mlview_app_context_get_xpm
                (a_app_context,
                 (char *) MLVIEW_STG_K_CLOSE_ELEMENT_NODE_XPM,
                 &close_element_node_xpm,
                 &close_element_node_bmp);

        mlview_app_context_get_xpm (a_app_context, (char *)
                                    MLVIEW_STG_K_TEXT_NODE_XPM,
                                    &text_node_xpm,
                                    &text_node_bmp);

        for (node = a_node; node != NULL; node = node->next) {
                /*walk thru the siblings and create sibling visual nodes */
                /*
                 *the string reprentation of the opening 
                 *tag of the current element node
                 */
                gchar *tag_str = NULL;

                /*
                 *builds the string representation 
                 *of the open tag represented by node.
                 */
                tag_str = node_to_start_tag (node);

                if (node->type == XML_ELEMENT_NODE) {
                        if (*a_visual_tree_ptr == NULL)
                                *a_visual_tree_ptr =
                                        GTK_CTREE
                                        (gtk_ctree_new_with_titles
                                         (TREE_EDITOR_NUMBER_OF_COLUMNS,
                                          TREE_EDITOR_TREE_COLUMN,
                                          (gchar **)
                                          p_tree_editor_titles));

                        if (node->children == NULL) { /*node is leaf node */
                                visual_node =
                                        gtk_ctree_insert_node
                                        (*a_visual_tree_ptr,
                                         a_parent_node, NULL,
                                         &tag_str,
                                         TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                                         close_element_node_xpm,
                                         close_element_node_bmp,
                                         open_element_node_xpm,
                                         open_element_node_bmp,
                                         FALSE, FALSE);

                                g_assert (visual_node != NULL);
                                gtk_ctree_node_set_row_data
                                        (*a_visual_tree_ptr,
                                         visual_node, node);

                                if (!result)
                                        result = visual_node;

                        } else { /*node is not leaf node */
                                visual_node =
                                        gtk_ctree_insert_node
                                        (*a_visual_tree_ptr,
                                         a_parent_node, NULL,
                                         &tag_str,
                                         TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                                         close_element_node_xpm,
                                         close_element_node_bmp,
                                         open_element_node_xpm,
                                         open_element_node_bmp,
                                         FALSE, FALSE);

                                g_assert (visual_node != NULL);

                                gtk_ctree_node_set_row_data
                                        (*a_visual_tree_ptr,
                                         visual_node, node);
                                if (!result)
                                        result = visual_node;

                                build_ctree_from_xml_tree
                                        (a_app_context,
                                         node->children,
                                         visual_node,
                                         a_visual_tree_ptr);
                        }
                } else if (node->type == XML_TEXT_NODE
                           && !xmlIsBlankNode (node)) {

                        visual_node =
                                gtk_ctree_insert_node
                                (*a_visual_tree_ptr,
                                 a_parent_node, NULL,
                                 (gchar **) & tag_str,
                                 TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                                 text_node_xpm, text_node_bmp,
                                 text_node_xpm, text_node_bmp,
                                 FALSE, FALSE);

                        g_assert (visual_node != NULL);
                        gtk_ctree_node_set_row_data
                                (*a_visual_tree_ptr, visual_node,
                                 node);

                        if (!result)
                                result = visual_node;

                } else if (node->type == XML_COMMENT_NODE
                           || node->type == XML_PI_NODE) {
                        visual_node = gtk_ctree_insert_node
                                (*a_visual_tree_ptr,
                                 a_parent_node, NULL, &tag_str,
                                 TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                                 text_node_xpm, text_node_bmp,
                                 text_node_xpm, text_node_bmp,
                                 FALSE, FALSE);

                        g_assert (visual_node != NULL);

                        gtk_ctree_node_set_row_data
                                (*a_visual_tree_ptr, visual_node,
                                 node);

                        if (!result)
                                result = visual_node;
                } else {
                        /*unknown node. Do not show it. */
                }

                if (tag_str != NULL) {
                        g_free (tag_str);
                        tag_str = NULL;
                }

        }                       /*end for */

        return result;
}                               /*end function */


/**
 *Takes an xml document in argument and builds the corresponding visual tree.
 *The visual tree is an instance of GtkCTree. 
 *
 */
static void
build_ctree_from_xml_doc (MlViewAppContext * a_app_context,
                          const xmlDocPtr a_doc,
                          GtkCTree ** a_visual_tree_ptr,
                          gchar * a_doc_name,
                          guint a_expansion_level)
{
        GtkCTreeNode *visual_node;

        xmlNodePtr xml_tree;
        const gchar *root = N_("XML Document");
        const gchar *tree_editor_titles[] =
                { "", N_("the xml document tags") };

        g_assert (*a_visual_tree_ptr == NULL);
        g_return_if_fail (a_doc != NULL);

        if (a_doc_name != NULL)
                tree_editor_titles[TREE_EDITOR_TREE_COLUMN] =
                        a_doc_name;

        else if (a_doc->name != NULL)
                tree_editor_titles[TREE_EDITOR_TREE_COLUMN] =
                        a_doc->name;
        else
                tree_editor_titles[TREE_EDITOR_TREE_COLUMN] =
                        _("the xml document");

        *a_visual_tree_ptr =
                GTK_CTREE (gtk_ctree_new_with_titles
                           (TREE_EDITOR_NUMBER_OF_COLUMNS,
                            TREE_EDITOR_TREE_COLUMN,
                            (gchar **) tree_editor_titles));
        /*
         *FIXME: add support of display all the 
         *types of node that could occur before the real so called xml tree.
         */
        visual_node =
                gtk_ctree_insert_node (*a_visual_tree_ptr, NULL,
                                       NULL, (gchar **) & root,
                                       TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                                       NULL, NULL, NULL, NULL,
                                       FALSE, FALSE);

        gtk_ctree_node_set_row_data (*a_visual_tree_ptr,
                                     visual_node, a_doc);
        xml_tree = a_doc->children;

        if (xml_tree != NULL) {
                build_ctree_from_xml_tree (a_app_context,
                                           xml_tree, visual_node,
                                           a_visual_tree_ptr);
                gtk_ctree_expand_to_depth (*a_visual_tree_ptr,
                                           visual_node,
                                           a_expansion_level);
        }
}


/**
 *Builds a new xml node which type is given in argument. 
 *
 */
static xmlNodePtr
new_xml_node (xmlElementType a_node_type,
              MlViewXMLDocument * a_xml_doc)
{
        xmlNodePtr result = NULL;
        xmlDoc *doc = NULL;

        if (a_xml_doc)
                doc = mlview_xml_document_get_xml_document
                        (a_xml_doc);

        switch (a_node_type) {

        case XML_ELEMENT_NODE:
                result = xmlNewNode (NULL, "");
                break;

        case XML_TEXT_NODE:
                result = xmlNewText ("");
                break;

        case XML_CDATA_SECTION_NODE:
                g_return_val_if_fail (doc != NULL, NULL);
                xmlNewCDataBlock (doc, "", 128);
                break;

        case XML_PI_NODE:
                result = xmlNewPI ("", "");
                break;

        case XML_COMMENT_NODE:
                result = xmlNewComment ("");
                break;

        case XML_DOCUMENT_NODE:
        case XML_DOCUMENT_TYPE_NODE:
        case XML_DOCUMENT_FRAG_NODE:
        case XML_NOTATION_NODE:
        case XML_DTD_NODE:
        case XML_ELEMENT_DECL:
        case XML_ATTRIBUTE_DECL:
        case XML_ENTITY_DECL:
        case XML_NAMESPACE_DECL:
        case XML_XINCLUDE_START:
        case XML_XINCLUDE_END:

        default:
                result = xmlNewNode (NULL, "");
                break;
        }

        return result;
}

/**
 *Searches in the name of the xml node given in argument for a string. 
 *@return 0 if the string has not been found, 
 *1 if the string has been found and -1 if the xml node 
 */
static gint
xml_node_name_contains_string (const xmlNode * a_xml_node,
                               const gchar * a_str,
                               gchar ** a_found_str)
{
        g_return_val_if_fail (a_xml_node != NULL, -1);

        *a_found_str = NULL;

        if (a_str == NULL)
                return 0;

        if (a_xml_node->type != XML_ELEMENT_NODE
            && a_xml_node->type != XML_PI_NODE) {
                return -1;
        }

        *a_found_str = strstr (a_xml_node->name, a_str);
        if (*a_found_str)
                return 1;
        return 0;
}

/**
 *
 */
static gint
xml_node_attributes_name_contains_string (const xmlNode *
                                          a_xml_node,
                                          const gchar * a_str,
                                          gchar ** a_found_str)
{
        xmlAttr *mobile_attr;

        g_return_val_if_fail (a_xml_node != NULL, -1);

        *a_found_str = NULL;
        if (a_str == NULL)
                return 0;

        mobile_attr = a_xml_node->properties;
        while (mobile_attr) {
                *a_found_str = strstr (mobile_attr->name, a_str);
                if (*a_found_str)
                        return 1;
                mobile_attr = mobile_attr->next;
        }
        return 0;
}


/**
 *
 */
static gint
xml_node_attributes_value_contains_string (const xmlNode *
                                           a_xml_node,
                                           const gchar * a_str,
                                           gchar ** a_found_str)
{
        xmlAttr *mobile_attr;

        g_return_val_if_fail (a_xml_node != NULL, -1);

        *a_found_str = NULL;

        if (a_str == NULL)
                return 0;

        mobile_attr = a_xml_node->properties;

        while (mobile_attr) {
                gchar *content = NULL;

                content =
                        xmlNodeGetContent (mobile_attr->
                                           children);

                if (content != NULL)
                        *a_found_str = strstr (content, a_str);

                if (*a_found_str != NULL)
                        return 1;

                mobile_attr = mobile_attr->next;

                if (content) {
                        xmlFree (content);
                        content = NULL;
                }
        }

        return 0;
}


/**
 *
 */
static gint
xml_node_content_contains_string (const xmlNode * a_xml_node,
                                  const gchar * a_str,
                                  gchar ** a_found_str)
{
        gchar *content;

        g_return_val_if_fail (a_xml_node != NULL, -1);

        *a_found_str = NULL;

        if (a_str == NULL)
                return 0;

        if (a_xml_node->type != XML_TEXT_NODE)
                return -1;

        content = xmlNodeGetContent ((xmlNode *) a_xml_node);

        if (content != NULL)
                *a_found_str = strstr (content, a_str);

        if (content) {
                xmlFree (content);
                content = NULL;
        }

        if (*a_found_str != NULL)
                return 1;

        return 0;
}

/**
 *
 */
static enum WhereInTheNodeBitmap
 xml_node_contains_string
        (const xmlNode * a_xml_node,
         const gchar * a_str,
         const enum WhereInTheNodeBitmap a_where_in_node_bitmap)
{
        enum WhereInTheNodeBitmap result = 0;
        gchar *string_found;

        g_return_val_if_fail (a_xml_node, 0);

        if (a_str == NULL)
                return 0;
        if (a_where_in_node_bitmap & NODE_NAME) {
                if (xml_node_name_contains_string (a_xml_node,
                                                   a_str,
                                                   &string_found)
                    > 0) {
                        result |= NODE_NAME;
                }
        }

        if (a_where_in_node_bitmap & NODE_ATTRIBUTE_NAME) {
                if (xml_node_attributes_name_contains_string
                    (a_xml_node, a_str, &string_found) > 0) {

                        result |= NODE_ATTRIBUTE_VALUE;
                }
        }
        if (a_where_in_node_bitmap & NODE_ATTRIBUTE_VALUE) {
                if (xml_node_attributes_value_contains_string
                    (a_xml_node, a_str, &string_found) > 0) {
                        result |= NODE_ATTRIBUTE_VALUE;
                }
        }
        if (a_where_in_node_bitmap & NODE_CONTENT) {
                if (xml_node_content_contains_string (a_xml_node,
                                                      a_str,
                                                      &string_found)
                    > 0) {
                        result |= NODE_CONTENT;
                }
        }

        return result;
}


/**
 *
 *@param a_where_in_node_bitmap the 
 *bitmap that specifies which part of the 
 *node (name, content, attributes name etc)the search should apply for. 
 *@param a_visual_tree the concerned GtkCTree.
 *@param a_str the string to search
 *@param a_starting_from the GtkCTReeNode from which the search starts
 *@param a_start_after wether to start the search before or after
 *@return the node that contains the searched string a_str. 
 */
static GtkCTreeNode *downward_find_visual_node_that_contains
        (GtkCTree * a_visual_tree,
         GtkCTreeNode * a_starting_from,
         const gchar * a_str,
         const enum WhereInTheNodeBitmap a_where_in_node_bitmap,
         gboolean a_start_after) {
        GtkCTreeNode *visual_node = NULL,
                *result = NULL;
        xmlNode *xml_node;

        g_return_val_if_fail (a_visual_tree != NULL, NULL);

        if (a_starting_from == NULL)
                return NULL;

        visual_node = a_starting_from;

        if (a_start_after == TRUE) {
                visual_node = GTK_CTREE_NODE_NEXT (visual_node);
        }

        while (visual_node) {
                xml_node =
                        gtk_ctree_node_get_row_data
                        (a_visual_tree, visual_node);

                if (xml_node_contains_string (xml_node,
                                              a_str,
                                              a_where_in_node_bitmap))
                        return visual_node;

                result = downward_find_visual_node_that_contains
                        (a_visual_tree,
                         GTK_CTREE_ROW (visual_node)->children,
                         a_str, a_where_in_node_bitmap, FALSE);

                if (result != NULL)
                        return result;

                visual_node = GTK_CTREE_NODE_NEXT (visual_node);
        }

        return result;
}


/**
 *Gets the "find" dialob box.
 *@param a_app_context the current "application context".
 *@return the "find" dialog box.
 */
static GtkDialog *
mlview_tree_editor_find_dialog_box_retrieve (MlViewAppContext *
                                             a_app_context)
{
        GtkWidget *search_in_names_check = NULL,
                *search_in_attribute_names_check = NULL,
                *searched_text_entry = NULL;
        GtkWidget *search_in_attribute_value_check = NULL,
                *search_in_content_check = NULL;
        GtkWidget *label = NULL,
                *table = NULL;

        if (p_find_dialog == NULL) {
                p_find_dialog =
                        GTK_DIALOG
                        (gtk_dialog_new_with_buttons
                         (_("Find an xml node in this document"),
                          NULL, GTK_DIALOG_MODAL, _("Find"),
                          GTK_RESPONSE_ACCEPT, _("Cancel"),
                          GTK_RESPONSE_REJECT, NULL));

                //gtk_window_set_modal (GTK_WINDOW (p_find_dialog), FALSE) ;

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     IS_THE_FIND_DIALOG,
                                     (gpointer) "yes");

                label = gtk_label_new (_("Text to look for:"));

                searched_text_entry = gtk_entry_new ();

                table = gtk_table_new (1, 2, TRUE);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           label, 0, 1, 0, 1);

                gtk_table_attach_defaults (GTK_TABLE (table),
                                           searched_text_entry,
                                           1, 2, 0, 1);

                gtk_box_pack_start (GTK_BOX
                                    (p_find_dialog->vbox), table,
                                    TRUE, FALSE, 0);

                search_in_names_check =
                        gtk_check_button_new_with_label
                        (_("Search among nodes name"));

                search_in_attribute_names_check =
                        gtk_check_button_new_with_label
                        (_("Search among attributes names"));

                search_in_attribute_value_check =
                        gtk_check_button_new_with_label
                        (_("Search among attributes values"));

                search_in_content_check =
                        gtk_check_button_new_with_label
                        (_("Search among content nodes text"));

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     SEARCH_IN_NAMES_CHECK_BUTTON,
                                     search_in_names_check);

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     SEARCH_IN_ATTRIBUTES_NAMES_CHECK_BUTTON,
                                     search_in_attribute_names_check);

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     SEARCH_IN_ATTRIBUTES_VALUES_CHECK_BUTTON,
                                     search_in_attribute_value_check);

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     SEARCH_IN_CONTENT_CHECK_BUTTON,
                                     search_in_content_check);

                gtk_object_set_data (GTK_OBJECT (p_find_dialog),
                                     SEARCHED_TEXT_ENTRY,
                                     searched_text_entry);

                gtk_box_pack_start (GTK_BOX
                                    (p_find_dialog->vbox),
                                    search_in_names_check, TRUE,
                                    FALSE, 0);

                gtk_box_pack_start (GTK_BOX
                                    (p_find_dialog->vbox),
                                    search_in_attribute_names_check,
                                    TRUE, FALSE, 0);

                gtk_box_pack_start (GTK_BOX
                                    (p_find_dialog->vbox),
                                    search_in_attribute_value_check,
                                    TRUE, FALSE, 0);

                gtk_box_pack_start (GTK_BOX
                                    (p_find_dialog->vbox),
                                    search_in_content_check,
                                    TRUE, FALSE, 0);
                /*
                   mlview_app_context_set_window_icon (a_app_context, 
                   GTK_WIDGET (p_find_dialog)) ;
                 */
                gtk_window_set_wmclass (GTK_WINDOW
                                        (p_find_dialog),
                                        "find-dialog-box",
                                        "MlView");

                gtk_widget_show_all (GTK_WIDGET
                                     (p_find_dialog->vbox));
        }

        return p_find_dialog;
}


/**
 *
 */
static gboolean
mlview_tree_editor_find_dialog_search_in_names_is_on (GtkDialog * a_find_dialog) 
{
        gchar *check_str;
        GtkCheckButton *search_in_names_check_button;

        g_return_val_if_fail (GTK_IS_DIALOG (a_find_dialog),
                              FALSE);

        /*
         *a SANITY check:
         *get a sanity check string associated 
         *to the search dialog by the function 
         *mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". 
         *If yes, the a_find_dialog is a 
         *find dialog box build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                     IS_THE_FIND_DIALOG);

        if (strcmp (check_str, "yes"))
                return FALSE;

        search_in_names_check_button =
                GTK_CHECK_BUTTON
                (gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                      SEARCH_IN_NAMES_CHECK_BUTTON));

        return gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON
                 (search_in_names_check_button));
}


/**
 *
 */
static gboolean
mlview_tree_editor_find_dialog_search_in_attributes_names_is_on (GtkDialog * a_find_dialog) 
{
        gchar *check_str;
        GtkCheckButton *search_in_attribute_names_check_button;

        g_return_val_if_fail (GTK_IS_DIALOG (a_find_dialog),
                              FALSE);

        /*a SANITY check:
         *get a sanity check string associated to 
         *the search dialog by the function 
         *mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". 
         *If yes, the a_find_dialog is a find 
         *dialog box build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                     IS_THE_FIND_DIALOG);
        if (strcmp (check_str, "yes"))
                return FALSE;

        search_in_attribute_names_check_button =
                GTK_CHECK_BUTTON
                (gtk_object_get_data
                 (GTK_OBJECT (a_find_dialog),
                  SEARCH_IN_ATTRIBUTES_NAMES_CHECK_BUTTON));
        return gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON
                 (search_in_attribute_names_check_button));
}

/**
 *
 */
static gboolean
mlview_tree_editor_find_dialog_search_in_attributes_values_is_on (GtkDialog * a_find_dialog) 
{
        gchar *check_str;
        GtkCheckButton *search_in_attribute_values_check_button;

        g_return_val_if_fail (GTK_IS_DIALOG (a_find_dialog),
                              FALSE);

        /*a SANITY check:
         *get a sanity check string associated 
         *to the search dialog by the function 
         *mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". 
         *If yes, the a_find_dialog is a find 
         *dialog box build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                     IS_THE_FIND_DIALOG);
        if (strcmp (check_str, "yes"))
                return FALSE;

        search_in_attribute_values_check_button =
                GTK_CHECK_BUTTON
                (gtk_object_get_data
                 (GTK_OBJECT (a_find_dialog),
                  SEARCH_IN_ATTRIBUTES_VALUES_CHECK_BUTTON));
        return gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON
                 (search_in_attribute_values_check_button));
}

/**
 *
 */
static gboolean
mlview_tree_editor_find_dialog_search_in_content_is_on (GtkDialog * a_find_dialog) 
{
        gchar *check_str;
        GtkCheckButton *search_in_content_check_button;

        g_return_val_if_fail (GTK_IS_DIALOG (a_find_dialog),
                              FALSE);

        /*
         *a SANITY check:
         *get a sanity check string associated to the search 
         *dialog by the function mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". 
         *If yes, the a_find_dialog is a 
         *find dialog box build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                     IS_THE_FIND_DIALOG);

        if (strcmp (check_str, "yes"))
                return FALSE;

        search_in_content_check_button =
                GTK_CHECK_BUTTON
                (gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                      SEARCH_IN_CONTENT_CHECK_BUTTON));
        return gtk_toggle_button_get_active
                (GTK_TOGGLE_BUTTON
                 (search_in_content_check_button));
}


/**
 *
 *@param a_find_dialog the dialog build by the 
 *function mlview_tree_editor_find_dialog_retrieve()
 */
static gchar *
mlview_tree_editor_find_dialog_get_searched_text (GtkDialog *a_find_dialog)
{
        gchar *check_str;
        gchar *result;
        GtkEntry *search_string_entry;

        g_return_val_if_fail (GTK_IS_DIALOG (a_find_dialog),
                              NULL);

        /*
         *a SANITY check:
         *get a sanity check string associated 
         *to the search dialog by the function 
         *mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". If yes, 
         *the a_find_dialog is a find dialog box 
         *build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_find_dialog),
                                     IS_THE_FIND_DIALOG);
        if (strcmp (check_str, "yes"))
                return FALSE;

        search_string_entry =
                GTK_ENTRY (gtk_object_get_data
                           (GTK_OBJECT (a_find_dialog),
                            SEARCHED_TEXT_ENTRY));
        g_return_val_if_fail (GTK_IS_ENTRY (search_string_entry),
                              NULL);

        result = NULL;
        /*  gtk_entry_get_text (search_string_entry) ; */

        if (result != NULL)
                return g_strdup (result);

        return result;
}

/**
 *
 *@param a_dialog the find dialog box.
 *
 *
 */
static GtkCTreeNode* 
mlview_tree_editor_get_data_from_find_dialog_and_perform_search (MlViewTreeEditor * a_tree_editor, GtkDialog * a_dialog,
                                                                 GtkCTreeNode * a_start_from, gboolean a_start_after) 
{
        gchar *check_str;
        gchar *str_to_search;
        GtkCTreeNode *start_from = NULL,
                *result = NULL;
        enum WhereInTheNodeBitmap bmp = 0;

        g_return_val_if_fail (a_dialog != NULL, NULL);
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor)->
                              viewable_doc != NULL, NULL);

        /*
         *a SANITY check:
         *get a sanity check string associated
         * to the search dialog by the function 
         *mlview_tree_editor_find_dialog_retrieve()
         *and see if it's equals "yes". 
         *If yes, the a_find_dialog is a find 
         *dialog box build by mlview_tree_editor_find_dialog_retrieve() ;
         *if no, return false... just in case ...
         */
        check_str =
                gtk_object_get_data (GTK_OBJECT (a_dialog),
                                     IS_THE_FIND_DIALOG);
        if (strcmp (check_str, "yes"))
                return NULL;

        /*
         *build the bitmap that describes 
         *where in the xml node 
         *(attribute names?, node names?, node content?) 
         *the search functions have to look ...
         */
        if (mlview_tree_editor_find_dialog_search_in_names_is_on
            (a_dialog) == TRUE)
                bmp |= NODE_NAME;

        if (mlview_tree_editor_find_dialog_search_in_attributes_names_is_on (a_dialog) == TRUE)
                bmp |= NODE_ATTRIBUTE_NAME;

        if (mlview_tree_editor_find_dialog_search_in_attributes_values_is_on (a_dialog) == TRUE)
                bmp |= NODE_ATTRIBUTE_VALUE;

        if (mlview_tree_editor_find_dialog_search_in_content_is_on (a_dialog) == TRUE)
                bmp |= NODE_CONTENT;

        str_to_search =
                mlview_tree_editor_find_dialog_get_searched_text
                (a_dialog);

        start_from = a_start_from;

        if (start_from == NULL) {
                start_from =
                        gtk_ctree_node_nth
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         0);

                g_return_val_if_fail (start_from != NULL, NULL);
        }
        if (str_to_search != NULL && strcmp (str_to_search, ""))
                result = downward_find_visual_node_that_contains
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         start_from, str_to_search, bmp,
                         a_start_after);

        if (result != NULL) {

                GtkAdjustment *vadj = NULL;
                gint node_ypos = 0;

                gtk_ctree_expand_to_depth
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         gtk_ctree_node_nth
                         (PRIVATE (a_tree_editor)->viewable_doc,
                          0), GTK_CTREE_ROW (result)->level);

                gtk_ctree_select (PRIVATE (a_tree_editor)->
                                  viewable_doc, result);

                vadj = gtk_clist_get_vadjustment
                        (GTK_CLIST
                         (PRIVATE (a_tree_editor)->
                          viewable_doc));

                if (vadj == NULL)
                        return NULL;

                node_ypos =
                        gtk_ctree_node_absolute_top_ypixel
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         result);

                gtk_adjustment_set_value (vadj, node_ypos);
        }

        return result;
}

/**
 *
 *@param a_nt_picker the concerned instance of MlViewNodeTypePicker.
 *@param a_tree_editor 
 *
 *
 */
static void
mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_add_child (MlViewTreeEditor * a_tree_editor) 
{
        guint selected_node_type,
		node_addion_status;
        MlViewNodeTypePicker *nt_picker;

        xmlNodePtr xml_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        xmlNs *ns = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;

        g_return_if_fail (a_tree_editor != NULL);

        nt_picker =
                mlview_tree_editor_get_node_type_picker
                (a_tree_editor);
        g_return_if_fail (nt_picker != NULL);

        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (nt_picker);

        if (node_name_or_content != NULL
            &&
            !mlview_utils_is_white_string (node_name_or_content))
        {

                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (nt_picker);

                xml_doc =
                        mlview_tree_editor_get_xml_document
                        (a_tree_editor);

                xml_node =
                        new_xml_node (selected_node_type,
                                      xml_doc);

                switch (selected_node_type) {

                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);

                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);

                                g_free (local_name);
                                local_name = NULL;
                        }

                        break;
                default:
                        /*
                           xmlNodeSetContent 
                           (xml_node, 
                           g_strdup (node_name_or_content)) ;
                         */
                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 FALSE);
                        break;
                }

                node_addion_status =
                        mlview_tree_editor_add_child_node
                        (a_tree_editor,
                         mlview_tree_editor_get_current_selected_node
                         (a_tree_editor), xml_node);

                if (!node_addion_status
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {
                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);

                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }
                        if (local_name) {
                                g_free (local_name);
                                local_name = NULL;
                        }
                }
        }
}


/**
 *Is called when the user clicks on the
 *OK button of the node type picker to say
 *that she wants to insert a sibling node.
 *@param a_tree_editor the current instance
 *of #MlViewTreeEditor.
 */
static void
mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_insert_sibling_node (MlViewTreeEditor * a_tree_editor) 
{
        guint selected_node_type;
        xmlNode *xml_node = NULL,
                *result_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;
        gpointer previous = NULL;
        MlViewNodeTypePicker *node_type_picker = NULL;
        guchar *node_name_or_content = NULL,
                *local_name = NULL;
        xmlNs *ns = NULL;
        GtkCTreeNode *visual_node = NULL;
        GtkCTree *visual_tree = NULL;

        g_return_if_fail (a_tree_editor != NULL);

        node_type_picker =
                mlview_tree_editor_get_node_type_picker
                (a_tree_editor);

        g_return_if_fail (node_type_picker != NULL);

        node_name_or_content =
                mlview_node_type_picker_get_node_name_or_content
                (node_type_picker);

        if (node_name_or_content != NULL
            &&
            !mlview_utils_is_white_string (node_name_or_content)) {
                selected_node_type =
                        mlview_node_type_picker_get_selected_node_type
                        (node_type_picker);

                xml_doc =
                        mlview_tree_editor_get_xml_document
                        (a_tree_editor);
                xml_node =
                        new_xml_node (selected_node_type,
                                      xml_doc);

                switch (selected_node_type) {
                case XML_ELEMENT_NODE:
                case XML_PI_NODE:
                        mlview_utils_parse_full_name (xml_node,
                                                      node_name_or_content,
                                                      &ns,
                                                      &local_name);
                        if (local_name != NULL) {
                                mlview_xml_document_set_node_name
                                        (xml_doc, xml_node,
                                         local_name, ISO8859_1,
                                         TRUE);

                                g_free (local_name);
                        }
                        break;

                default:

                        mlview_xml_document_set_node_content
                                (xml_doc, xml_node,
                                 node_name_or_content, ISO8859_1,
                                 TRUE);
                        break;
                }

                /*
                 *retrieve a flag set to indicate 
                 *if the sibling has to be inserted 
                 *before or after the current xml node
                 */
                previous =
                        gtk_object_get_data (GTK_OBJECT
                                             (a_tree_editor),
                                             "prev");

                result_node =
                        mlview_tree_editor_insert_sibling_node
                        (a_tree_editor,
                         mlview_tree_editor_get_current_selected_node
                         (a_tree_editor), xml_node,
                         GPOINTER_TO_INT (previous));

                if (result_node != NULL
                    && (selected_node_type == XML_ELEMENT_NODE
                        || selected_node_type == XML_PI_NODE)) {

                        visual_tree =
                                mlview_tree_editor_get_visual_tree
                                (a_tree_editor);
                        g_return_if_fail (visual_tree != NULL);

                        visual_node =
                                gtk_ctree_find_by_row_data
                                (visual_tree, NULL, result_node);
                        g_return_if_fail (visual_node != NULL);

                        mlview_utils_parse_full_name
                                (xml_node, node_name_or_content,
                                 &ns, &local_name);

                        if (ns) {
                                xmlSetNs (xml_node, ns);
                        } else {
                                xml_node->ns = NULL;
                        }

                        if (local_name != NULL)
                                g_free (local_name);

                        mlview_tree_editor_update_visual_node
                                (a_tree_editor, visual_node);
                }
        }
}


static gint
mlview_tree_editor_build_node_insertion_submenu (MlViewTreeEditor * a_editor,
                                                 enum NODE_INSERTION_SCHEME a_insertion_scheme,
                                                 GtkMenu ** a_submenu) 
{
        gchar *validation_is_on = NULL;
        GList *attr_names = NULL;
        xmlNode *xml_node = NULL;
        gint nb_names = 0;

        g_return_val_if_fail (a_editor != NULL, -5);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR (a_editor),
                              -5);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, -5);
        g_return_val_if_fail (a_submenu != NULL, -5);

        *a_submenu = NULL;
        if (PRIVATE (a_editor)->app_context == NULL)
                return -4;

        if (!PRIVATE (a_editor)->current_selected_node
            || !PRIVATE (a_editor)->viewable_doc)
                return -3;

        xml_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_editor)->viewable_doc,
                 PRIVATE (a_editor)->current_selected_node);

        if (xml_node == NULL)
                return -3;

        validation_is_on =
                mlview_app_context_get_settings_value
                (PRIVATE (a_editor)->app_context,
                 MLVIEW_STG_K_IS_VALIDATION_ON);

        if (!validation_is_on
            || strcmp (validation_is_on, MLVIEW_STG_V_YES)) {
                return -2;
        }

        if (xml_node->type == XML_ELEMENT_NODE)
                nb_names =
                        mlview_parsing_utils_build_element_name_completion_list
                        (PRIVATE (a_editor)->app_context,
                         a_insertion_scheme, xml_node,
                         &attr_names);

        if (nb_names != 0) {
                GList *cur = attr_names;
                GtkMenuItem *menu_item = NULL;

                *a_submenu = GTK_MENU (gtk_menu_new ());

                while (cur) {
                        menu_item =
                                GTK_MENU_ITEM
                                (gtk_menu_item_new_with_label
                                 (cur->data));

                        gtk_object_set_data (GTK_OBJECT
                                             (menu_item),
                                             "element-name",
                                             cur->data);

                        switch (a_insertion_scheme) {
                        case ADD_CHILD:
                                gtk_signal_connect
                                        (GTK_OBJECT (menu_item),
                                         "activate",
                                         GTK_SIGNAL_FUNC
                                         (menu_add_child_node_activate_cb),
                                         a_editor);
                                break;

                        case INSERT_BEFORE:
                                gtk_signal_connect
                                        (GTK_OBJECT (menu_item),
                                         "activate",
                                         GTK_SIGNAL_FUNC
                                         (menu_insert_prev_sibling_node_activate_cb),
                                         a_editor);
                                break;
                        case INSERT_AFTER:
                                gtk_signal_connect
                                        (GTK_OBJECT (menu_item),
                                         "activate",
                                         GTK_SIGNAL_FUNC
                                         (menu_insert_next_sibling_node_activate_cb),
                                         a_editor);
                                break;
                        default:
                                break;
                        }
                        gtk_widget_show (GTK_WIDGET (menu_item));
                        gtk_menu_append (*a_submenu,
                                         GTK_WIDGET (menu_item));

                        cur = cur->next;
                }
        }
        return 0;
}



/*===========================================================
 *callbacks and signal handlers
 *===========================================================
 */

/**
 *The default handler of the signal "node-selected" emitted by MlViewTreeEditor when a node of the visual tree is selected.
 *So far, this function has been used for debug purposes. 
 *
 *@param a_visual_tree the current visual tree, 
 *image of the xml doc being edited
 *@param a_editor the current instance of MlViewTreeEditor
 *@param a_visual_node the selected node, member of a_visual_tree
 */
static void
 mlview_tree_editor_node_selected_default_signal_handler
        (MlViewTreeEditor * a_editor,
         GtkCTreeNode * a_visual_node, gpointer a_user_data) {
        g_assert (a_editor != NULL);
        g_assert (PRIVATE (a_editor) != NULL);
        g_assert (PRIVATE (a_editor)->viewable_doc != NULL);
}

/**
 *
 *@param a_visual_tree the instance 
 *of GtkCTree that emitted the signal
 *@param a_tree_editor the instance of 
 *MlViewTreeEditor that owns the GtkCTree that emited the signal.
 *@param a_node the node that has just been selected.
 *@param a_column the column of the selected node.
 *
 *
 */
static void
mlview_tree_editor_tree_select_row_cb (GtkCTree * a_visual_tree,
                                       GList * a_node,
                                       gint a_column,
                                       gpointer a_tree_editor)
{
        MlViewTreeEditor *editor;

        g_assert (a_tree_editor != NULL);
        g_assert (a_node != NULL);
        g_assert (a_node->data != NULL);

        editor = MLVIEW_TREE_EDITOR (a_tree_editor);
        g_assert (PRIVATE (editor) != NULL);

        PRIVATE (editor)->current_selected_node =
                GTK_CTREE_NODE (a_node);

        gtk_signal_emit (GTK_OBJECT (MLVIEW_TREE_EDITOR (a_tree_editor)), mlview_tree_editor_signals[NODE_SELECTED], GTK_CTREE_NODE (a_node) /*visual_node */
                );
}


static void
menu_add_child_node_activate_cb (GtkMenuItem * a_menu_item,
                                 gpointer a_user_data)
{
        MlViewTreeEditor *tree_editor = NULL;
        gchar *element_name = NULL;

        g_return_if_fail (a_menu_item != NULL);
        g_return_if_fail (GTK_IS_MENU_ITEM (a_menu_item));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_user_data));

        tree_editor = MLVIEW_TREE_EDITOR (a_user_data);
        g_return_if_fail (PRIVATE (tree_editor) != NULL);
        g_return_if_fail (PRIVATE (tree_editor)->
                          current_selected_node != NULL);
        g_return_if_fail (PRIVATE (tree_editor)->viewable_doc !=
                          NULL);

        element_name =
                gtk_object_get_data (GTK_OBJECT (a_menu_item),
                                     "element-name");

        if (element_name) {
                xmlNode *xml_node =
                        xmlNewNode (NULL, element_name);

                mlview_tree_editor_add_child_node
                        (tree_editor,
                         PRIVATE (tree_editor)->
                         current_selected_node, xml_node);
        }
}


static void
menu_insert_prev_sibling_node_activate_cb (GtkMenuItem *
                                           a_menu_item,
                                           gpointer a_user_data)
{
        MlViewTreeEditor *tree_editor = NULL;
        gchar *element_name = NULL;

        g_return_if_fail (a_menu_item != NULL);
        g_return_if_fail (GTK_IS_MENU_ITEM (a_menu_item));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_user_data));

        tree_editor = MLVIEW_TREE_EDITOR (a_user_data);

        g_return_if_fail (PRIVATE (tree_editor) != NULL);

        g_return_if_fail
                (PRIVATE (tree_editor)->current_selected_node !=
                 NULL);

        g_return_if_fail (PRIVATE (tree_editor)->viewable_doc !=
                          NULL);

        element_name =
                gtk_object_get_data (GTK_OBJECT (a_menu_item),
                                     "element-name");

        if (element_name) {
                xmlNode *xml_node =
                        xmlNewNode (NULL, element_name);

                mlview_tree_editor_insert_sibling_node
                        (tree_editor,
                         PRIVATE (tree_editor)->
                         current_selected_node, xml_node, TRUE);
        }
}

static void
menu_insert_next_sibling_node_activate_cb (GtkMenuItem *
                                           a_menu_item,
                                           gpointer a_user_data)
{
        MlViewTreeEditor *tree_editor = NULL;
        gchar *element_name = NULL;

        g_return_if_fail (a_menu_item != NULL);
        g_return_if_fail (GTK_IS_MENU_ITEM (a_menu_item));
        g_return_if_fail (a_user_data != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_user_data));

        tree_editor = MLVIEW_TREE_EDITOR (a_user_data);
        g_return_if_fail (PRIVATE (tree_editor) != NULL);
        g_return_if_fail (PRIVATE (tree_editor)->
                          current_selected_node != NULL);
        g_return_if_fail (PRIVATE (tree_editor)->viewable_doc !=
                          NULL);

        element_name =
                gtk_object_get_data (GTK_OBJECT (a_menu_item),
                                     "element-name");

        if (element_name) {
                xmlNode *xml_node =
                        xmlNewNode (NULL, element_name);

                mlview_tree_editor_insert_sibling_node
                        (tree_editor,
                         PRIVATE (tree_editor)->
                         current_selected_node, xml_node, FALSE);
        }
}

static gboolean
mlview_tree_editor_event_cb (GtkWidget * a_widget,
                             GdkEvent * a_event,
                             gpointer a_user_data)
{

        MlViewTreeEditor *tree_editor = NULL;

        g_return_val_if_fail (a_widget != NULL, FALSE);
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget), FALSE);
        g_return_val_if_fail (a_user_data != NULL, FALSE);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR
                              (a_user_data), FALSE);
        g_return_val_if_fail (a_event != NULL, FALSE);

        tree_editor = MLVIEW_TREE_EDITOR (a_user_data);
        g_return_val_if_fail (tree_editor != NULL, FALSE);
        g_return_val_if_fail (PRIVATE (tree_editor), FALSE);

        switch (a_event->type) {
        case GDK_BUTTON_PRESS:
                if (a_event->button.button == 3) {
                        /*user pressed the right mouse button */
                        GtkWidget *root_widget =
                                gtk_widget_get_toplevel
                                (GTK_WIDGET (tree_editor));

                        g_return_val_if_fail (root_widget !=
                                              NULL, FALSE);

                        gtk_propagate_event (root_widget,
                                             a_event);
                }
                break;
        default:
                break;
        }
        return TRUE;
}


/*===============================================================
 *public methods
 *===============================================================
 */

/**
 *the standard type id builder of the MlViewTreeEditor object. 
 *
 *@return the type id of the MlViewTreeEditor object. 
 */
guint
mlview_tree_editor_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewTreeEditorClass),
                        NULL, NULL,
                        (GClassInitFunc)
                                mlview_tree_editor_init_class,
                        NULL, NULL,
                        sizeof (MlViewTreeEditor),
                        0,
                        (GInstanceInitFunc)
                        mlview_tree_editor_init
                };
                type = g_type_register_static (GTK_TYPE_VBOX,
                                               "MlViewTreeEditor",
                                               &type_info, 0);
        }
        return type;
}


/**
 *the public constructor of the MlViewTreeEditor. 
 *
 *@param a_context the application context.
 *
 *
 *@return the newly build instance of MlViewTreeEditor. 
 */
GtkWidget *
mlview_tree_editor_new (MlViewAppContext * a_context)
{
        MlViewTreeEditor *editor;

        editor = g_object_new (MLVIEW_TYPE_TREE_EDITOR, NULL);
        PRIVATE (editor)->app_context = a_context;

        gtk_signal_connect (GTK_OBJECT (editor),
                            "button_press_event",
                            GTK_SIGNAL_FUNC
                            (mlview_tree_editor_event_cb),
                            editor);

        return GTK_WIDGET (editor);
}

void
mlview_tree_editor_set_application_context (MlViewTreeEditor *
                                            a_tree_editor,
                                            MlViewAppContext *
                                            a_app_context)
{
        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);

        PRIVATE (a_tree_editor)->app_context = a_app_context;
}


MlViewAppContext *
mlview_tree_editor_get_application_context (MlViewTreeEditor *
                                            a_tree_editor)
{
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR
                              (a_tree_editor), NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);

        return PRIVATE (a_tree_editor)->app_context;
}


/**
 *Sets the title of the xml DOM to @a_file_path.
 *Updates this information in the tree editor and in the
 *XML DOM. 
 *
 *@param a_file_path the new file path of xml document.
 *@param a_tree_editor the current tree editor.
 *
 *
 */
void
mlview_tree_editor_set_xml_document_path (MlViewTreeEditor *
                                          a_tree_editor,
                                          gchar * a_file_path)
{
        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (a_file_path != NULL);

        if (!PRIVATE (a_tree_editor)->xml_doc)
                return;

        if (PRIVATE (a_tree_editor)->xml_doc->name)
                g_free (PRIVATE (a_tree_editor)->xml_doc->name);

        PRIVATE (a_tree_editor)->xml_doc->name =
                g_strdup (a_file_path);
        gtk_clist_set_column_title (GTK_CLIST
                                    (PRIVATE (a_tree_editor)->
                                     viewable_doc),
                                    TREE_EDITOR_TREE_COLUMN,
                                    PRIVATE (a_tree_editor)->
                                    xml_doc->name);
}


/**
 *A very important method in this class (and even in the whole MlView software).
 *It takes an xml document in parameter, 
 *builds the graphical view that matches it and displays it. 
 *
 *@param a_xmldoc the xml document to edit.
 *@param a_editor the current instance of MlViewTreeEditor.
 *
 *
 *@return 0 if OK or the error code. 
 */
guint
mlview_tree_editor_edit_xml_doc (MlViewTreeEditor * a_editor,
                                 MlViewXMLDocument *
                                 a_mlview_xml_doc,
                                 gchar * a_doc_name)
{
        GtkCTree *viewable_doc = NULL;
        GtkWidget *scr_win = NULL;
        glong expansion_depth =
                MLVIEW_TREE_EDITOR_DEFAULT_EXPANSION_DEPTH;
        gchar *expansion_depth_str = NULL;
        GHashTable *settings = NULL;
        xmlDoc *xml_doc = NULL;

        g_return_val_if_fail (a_editor != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR (a_editor),
                              -1);
        g_return_val_if_fail (PRIVATE (a_editor) != NULL, -1);
        g_return_val_if_fail (a_mlview_xml_doc != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_XML_DOCUMENT
                              (a_mlview_xml_doc), -1);

        xml_doc =
                mlview_xml_document_get_xml_document
                ((MlViewXMLDocument *) a_mlview_xml_doc);
        g_return_val_if_fail (xml_doc != NULL, -1);

        PRIVATE (a_editor)->mlview_xml_doc = a_mlview_xml_doc;

        if (PRIVATE (a_editor)->app_context)
                settings =
                        mlview_app_context_get_settings_hash_table
                        (PRIVATE (a_editor)->app_context);

        if (settings
            && (expansion_depth_str =
                g_hash_table_lookup
                (settings,
                 MLVIEW_STG_K_DEFAULT_TREE_EXPANSION_LEVEL))) {
                expansion_depth =
                        strtol (expansion_depth_str, NULL, 10);
        }

        if ((expansion_depth == LONG_MIN)
            || (expansion_depth == LONG_MAX)) {
                expansion_depth =
                        MLVIEW_TREE_EDITOR_DEFAULT_EXPANSION_DEPTH;
        }

        build_ctree_from_xml_doc (PRIVATE (a_editor)->
                                  app_context, xml_doc,
                                  &viewable_doc, a_doc_name,
                                  expansion_depth);

        g_assert (viewable_doc != NULL);

        if (PRIVATE (a_editor)->viewable_doc) {

                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (a_editor)->viewable_doc));
        }

        PRIVATE (a_editor)->viewable_doc = viewable_doc;

        gtk_signal_connect (GTK_OBJECT (viewable_doc),
                            "tree-select-row",
                            GTK_SIGNAL_FUNC
                            (mlview_tree_editor_tree_select_row_cb),
                            a_editor);

        scr_win = gtk_scrolled_window_new (NULL, NULL);
        gtk_container_add (GTK_CONTAINER (scr_win),
                           GTK_WIDGET (viewable_doc));
        gtk_box_pack_start (GTK_BOX (a_editor), scr_win, TRUE,
                            TRUE, 0);
        gtk_widget_show_all (GTK_WIDGET (a_editor));
        PRIVATE (a_editor)->xml_doc = xml_doc;
        return 0;
}

/**
 *Updates the visual node (the node that the user sees in the tree)
 *that matches a given instance of xmlNode.
 *
 *@param a_tree_editor the "this pointer" of the current instance.
 *@param a_node the instance of xmlNode which matching visual node
 *is to be updated. This is the actual data structure that modelizes
 *the xml node.
 */
void
mlview_tree_editor_update_visual_xml_node (MlViewTreeEditor *
                                           a_tree_editor,
                                           xmlNode * a_node)
{
        GtkCTreeNode *visual_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          viewable_doc != NULL);
        g_return_if_fail (a_node != NULL);

        visual_node =
                gtk_ctree_find_by_row_data
                (PRIVATE (a_tree_editor)->viewable_doc, NULL,
                 a_node);

        if (visual_node == NULL)
                return;

        mlview_tree_editor_update_visual_node (a_tree_editor,
                                               visual_node);

}

/**
 *Given a GtkCTreeNode which has been 
 *associated to an xmlNode 
 *(by calling gtk_ctree_node_set_row_data 
 *on the GtkCTreeNode with the xmlNode in param),
 *this function redraws the 
 *GtkCTreeNode by updating the string 
 *representation of the xml node that apears in the 
 *column at the right of the node.
 *This method is to be called each 
 *time an xml node is modified (i.e name, attribute, or namespace def changes) 
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor to consider.
 *@param a_visual_node the visual node to update
 *
 */
void
mlview_tree_editor_update_visual_node (MlViewTreeEditor *
                                       a_tree_editor,
                                       GtkCTreeNode *
                                       a_visual_node)
{
        xmlNodePtr xml_node = NULL;
        gchar *tag_string = NULL;
        GdkPixmap *pixmap = NULL;
        GdkBitmap *bitmap = NULL;
        guint8 spacing;
        gchar *text = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          viewable_doc != NULL);
        g_return_if_fail (a_visual_node != NULL);

        xml_node =
                (xmlNodePtr) gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 a_visual_node);

        g_return_if_fail (xml_node != NULL);

        tag_string = node_to_start_tag (xml_node);

        if (tag_string != NULL) {

                gtk_ctree_node_get_pixtext
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         a_visual_node,
                         TREE_EDITOR_TREE_COLUMN, &text,
                         &spacing, &pixmap, &bitmap);

                gtk_ctree_node_set_pixtext
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         a_visual_node, TREE_EDITOR_TREE_COLUMN,
                         tag_string,
                         TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                         pixmap, bitmap);
                g_free (tag_string);

                gtk_widget_show
                        (GTK_WIDGET
                         (PRIVATE (a_tree_editor)->
                          viewable_doc));
        }
}

/**
 *the destroy signal handler of the mlviewTreeEditorClass. 
 *
 *@param a_object the current instance of MlViewTreeEditor
 *
 *
 */
void
mlview_tree_editor_destroy (GtkObject * a_object)
{
        MlViewTreeEditor *editor;

        g_return_if_fail (a_object);

        editor = MLVIEW_TREE_EDITOR (a_object);
        if (PRIVATE (editor) == NULL)
                return;

        if (p_find_dialog != NULL) {
                gtk_widget_destroy (GTK_WIDGET (p_find_dialog));
                p_find_dialog = NULL;
        }

        if (PRIVATE (editor)->node_type_picker != NULL) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (editor)->node_type_picker));
                PRIVATE (editor)->node_type_picker = NULL;
        }

        if (PRIVATE (editor) != NULL) {
                g_free (PRIVATE (editor));
                PRIVATE (editor) = NULL;
        }

        if (GTK_OBJECT_CLASS (parent_class)->destroy) {
                (*GTK_OBJECT_CLASS (parent_class)->
                 destroy) (a_object);
        }
}


/**
 *This method visually updates the tree 
 *editor to make the addition of child node visible.
 *If the node addition has been 
 *already updated, this method does nothing.
 *Upon successfull update, this method makes 
 *a_tree_editor fire two signals :
 *"node-added", "tree-changed". 
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor.
 *@param a_added_node the added node.
 *@param a_parent_node the parent node of the node added.
 *@param a_user_data 
 *
 *
 */
void
mlview_tree_editor_update_child_node_added (MlViewTreeEditor *
                                            a_tree_editor,
                                            xmlNode *
                                            a_parent_node,
                                            xmlNode *
                                            a_added_node)
{
        GtkCTreeNode *result = NULL,
                *parent_visual_node = NULL,
                *visual_added_node = NULL;
        GtkCTree *visual_tree = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);

        g_return_if_fail (a_parent_node != NULL);
        g_return_if_fail (a_added_node != NULL);

        /*only an element node should have children */
        g_return_if_fail (a_parent_node->type ==
                          XML_ELEMENT_NODE);
        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        parent_visual_node =
                gtk_ctree_find_by_row_data (visual_tree,
                                            NULL, a_parent_node);

        /*make sure the parent node has been drawn on the editor already. */
        g_return_if_fail (parent_visual_node != NULL);

        visual_added_node =
                gtk_ctree_find_by_row_data (visual_tree, NULL,
                                            a_added_node);

        /*Make sure this node hasn't been drawn yet */
        g_return_if_fail (visual_added_node == NULL);

        result = build_ctree_from_xml_tree
                (PRIVATE (a_tree_editor)->app_context,
                 a_added_node, parent_visual_node, &visual_tree);

        mlview_tree_editor_update_visual_node
                (a_tree_editor, parent_visual_node);

        if (!gtk_ctree_is_viewable (visual_tree, result))
                mlview_tree_editor_expand_tree_to_depth
                        (a_tree_editor, 1);

        gtk_ctree_select (visual_tree, result);

        /*emit the appropriate signals */
        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals[NODE_ADDED],
                         (gpointer) result);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals
                         [TREE_CHANGED]);
}

/**
 *
 *@param a_cut_node 
 *@param a_tree_editor 
 *@param a_parent_node 
 *
 *
 */
void
mlview_tree_editor_update_node_cut (MlViewTreeEditor *
                                    a_tree_editor,
                                    xmlNode * a_parent_node,
                                    xmlNode * a_cut_node)
{

        GtkCTree *visual_tree = NULL;
        GtkCTreeNode *parent_visual_node = NULL,
                *visual_cut_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          viewable_doc != NULL);
        g_return_if_fail (a_parent_node != NULL);
        g_return_if_fail (a_cut_node != NULL);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        visual_cut_node = gtk_ctree_find_by_row_data
                (visual_tree, NULL, a_cut_node);

        /*Make sure this node has been drawn already. */
        g_return_if_fail (visual_cut_node != NULL);

        parent_visual_node = gtk_ctree_find_by_row_data
                (visual_tree, NULL, a_parent_node);

        g_return_if_fail (parent_visual_node != NULL);

        gtk_ctree_remove_node (visual_tree, visual_cut_node);

        mlview_tree_editor_update_visual_node (a_tree_editor,
                                               parent_visual_node);

        gtk_ctree_select (visual_tree, parent_visual_node);

        /*emit the appropriate signals */
        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals[NODE_CUT],
                         visual_cut_node);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals
                         [TREE_CHANGED]);
}


/**
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor.
 *@param a_added_node the newly pasted node.
 *@param a_parent_node the parent node of the pasted node.
 *
 *
 */
void
mlview_tree_editor_update_node_pasted (MlViewTreeEditor *
                                       a_tree_editor,
                                       xmlNode * a_parent_node,
                                       xmlNode * a_pasted_node)
{
        GtkCTree *visual_tree = NULL;
        GtkCTreeNode *parent_visual_node = NULL,
                *visual_node = NULL,
                *visual_pasted_node = NULL;


        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor));
        g_return_if_fail (a_parent_node != NULL);
        g_return_if_fail (a_pasted_node != NULL);

        visual_pasted_node = gtk_ctree_find_by_row_data
                (visual_tree, NULL, a_pasted_node);

        /*Make sure this node hasn't been drawn yet */
        g_return_if_fail (visual_pasted_node == NULL);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        parent_visual_node =
                gtk_ctree_find_by_row_data (visual_tree, NULL,
                                            a_parent_node);
        g_return_if_fail (parent_visual_node != NULL);

        mlview_tree_editor_update_child_node_added
                (a_tree_editor, a_parent_node, a_pasted_node);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals[NODE_PASTED],
                         visual_node);
}


void mlview_tree_editor_update_sibling_node_inserted
        (MlViewTreeEditor * a_tree_editor,
         xmlNode * a_sibling_node,
         xmlNode * a_inserted_node, gboolean a_previous) {
        GtkCTreeNode *visual_sibling_node = NULL,
                *visual_inserted_node = NULL,
                *result = NULL,
                *visual_parent_node = NULL;
        GtkCTree *visual_tree = NULL;
        gchar *tag_str = NULL;

        GdkPixmap *open_element_node_xpm = NULL;
        GdkBitmap *open_element_node_bmp = NULL;
        GdkPixmap *close_element_node_xpm = NULL;
        GdkBitmap *close_element_node_bmp = NULL;
        GdkPixmap *text_node_xpm = NULL;
        GdkBitmap *text_node_bmp = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (a_sibling_node != NULL);
        g_return_if_fail (a_inserted_node != NULL);

        mlview_app_context_get_xpm (PRIVATE (a_tree_editor)->
                                    app_context, (char *)
                                    MLVIEW_STG_K_OPEN_ELEMENT_NODE_XPM,
                                    &open_element_node_xpm,
                                    &open_element_node_bmp);

        mlview_app_context_get_xpm (PRIVATE (a_tree_editor)->
                                    app_context, (char *)
                                    MLVIEW_STG_K_CLOSE_ELEMENT_NODE_XPM,
                                    &close_element_node_xpm,
                                    &close_element_node_bmp);

        mlview_app_context_get_xpm (PRIVATE (a_tree_editor)->
                                    app_context, (char *)
                                    MLVIEW_STG_K_TEXT_NODE_XPM,
                                    &text_node_xpm,
                                    &text_node_bmp);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        visual_inserted_node =
                gtk_ctree_find_by_row_data (visual_tree, NULL,
                                            a_inserted_node);
        /*Make sure the node has not been drawn before */
        g_return_if_fail (visual_inserted_node == NULL);

        tag_str = node_to_start_tag (a_inserted_node);
        g_return_if_fail (tag_str != NULL);

        visual_sibling_node = gtk_ctree_find_by_row_data
                (visual_tree, NULL, a_sibling_node);
        g_return_if_fail (visual_sibling_node != NULL);

        visual_parent_node =
                GTK_CTREE_ROW (visual_sibling_node)->parent;

        if (a_previous == FALSE) {
                visual_sibling_node =
                        (GTK_CTREE_ROW (visual_sibling_node)->
                         sibling ==
                         NULL) ? NULL :
                        GTK_CTREE_ROW (visual_sibling_node)->
                        sibling;
        }

        if (a_inserted_node->type == XML_ELEMENT_NODE)
                result = gtk_ctree_insert_node
                        (visual_tree, visual_parent_node,
                         visual_sibling_node,
                         &tag_str,
                         TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                         close_element_node_xpm,
                         close_element_node_bmp,
                         open_element_node_xpm,
                         open_element_node_bmp, FALSE, FALSE);
        else
                result = gtk_ctree_insert_node
                        (visual_tree, visual_parent_node,
                         visual_sibling_node, &tag_str,
                         TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                         text_node_xpm, text_node_bmp,
                         text_node_xpm, text_node_bmp, FALSE,
                         FALSE);

        g_return_if_fail (result != NULL);

        if (a_inserted_node->children != NULL) {
                build_ctree_from_xml_tree (PRIVATE
                                           (a_tree_editor)->
                                           app_context,
                                           a_inserted_node->
                                           children, result,
                                           &visual_tree);
        }

        gtk_ctree_node_set_row_data (visual_tree, result,
                                     a_inserted_node);
        mlview_tree_editor_update_visual_node (a_tree_editor,
                                               visual_parent_node);
        gtk_ctree_select (visual_tree, result);
}


/**
 *Adds a child node to the GtkCTreeNode a_parent_node. 
 *This function also associates 
 *the newly build child node (instance of GtkCTreeNode)
 *to the xmlNodePtr given in 
 *argument by calling gtk_ctree_node_set_row_data().
 *Note that the child node is added at the end of 
 *the existing children nodes of a_parent_node.
 *This method is not interactive. 
 *
 *@param a_xml_node the xml node 
 *that is to be associated to the newly created visual node.
 *@param a_tree_editor the current instance of GtkCTreeNode.
 *@param a_parent_node the parent node of the node that is to be created.
 *
 */
gint
mlview_tree_editor_add_child_node (MlViewTreeEditor *
                                   a_tree_editor,
                                   GtkCTreeNode * a_parent_node,
                                   xmlNode * a_xml_node)
{
        xmlNode *parent_xml_node = NULL;
        xmlNode *added_node = NULL;
        GtkCTree *visual_tree = NULL;

        g_return_val_if_fail (a_tree_editor != NULL, -1);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR
                              (a_tree_editor), -1);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              -1);
        g_return_val_if_fail (PRIVATE (a_tree_editor)->
                              mlview_xml_doc != NULL, -1);

        g_return_val_if_fail (a_parent_node != NULL, -1);
        g_return_val_if_fail (a_xml_node != NULL, -1);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_val_if_fail (visual_tree != NULL, -1);

        parent_xml_node = (xmlNodePtr)
                gtk_ctree_node_get_row_data (visual_tree,
                                             a_parent_node);

        added_node =
                mlview_xml_document_add_child_node
                (PRIVATE (a_tree_editor)->mlview_xml_doc,
                 parent_xml_node, a_xml_node, TRUE,
                 TRUE /*emit signal */ );

        if (added_node)
                return 0;

        return 1;
}


/**
 *Builds and inserts a sibling node 
 *after the a_sibling node in the 
 *current visual xml tree. The newly build node is a child node
 *of a_parent_node and reflects the xml node a_xml_node given in argument.
 *Emit the signals "node-added" and the signal "tree-changed". 
 *
 *@param a_xml_node the xml node to insert.
 *@param a_tree_editor the current instance of MlViewTreeEditor
 *@param a_sibling_node the reference node 
 *after wich the newly build node will be inserted
 *@param a_parent_node the parent of the viusal node to insert
 *
 *
 */
xmlNode *
mlview_tree_editor_insert_sibling_node (MlViewTreeEditor *
                                        a_tree_editor,
                                        GtkCTreeNode *
                                        a_sibling_node,
                                        xmlNodePtr a_xml_node,
                                        gboolean a_previous)
{
        xmlNode *sibling_xml_node = NULL,
                *result = NULL;
        GtkCTree *visual_tree = NULL;

        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (a_sibling_node != NULL, NULL);
        g_return_val_if_fail (a_xml_node != NULL, NULL);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_val_if_fail (visual_tree != NULL, NULL);

        sibling_xml_node =
                (xmlNodePtr) gtk_ctree_node_get_row_data
                (visual_tree, a_sibling_node);

        g_return_val_if_fail (sibling_xml_node != NULL, NULL);

        if (a_previous == TRUE) {
                /*insert the node as previous sibling */
                result = mlview_xml_document_insert_prev_sibling_node (PRIVATE (a_tree_editor)->mlview_xml_doc, sibling_xml_node, a_xml_node, TRUE, TRUE);

        } else {
                /*insert the node as next sibling */
                result = mlview_xml_document_insert_next_sibling_node (PRIVATE (a_tree_editor)->mlview_xml_doc, sibling_xml_node, a_xml_node, TRUE, TRUE);
        }

        return result;
}


/**
 *Removes the node a_visual_node given 
 *in argument and put it in the clipboard.
 *Cuts also the correponding xmlNode from the dom.
 *Emits the signals "node-cut" and "tree-changed". 
 */
void
mlview_tree_editor_cut_node (MlViewTreeEditor * a_tree_editor,
                             GtkCTreeNode * a_visual_node)
{
        xmlNodePtr xml_node = NULL;
        GtkCTree *visual_tree = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          mlview_xml_doc != NULL);
        g_return_if_fail (a_visual_node != NULL);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        xml_node = (xmlNodePtr)
                gtk_ctree_node_get_row_data (visual_tree,
                                             a_visual_node);
        g_return_if_fail (xml_node);

        mlview_xml_document_cut_node (PRIVATE (a_tree_editor)->
                                      mlview_xml_doc, xml_node,
                                      TRUE);
}

void
mlview_tree_editor_copy_node (MlViewTreeEditor * a_tree_editor,
                              GtkCTreeNode * a_visual_node,
                              gboolean a_recursive)
{
        GtkCTree *visual_tree = NULL;
        xmlNodePtr xml_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (a_visual_node != NULL);

        visual_tree =
                mlview_tree_editor_get_visual_tree
                (a_tree_editor);
        g_return_if_fail (visual_tree != NULL);

        xml_node = (xmlNodePtr)
                gtk_ctree_node_get_row_data (visual_tree,
                                             a_visual_node);
        g_return_if_fail (xml_node);

        mlview_xml_document_copy_node_to_clipboard
                (xml_node, PRIVATE (a_tree_editor)->xml_doc);
}


/**
 *
 *@param a_tree_editor the current instance of MlViewTreEditor
 *
 *
 */
GtkCTreeNode *
mlview_tree_editor_get_current_selected_node (MlViewTreeEditor *
                                              a_tree_editor)
{
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);

        return PRIVATE (a_tree_editor)->current_selected_node;
}


/**
 *Creates a new xml doc 
 *
 *@param a_xml_doc 
 *@param a_tree_editor 
 *
 *
 */
void
mlview_tree_editor_create_new_xml_doc (MlViewTreeEditor *
                                       a_tree_editor,
                                       MlViewXMLDocument *
                                       a_xml_doc)
{
        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);


        mlview_tree_editor_edit_xml_doc (a_tree_editor,
                                         a_xml_doc, NULL);
}


/**
 *Set the root element to a visual node build from a_xml_node.
 *Also set the root element of the dom to a_xml_node. 
 *
 *@param a_xml_root_node the xml node to set as root element.
 *@param a_tree_editor the current instance of MlViewTreeEditor.
 *
 *
 */
GtkCTreeNode *
mlview_tree_editor_set_root_element (MlViewTreeEditor *
                                     a_tree_editor,
                                     xmlNodePtr a_xml_root_node)
{
        GtkCTreeNode *visual_doc_node = NULL,
                *result = NULL;
        gchar *tag_string;

        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (a_xml_root_node != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor)->
                              viewable_doc != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor)->xml_doc !=
                              NULL, NULL);
        g_return_val_if_fail (visual_doc_node != NULL, NULL);

        xmlDocSetRootElement (PRIVATE (a_tree_editor)->xml_doc,
                              a_xml_root_node);
        tag_string = node_to_start_tag (a_xml_root_node);
        result = gtk_ctree_insert_node
                (PRIVATE (a_tree_editor)->viewable_doc,
                 NULL,
                 NULL, &tag_string,
                 TREE_EDITOR_SPACE_BETWEEN_PIXMAP_AND_TEXT,
                 NULL, NULL, NULL, NULL, FALSE, TRUE);

        if (tag_string != NULL)
                g_free (tag_string);

        g_return_val_if_fail (result != NULL, NULL);
        gtk_ctree_node_set_row_data (PRIVATE (a_tree_editor)->
                                     viewable_doc, result,
                                     a_xml_root_node);

        gtk_ctree_select (PRIVATE (a_tree_editor)->viewable_doc,
                          result);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals[NODE_ADDED],
                         (gpointer) result);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals
                         [TREE_CHANGED]);

        return result;
}


/**
 *Get the last node put into the clipboard 
 *and insert it into the visual dom. 
 *
 */
void
mlview_tree_editor_paste_node_as_child (MlViewTreeEditor *
                                        a_tree_editor,
                                        GtkCTreeNode *
                                        a_visual_parent_node)
{
        xmlNode *parent_xml_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->viewable_doc);

        g_return_if_fail (a_visual_parent_node != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->viewable_doc);

        parent_xml_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 a_visual_parent_node);
        g_return_if_fail (parent_xml_node != NULL);

        mlview_xml_document_paste_node_as_child
                (PRIVATE (a_tree_editor)->mlview_xml_doc,
                 parent_xml_node, TRUE);

        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals
                         [TREE_CHANGED]);
}

/**
 *Gets an xml node from the clipboard 
 *and pastes it as the previous sibling of the visual node given in argument.
 *If the node retrieved from the clipboard is NULL, does nothing. 
 *
 *@param a_tree_editor 
 *@param a_visual_sibling_node 
 *
 *
 */
void
mlview_tree_editor_paste_node_as_sibling (MlViewTreeEditor *
                                          a_tree_editor,
                                          GtkCTreeNode *
                                          a_visual_sibling_node,
                                          gboolean a_previous)
{

        xmlNode *sibling_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          mlview_xml_doc != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor)->
                          viewable_doc != NULL);
        g_return_if_fail (a_visual_sibling_node != NULL);

        sibling_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 a_visual_sibling_node);
        g_return_if_fail (sibling_node != NULL);
        g_return_if_fail (sibling_node->parent != NULL);

        mlview_xml_document_paste_node_as_sibling
                (PRIVATE (a_tree_editor)->mlview_xml_doc,
                 sibling_node->parent, sibling_node, a_previous,
                 TRUE);

        /*emit the appropriate signals */
        gtk_signal_emit (GTK_OBJECT (a_tree_editor),
                         mlview_tree_editor_signals
                         [TREE_CHANGED]);
}


/**
 *Gets the instance of
 *GtkCTree that features
 *the xml document graphically
 *@param a_tree_editor the current instance
 *of #MlViewTreeEditor.
 *@return the GtkCTree graphical tree.
 */
GtkCTree *
mlview_tree_editor_get_visual_tree (MlViewTreeEditor *
                                    a_tree_editor)
{
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR
                              (a_tree_editor), NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);

        return PRIVATE (a_tree_editor)->viewable_doc;
}

/**
 *Asks the user for the type of node he wants to add and adds it. 
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor
 *
 *
 */
void
mlview_tree_editor_add_child_node_interactive (MlViewTreeEditor *
                                               a_tree_editor)
{
        MlViewNodeTypePicker *nt_picker;
        gint button;
        xmlNode *current_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);

        current_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 PRIVATE (a_tree_editor)->current_selected_node);

        /*
         *only an element node should have children.
         */
        if (current_node->type != XML_ELEMENT_NODE)
                return;

        nt_picker =
                mlview_tree_editor_get_node_type_picker
                (a_tree_editor);

        if (nt_picker == NULL) {
                nt_picker =
                        MLVIEW_NODE_TYPE_PICKER
                        (mlview_node_type_picker_new
                         (PRIVATE (a_tree_editor)->app_context));

                gtk_window_set_modal (GTK_WINDOW (nt_picker),
                                      TRUE);
                mlview_tree_editor_set_node_type_picker
                        (a_tree_editor, nt_picker);
        }

        mlview_node_type_picker_set_title (nt_picker,
                                           _("add a child node"));
        mlview_node_type_picker_build_element_name_choice_list
                (nt_picker, ADD_CHILD, current_node);

        mlview_node_type_picker_select_node_name_or_content_entry_text
                (nt_picker);

        /*gtk_widget_realize (GTK_WIDGET (nt_picker));*/

        mlview_app_context_set_window_icon
                (PRIVATE (a_tree_editor)->app_context,
                 GTK_WINDOW (nt_picker));

        button = gtk_dialog_run (GTK_DIALOG (nt_picker));

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*OK button */
                mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_add_child
                        (a_tree_editor);
                gtk_widget_hide (GTK_WIDGET
                                 (PRIVATE (a_tree_editor)->
                                  node_type_picker));
                break;
        case GTK_RESPONSE_REJECT: /*cancel button */
                gtk_widget_hide
                        (GTK_WIDGET
                         (PRIVATE (a_tree_editor)->
                          node_type_picker));
                break;
        default:
                gtk_widget_hide
                        (GTK_WIDGET
                         (PRIVATE (a_tree_editor)->
                          node_type_picker));
                break;
        }
}


/**
 *Asks the user for the type of node he wants to insert and inserts it. 
 *
 *@param a_tree_editor 
 *
 *
 */
void 
mlview_tree_editor_insert_prev_sibling_node_interactive (MlViewTreeEditor * a_tree_editor) 
{
        MlViewNodeTypePicker *nt_picker;
        gint button;
        xmlNode *current_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);

        nt_picker =
                mlview_tree_editor_get_node_type_picker
                (a_tree_editor);

        if (nt_picker == NULL) {
                nt_picker =
                        MLVIEW_NODE_TYPE_PICKER
                        (mlview_node_type_picker_new
                         (PRIVATE (a_tree_editor)->app_context));

                gtk_window_set_modal (GTK_WINDOW (nt_picker),
                                      TRUE);

                mlview_tree_editor_set_node_type_picker
                        (a_tree_editor, nt_picker);
        }
        mlview_node_type_picker_set_title (nt_picker,
                                           _("insert a previous sibling node"));
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (nt_picker);

        /*
         *insert a flag to indicate that 
         *the picker is used to insert node as previous node
         *this is used in function
         */
        gtk_object_set_data (GTK_OBJECT (a_tree_editor),
                             "prev", GINT_TO_POINTER (TRUE));
        current_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 PRIVATE (a_tree_editor)->current_selected_node);

        mlview_node_type_picker_build_element_name_choice_list
                (nt_picker, INSERT_BEFORE, current_node);
        mlview_app_context_set_window_icon
                (PRIVATE (a_tree_editor)->app_context,
                 GTK_WINDOW (nt_picker));
        button = gtk_dialog_run (GTK_DIALOG (nt_picker));

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*OK button */
                mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_tree_editor);
                break;
        default:
                break;
        }
        gtk_widget_hide (GTK_WIDGET
                         (PRIVATE (a_tree_editor)->node_type_picker));
}


/**
 *Asks the user for the type of node he wants
 *to insert and inserts it as a next sibling of the
 *currently selected node.
 *@param a_tree_editor the current instance of #MlViewTreeEditor.
 */
void 
mlview_tree_editor_insert_next_sibling_node_interactive (MlViewTreeEditor * a_tree_editor) 
{
        MlViewNodeTypePicker *nt_picker;
        gint button;
        xmlNode *current_node = NULL;

        g_return_if_fail (a_tree_editor != NULL);

        nt_picker =
                mlview_tree_editor_get_node_type_picker
                (a_tree_editor);

        if (nt_picker == NULL) {
                nt_picker =
                        MLVIEW_NODE_TYPE_PICKER
                        (mlview_node_type_picker_new
                         (PRIVATE (a_tree_editor)->app_context));

                gtk_window_set_modal (GTK_WINDOW (nt_picker),
                                      TRUE);
                mlview_tree_editor_set_node_type_picker
                        (a_tree_editor, nt_picker);
        }

        mlview_node_type_picker_set_title (nt_picker,
                                           _("insert a next sibling node"));
        mlview_node_type_picker_select_node_name_or_content_entry_text
                (nt_picker);
        current_node =
                gtk_ctree_node_get_row_data
                (PRIVATE (a_tree_editor)->viewable_doc,
                 PRIVATE (a_tree_editor)->current_selected_node);
        mlview_node_type_picker_build_element_name_choice_list
                (nt_picker, INSERT_AFTER, current_node);
        gtk_object_set_data (GTK_OBJECT (a_tree_editor),
                             "prev", GINT_TO_POINTER (FALSE));
        mlview_app_context_set_window_icon
                (PRIVATE (a_tree_editor)->app_context,
                 GTK_WINDOW (nt_picker));
        button = gtk_dialog_run (GTK_DIALOG (nt_picker));
        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*OK button */
                mlview_tree_editor_handle_nt_picker_ok_button_clicked_to_insert_sibling_node
                        (a_tree_editor);
                break;
        default:
                break;
        }
        gtk_widget_hide (GTK_WIDGET
                         (PRIVATE (a_tree_editor)->node_type_picker));
}


/**
 *Returns the instance of #MlViewXMLDocument associated to
 *the current instance of #MlViewTreeEditor.
 *
 */
MlViewXMLDocument *
mlview_tree_editor_get_xml_document (MlViewTreeEditor *
                                     a_tree_editor)
{
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);

        return PRIVATE (a_tree_editor)->mlview_xml_doc;
}

/**
 *
 *@param a_tree_editor 
 *@param a_node_type_picker 
 *
 *
 */
void 
mlview_tree_editor_set_node_type_picker (MlViewTreeEditor * a_tree_editor,
					 MlViewNodeTypePicker * a_node_type_picker) 
{
        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);

        PRIVATE (a_tree_editor)->node_type_picker =
                a_node_type_picker;
}

/**
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor
 *@return the current node type picker 
 *associated to this tree editor. NULL if no node type picker is associated. 
 */
MlViewNodeTypePicker *
mlview_tree_editor_get_node_type_picker (MlViewTreeEditor *
                                         a_tree_editor)
{
        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);
        return PRIVATE (a_tree_editor)->node_type_picker;
}

/**
 *
 *@param a_tree_editor the current instance of MlViewTreeEditor
 *@param a_str the string to search.
 *
 *
 */
GtkCTreeNode *
mlview_tree_editor_find_xml_node_that_contains_str (MlViewTreeEditor * a_tree_editor, const gchar * a_str,
                                                    const enum WhereInTheNodeBitmap a_where_in_node_bitmap,
                                                    const gboolean a_start_after) 
{
        GtkCTreeNode *start_node = NULL;
        MlViewXMLDocument *xml_doc = NULL;

        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor)->
                              viewable_doc != NULL, NULL);

        if (a_str == NULL)
                return NULL;
        xml_doc =
                mlview_tree_editor_get_xml_document
                (a_tree_editor);
        g_return_val_if_fail (xml_doc != NULL, NULL);

        start_node =
                mlview_tree_editor_get_current_selected_node
                (a_tree_editor);

        if (start_node == NULL) { /*set start node to the doc root element. */
                start_node =
                        gtk_ctree_node_nth
                        (PRIVATE (a_tree_editor)->viewable_doc,
                         0);
                g_return_val_if_fail (start_node != NULL, NULL);
        }

        return downward_find_visual_node_that_contains
                (PRIVATE (a_tree_editor)->viewable_doc,
                 start_node, a_str, a_where_in_node_bitmap,
                 a_start_after);
}


/**
 *mlview_tree_editor_find_xml_node_that_contains_str_interactive 
 *
 */
GtkCTreeNode *
mlview_tree_editor_find_xml_node_that_contains_str_interactive (MlViewTreeEditor * a_tree_editor) 
{
        GtkDialog *find_dialog;
        GtkCTreeNode *previously_found_node = NULL;
        gint button;
        gboolean go = FALSE,
                start_after = FALSE;

        g_return_val_if_fail (a_tree_editor != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_EDITOR
                              (a_tree_editor), NULL);
        g_return_val_if_fail (PRIVATE (a_tree_editor) != NULL,
                              NULL);

        find_dialog =
                mlview_tree_editor_find_dialog_box_retrieve
                (PRIVATE (a_tree_editor)->app_context);
        g_return_val_if_fail (find_dialog != NULL, NULL);
        gtk_window_set_modal (GTK_WINDOW (find_dialog), FALSE);

        go = TRUE;
        previously_found_node = NULL;

        while (go) {

                gtk_widget_realize (GTK_WIDGET (find_dialog));
                mlview_app_context_set_window_icon
                        (PRIVATE (a_tree_editor)->app_context,
                         GTK_WINDOW (find_dialog));
                button = gtk_dialog_run (find_dialog);

                switch (button) {
                case GTK_RESPONSE_ACCEPT:
                        start_after =
                                (previously_found_node !=
                                 NULL) ? TRUE : FALSE;
                        previously_found_node =
                                mlview_tree_editor_get_data_from_find_dialog_and_perform_search
                                (a_tree_editor, find_dialog,
                                 previously_found_node,
                                 start_after);
                        break;
                case GTK_RESPONSE_CLOSE:
                default:
                        gtk_widget_hide (GTK_WIDGET
                                         (find_dialog));
                        go = FALSE;
                        break;
                }
        }
        return NULL;
}

/**
 *Expands the current selected 
 *visual xml tree to the depth specified by @a_a_depth. If @a_depth
 *is set to -1 the tree is expanded to the leaves. 
 *
 *@param a_editor the current instance of MlViewTreeEditor.
 *@param a_depth expansion depth. 
 */
void
mlview_tree_editor_expand_tree_to_depth (MlViewTreeEditor *
                                         a_editor, gint a_depth)
{
        g_return_if_fail (a_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_editor));
        g_return_if_fail (PRIVATE (a_editor) != NULL);

        if (PRIVATE (a_editor)->current_selected_node == NULL)
                return;

        if (a_depth >= 0) {
                gtk_ctree_collapse_recursive
                        (PRIVATE (a_editor)->viewable_doc,
                         PRIVATE (a_editor)->
                         current_selected_node);

                if (a_depth != 0) {
                        gtk_ctree_expand_to_depth
                                (PRIVATE (a_editor)->
                                 viewable_doc,
                                 PRIVATE (a_editor)->
                                 current_selected_node,
                                 GTK_CTREE_ROW (PRIVATE
                                                (a_editor)->
                                                current_selected_node)->
                                 level + a_depth - 1);
                }

        } else {                /*expand to leaves */
                gtk_ctree_expand_recursive (PRIVATE (a_editor)->
                                            viewable_doc,
                                            PRIVATE (a_editor)->
                                            current_selected_node);
        }
        gtk_widget_show_all (GTK_WIDGET (a_editor));
}

/**
 *
 *@param a_menu_ptr 
 *@param a_tree_editor 
 *
 *
 */
void
mlview_tree_editor_update_contextual_menu (MlViewTreeEditor *
                                           a_tree_editor,
                                           GtkMenu ** a_menu_ptr)
{
        GtkWidget *menu_item = NULL;

        g_return_if_fail (a_tree_editor != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_EDITOR (a_tree_editor));
        g_return_if_fail (PRIVATE (a_tree_editor) != NULL);
        g_return_if_fail (a_menu_ptr != NULL);
        g_return_if_fail (*a_menu_ptr != NULL);
        g_return_if_fail (GTK_IS_MENU (*a_menu_ptr));

        if (PRIVATE (a_tree_editor)->current_selected_node !=
            NULL) {
                GtkMenu *submenu = NULL;

                /*separator */
                menu_item = gtk_menu_item_new ();
                gtk_menu_append (*a_menu_ptr, menu_item);
                gtk_widget_show (menu_item);

                menu_item =
                        gtk_menu_item_new_with_label (_
                                                      ("Add child node"));
                gtk_menu_append (GTK_MENU (*a_menu_ptr),
                                 menu_item);
                gtk_widget_show (menu_item);
                if (!mlview_tree_editor_build_node_insertion_submenu (a_tree_editor, ADD_CHILD, &submenu) && submenu) {
                        gtk_menu_item_set_submenu
                                (GTK_MENU_ITEM (menu_item),
                                 GTK_WIDGET (submenu));
                }

                menu_item =
                        gtk_menu_item_new_with_label
                        (_("Insert next sibling node"));
                gtk_menu_append (GTK_MENU (*a_menu_ptr),
                                 menu_item);
                gtk_widget_show (menu_item);
                if (!mlview_tree_editor_build_node_insertion_submenu (a_tree_editor, INSERT_AFTER, &submenu) && submenu) {
                        gtk_menu_item_set_submenu
                                (GTK_MENU_ITEM (menu_item),
                                 GTK_WIDGET (submenu));
                }

                menu_item = gtk_menu_item_new_with_label
                        (_("Insert previous sibling node"));
                gtk_menu_append (GTK_MENU (*a_menu_ptr),
                                 menu_item);
                gtk_widget_show (menu_item);

                if (!mlview_tree_editor_build_node_insertion_submenu (a_tree_editor, INSERT_BEFORE, &submenu) && submenu) {
                        gtk_menu_item_set_submenu
                                (GTK_MENU_ITEM (menu_item),
                                 GTK_WIDGET (submenu));
                }
        }
}
