/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it and/or 
 *modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView.
 *see the file COPYING. 
 *If not, write to the 
 *Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <stdio.h>
#include <gnome.h>
#include "config.h"
#include "mlview-editor.h"
#include "mlview-app.h"
#include "mlview-app-context.h"
#include "mlview-preferences.h"


/**
 *@file 
 *the mlview application class definition.
 *It instanciates mlview's instance of  GnomeApp, 
 *fills it with an instance of MlViewEditor, builds the application menus
 *and toolbars and creates an instance of MlViewAppContext.
 */

#define PRIVATE(a_obj) a_obj->priv

struct _MlViewAppPriv {
        struct MlViewWidgetsHandle *widgets ;
} ;

/*
 *The current instance of the mlview application context.
 *Must be set by mlview_app. For the moment, 
 *there must be one application
 *context per process. That should be changed in the 
 *future to enforce
 *reentrency.
 */

/*common function declarations*/

/*callbacks ...*/
static void
display_about_dialog (void);


static void create_internal_subset_cb (GtkWidget *a_menu_item,
                                       MlViewAppContext *a_context) ;


static enum MlViewStatus
build_widgets_handle (MlViewApp *a_app,
                      GladeXML *a_glade_xml) ;

static enum MlViewStatus
init_editor (MlViewApp *a_app,
             GladeXML *a_glade_xml) ;

/*
 *Displays the about dialog.
 */
static void
display_about_dialog ()
{
        static GtkWidget *about_dialog = NULL;
        guchar *path = NULL ;
        GdkPixbuf *pixbuf = NULL;

        const gchar *authors[] =
                {"Dodji Seketeli<dodji@mlview.org>",
                 "Gael Chamoulaud<strider@mlview.org>",
                 "Nicolas Centa<happypeng@free.fr>",
                 "Baptiste Mille-Mathias<bmm80@ifrance.com>",
                 "Stephane Bonhomme<s.bonhomme@wanadoo.fr>",
                 NULL
        };

        const gchar *documenters[] =
                {"Dodji Seketeli<dodji@mlview.org>",
                 NULL
        };

        const gchar *translator_credits = _("translator_credits");

        if (about_dialog) {
                gtk_window_present (GTK_WINDOW (about_dialog));
                return;
        }
        path = gnome_program_locate_file (NULL, 
                                          GNOME_FILE_DOMAIN_APP_DATADIR,
                                          "mlview/mlview-app-icon.xpm",
                                          TRUE, NULL) ;
        if (path) {
                pixbuf = gdk_pixbuf_new_from_file (path, NULL);
                g_free (path) ;
                path = NULL ;
        }
        about_dialog = gnome_about_new 
                (PACKAGE, VERSION,
                 "Copyright \xc2\xa9 2001-2004 Dodji Seketeli\n",
                 _("A simple xml editor for GNOME"),
                 (const char **) authors,
                 (const char **) documenters,
                 strcmp
                 (translator_credits,
                  "translator_credits") !=
                 0 ? translator_credits :
                 "No translators, English by\n"
                 "Dodji Seketeli  <dodji@mlview.org>\n"
                 "Gael Chamoulaud <strider@mlview.org>",
                 pixbuf);

        if (pixbuf != NULL) {
                g_object_unref (pixbuf) ;
                pixbuf = NULL ;
        }
        g_object_add_weak_pointer (G_OBJECT (about_dialog),
                                   (void **) &about_dialog);
        gtk_widget_show (about_dialog);
}

/*=========================================================
 *General signal handlers for the differents menuitems
 *========================================================*/
static void
open_doc_menu_item_clicked_cb (GtkWidget * a_widget, 
                               MlViewAppContext * a_context)
{
        g_return_if_fail (a_widget != NULL);
        g_return_if_fail (a_context != NULL);

        mlview_editor_open_local_xml_document_interactive
                (mlview_app_context_get_element
                 (a_context, "MlViewEditor"));
}

static void
about_menu_item_clicked_cb (GtkWidget * a_widget, 
                            MlViewAppContext * a_context)
{
        display_about_dialog ();
}


static void
new_doc_menu_item_clicked_cb (GtkWidget * a_menu_item, 
                              MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        g_return_if_fail (editor) ;
        mlview_editor_create_new_xml_document (editor);
}


static void
new_view_on_document_menu_item_clicked_cb (GtkWidget * a_menu_item,
                                           MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_create_new_view_on_current_document_interactive
                (editor);
}

static void 
create_internal_subset_cb (GtkWidget *a_menu_item,
                           MlViewAppContext *a_context)
{
        MlViewEditor *editor;
        MlViewAction action = {0} ;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item);
        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        action.name = (guchar*) 
                "create-internal-subset-node-interactive" ;
        mlview_editor_execute_action (editor, &action) ;

}

static void
close_menu_item_clicked_cb (GtkWidget * a_menu_item, MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_close_xml_document_interactive (editor);
}

static void
close_application (GtkWidget *a_widget, MlViewAppContext *a_context)
{
        GtkWindow *app_win = NULL;
        MlViewEditor *editor = NULL;
        gboolean is_ok = FALSE;

        g_return_if_fail (a_widget
                          && MLVIEW_IS_APP_CONTEXT (a_context));

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        if (editor)
                is_ok = mlview_editor_close_all_xml_documents_interactive
                        (editor);

        if (is_ok == FALSE) {
                gtk_widget_show (a_widget) ;
                return;
        }
        app_win = mlview_app_context_get_element (a_context,
                                                  "MlViewMainWindow");
        g_return_if_fail (app_win) ;
        gtk_widget_destroy (GTK_WIDGET (app_win));
        gtk_main_quit ();
}

static gboolean
delete_event_cb (GtkWidget *a_widget,
                 GdkEvent *a_event,
                 MlViewAppContext *a_context)
{
        g_return_val_if_fail (GTK_IS_WIDGET (a_widget)
                              && MLVIEW_IS_APP_CONTEXT (a_context),
                              FALSE) ;

        close_application (a_widget, a_context) ;
        return TRUE ;
}

static gint
quit_menu_item_clicked_cb (GtkWidget * a_widget, 
                           MlViewAppContext * a_context)
{
        
        close_application (a_widget, a_context) ;
        return FALSE;
}


static void
save_document_menu_item_clicked_cb (GtkWidget * a_menu_item, 
                                    MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_save_xml_document (editor);
}

static void
save_as_menu_item_clicked_cb (GtkWidget * a_menu_item,
                              MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_save_xml_document_as_interactive (editor);
}



static void
associate_a_dtd_menu_item_clicked_cb (GtkWidget * a_menu_item,
                                      MlViewAppContext * a_context)
{
        MlViewEditor *editor;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        mlview_editor_associate_dtd_interactive (editor);
}

static void
validate_document_menu_item_clicked_cb (GtkWidget * a_menu_item,
                                        MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_validate (editor);
}

static void
apply_xslt_menu_item_clicked_cb (GtkWidget * a_menu_item,
                                 MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor") ;
        mlview_editor_xslt_transform_document_interactive 
                (editor) ;
}

static void
edit_settings_cb (GtkWidget * a_menu_item,
                  MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");

        mlview_preferences_dialog_show (a_context);
}

static void
rename_view_menu_item_clicked_cb (GtkWidget * a_menu_item,
                                  MlViewAppContext * a_context)
{
        MlViewEditor *editor = NULL;

        g_return_if_fail (a_context != NULL);
        g_return_if_fail (a_menu_item != NULL);

        editor = mlview_app_context_get_element (a_context,
                                                 "MlViewEditor");
        mlview_editor_set_current_view_name_interactive (editor);
}

static void
set_editing_enabled (MlViewApp *a_this,
                     gboolean a_enable)
{
        struct MlViewWidgetsHandle *widgets = NULL;
        
        g_return_if_fail (a_this && PRIVATE (a_this)) ;


        widgets = mlview_app_get_widgets_handle (a_this) ;
        g_return_if_fail (widgets) ;

        if (a_enable == TRUE) {
                if (widgets->save_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->save_menu_item, TRUE) ;

                if (widgets->saveas_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->saveas_menu_item, TRUE) ;

                if (widgets->close_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->close_menu_item, TRUE) ;

                if (widgets->action_menu)
                        gtk_widget_set_sensitive 
                                (widgets->action_menu, TRUE) ;

                if (widgets->close_button)
                        gtk_widget_set_sensitive 
                                (widgets->close_button, TRUE) ;

                if (widgets->save_button)
                        gtk_widget_set_sensitive 
                                (widgets->save_button, TRUE) ;
        } else {
                if (widgets->save_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->save_menu_item, FALSE) ;

                if (widgets->saveas_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->saveas_menu_item, FALSE) ;

                if (widgets->close_menu_item)
                        gtk_widget_set_sensitive 
                                (widgets->close_menu_item, FALSE) ;

                if (widgets->action_menu)
                        gtk_widget_set_sensitive 
                                (widgets->action_menu, FALSE) ;

                if (widgets->close_button)
                        gtk_widget_set_sensitive 
                                (widgets->close_button, FALSE) ;

                if (widgets->save_button)
                        gtk_widget_set_sensitive 
                                (widgets->save_button, FALSE) ;
        }
}

static void
application_initialized_cb (MlViewAppContext *a_this,
                            gpointer a_user_data)
{
        MlViewApp *app = NULL ;
        g_return_if_fail (a_this && MLVIEW_IS_APP_CONTEXT (a_this)
                          && a_user_data) ;

        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, FALSE) ;
}

static void
last_view_removed_cb (MlViewEditor *a_editor,
                      gpointer a_user_data)
{
        MlViewApp *app = NULL ;
        g_return_if_fail (a_editor 
                          && MLVIEW_IS_EDITOR (a_editor)
                          && a_user_data) ;
        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, FALSE) ;
}

static void
first_view_added_cb (MlViewEditor *a_editor,
                     MlViewIView *a_view,
                     gpointer a_user_data)
{
        MlViewApp *app = NULL ;

        g_return_if_fail (a_editor && MLVIEW_IS_EDITOR (a_editor)
                          && MLVIEW_IS_IVIEW (a_view)
                          && a_user_data) ;        
        app = (MlViewApp*) a_user_data ;
        set_editing_enabled (app, TRUE) ;
}


static void
go_url_home_cb (void)
{
	gnome_url_show ("http://www.mlview.org", NULL);
}

static enum MlViewStatus
connect_menu_and_toolbar_signals (MlViewApp *a_app,
                                  GladeXML *a_glade_xml)
{
        MlViewAppContext *ctxt = NULL ;

        g_return_val_if_fail (a_app && a_glade_xml,
                              MLVIEW_BAD_PARAM_ERROR) ;

        ctxt = mlview_app_get_application_context (a_app) ;
        g_return_val_if_fail (ctxt, MLVIEW_ERROR) ;

        /*connect to menu signals*/
        glade_xml_signal_connect_data (a_glade_xml, 
                                       "new_doc_menu_item_clicked_cb",
                                       G_CALLBACK (new_doc_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "open_doc_menu_item_clicked_cb",
                                       G_CALLBACK (open_doc_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "save_document_menu_item_clicked_cb",
                                       G_CALLBACK (save_document_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "save_as_menu_item_clicked_cb",
                                       G_CALLBACK (save_as_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "close_menu_item_clicked_cb",
                                       G_CALLBACK (close_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "quit_menu_item_clicked_cb",
                                       G_CALLBACK (quit_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "new_view_on_document_menu_item_clicked_cb",
                                       G_CALLBACK (new_view_on_document_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "rename_view_menu_item_clicked_cb",
                                       G_CALLBACK (rename_view_menu_item_clicked_cb),
                                       ctxt) ;
        
        glade_xml_signal_connect_data (a_glade_xml,
                                       "associate_a_dtd_menu_item_clicked_cb",
                                       G_CALLBACK (associate_a_dtd_menu_item_clicked_cb),
                                       ctxt) ;

        glade_xml_signal_connect_data (a_glade_xml,
                                       "apply_xslt_menu_item_clicked_cb",
                                       G_CALLBACK (apply_xslt_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "validate_document_menu_item_clicked_cb",
                                       G_CALLBACK (validate_document_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "about_menu_item_clicked_cb",
                                       G_CALLBACK (about_menu_item_clicked_cb),
                                       ctxt) ;

        /*connect to toolbar signals*/
        glade_xml_signal_connect_data (a_glade_xml,
                                       "new_button_clicked_cb",
                                       G_CALLBACK (new_doc_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "open_button_clicked_cb",
                                       G_CALLBACK (open_doc_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "close_button_clicked_cb",
                                       G_CALLBACK (close_menu_item_clicked_cb),
                                       ctxt) ;
        glade_xml_signal_connect_data (a_glade_xml,
                                       "save_button_clicked_cb",
                                       G_CALLBACK (save_document_menu_item_clicked_cb),
                                       ctxt) ;

        return MLVIEW_OK ;
}

/**
 *Make sure all the signals of the menu items
 *the right function callbacks are correctly
 *connected to the menu items.
 *@param a_menu_bar
 */
static enum MlViewStatus
init_menu_and_tool_bar (MlViewApp *a_app,
                        GladeXML *a_glade_xml)
{
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_glade_xml && a_app,
                              MLVIEW_BAD_PARAM_ERROR) ;

        connect_menu_and_toolbar_signals (a_app, a_glade_xml) ;
        return status ;
}

static enum MlViewStatus
init_app_win (MlViewApp *a_app,
              GladeXML *a_glade_xml)
{
        MlViewAppContext *ctxt = NULL ;
        GtkWidget *editor_container = NULL ;
        gchar *mlview_icon_filename = NULL ;
        enum MlViewStatus status = MLVIEW_ERROR ;

        g_return_val_if_fail (a_glade_xml && a_app,
                              MLVIEW_BAD_PARAM_ERROR) ;
        build_widgets_handle (a_app, a_glade_xml) ;
        /*init the editor widget*/
        status = init_editor (a_app, a_glade_xml) ;
        /*init the menubar and toolbar*/
        status = init_menu_and_tool_bar (a_app, a_glade_xml) ;
        g_return_val_if_fail (status == MLVIEW_OK, status) ;

        g_return_val_if_fail (PRIVATE (a_app)->widgets->editor,
                              MLVIEW_ERROR) ;
        ctxt = mlview_editor_get_app_context 
                (MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor)) ;
        g_return_val_if_fail (ctxt, MLVIEW_ERROR) ;
        /*
         *set the main window in the context.
         */
        g_return_val_if_fail (PRIVATE (a_app)->widgets->app_win,
                              MLVIEW_ERROR) ;
        mlview_app_context_set_element
                (ctxt, "MlViewMainWindow",
                 PRIVATE (a_app)->widgets->app_win) ;
        mlview_app_context_set_element 
                (ctxt, "MlViewEditor",
                 PRIVATE (a_app)->widgets->editor) ;
        mlview_app_context_set_element
                (ctxt, "MlViewAppMainMenuBar",
                 PRIVATE (a_app)->widgets->main_menu_bar) ;
        mlview_icon_filename = gnome_program_locate_file
                (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                 PACKAGE "/mlview-app-icon.png", TRUE,
                 NULL) ;
        g_return_val_if_fail (mlview_icon_filename, MLVIEW_ERROR) ;
        gtk_window_set_icon_from_file 
                (GTK_WINDOW (PRIVATE (a_app)->widgets->app_win),
                 mlview_icon_filename, NULL) ;
        gtk_window_set_default_icon_from_file 
                (mlview_icon_filename, NULL) ;
        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->app_win),
                          "delete-event",
                          G_CALLBACK (delete_event_cb),
                          ctxt) ;
        g_signal_connect (G_OBJECT (ctxt), 
                          "application-initialized",
                          G_CALLBACK (application_initialized_cb),
                          a_app) ;

        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->editor),
                          "last-view-removed",
                          G_CALLBACK (last_view_removed_cb),
                          a_app) ;
        g_signal_connect (G_OBJECT (PRIVATE (a_app)->widgets->editor),
                          "first-view-added",
                          G_CALLBACK (first_view_added_cb),
                          a_app) ;

        editor_container = NULL ;
        status = MLVIEW_OK ;
        /*
         *notify whomever listens that the application has
         *been initialized.
         */
        mlview_app_context_notify_application_initialized (ctxt) ;
        return status ;
}


/**
 *populate the main container with the menu bar
 *a MlViewEditor widget.
 *@param a_glade_xml the glade xml object that contains
 *the mlview app graphical layout.
 */
static enum MlViewStatus
init_editor (MlViewApp *a_app,
             GladeXML *a_glade_xml)
{
        GtkWidget *mlview_editor = NULL ;
        GtkWidget *editor_container = NULL ;
        MlViewAppContext *ctxt = NULL ;

        g_return_val_if_fail (a_glade_xml && a_app, 
                              MLVIEW_ERROR) ;

        editor_container = glade_xml_get_widget (a_glade_xml,
                                                 "EditorContainer") ;
        g_return_val_if_fail (editor_container, MLVIEW_ERROR) ;
        
        /*create and pack the MlViewEditor widget*/
        mlview_editor = mlview_editor_new ("mlview:empty", NULL) ;
        g_return_val_if_fail (mlview_editor, MLVIEW_ERROR) ;
        /*
         *associate the application context to
         *the main container so that we can grab
         *that container from the container.
         */
        ctxt = mlview_editor_get_app_context
                (MLVIEW_EDITOR (mlview_editor)) ;
        g_return_val_if_fail (ctxt, MLVIEW_ERROR) ;
        g_object_ref (G_OBJECT (ctxt)) ;
        /*
         *Pack the editor widget into the editor container
         */
        gtk_box_pack_start_defaults (GTK_BOX (editor_container),
                                     mlview_editor) ;
        PRIVATE (a_app)->widgets->editor = mlview_editor ;

        return MLVIEW_OK ;
}

static enum MlViewStatus
build_widgets_handle (MlViewApp *a_app,
                      GladeXML *a_glade_xml)
{
        g_return_val_if_fail (a_glade_xml 
                              && a_app
                              && PRIVATE (a_app)
                              && PRIVATE (a_app)->widgets,
                              MLVIEW_BAD_PARAM_ERROR) ;

        PRIVATE (a_app)->widgets->app_win = glade_xml_get_widget 
                (a_glade_xml, "AppWin") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->app_win,
                              MLVIEW_BAD_PARAM_ERROR) ;
        PRIVATE (a_app)->widgets->main_menu_bar = 
                glade_xml_get_widget (a_glade_xml,
                                      "MainMenuBar") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->main_menu_bar,
                              MLVIEW_ERROR) ;   
        PRIVATE (a_app)->widgets->file_menu =
                glade_xml_get_widget (a_glade_xml,
                                      "FileMenu") ;
        PRIVATE (a_app)->widgets->new_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "NewDocumentMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->new_menu_item,
                              MLVIEW_ERROR) ;
        PRIVATE (a_app)->widgets->open_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "OpenMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->open_menu_item,
                              MLVIEW_ERROR) ;
        PRIVATE (a_app)->widgets->save_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "SaveMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->save_menu_item,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->saveas_menu_item =
                glade_xml_get_widget (a_glade_xml,
                                      "SaveAsMenuItem") ;
        g_return_val_if_fail 
                (PRIVATE (a_app)->widgets->saveas_menu_item,
                 MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->quit_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "QuitMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->quit_menu_item,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->close_menu_item = 
                glade_xml_get_widget (a_glade_xml,
                                      "CloseMenuItem") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->quit_menu_item,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->action_menu =
                glade_xml_get_widget (a_glade_xml, "ActionMenu") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->action_menu,
                              MLVIEW_ERROR) ;
        PRIVATE (a_app)->widgets->main_toolbar = glade_xml_get_widget 
                (a_glade_xml, "MainToolbar") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->main_toolbar,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->new_button = glade_xml_get_widget
                (a_glade_xml, "NewButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->new_button,
                              MLVIEW_ERROR) ;
        
        PRIVATE (a_app)->widgets->open_button = glade_xml_get_widget
                (a_glade_xml, "OpenButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->open_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->close_button = glade_xml_get_widget
                (a_glade_xml, "CloseButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->close_button,
                              MLVIEW_ERROR) ;

        PRIVATE (a_app)->widgets->save_button = glade_xml_get_widget 
                (a_glade_xml, "SaveButton") ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->save_button,
                              MLVIEW_ERROR) ;

        return MLVIEW_OK ;
}

/****************
 *Public methods
 ***************/

/**
 *Getter of the application context 
 *associated to the current mlview application.
 *@param a_appname
 *@return the newly instanciated GtkWindow that
 *represents the mlview application.
 */
MlViewApp *
mlview_app_new (const guchar *a_appname)
{
        MlViewApp *result = NULL, *mlview_app = NULL;
        gchar *glade_file = NULL ;
        GladeXML *glade_xml = NULL ;

        mlview_app = g_try_malloc (sizeof (MlViewApp)) ;
        if (!mlview_app) {
                mlview_utils_trace_info ("Out of memory") ;
                return NULL ;
        }
        memset (mlview_app, 0, sizeof (MlViewApp)) ;        
        
        PRIVATE (mlview_app) = g_try_malloc (sizeof (MlViewAppPriv)) ;
        if (!PRIVATE (mlview_app)) {
                mlview_utils_trace_info ("Out of memory") ;
                goto end ;
        }
        memset (PRIVATE (mlview_app), 0, sizeof (MlViewAppPriv)) ;
        PRIVATE (mlview_app)->widgets = 
                g_try_malloc (sizeof (struct MlViewWidgetsHandle)) ;
        if (!PRIVATE (mlview_app)->widgets) {
                mlview_utils_trace_info ("Out of memory") ;
                goto end ;
        }
        memset (PRIVATE (mlview_app)->widgets, 0, 
                sizeof (struct MlViewWidgetsHandle)) ;
        glade_file = gnome_program_locate_file
                (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
                 PACKAGE "/mlview-main-app-win.glade", TRUE,
                 NULL) ;
        g_return_val_if_fail (glade_file, NULL) ;

        glade_xml = glade_xml_new (glade_file,
                                   "AppWin",
                                   NULL) ;
        g_return_val_if_fail (glade_xml, NULL) ;
        init_app_win (mlview_app, glade_xml) ;
        result = mlview_app ;
        /*
         *connect to MlViewEditor contextual
         *signals view and to enable/disable
         *editing.
         */
        mlview_app = NULL ;

 end:
        if (glade_xml) {
                g_object_unref (G_OBJECT (glade_xml)) ;
                glade_xml = NULL ;
        }
        if (mlview_app && PRIVATE (mlview_app) 
            && PRIVATE (mlview_app)
            && PRIVATE (mlview_app)->widgets) {
                g_free (PRIVATE (mlview_app)->widgets) ;
                PRIVATE (mlview_app)->widgets = NULL ;
        }
        if (mlview_app && PRIVATE (mlview_app) 
            && PRIVATE (mlview_app)) {
                g_free (PRIVATE (mlview_app)) ;
                PRIVATE (mlview_app) = NULL ;
        }        
        if (mlview_app) {
                g_free (mlview_app) ;
                mlview_app = NULL ;
        }
        return  result ;
}


MlViewAppContext *
mlview_app_get_application_context (MlViewApp *a_app)
{
        MlViewAppContext *ctxt = NULL ;

        g_return_val_if_fail (a_app, NULL) ;
        g_return_val_if_fail (PRIVATE (a_app)->widgets->editor,
                              NULL) ;
        ctxt = mlview_editor_get_app_context
                (MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor)) ;
        return ctxt ;
}

/**
 *Getter of the instance of MlViewEditor associated
 *to the current instance of GnomeApp.
 *
 *@param a_app the instance of GnomeApp to consider.
 *@return the instance of MlViewEditor associated to a_app or
 *NULL if nothing is associated.
 */
MlViewEditor *
mlview_app_get_editor (MlViewApp * a_app)
{
        MlViewEditor *result = NULL;

        g_return_val_if_fail (a_app 
                              && PRIVATE (a_app)->widgets->editor, 
                              NULL);
        result = MLVIEW_EDITOR (PRIVATE (a_app)->widgets->editor) ;
        return result;
}

/**
 *Returns the instance of #MlViewWidgetsHandle associated
 *to the current instance of #MlViewApp.
 *@param a_this the instance of #MlViewApp to consider.
 *@return a pointer to the widgets handle, NULL in case of
 *an error.
 */
struct MlViewWidgetsHandle *
mlview_app_get_widgets_handle (MlViewApp *a_this)
{
        g_return_val_if_fail (a_this, NULL) ;
        return PRIVATE (a_this)->widgets ;
}
