/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */

/*
 *This file is part of MlView.
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published by the 
 *Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *GNU MlView is distributed in the hope that it will 
 *be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the 
 *GNU General Public License along with MlView; 
 *see the file COPYING. 
 *If not, write to the Free Software Foundation, 
 *Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include "mlview-iview.h"
#include "mlview-marshal.h"

/**
 *@file
 *The definition of the #MlViewIView interface.
 */
enum MlVIXMLDocViewSignals {
	NAME_CHANGED,
	NB_SIGNALS
} ;

static guint gv_signals[NB_SIGNALS] = {0} ;

static void mlview_iview_base_init (gpointer a_klass) ;

static void
mlview_iview_base_init (gpointer a_klass)
{
	static gboolean initialized = FALSE ;

	if (initialized == TRUE)
		return ;

	gv_signals[NAME_CHANGED] =
		g_signal_new ("name-changed",
			      MLVIEW_TYPE_IVIEW,
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MlViewIView,
					       name_changed),
			      NULL, NULL,
			      mlview_marshal_VOID__VOID,
			      G_TYPE_NONE, 0, NULL) ;
	initialized = TRUE ;
}


/**
 *Getter/builder of the type identifier of the
 *the #MlViewIView class.
 *@return the new type identifier.
 */
GType
mlview_iview_get_type (void)
{
	static GType type = 0 ;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MlViewIView), /*class size*/
			mlview_iview_base_init, /*base_init*/
			NULL, /*base_finalize*/
			NULL, /*class init*/
			NULL, /*class finalize*/
			NULL, /*class data*/
			0, /*instance size*/
			0, /*n_prealloc*/
			NULL,/*instance_init*/
		} ;
		type = g_type_register_static (G_TYPE_INTERFACE,
					       "MlViewIView",
					       &info, 0) ;
		g_type_interface_add_prerequisite (type,
						   G_TYPE_OBJECT) ;
	}
	return type ;
}

/**
 *Connects the view to the document object model.
 *@param a_this the current instance of #MlViewIView.
 *@param a_doc the current document object model.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_iview_connect_to_doc (MlViewIView *a_this,
                             MlViewXMLDocument *a_doc)
{
	MlViewIView *iface = NULL;
	enum MlViewStatus status = MLVIEW_IFACE_NOT_DEFINED_ERROR ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;

	if (iface->connect_to_doc) {
		status = iface->connect_to_doc (a_this, a_doc) ;
	}
	return status ;
}

/**
 *Disconnects from the document object model.
 *@param a_this the current instance of #MlViewIView
 *@param a_doc the document object model to disconnect from
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus
mlview_iview_disconnect_from_doc (MlViewIView *a_this,
                                  MlViewXMLDocument *a_doc)
{
	MlViewIView *iface = NULL;
	enum MlViewStatus status = MLVIEW_IFACE_NOT_DEFINED_ERROR ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_ERROR) ;
	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;

	if (iface->disconnect_from_doc) {
		status = iface->disconnect_from_doc (a_this, a_doc) ;
	}
	return status ;
}


/**
 *Let's this view execute an editing action.
 *@param a_this the current instance of #MlViewIView.
 *@param a_action the action to execute.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_execute_action (MlViewIView *a_this,
                             MlViewAction *a_action)
{
	MlViewIView *iface = NULL ;
	enum MlViewStatus status = MLVIEW_IFACE_NOT_DEFINED_ERROR ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
			      MLVIEW_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;

	if (iface->execute_action) {
		status = iface->execute_action (a_this, a_action) ;
	}
	return status ;
}

/**
 *Sets the name of the view.
 *@param a_this the current instance of #MlViewIView
 *@param a_name the new name of the view.
 *@return MLIVEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_iview_set_name (MlViewIView *a_this,
                       guchar *a_name)
{
	MlViewIView *iface = NULL ;
	enum MlViewStatus status = MLVIEW_IFACE_NOT_DEFINED_ERROR ;

	g_return_val_if_fail (a_this
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_name
			      && a_name, 
			      MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->set_name) {
		status = iface->set_name (a_this, a_name) ;
		if (status == MLVIEW_OK) {
			g_signal_emit (G_OBJECT (a_this),
				       gv_signals[NAME_CHANGED],0) ;
		}
	}
	return status ;
}

/**
 *Gets the name of the current editing view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_name out parameter. Points to where to store the name
 *of the view. Caller must not free *a_name.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_get_name (MlViewIView *a_this,
                       guchar **a_name)
{
	MlViewIView *iface = NULL ;

	g_return_val_if_fail (a_this
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_name
			      && a_name, 
			      MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->get_name) {
		return iface->get_name (a_this, a_name) ;
	}
	return MLVIEW_OK ;
}

/**
 *Interactively sets the name of the current view.
 *@param a_this the current instance of #MlViewIView.
 *@return MLVIEW_OK upon successful completion, an error code otherwise.
 */
enum MlViewStatus 
mlview_iview_set_name_interactive (MlViewIView *a_this)
{
        MlViewIView *iface = NULL ;

        g_return_val_if_fail (a_this && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
        g_return_val_if_fail (iface, MLVIEW_ERROR) ;

        if (iface->set_name_interactive) {
                return iface->set_name_interactive (a_this) ;
        }
        return MLVIEW_OK ;
}


/**
 *Gets the document being edited by the current
 *view.
 *@param a_this the current instance of #MlViewIView .
 *@param a_doc out parameter.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_get_document (MlViewIView *a_this,
                           MlViewXMLDocument **a_doc)
{
	MlViewIView *iface = NULL ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_doc, MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->get_document) {
		return iface->get_document (a_this, a_doc) ;
	}
	return MLVIEW_OK ;
}


/**
 *Sets associates a new document to the current view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_doc the new document.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_iview_set_document (MlViewIView *a_this,
                           MlViewXMLDocument *a_doc)
{
	MlViewIView *iface = NULL ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this)
			      && a_doc, MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->set_document) {
		return iface->set_document (a_this, a_doc) ;
	}
	return MLVIEW_OK ;

}

/**
 *Edit the contextual menu according to the
 *services proposed by this view.
 *@param a_this the current instance of #MlViewIView.
 *@param a_menu out parameter.
 *@return MLVIEW_OK upon successful completion, an error
 *code otherwise.
 */
enum MlViewStatus
mlview_iview_update_contextual_menu (MlViewIView *a_this,
                                     GtkWidget **a_menu)
{
	MlViewIView *iface = NULL ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->update_contextual_menu) {
		return iface->update_contextual_menu (a_this, a_menu) ;
	}
	return MLVIEW_OK ;
}


/**
 *Gets the implementation of the current view interface.
 *@param a_this the current instance of #MlViewIView.
 *@param a_impl out parameter. 
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus 
mlview_iview_get_impl (MlViewIView *a_this, GtkWidget **a_impl)
{
        MlViewIView *iface = NULL ;

	g_return_val_if_fail (a_this 
			      && MLVIEW_IS_IVIEW (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

	iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
	g_return_val_if_fail (iface, MLVIEW_ERROR) ;
	if (iface->get_impl) {
		return iface->get_impl (a_this, a_impl) ;
	}
	return MLVIEW_OK ;
}
