/* -*- Mode: C; indent-tabs-mode:nil; c-basic-offset: 8-*- */
/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute 
 *it and/or modify it under the terms of 
 *the GNU General Public License as published 
 *by the Free Software Foundation; either version 2, 
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it 
 *will be useful, but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of 
 *MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the GNU 
 *General Public License along with MlView; 
 *see the file COPYING. If not, write to 
 *the Free Software Foundation, Inc., 
 *59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright information.
 */

#include <string.h>

#include <glade/glade.h>
#include <gtk/gtk.h>
#include "mlview-editor.h"
#include "mlview-tree-view.h"
#include "mlview-tree-editor.h"
#include "mlview-icon-tree.h"
#include "mlview-completion-table.h"


/**
 *@file
 *The definition of the #MlViewTreeView class.
 */



static void comment_node_action_cb (GtkAction *a_action,
                                    MlViewTreeView *a_this) ;

static void uncomment_node_action_cb (GtkAction *a_action,
		          MlViewTreeView *a_this) ;

static void add_child_node_action_cb (GtkAction *a_action,
                                      MlViewTreeView *a_this) ;

static void
insert_element_action_cb (GtkAction *a_action,
                          MlViewTreeView *a_this) ;

static void insert_next_sibling_node_action_cb (GtkAction *a_action,
                                                MlViewTreeView *a_this) ;

static void insert_prev_sibling_node_action_cb (GtkAction *a_action,
                                                MlViewTreeView *a_this) ;

static void copy_node_action_cb (GtkAction *a_action,
                                MlViewTreeView *a_this) ;

static void cut_node_action_cb (GtkAction *a_action,
                                MlViewTreeView *a_this) ;

static void paste_node_as_child_action_cb (GtkAction *a_action,
                                           MlViewTreeView *a_this) ;

static void paste_node_as_prev_action_cb (GtkAction *a_action,
                                          MlViewTreeView *a_this) ;

static void paste_node_as_next_action_cb (GtkAction *a_action,
                                          MlViewTreeView *a_this) ;

static void select_next_sibling_node_action_cb (GtkAction *a_action,
                                                MlViewTreeView *a_this) ;

static void select_prev_sibling_node_action_cb (GtkAction *a_action,
                                                MlViewTreeView *a_this) ;

static void select_parent_node_action_cb (GtkAction *a_action,
                                          MlViewTreeView *a_this) ;

static void find_node_action_cb (GtkAction *a_action,
                                 MlViewTreeView *a_this) ;


static void mlview_tree_view_is_swapped_out (MlViewIView *a_this,
                                             gpointer a_user_data) ;

static void mlview_tree_view_is_swapped_in (MlViewIView *a_this,
                                            gpointer a_user_data) ;

static void mlview_tree_view_application_menu_populating_requested (MlViewIView *a_this,
                                                                    gpointer a_user_data) ;


static void clear_completion_popup_submenus (MlViewTreeView *a_this) ;

static void set_tree_view_proportions_cb (MlViewTreeView *a_this,
					  gpointer a_user_data) ;

struct CompletionMenuItem {
        GtkWidget *item;
        gchar *text;
        MlViewTreeView *view;
};

struct _MlViewTreeViewPrivate {
        guchar *name ;
        GtkWidget *set_name_dialog ;

        /* all tree editors */
        GtkNotebook *tree_editors; 

        /* current node */
        xmlNode *current_node;

        /* current tree editor*/
        MlViewTreeEditor *current_tree_editor;

        /**/
        MlViewTreeEditor *raw_tree_editor ;

        /**/
        MlViewTreeEditor *icon_tree_editor ;

        /*the element/attribute names completion widget*/
        MlViewCompletionTable *completion_widget ; 

        /*the xml node editor*/
        MlViewNodeEditor *node_editor;

        /*the application context*/
        MlViewAppContext *app_context;

        /*xml document being edited*/
        MlViewXMLDocument *xml_doc ;

        GtkDialog *expand_tree_dialog;

        /*The main paned that divides the view in two main parts */
        GtkPaned *main_paned;

        GtkPaned *upper_paned1;

        guint main_paned_percentage;

        gboolean dispose_has_run;

        struct TreeEditorContextualMenuHandle *contextual_menu_handle ;

        GtkUIManager *ui_manager ;
        GtkActionGroup *action_group ;
        GtkActionGroup *add_child_element_action_group ;
        GtkActionGroup *insert_next_element_action_group ;
        GtkActionGroup *insert_prev_element_action_group ;
        guint add_child_element_merge_id ;
        guint insert_prev_element_merge_id ;
        guint insert_next_element_merge_id ;
        guint edit_menu_merge_id ;
        guint tree_view_popup_edit_menu_merge_id ;

} ;

enum {
        DOCUMENT_CHANGED,
        SIGNAL_NUM
};


static void
tree_selected_cb (GtkNotebook *notebook,
                  GtkNotebookPage *page, 
                  guint page_num,
                  gpointer data);

static void xml_node_changed_cb (MlViewNodeEditor * a_editor,
                                 gpointer a_data);

static void toggle_expand_to_leaves_cb (GtkToggleButton * a_toggle_button,
                                        gpointer * a_depth_entry);

static void doc_path_changed_cb (MlViewXMLDocument * a_xml_doc,
                                 gpointer a_xml_doc_tree_view);

static void view_swapped_cb (MlViewAppContext *a_ctxt, 
                             MlViewIView *a_old_view,
                             MlViewIView *a_new_view,
                             gpointer a_user_data) ;

static GtkDialog *get_expand_tree_dialog (MlViewTreeView * a_this);

static void mlview_tree_view_init (MlViewTreeView * a_tree_view);

static void mlview_tree_view_class_init (MlViewTreeViewClass * a_class);

static enum MlViewStatus mlview_tree_view_execute_action (MlViewIView *a_this,
                                                          MlViewAction *a_action) ;

static enum MlViewStatus mlview_tree_view_connect_to_doc (MlViewIView *a_this,
                                                          MlViewXMLDocument *a_doc)  ;

static enum MlViewStatus mlview_tree_view_disconnect_from_doc (MlViewIView *a_this,
                                                               MlViewXMLDocument *a_doc) ;

static void mlview_tree_view_dispose (GObject *a_this);

static void mlview_tree_view_finalise (GObject *a_this) ;

static void document_ext_subset_changed_cb (MlViewXMLDocument *a_xml_doc,                                            
                                            gpointer a_user_data) ;

static enum MlViewStatus mlview_tree_view_handle_contextual_menu_request (MlViewTreeView *a_this,
                                                                          GtkWidget *a_source_widget,    
                                                                          GdkEvent *a_event) ;


static gboolean contextual_menu_requested_cb (MlViewAppContext *a_ctxt,
                                              GtkWidget *a_source_widget,
                                              GdkEvent *a_event,
                                              gpointer a_user_data);


static void tree_editor_ungrab_focus_requested_cb (MlViewTreeEditor *a_this,
                                                   gpointer a_user_data) ;

static void node_editor_ungrab_focus_requested_cb (MlViewNodeEditor *a_this,
                                                   gpointer a_user_data) ;

static GtkActionEntry gv_edit_menu_actions [] = {
        {
                "CommentNodeAction", NULL, N_("Comment"),
                NULL, N_("Comment the currently selected node"),
                G_CALLBACK (comment_node_action_cb)
        },
 
        {
                "UncommentNodeAction", NULL, N_("Uncomment"),
                NULL, N_("Uncomment the currently selected node"),
                G_CALLBACK (uncomment_node_action_cb)
        },

        {
                "AddChildNodeMenuAction", NULL, N_("Add child node"),
                NULL, N_("Add a child node to the currently selected node"),
                NULL
        },
         
        {
                "AddChildNodeAction", NULL, N_("Add custom child node"),
                NULL, N_("Add a custom child node to the currently selected node"),
                G_CALLBACK (add_child_node_action_cb)
        },
         
 
        {
                "InsertNextSiblingNodeMenuAction", NULL, N_("Insert a node after"),
                NULL, N_("Insert a node after the currently selected node"),
                NULL
        },
 
        {
                "InsertNextSiblingNodeAction", NULL, N_("Insert a custom node after"),
                NULL, N_("Insert a node after the currently selected node"),
                G_CALLBACK (insert_next_sibling_node_action_cb)
        },
 
        {
                "InsertPrevSiblingNodeMenuAction", NULL, N_("Insert a  node before"),
                NULL, N_("Insert a node after the currently selected node"),
                NULL
        },
         
        {
                "InsertPrevSiblingNodeAction", NULL, N_("Insert a custom node before"),
                NULL, N_("Insert a node after the currently selected node"),
                G_CALLBACK (insert_prev_sibling_node_action_cb)
        },
 
 
        {
                "CopyNodeAction", NULL, N_("Copy node"),
                NULL, N_("Copy the current node to the internal clipboard"),
                G_CALLBACK (copy_node_action_cb)
        },
 
        {
                "CutNodeAction", NULL, N_("Cut node"),
                NULL, N_("Cuts the current node to the internal clipboard"),
                G_CALLBACK (cut_node_action_cb)
        },
         
        {
                "PasteNodeAsChildAction", NULL, N_("Paste node as child"),
                NULL, N_("Paste node as a child of the currently selected node"),
                G_CALLBACK (paste_node_as_child_action_cb)
        },
 
        {
                "PasteNodeAsPrevAction", NULL, N_("Paste node as previous sibling"),
                NULL, N_("Paste node as the previous sibling of the currently selected node"),
                G_CALLBACK (paste_node_as_prev_action_cb)
        },
 
        {
                "PasteNodeAsNextAction", NULL, N_("Paste node as next sibling"),
                NULL, N_("Paste node as the next sibling of the currently selected node"),
                G_CALLBACK (paste_node_as_next_action_cb)
        },
 
        {
                "SelectNextSiblingAction", NULL, N_("Select next sibling"),
                NULL, N_("Select the next sibling node"),
                G_CALLBACK (select_next_sibling_node_action_cb)
        },
 
        {
                "SelectPrevSiblingAction", NULL, N_("Select prev sibling"),
                NULL, N_("Select the previous sibling node"),
                G_CALLBACK (select_prev_sibling_node_action_cb)
        },
 
        {
                "SelectParentNodeAction", NULL, N_("Select parent node"),
                NULL, N_("Select the parent of the currently selected node"),
                G_CALLBACK (select_parent_node_action_cb)
        },
 
 
        {
                "FindNodeAction", GTK_STOCK_FIND, N_("Find node"),
                NULL, N_("Find a node"),
                G_CALLBACK (find_node_action_cb)
        },
 
} ;
 
#define PRIVATE(tree_view) (tree_view->priv)
  
static MlViewViewAdapterClass *parent_class = NULL;
static guint p_signals[SIGNAL_NUM] = { 0 };


/*========================================
 *private (static) gtk framework functions
 *========================================*/

static void
mlview_tree_view_init (MlViewTreeView *a_this)
{
        MlViewIView *view_iface = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && MLVIEW_IS_IVIEW (a_this));

        if (PRIVATE (a_this) == NULL) {
                PRIVATE (a_this) =
                        g_try_malloc (sizeof (MlViewTreeViewPrivate));
                if (!PRIVATE (a_this)) {
                        mlview_utils_trace_info ("g_try_malloc failed") ;
                        return ;
                }
                memset (PRIVATE (a_this), 0, sizeof (MlViewTreeViewPrivate)) ;
        }

        view_iface = MLVIEW_IVIEW_GET_IFACE (a_this) ;
        g_return_if_fail (view_iface) ;
        view_iface->execute_action = mlview_tree_view_execute_action ;
        view_iface->connect_to_doc = mlview_tree_view_connect_to_doc ;
        view_iface->disconnect_from_doc = mlview_tree_view_disconnect_from_doc ;
        view_iface->is_swapped_out = mlview_tree_view_is_swapped_out ;
        view_iface->is_swapped_in = mlview_tree_view_is_swapped_in ;
        view_iface->application_menu_populating_requested = 
                mlview_tree_view_application_menu_populating_requested ;
}


/**
 *Class initialyzer.
 *Defines the signal "document-changed" and sets it default handler to NULL. 
 *
 */
static void
mlview_tree_view_class_init (MlViewTreeViewClass *a_class)
{
        GObjectClass *gobject_class = NULL ;

        g_return_if_fail (a_class != NULL);

        parent_class = g_type_class_peek_parent (a_class);
        gobject_class = G_OBJECT_CLASS (a_class);
        g_return_if_fail (gobject_class) ;

        gobject_class->dispose = mlview_tree_view_dispose ;
        gobject_class->finalize = mlview_tree_view_finalise ;

        p_signals[DOCUMENT_CHANGED] =
                g_signal_new ("document-changed",
                              G_TYPE_FROM_CLASS (gobject_class),
                              GTK_RUN_FIRST,
                              GTK_SIGNAL_OFFSET
                              (MlViewTreeViewClass,
                               document_changed), NULL, NULL,
                              gtk_marshal_NONE__NONE,
                              GTK_TYPE_NONE, 0, NULL);
        a_class->document_changed = NULL;
}


/**********************************************************
 *private helper function and other private methods.
 *********************************************************/
static void
doc_path_changed_cb (MlViewXMLDocument * a_xml_doc,
                     gpointer a_xml_doc_tree_view)
{
        MlViewFileDescriptor *file_desc = NULL;
        MlViewTreeView *tree_view = NULL;
        gchar *path = NULL;

        g_return_if_fail (a_xml_doc != NULL);
        g_return_if_fail (MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_xml_doc_tree_view != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_xml_doc_tree_view));

        tree_view =
                MLVIEW_TREE_VIEW (a_xml_doc_tree_view);

        file_desc =
                mlview_xml_document_get_file_descriptor
                (a_xml_doc);

        g_return_if_fail (file_desc != NULL);

        path = mlview_file_descriptor_get_file_path (file_desc);

        g_return_if_fail (path != NULL);

        mlview_tree_view_set_xml_document_path
                (tree_view, path);

}

static void
mlview_tree_view_dispose (GObject *a_this)
{
        MlViewTreeView *thiz = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && G_IS_OBJECT (a_this)) ;

        thiz = MLVIEW_TREE_VIEW (a_this) ;
        if (PRIVATE (thiz)->dispose_has_run == TRUE)
                return ;
	clear_completion_popup_submenus (thiz) ;

        if (PRIVATE (thiz)->expand_tree_dialog) {
                gtk_widget_destroy
                        (GTK_WIDGET
                         (PRIVATE (thiz)->expand_tree_dialog));
                PRIVATE (thiz)->expand_tree_dialog = NULL;
        }
        if (PRIVATE (thiz)->xml_doc) {
                mlview_iview_disconnect_from_doc
                        (MLVIEW_IVIEW (thiz),
                         PRIVATE (thiz)->xml_doc) ;
                g_signal_handlers_disconnect_by_func
                        (G_OBJECT (PRIVATE (thiz)->xml_doc),
                         G_CALLBACK (document_ext_subset_changed_cb),
                         thiz) ;
                g_signal_handlers_disconnect_by_func (G_OBJECT (PRIVATE (thiz)->xml_doc),
                                                      G_CALLBACK (doc_path_changed_cb),
                                                      thiz) ;                
                mlview_xml_document_unref
                        (PRIVATE (thiz)->xml_doc) ;
                PRIVATE (thiz)->xml_doc = NULL ;
        }

        PRIVATE (thiz)->dispose_has_run = TRUE ;

        g_signal_handlers_disconnect_by_func
                (G_OBJECT (PRIVATE (thiz)->app_context),
                 G_CALLBACK (contextual_menu_requested_cb),
                 thiz) ;
        g_signal_handlers_disconnect_by_func
                (G_OBJECT (PRIVATE (thiz)->app_context),
                 G_CALLBACK (view_swapped_cb),
                 thiz) ;
        /*call the destroy method of the parent */
        if (G_OBJECT_CLASS (parent_class)->dispose)
                G_OBJECT_CLASS (parent_class)->dispose
                        (a_this) ;
}

static void
mlview_tree_view_finalise (GObject *a_this)
{
        MlViewTreeView *thiz = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_TREE_VIEW (a_this)) ;
        thiz = MLVIEW_TREE_VIEW (a_this) ;
        g_return_if_fail (thiz) ;
        if (PRIVATE (thiz)) {
                g_free (PRIVATE (thiz)) ;
                PRIVATE (thiz) = NULL ;
        }
}


/*========================================
 *private (static) helper functions
 *========================================*/

static void 
tree_editor_ungrab_focus_requested_cb (MlViewTreeEditor *a_this,
                                       gpointer a_user_data)
{
        MlViewTreeView *thiz = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_EDITOR (a_this)) ;

        thiz = MLVIEW_TREE_VIEW (a_user_data) ;
        g_return_if_fail (thiz) ;
        g_return_if_fail (PRIVATE (thiz) && PRIVATE (thiz)->node_editor) ;

        mlview_node_editor_grab_focus (PRIVATE (thiz)->node_editor) ;
}


static void 
node_editor_ungrab_focus_requested_cb (MlViewNodeEditor *a_this,
                                       gpointer a_user_data)
{
        MlViewTreeView *thiz = NULL ;
        
        g_return_if_fail (a_this 
                          && MLVIEW_IS_NODE_EDITOR (a_this)
                          && a_user_data) ;

        thiz = MLVIEW_TREE_VIEW (a_user_data) ;
        g_return_if_fail (thiz) ;
        g_return_if_fail (PRIVATE (thiz) && PRIVATE (thiz)->current_tree_editor) ;

        mlview_tree_editor_grab_focus (PRIVATE (thiz)->current_tree_editor) ;

}

/**
  *This callback is called when the user
  *selects a tree in the trees' notebook.
  */
static void
tree_selected_cb (GtkNotebook *notebook,
                  GtkNotebookPage *page, 
                  guint page_num,
                  gpointer data)
{
        MlViewTreeEditor *tree;
        MlViewTreeView *thiz = data;

        g_return_if_fail (thiz && MLVIEW_IS_TREE_VIEW (thiz));

        tree = MLVIEW_TREE_EDITOR (gtk_notebook_get_nth_page (notebook, page_num));
        
        g_return_if_fail (tree && MLVIEW_IS_TREE_EDITOR (tree));
        
        PRIVATE (thiz)->current_tree_editor = tree;
}



/**
 *The callback of the "node-changed signal" 
 *emited by the #MlViewXMLDocument 
 */
static void
xml_node_changed_cb (MlViewNodeEditor * a_node_editor,
                     gpointer a_data)
{
        GtkTreeView *visual_tree=NULL;
        xmlNodePtr xml_node=NULL;
        MlViewTreeView *view=NULL;
        gint i, nb_pages;
        MlViewTreeEditor *tree;

        g_return_if_fail (a_node_editor
                          && MLVIEW_IS_NODE_EDITOR (a_node_editor)
                          && a_data
                          && MLVIEW_IS_TREE_VIEW (a_data)) ;
        xml_node =
                mlview_node_editor_get_current_xml_node 
                (a_node_editor);
        g_return_if_fail (xml_node != NULL);

        /*get the visual tree */
        view = MLVIEW_TREE_VIEW (a_data);
        
        g_return_if_fail (view && PRIVATE (view)
                          && PRIVATE (view)->tree_editors);
        nb_pages = gtk_notebook_get_n_pages (PRIVATE (view)->tree_editors);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR (gtk_notebook_get_nth_page 
                                            (PRIVATE (view)->tree_editors, i));
                g_return_if_fail (tree && MLVIEW_IS_TREE_EDITOR (tree));
                visual_tree = mlview_tree_editor_get_tree_view (tree);
                g_return_if_fail (visual_tree && GTK_IS_TREE_VIEW (visual_tree));
                mlview_tree_editor_update_visual_node2 (tree, xml_node); }
}


/**
 *
 */
static void
toggle_expand_to_leaves_cb (GtkToggleButton * a_toggle_button,
                            gpointer * a_depth_entry)
{
        g_return_if_fail (a_toggle_button != NULL);
        g_return_if_fail (GTK_IS_TOGGLE_BUTTON
                          (a_toggle_button));
        g_return_if_fail (a_depth_entry != NULL);
        g_return_if_fail (GTK_IS_WIDGET (a_depth_entry));

        if (gtk_toggle_button_get_active (a_toggle_button) ==
            TRUE)
                gtk_widget_set_sensitive (GTK_WIDGET
                                          (a_depth_entry),
                                          FALSE);
        else
                gtk_widget_set_sensitive (GTK_WIDGET
                                          (a_depth_entry), TRUE);
}

/**
 *Callback called when the MlViewXMLDocument emits
 *a "node-selected" signal ; updates the MlViewCompletionTable.
 */
static void
update_completion_widget_cb (MlViewXMLDocument *a_xml_doc,
                             xmlNode *a_node_found,
                             gpointer a_user_data)
{
        g_return_if_fail (a_xml_doc && MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (a_node_found);
        g_return_if_fail (a_user_data && MLVIEW_IS_COMPLETION_TABLE (a_user_data));

        mlview_completion_table_select_node (MLVIEW_COMPLETION_TABLE (a_user_data),
                                             a_node_found);
}


static gboolean
contextual_menu_requested_cb (MlViewAppContext *a_ctxt,
                              GtkWidget *a_source_widget,
                              GdkEvent *a_event,
                              gpointer a_user_data)
{
        MlViewTreeView *tree_view = a_user_data ;

        g_return_val_if_fail (a_ctxt, FALSE) ;
        g_return_val_if_fail (GTK_IS_WIDGET (a_source_widget), FALSE) ;
        g_return_val_if_fail (tree_view, FALSE) ;
        g_return_val_if_fail (MLVIEW_IS_TREE_VIEW (tree_view), FALSE) ;

        mlview_tree_view_handle_contextual_menu_request 
                (tree_view, a_source_widget, a_event) ;
        
        return FALSE ;
}

static void 
set_tree_view_proportions_cb (MlViewTreeView *a_this,
			      gpointer a_user_data)
{
	g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
			  && GTK_IS_WIDGET (a_this)) ;

	mlview_tree_view_set_upper_paned1_proportions (a_this, 75) ;
	mlview_tree_view_set_main_paned_proportions (a_this, 44) ;
}

/**
 *This function builds a GtkDialog that will ask the user
 *to entrer the depth to which she wants the tree to be expanded.
 *The expansion depth is entered in a GtkEntry widget 
 *associated with the returned
 *dialog (using the function gtk_object_set_data ()) via the key "depth-entry".
 *The expand-to-leaves info is intered in a 
 *GtkCheckButton widget associated with the
 *returned dialg (using the function gtk_object_set_data ()) 
 *via the key "expand-to-leave".
 *The info intered by the user can then be recovered by using the function
 *gtk_object_get_data () using the key "depth-entry" 
 *the get the GtkEntry widget that
 *captured the depth entry if any or the 
 *key "expand-to-leaves" to get the GtkCheckButton widget
 *that captured the "expand-to-leaves" info. 
 *
 */
static GtkDialog *
get_expand_tree_dialog (MlViewTreeView * a_this) 
{
        GtkWidget *table = NULL,
                *label = NULL,
                *depth_entry = NULL;
        GtkWidget *expand_to_leaves = NULL;

        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (MLVIEW_IS_TREE_VIEW
                              (a_this), NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL, NULL);

        if (PRIVATE (a_this)->expand_tree_dialog == NULL) {
                PRIVATE (a_this)->expand_tree_dialog =
                        GTK_DIALOG
                        (gtk_dialog_new_with_buttons
                         (_("Choose the depth of the tree expansion"),
                          NULL, GTK_DIALOG_MODAL,
                          GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                          GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,
                          NULL));
                g_return_val_if_fail 
                        (PRIVATE (a_this)->expand_tree_dialog, NULL) ;
                gtk_dialog_set_default_response 
                        (GTK_DIALOG 
                         (PRIVATE (a_this)->expand_tree_dialog),
                          GTK_RESPONSE_ACCEPT) ;
                expand_to_leaves =
                        gtk_check_button_new_with_label 
                        (_("expand to leaves"));

                gtk_box_pack_start_defaults
                        (GTK_BOX
                         (PRIVATE (a_this)->expand_tree_dialog->
                          vbox), expand_to_leaves);

                label = gtk_label_new (_
                                       ("absolute expansion depth:"));
                depth_entry = gtk_entry_new ();
                gtk_entry_set_text (GTK_ENTRY (depth_entry),
                                    "2");

                table = gtk_table_new (1, 2, FALSE);
                gtk_table_attach_defaults (GTK_TABLE (table),
                                           label, 0, 1, 0, 1);
                gtk_table_attach_defaults (GTK_TABLE (table),
                                           depth_entry, 1, 2, 0,
                                           1);

                gtk_box_pack_start_defaults
                        (GTK_BOX
                         (PRIVATE (a_this)->expand_tree_dialog->
                          vbox), table);

                g_signal_connect (G_OBJECT (expand_to_leaves),
                                  "toggled",
                                  G_CALLBACK
                                  (toggle_expand_to_leaves_cb),
                                  depth_entry);

                gtk_widget_show_all
                        (PRIVATE (a_this)->expand_tree_dialog->
                         vbox);

                gtk_object_set_data
                        (GTK_OBJECT
                         (PRIVATE (a_this)->expand_tree_dialog),
                         "expand-to-leaves", expand_to_leaves);

                gtk_object_set_data
                        (GTK_OBJECT
                         (PRIVATE (a_this)->expand_tree_dialog),
                         "depth-entry", depth_entry);
        }

        return PRIVATE (a_this)->expand_tree_dialog;
}


#define CHECK_OBJECT(obj) \
if (!obj) { \
        mlview_utils_trace_info \
                ("Out of memory") ; \
        status = MLVIEW_OUT_OF_MEMORY_ERROR ; \
        goto end ; \
}


/*****************************
 *activated Actions callbacks
 *****************************/

static void 
comment_node_action_cb (GtkAction *a_action,
                        MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;
        
        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)) ;
        action.name = (guchar*) "comment-current-node";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void
uncomment_node_action_cb (GtkAction *a_action,
		          MlViewTreeView *a_this)
{
	MlViewAction action = {0} ;

	g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
			  && PRIVATE (a_this)) ;
	action.name = (guchar*) "uncomment-current-node" ;
	mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
			                 &action) ;
}

static void 
add_child_node_action_cb (GtkAction *a_action,
                          MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        
        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "add-child-node-interactive";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void
insert_element_action_cb (GtkAction *a_action,
                          MlViewTreeView *a_this)
{
        gchar *action_name = NULL, *label = NULL ;
        
        g_return_if_fail (a_action && GTK_IS_ACTION (a_action)
                          && a_this && MLVIEW_IS_TREE_VIEW (a_this)) ;

        action_name = (gchar*) gtk_action_get_name (a_action) ;
        g_object_get (G_OBJECT (a_action),
                      "label", &label,
                      NULL) ;
        g_return_if_fail (label) ;

        if (g_str_has_prefix (action_name,
                              "AddElementAction")) {
                if (!strcmp (label, "#PCDATA")) {
                        mlview_tree_view_add_child_text_node (a_this, "text node") ;
                } else {
                        mlview_tree_view_add_child_element_node (a_this,
                                                                 label) ;
                }
        }
        if (g_str_has_prefix (action_name,
                              "InsertPrevSiblingElementAction")) {
                if (!strcmp (label, "#PCDATA")) {
                        mlview_tree_view_insert_prev_sibling_text_node (a_this, "text node") ;
                } else {
                        mlview_tree_view_insert_prev_sibling_element_node (a_this,
                                                                           label) ;
                }
        }
        if (g_str_has_prefix (action_name,
                              "InsertNextSiblingElementAction")) {
                if (!strcmp (label, "#PCDATA")) {
                        mlview_tree_view_insert_next_sibling_text_node (a_this, 
                                                                        "text node") ;
                } else {
                        mlview_tree_view_insert_next_sibling_element_node (a_this,
                                                                           label) ;
                }
        }
}

static void 
insert_next_sibling_node_action_cb (GtkAction *a_action,
                                    MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "insert-next-sibling-node-interactive";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
insert_prev_sibling_node_action_cb (GtkAction *a_action,
                                    MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "insert-prev-sibling-node-interactive";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
copy_node_action_cb (GtkAction *a_action,
                     MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "copy-node";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
cut_node_action_cb (GtkAction *a_action,
                    MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "cut-node";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
paste_node_as_child_action_cb (GtkAction *a_action,
                               MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)) ;

        action.name = (guchar*) "paste-node-as-child";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
paste_node_as_prev_action_cb (GtkAction *a_action,
                              MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "paste-node-as-prev-sibling";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
paste_node_as_next_action_cb (GtkAction *a_action,
                              MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)) ;

        action.name = (guchar*) "past-node-as-next-sibling";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
select_next_sibling_node_action_cb (GtkAction *a_action,
                                    MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "select-next-sibling-node";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
select_prev_sibling_node_action_cb (GtkAction *a_action,
                                    MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "select-prev-sibling-node";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void 
select_parent_node_action_cb (GtkAction *a_action,
                              MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "select-parent-node";

        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}


static void 
find_node_action_cb (GtkAction *a_action,
                     MlViewTreeView *a_this)
{
        MlViewAction action = {0} ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        action.name = (guchar*) "find-node-that-contains-str-interactive";
        mlview_tree_view_execute_action (MLVIEW_IVIEW (a_this),
                                         &action) ;
}

static void
view_swapped_cb (MlViewAppContext *a_ctxt, 
                 MlViewIView *a_old_view,
                 MlViewIView *a_new_view,
                 gpointer a_user_data)
{
        MlViewTreeView *tree_view = NULL ;
        GtkWidget *app_menu_bar = NULL ;

        g_return_if_fail (a_ctxt
                          && MLVIEW_IS_APP_CONTEXT (a_ctxt)) ;

        /*
         *if we are not the currently selected view,
         *get out!
         */
        if (a_new_view != MLVIEW_IVIEW (tree_view))
                return ;

        app_menu_bar = mlview_app_context_get_element 
                (a_ctxt, "MlViewAppMainMenuBar") ;
        g_return_if_fail (app_menu_bar) ;
}


static const gchar*
build_edit_menu_root_path (MlViewTreeView *a_this,
                           gboolean a_popup)
{
        gchar *menu_root_path = NULL ;

        g_return_val_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this),
                              NULL) ;

        if (a_popup == TRUE) {
                menu_root_path = (gchar*)"/TreeViewPopupEditMenu" ;
        } else {
                menu_root_path = (gchar*)"/MainMenubar/EditMenu" ;
        }
        return menu_root_path ;
}

static enum MlViewStatus
build_edit_menu_body (MlViewTreeView *a_this,
                      const gchar *a_menu_root_path)
{
        guint *merge_id = NULL ;

        gchar * parent_menu_path = NULL ;
        GtkUIManager *ui_manager = NULL ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_val_if_fail (ui_manager, MLVIEW_ERROR) ;

        if (!strcmp (a_menu_root_path, "/MainMenubar/EditMenu")) {
                if (!PRIVATE (a_this)->edit_menu_merge_id ) {
                        PRIVATE (a_this)->edit_menu_merge_id = 
                                gtk_ui_manager_new_merge_id (ui_manager) ;
                }
                merge_id = &PRIVATE (a_this)->edit_menu_merge_id ;
        } else if (!strcmp (a_menu_root_path, "/TreeViewPopupEditMenu")) {
                merge_id = &PRIVATE (a_this)->tree_view_popup_edit_menu_merge_id ;
        } else {
                mlview_utils_trace_info ("Unknown menu root path:") ;
                mlview_utils_trace_info (a_menu_root_path) ;
                return MLVIEW_ERROR ;
        }

        /*
         *build the contextual menu in here !
         */
        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "CommentNodeMenuitem",
                               "CommentNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "UncommentNodeMenuitem",
                               "UncommentNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

         gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator0",
                                NULL,
                                GTK_UI_MANAGER_SEPARATOR,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "AddChildNodeMenu",
                               "AddChildNodeMenuAction",
                               GTK_UI_MANAGER_MENU,
                               FALSE) ;

        parent_menu_path = g_strjoin ("/",
                                      a_menu_root_path,
                                      "AddChildNodeMenu",
                                      NULL) ;
        g_return_val_if_fail (parent_menu_path, MLVIEW_ERROR) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               parent_menu_path,
                               "AddChildNodeMenuitem",
                               "AddChildNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;
        if (parent_menu_path) {
                g_free (parent_menu_path) ;
                parent_menu_path = NULL ;
        }
        
        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "InsertNextSiblingNodeMenu",
                               "InsertNextSiblingNodeMenuAction",
                               GTK_UI_MANAGER_MENU,
                               FALSE) ;
        parent_menu_path = g_strjoin ("/",
                                      a_menu_root_path,
                                      "InsertNextSiblingNodeMenu",
                                      NULL) ;
        g_return_val_if_fail (parent_menu_path, MLVIEW_ERROR) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               parent_menu_path,
                               "InsertNextSiblingNodeMenuitem",
                               "InsertNextSiblingNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;
        if (parent_menu_path) {
                g_free (parent_menu_path) ;
                parent_menu_path = NULL ;
        }

        
        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "InsertPrevSiblingNodeMenu",
                               "InsertPrevSiblingNodeMenuAction",
                               GTK_UI_MANAGER_MENU,
                               FALSE) ;
        
        parent_menu_path = g_strjoin ("/",
                                      a_menu_root_path,
                                      "InsertPrevSiblingNodeMenu",
                                      NULL) ;
        g_return_val_if_fail (parent_menu_path, MLVIEW_ERROR) ;
        
        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               parent_menu_path,
                               "InsertPrevSiblingNodeMenuitem",
                               "InsertPrevSiblingNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;
        if (parent_menu_path) {
                g_free (parent_menu_path) ;
                parent_menu_path = NULL ;
        }

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator1",
                               NULL,
                               GTK_UI_MANAGER_SEPARATOR,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "CopyNodeMenuitem",
                               "CopyNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;


        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "CutNodeMenuitem",
                               "CutNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

         gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator2",
                                NULL,
                               GTK_UI_MANAGER_SEPARATOR,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "PasteNodeAsChildMenuitem",
                               "PasteNodeAsChildAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "PasteNodeAsPrevMenuitem",
                               "PasteNodeAsPrevAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "PasteNodeAsNextMenuitem",
                               "PasteNodeAsNextAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

         gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator3",
                                NULL,
                                GTK_UI_MANAGER_SEPARATOR,
                                FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "SelectNextSiblingNodeMenuitem",
                               "SelectNextSiblingAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "SelectPrevSiblingNodeMenuitem",
                               "SelectPrevSiblingAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator4",
                               NULL,
                               GTK_UI_MANAGER_SEPARATOR,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "SelectParentNodeMenuitem",
                               "SelectParentNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "TreeViewEditMenuSeparator5",
                               NULL,
                               GTK_UI_MANAGER_SEPARATOR,
                               FALSE) ;

        gtk_ui_manager_add_ui (ui_manager,
                               *merge_id,
                               a_menu_root_path,
                               "FindNodeMenuitem",
                               "FindNodeAction",
                               GTK_UI_MANAGER_AUTO,
                               FALSE) ;

        gtk_ui_manager_ensure_update (ui_manager) ;        
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_build_contextual_menu2 (MlViewTreeView *a_this)
{
        gchar * menu_root_path = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_val_if_fail (a_this && MLVIEW_TREE_VIEW (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        menu_root_path = (gchar*)build_edit_menu_root_path (a_this, TRUE) ;
        g_return_val_if_fail (menu_root_path, MLVIEW_BAD_PARAM_ERROR) ;
        status = build_edit_menu_body (a_this, menu_root_path) ;
        g_return_val_if_fail (status == MLVIEW_OK, status) ;

        return MLVIEW_OK ;
}


static enum MlViewStatus
mlview_tree_view_build_app_edit_menu (MlViewTreeView *a_this)
{
        enum MlViewStatus status = MLVIEW_OK ;
        gchar *menu_root_path = NULL ;

        menu_root_path = (gchar*)build_edit_menu_root_path (a_this, FALSE) ;
        g_return_val_if_fail (menu_root_path, MLVIEW_ERROR) ;
        status = build_edit_menu_body (a_this, menu_root_path) ;
        g_return_val_if_fail (status == MLVIEW_OK, status) ;
        return MLVIEW_OK ;        
}


static void
clear_completion_popup_submenus (MlViewTreeView *a_this)
{
        GtkUIManager *ui_manager = NULL ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_if_fail (ui_manager) ;
        
        gtk_ui_manager_remove_ui (ui_manager,
                                  PRIVATE (a_this)->add_child_element_merge_id) ;
	PRIVATE (a_this)->add_child_element_merge_id= 0 ;
        gtk_ui_manager_remove_ui (ui_manager,
                                  PRIVATE (a_this)->insert_next_element_merge_id) ;
	PRIVATE (a_this)->insert_next_element_merge_id = 0 ;
        gtk_ui_manager_remove_ui (ui_manager,
                                  PRIVATE (a_this)->insert_prev_element_merge_id) ;
	PRIVATE (a_this)->insert_prev_element_merge_id = 0 ;
	gtk_ui_manager_ensure_update (ui_manager) ;
}

static void
update_completion_popup_submenu2 (MlViewTreeView *a_this,
                                  MlViewAppContext *a_context,
                                  const gchar *a_menu_root_path,
                                  xmlNodePtr a_node,
                                  enum NODE_INSERTION_SCHEME a_insertion_scheme)
{
        GtkAction *action = NULL ;
        GtkUIManager *ui_manager = NULL ;
        GtkActionGroup *action_group = NULL ;
        GList *children_name_list = NULL ;
        gint nb_of_names = 0;
        gchar *cur_name = NULL ;
        GList *cur = NULL ;
        guint *merge_id_ptr = 0 ;
        gchar *menu_path = NULL, 
                *action_name_prefix = NULL,
                *action_name = NULL ;

        g_return_if_fail (a_context && 
                          MLVIEW_IS_APP_CONTEXT (a_context) 
                          && a_node
                          && a_node->doc
                          && a_node->doc->extSubset);
        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this));

        nb_of_names =
                mlview_parsing_utils_build_element_name_completion_list
                (a_context,
                 a_insertion_scheme,
                 a_node, &children_name_list);
        
        if (nb_of_names <= 0)
                return ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_if_fail (ui_manager) ;

        if (!PRIVATE (a_this)->add_child_element_action_group) {
                PRIVATE (a_this)->add_child_element_action_group =
                        gtk_action_group_new ("AddChildElementActionsGroup") ;
		gtk_action_group_set_translation_domain (PRIVATE (a_this)->add_child_element_action_group,
				                         GETTEXT_PACKAGE) ;
                g_return_if_fail (PRIVATE (a_this)->add_child_element_action_group) ;
                gtk_ui_manager_insert_action_group (ui_manager,
                                                    PRIVATE (a_this)->add_child_element_action_group,
                                                    0) ;
        }
        if (!PRIVATE (a_this)->add_child_element_merge_id) {
                PRIVATE (a_this)->add_child_element_merge_id =
                        gtk_ui_manager_new_merge_id (ui_manager) ;
        }
        if (!PRIVATE (a_this)->insert_next_element_merge_id) {
                PRIVATE (a_this)->insert_next_element_merge_id =
                        gtk_ui_manager_new_merge_id (ui_manager) ;
        }
        if (!PRIVATE (a_this)->insert_prev_element_merge_id) {
                PRIVATE (a_this)->insert_prev_element_merge_id =
                        gtk_ui_manager_new_merge_id (ui_manager) ;
        }
        if (!PRIVATE (a_this)->insert_prev_element_action_group) {
                PRIVATE (a_this)->insert_prev_element_action_group = 
                        gtk_action_group_new ("InsertNextElementActionsGroup") ;
		gtk_action_group_set_translation_domain (PRIVATE (a_this)->insert_prev_element_action_group,
							 GETTEXT_PACKAGE) ;
                g_return_if_fail (PRIVATE (a_this)->insert_prev_element_action_group) ;
                gtk_ui_manager_insert_action_group (ui_manager,
                                                    PRIVATE (a_this)->insert_prev_element_action_group,
                                                    0) ;
        }
        if (!PRIVATE (a_this)->insert_next_element_action_group) {
                PRIVATE (a_this)->insert_next_element_action_group = 
                        gtk_action_group_new ("InsertNextElementActionsGroup") ;
		gtk_action_group_set_translation_domain (PRIVATE (a_this)->insert_next_element_action_group,
							 GETTEXT_PACKAGE) ;
                g_return_if_fail (PRIVATE (a_this)->insert_prev_element_action_group) ;
                gtk_ui_manager_insert_action_group (ui_manager,
                                                    PRIVATE (a_this)->insert_next_element_action_group,
                                                    0) ;
        }


        switch (a_insertion_scheme) {
        case ADD_CHILD:
                action_name_prefix = (gchar*) "AddElementAction" ;
                merge_id_ptr = &PRIVATE (a_this)->add_child_element_merge_id ;
                menu_path = g_strjoin ("/", 
                                       a_menu_root_path,
                                       "AddChildNodeMenu",
                                       NULL) ;
                g_return_if_fail (menu_path) ;
                action_group = PRIVATE (a_this)->add_child_element_action_group ;
                break ;
        case INSERT_BEFORE:
                action_name_prefix = (gchar*) "InsertPrevSiblingElementAction" ;
                merge_id_ptr = &PRIVATE (a_this)->insert_prev_element_merge_id ;
                menu_path = g_strjoin ("/", 
                                       a_menu_root_path,
                                       "InsertPrevSiblingNodeMenu",
                                       NULL) ;
                g_return_if_fail (menu_path) ;
                action_group = PRIVATE (a_this)->insert_prev_element_action_group ;
                break ;
        case INSERT_AFTER:
                action_name_prefix = (gchar*) "InsertNextSiblingElementAction" ;
                merge_id_ptr = &PRIVATE (a_this)->insert_next_element_merge_id ;
                menu_path = g_strjoin ("/", 
                                       a_menu_root_path,
                                       "InsertNextSiblingNodeMenu",
                                       NULL) ;
                g_return_if_fail (menu_path) ;
                action_group = PRIVATE (a_this)->insert_next_element_action_group ;
                break ;
        case CHANGE_CUR_ELEMENT_NAME:
                g_assert_not_reached () ;
                break ;
        default:
                break ;
        }

        cur = children_name_list ;
        cur_name = cur->data ;
        while (cur_name) {
                /*TODO, finish this*/
                action_name = mlview_utils_get_unique_string 
                        (action_name_prefix) ;
                action = gtk_action_new (action_name,
                                         cur_name,
                                         NULL,
                                         NULL) ;
                g_signal_connect (G_OBJECT (action),
                                  "activate",
                                  G_CALLBACK (insert_element_action_cb),
                                  a_this) ;
                gtk_action_group_add_action (action_group,
                                              action) ;
                gtk_ui_manager_add_ui (ui_manager,
                                       *merge_id_ptr,
                                       menu_path,
                                       cur_name,
                                       action_name,
                                       GTK_UI_MANAGER_MENUITEM,
                                       FALSE) ;

                if (action_name) {
                        g_free (action_name) ;
                        action_name = NULL ;
                }
                cur = (cur->next) ? cur->next : NULL;
                cur_name = (cur) ? (gchar *) cur->data : NULL;
        }

        if (menu_path) {
                g_free (menu_path) ;
                menu_path = NULL ;
        }
}

static enum MlViewStatus
mlview_tree_view_handle_contextual_menu_request (MlViewTreeView *a_this,
                                                 GtkWidget *a_source_widget,
                                                 GdkEvent *a_event)
{
        GtkWidget *menu = NULL ;
        enum MlViewStatus status = MLVIEW_OK ;
        MlViewAppContext *ctxt = NULL ;
        GdkEventButton *event_button = NULL;

        g_return_val_if_fail (a_this 
                              && MLVIEW_TREE_VIEW (a_this)
                              && a_source_widget
                              && GTK_IS_WIDGET (a_source_widget)
                              && a_event, 
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (a_source_widget 
            == GTK_WIDGET (PRIVATE (a_this)->node_editor)
            || a_source_widget 
            == GTK_WIDGET (PRIVATE (a_this)->current_tree_editor)) {

                if (a_event->type == GDK_BUTTON_PRESS) {
                        event_button = (GdkEventButton*)a_event ;

                        status = mlview_tree_view_get_contextual_menu
                                (a_this, &menu) ;
                        g_return_val_if_fail (status == MLVIEW_OK
                                              && menu
                                              && GTK_IS_MENU (menu), 
                                              status) ;

                        status = mlview_tree_view_get_app_context 
                                (a_this, &ctxt) ;
                        g_return_val_if_fail 
                                (status == MLVIEW_OK && ctxt, status) ;

                        gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL,
                                        ctxt, event_button->button,
                                        event_button->time) ;
                }
                
        }
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_execute_action (MlViewIView *a_this,
                                 MlViewAction *a_action)
{
        MlViewTreeView *view = NULL ;
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_action,
                              MLVIEW_BAD_PARAM_ERROR) ;

        view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (PRIVATE (view), MLVIEW_BAD_PARAM_ERROR) ;

        if (!strcmp
            (a_action->name,
             "add-child-node-interactive")) {
                mlview_tree_view_add_child_node_interactive
                        (view) ;
        } else if (!strcmp
                   (a_action->name, 
                    "insert-prev-sibling-node-interactive") ) {
                mlview_tree_view_insert_prev_sibling_node_interactive 
                        (view) ;
        } else if (!strcmp 
                   (a_action->name,
                    "insert-next-sibling-node-interactive") ) {
                mlview_tree_view_insert_next_sibling_node_interactive
                        (view);
        } else if (!strcmp (a_action->name,
                            "cut-node") ) {
                mlview_tree_view_cut_node (view) ;
        } else if (!strcmp (a_action->name, 
                            "copy-node") ) {
                mlview_tree_view_copy_node (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-child") ) {
                mlview_tree_view_paste_node_as_child 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-prev-sibling") ) {
                mlview_tree_view_paste_node_as_prev_sibling 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "paste-node-as-next-sibling") ) {
                mlview_tree_view_paste_node_as_next_sibling 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "expand-tree-to-depth-interactive") ) {
                mlview_tree_view_expand_tree_to_depth_interactive 
                        (view) ;
        } else if (!strcmp (a_action->name, 
                            "find-node-that-contains-str-interactive") ) {
                mlview_tree_view_find_xml_node_that_contains_str_interactive
                        (view) ;
        } else if (!strcmp (a_action->name,
                            "create-internal-subset-node-interactive")) {
                mlview_tree_view_create_internal_subset_node_interactive 
                        (view) ;
        } else if (!strcmp (a_action->name,
                            "comment-current-node")){
                mlview_tree_view_comment_current_node (view) ;
        } else if (!strcmp (a_action->name,
                            "uncomment-current-node")){
                mlview_tree_view_uncomment_current_node (view) ;
        }else if (!strcmp (a_action->name,
                            "select-parent-node")) {
                mlview_tree_view_select_parent_node (view) ;
        } else if (!strcmp (a_action->name,
                            "select-prev-sibling-node")) {
                mlview_tree_view_select_prev_sibling_node (view) ;
        } else if (!strcmp (a_action->name,
                            "select-next-sibling-node")) {
                mlview_tree_view_select_next_sibling_node (view) ;
        }else  {
                guchar *err_msg = NULL ;
                g_strconcat ("Unknown edition action: ",
                             a_action->name,
                             NULL) ;
                mlview_utils_trace_info (err_msg) ;
                if (err_msg) {
                        g_free (err_msg) ;
                        err_msg = NULL ;
                }

        }
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_connect_to_doc (MlViewIView *a_this,
                                 MlViewXMLDocument *a_doc)
{
        MlViewTreeView *tree_view = NULL ;
        MlViewTreeEditor *tree;
        gint i, nb_pages;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        tree_view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (tree_view 
                              && PRIVATE (tree_view)->tree_editors,
                              MLVIEW_BAD_PARAM_ERROR) ;

        nb_pages = gtk_notebook_get_n_pages (PRIVATE (tree_view)->tree_editors);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR 
                        (gtk_notebook_get_nth_page (PRIVATE (tree_view)->tree_editors, i));
                g_return_val_if_fail (tree && MLVIEW_IS_TREE_EDITOR (tree),
                                      MLVIEW_BAD_PARAM_ERROR);
                mlview_tree_editor_connect_to_doc (tree, a_doc); 
        }
      
        mlview_node_editor_connect_to_doc (PRIVATE
                                           (tree_view)->node_editor,
                                           a_doc) ;
        return MLVIEW_OK ;
}

static enum MlViewStatus
mlview_tree_view_disconnect_from_doc (MlViewIView *a_this,
                                      MlViewXMLDocument *a_doc)
{
        MlViewTreeView *tree_view = NULL ;
        MlViewTreeEditor *tree;
        gint i, nb_pages;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_IVIEW (a_this)
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && a_doc
                              && MLVIEW_IS_XML_DOCUMENT (a_doc),
                              MLVIEW_BAD_PARAM_ERROR) ;

        tree_view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_val_if_fail (tree_view 
                              && PRIVATE (tree_view)->tree_editors,
                              MLVIEW_BAD_PARAM_ERROR) ;

        nb_pages = gtk_notebook_get_n_pages (PRIVATE (tree_view)->tree_editors);
        for (i = 0; i < nb_pages; i++) {
                tree = MLVIEW_TREE_EDITOR 
                        (gtk_notebook_get_nth_page (PRIVATE (tree_view)->tree_editors, i));
                g_return_val_if_fail (tree && MLVIEW_IS_TREE_EDITOR (tree),
                                      MLVIEW_BAD_PARAM_ERROR);
                mlview_tree_editor_disconnect_from_doc (tree, a_doc); }
      
        mlview_node_editor_disconnect_from_doc (PRIVATE
                                                (tree_view)->node_editor,
                                                a_doc) ;
        
        g_signal_handlers_disconnect_by_func (G_OBJECT (a_doc),
                                              G_CALLBACK (update_completion_widget_cb),
                                              PRIVATE (tree_view)->completion_widget);
        
        g_signal_handlers_disconnect_by_func (G_OBJECT (a_doc),
                                              G_CALLBACK (doc_path_changed_cb),
                                              tree_view);

        return MLVIEW_OK ;
}



static enum MlViewStatus
activate_or_deactivate_proper_menu_items2 (MlViewTreeView *a_this,
                                           const gchar *a_menu_root_path)
{
        GtkUIManager *ui_manager = NULL ;
        GtkAction *action = NULL ;
        gchar *menu_root_path = NULL ;

        struct MlViewAppSettings *settings = NULL;
        xmlNode *cur_node = NULL, *root_element = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->current_tree_editor,
                              MLVIEW_BAD_PARAM_ERROR) ;

        g_return_val_if_fail (PRIVATE (a_this)->app_context, MLVIEW_ERROR);

        settings = mlview_app_context_get_settings 
                (PRIVATE (a_this)->app_context);

        g_return_val_if_fail (settings, MLVIEW_ERROR);
        
        cur_node = mlview_tree_editor_get_cur_sel_xml_node 
                (PRIVATE (a_this)->current_tree_editor) ;
        if (!cur_node)
                return MLVIEW_ERROR ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_val_if_fail (ui_manager, MLVIEW_ERROR) ;


        mlview_xml_document_get_root_element (PRIVATE (a_this)->xml_doc,
                                              &root_element) ;

        clear_completion_popup_submenus (a_this) ;
        if (cur_node->type == XML_ELEMENT_NODE &&
            settings->general.validation_is_on &&
            cur_node->doc->extSubset) {

                  update_completion_popup_submenu2 (a_this,
                                                    PRIVATE (a_this)->app_context,
                                                    a_menu_root_path,
                                                    cur_node, 
                                                    ADD_CHILD) ;
                  update_completion_popup_submenu2 (a_this,
                                                    PRIVATE (a_this)->app_context,
                                                    a_menu_root_path,
                                                    cur_node, 
                                                    INSERT_AFTER) ;
                  update_completion_popup_submenu2 (a_this,
                                                    PRIVATE (a_this)->app_context,
                                                    a_menu_root_path,
                                                    cur_node, 
                                                    INSERT_BEFORE) ;
        } else {
        }

        switch (cur_node->type) {
        case XML_DOCUMENT_NODE:
                
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CommentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "UncommentNodeAction",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "InsertNextSiblingNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "InsertPrevSiblingNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CutNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }                                
                g_return_val_if_fail (action, MLVIEW_ERROR) ;

                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsChildMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsPrevMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsNextMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CopyNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectNextSiblingNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectParentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectPrevSiblingNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                break ;

        case XML_ELEMENT_NODE:
        case XML_DTD_NODE:
                

		if (cur_node->type == XML_ELEMENT_NODE) {

			menu_root_path = g_strjoin ("/",
						    a_menu_root_path,
						    "CommentNodeMenuitem",
						    NULL) ;
			action = gtk_ui_manager_get_action 
				(ui_manager,
				 menu_root_path) ;
			if (menu_root_path) {
				g_free (menu_root_path) ;
				menu_root_path = NULL ;
			}

			g_return_val_if_fail (action, MLVIEW_ERROR) ;
			
			if (cur_node == root_element) {
				g_object_set (G_OBJECT (action), 
				      "sensitive", FALSE,
				      NULL) ;
			} else {
				g_object_set (G_OBJECT (action), 
					      "sensitive", TRUE,
					      NULL) ;
			}
                
			menu_root_path = g_strjoin ("/",
						    a_menu_root_path,
						    "UncommentNodeMenuitem",
						    NULL) ;
			action = gtk_ui_manager_get_action 
				(ui_manager,
				 menu_root_path) ;
			g_return_val_if_fail (action, MLVIEW_ERROR) ;

			g_object_set (G_OBJECT (action), 
				      "sensitive", FALSE,
				      NULL) ;
                
			if (menu_root_path) {
				g_free (menu_root_path) ;
				menu_root_path = NULL ;
			}
		}
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "InsertNextSiblingNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;                
                if (cur_node == root_element) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                }
                

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "InsertPrevSiblingNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (cur_node == root_element) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CopyNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "AddChildNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsChildMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;
                

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsPrevMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;

                if (cur_node == root_element) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsNextMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (cur_node == root_element) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;

                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CopyNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectNextSiblingNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (cur_node->next) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                }
                
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectPrevSiblingNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (cur_node->prev) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                }


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "SelectParentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                if (cur_node->parent) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                }

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CutNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                                (ui_manager,
                                 menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;

                if (cur_node->parent
                    && cur_node->parent->type == XML_DOCUMENT_NODE
                    && cur_node->next == NULL
                    && cur_node->prev == NULL) {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", FALSE,
                                      NULL) ;
                } else {
                        g_object_set (G_OBJECT (action), 
                                      "sensitive", TRUE,
                                      NULL) ;
                }
                break ;
        case XML_COMMENT_NODE:
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CommentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "UncommentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                break ;
        default:
                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CommentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "UncommentNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE ,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CopyNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "CutNodeMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", TRUE,
                              NULL) ;


                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "AddChildNodeMenu",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                if (menu_root_path) {
                        g_free (menu_root_path) ;
                        menu_root_path = NULL ;
                }
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;

                menu_root_path = g_strjoin ("/",
                                            a_menu_root_path,
                                            "PasteNodeAsChildMenuitem",
                                            NULL) ;
                action = gtk_ui_manager_get_action 
                        (ui_manager,
                         menu_root_path) ;
                g_return_val_if_fail (action, MLVIEW_ERROR) ;
                g_object_set (G_OBJECT (action), 
                              "sensitive", FALSE,
                              NULL) ;
                break ;
        }
        return MLVIEW_OK ;
}

static void 
mlview_tree_view_is_swapped_out (MlViewIView *a_this,
                                 gpointer a_user_data)
{
        MlViewTreeView *tree_view = NULL ;
        GtkUIManager *ui_manager = NULL ;
        MlViewAppContext *app_context = NULL ;

        g_return_if_fail (a_this 
                          && MLVIEW_IS_IVIEW (a_this)
                          && MLVIEW_IS_TREE_VIEW (a_this) ) ;

        tree_view = MLVIEW_TREE_VIEW (a_this) ;
        g_return_if_fail (tree_view) ;

        mlview_tree_view_get_app_context (tree_view, 
                                          &app_context) ;
        g_return_if_fail (app_context) ;

        ui_manager = mlview_app_context_get_element (app_context,
                                                     "MlViewUIManager") ;
        g_return_if_fail (ui_manager) ;
        
        if (PRIVATE (tree_view)->edit_menu_merge_id) {
                gtk_ui_manager_remove_ui (ui_manager, 
                                          PRIVATE (tree_view)->edit_menu_merge_id)  ;
                        PRIVATE (tree_view)->edit_menu_merge_id = 0;
        }
	clear_completion_popup_submenus (tree_view) ;
}

static void 
mlview_tree_view_is_swapped_in (MlViewIView *a_this,
                                     gpointer a_user_data)
{
        g_return_if_fail (a_this
                          && MLVIEW_IS_IVIEW (a_this)
                          && MLVIEW_IS_TREE_VIEW (a_this)) ;

        /*
         *TODO: go edit the  applicatin edit menu to get our stuffs
         *out in there.
         */
        
}

static void
mlview_tree_view_application_menu_populating_requested (MlViewIView *a_this,
                                                        gpointer a_user_data)
{
        GtkWidget *menu = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_IVIEW (a_this)
                          && MLVIEW_IS_TREE_VIEW (a_this)) ;
        
        mlview_tree_view_get_edit_menu_for_application (MLVIEW_TREE_VIEW (a_this), 
                                                        &menu) ;
        g_return_if_fail (menu) ;
                
}


/**************************************
 *public gtk framework functions
 ***************************************/

GType
mlview_tree_view_get_type (void)
{
        static guint type = 0;

        if (!type) {
                static const GTypeInfo type_info = {
                        sizeof (MlViewTreeViewClass),
                        NULL, /*base_init*/
                        NULL, /*base_finalize*/
                        (GClassInitFunc)
                        mlview_tree_view_class_init,
                        NULL, /*class_finalize*/
                        NULL, /*class data*/
                        sizeof (MlViewTreeView),
                        0, /*n_prealloc*/
                        (GInstanceInitFunc)
                        mlview_tree_view_init
                };                
                type = g_type_register_static
                        (MLVIEW_TYPE_VIEW_ADAPTER,
                         "MlViewTreeView", &type_info, 0);
        }
        return type;
}

/**
 *Getter of the current tree editor used by this tree view and
 *that is currently selected.
 *@param a_this the current instance of #MlViewTreeView
 *@return the tree editor.
 */
MlViewTreeEditor *
mlview_tree_view_get_current_tree_editor (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return PRIVATE (a_this)->current_tree_editor;
}

/**
 *Returns the notebook that contains the tree editors used
 *by the tree view.
 *@param a_this the current instance of #MlViewTreeEditor.
 *@return the notebook that contains the tree editors, or
 *NULL in case of error.
 */
GtkNotebook *
mlview_tree_view_get_trees (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);
        
        return PRIVATE (a_this)->tree_editors;
}

/**
 *Gets the node editor used by the this tree view.
 *@param a_this the current instance of #MlViewTreeView.
 *@return the instance of #MlViewNodeEditor used by the tree view, or
 *NULL in case of error.
 */
MlViewNodeEditor *
mlview_tree_view_get_node_editor (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_this) != NULL,
                              NULL);

        return PRIVATE (a_this)->node_editor;
}


/**
 *A constructor of the #MlViewTreeView class.
 *@param a_mlview_xml_doc the xml document object model 
 *the view must work on.
 *@param a_name the name of the view
 *@param a_app_context the mlview application context.
 *@return the newly created tree view, or NULL in case of an error.
 */
GtkWidget *
mlview_tree_view_new (MlViewXMLDocument * a_mlview_xml_doc, 
                      gchar * a_name,
                      MlViewAppContext * a_app_context)
{
        MlViewTreeView *tree_view = NULL;
        xmlDoc *xml_doc = NULL ;

        g_return_val_if_fail (a_mlview_xml_doc != NULL, NULL);
        g_return_val_if_fail (PRIVATE (a_mlview_xml_doc) != NULL,
                              NULL);

        tree_view =
                g_object_new (MLVIEW_TYPE_TREE_VIEW,
                              NULL);
        mlview_tree_view_construct (tree_view, a_mlview_xml_doc,
                                    a_name, a_app_context) ;
        mlview_iview_connect_to_doc (MLVIEW_IVIEW (tree_view),
                                     a_mlview_xml_doc) ;
        xml_doc = mlview_xml_document_get_xml_document
                (a_mlview_xml_doc) ;
        if (xml_doc) {
                mlview_xml_document_select_node (a_mlview_xml_doc,
                                                 (xmlNode*)xml_doc) ;
        }
        return GTK_WIDGET (tree_view);
}

enum MlViewStatus
mlview_tree_view_get_app_context (MlViewTreeView *a_this,
                                  MlViewAppContext **a_app_context_ptr)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)
                              && a_app_context_ptr,
                              MLVIEW_BAD_PARAM_ERROR) ;

        *a_app_context_ptr = PRIVATE (a_this)->app_context ;

        return MLVIEW_OK ;
}

/**
 *Setter of the mlview application context.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_app_context the new mlview application context.
 */
void
mlview_tree_view_set_app_context (MlViewTreeView *a_this,
                                  MlViewAppContext *a_app_context)
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        PRIVATE (a_this)->app_context = a_app_context;

        if (PRIVATE (a_this)->current_tree_editor) {
                mlview_tree_editor_set_application_context 
                        (PRIVATE (a_this)->current_tree_editor,
                         a_app_context) ;
        }

        if (PRIVATE (a_this)->node_editor) {
                mlview_node_editor_set_application_context
                        (PRIVATE (a_this)->node_editor,
                         a_app_context);
        }
}

static void
document_ext_subset_changed_cb (MlViewXMLDocument *a_xml_doc,
                                gpointer a_user_data)
{
        xmlNode *current_node = NULL ;
        MlViewTreeView *tree_view = a_user_data ;
        
        g_return_if_fail (a_xml_doc 
                          && MLVIEW_IS_XML_DOCUMENT (a_xml_doc));
        g_return_if_fail (tree_view
                          && MLVIEW_IS_TREE_VIEW (tree_view));
        
        if (mlview_xml_document_is_completion_possible_global 
            (a_xml_doc)
            && PRIVATE (tree_view)->completion_widget) {
                gtk_widget_show
                        (GTK_WIDGET (PRIVATE (tree_view)->completion_widget)) ;
                current_node = mlview_tree_editor_get_cur_sel_xml_node 
                        (PRIVATE (tree_view)->current_tree_editor) ;
                if (current_node) {
                        mlview_completion_table_select_node (PRIVATE (tree_view)->completion_widget,
                                                             current_node) ;
                }
        }
}

static gboolean
completion_widget_mapped_cb (GtkWidget *a_widget,
                             gpointer a_user_data)
{
        MlViewTreeView *tree_view = NULL ;
        
        g_return_val_if_fail (a_user_data 
                              && MLVIEW_IS_TREE_VIEW (a_user_data),
                              FALSE) ;

        tree_view = MLVIEW_TREE_VIEW (a_user_data) ;
        g_return_val_if_fail (a_widget, FALSE) ;

        if (!mlview_xml_document_is_completion_possible_global 
            (PRIVATE (tree_view)->xml_doc))
                gtk_widget_hide (a_widget);   

        return FALSE ;
}


/**
 *Instance initialyzer of the #MlViewTreeView class.
 *Should be called by the constructor.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_mlview_xml_doc the xml document object model the
 *view must "work" on.
 *@param a_name the name of the view
 *@param a_app_context the mlview application context to
 *be associated with the tree view.
 */
void
mlview_tree_view_construct (MlViewTreeView *a_this,
                            MlViewXMLDocument *a_mlview_xml_doc,
                            gchar * a_name,
                            MlViewAppContext *a_app_context)
{
        MlViewTreeEditor *raw_xml = NULL;
        MlViewTreeEditor *elements = NULL;
        gint i, nb_pages;
        MlViewTreeEditor *tree_editor;
        struct MlViewAppSettings *settings = NULL;    
        enum MLVIEW_VIEW_ADAPTER_STATUS status = NOK;
        MlViewFileDescriptor *file_desc = NULL ;
        gchar *path = NULL ;

        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (a_mlview_xml_doc != NULL) ;
        PRIVATE (a_this)->xml_doc = a_mlview_xml_doc ;
        mlview_xml_document_ref (a_mlview_xml_doc) ;

        status = mlview_view_adapter_construct
                (MLVIEW_VIEW_ADAPTER (a_this),
                 a_mlview_xml_doc);
        g_return_if_fail (status == MLVIEW_VIEW_ADAPTER_OK);

        settings = mlview_app_context_get_settings (a_app_context);

        g_return_if_fail (settings);

        /*The main vpaned of the view */
        PRIVATE (a_this)->main_paned =
                GTK_PANED (gtk_vpaned_new ());

        gtk_box_pack_start (GTK_BOX (a_this),
                            GTK_WIDGET (PRIVATE (a_this)->
                                        main_paned), TRUE, TRUE,
                            0);
        gtk_widget_show (GTK_WIDGET
                         (PRIVATE (a_this)->main_paned));

        /*The tree editors and the tree editors' notebook */

        /* Notebook */ 
        PRIVATE (a_this)->upper_paned1 =
                GTK_PANED (gtk_hpaned_new ());
        gtk_paned_add1 (GTK_PANED
                        (PRIVATE (a_this)->main_paned),
                        GTK_WIDGET (PRIVATE (a_this)->
                                    upper_paned1));
        PRIVATE (a_this)->tree_editors = GTK_NOTEBOOK (gtk_notebook_new ());
        gtk_notebook_set_tab_pos (PRIVATE (a_this)->tree_editors, GTK_POS_BOTTOM);
        gtk_paned_add1 (GTK_PANED (PRIVATE (a_this)->upper_paned1),
                        GTK_WIDGET (PRIVATE (a_this)->tree_editors));
        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->tree_editors),
                          "switch-page",
                          G_CALLBACK (tree_selected_cb),
                          a_this);
        
        /* Tree editing widgets */
        raw_xml = MLVIEW_TREE_EDITOR (mlview_tree_editor_new (a_app_context));
        elements = MLVIEW_TREE_EDITOR (mlview_icon_tree_new (a_app_context));

        PRIVATE (a_this)->raw_tree_editor = raw_xml;
        PRIVATE (a_this)->icon_tree_editor = elements ;

        gtk_notebook_append_page (PRIVATE (a_this)->tree_editors,
                                  GTK_WIDGET (elements),
                                  gtk_label_new (_("Elements"))); 
        gtk_notebook_append_page (PRIVATE (a_this)->tree_editors, 
                                  GTK_WIDGET (raw_xml),
                                  gtk_label_new (_("Raw XML")));

        /* The element name completion widget*/
        PRIVATE (a_this)->completion_widget =
                MLVIEW_COMPLETION_TABLE 
                (mlview_completion_table_new (a_mlview_xml_doc)) ;

        g_signal_connect 
                (a_mlview_xml_doc, "node-selected",
                 (GCallback)update_completion_widget_cb, 
                 PRIVATE (a_this)->completion_widget);

        gtk_paned_pack2 (GTK_PANED
                         (PRIVATE (a_this)->upper_paned1),
                         GTK_WIDGET (PRIVATE (a_this)->completion_widget), 
                         FALSE, TRUE);

        /*The node editor */
        PRIVATE (a_this)->node_editor =
                MLVIEW_NODE_EDITOR 
                (mlview_node_editor_new
                 (a_app_context, a_mlview_xml_doc));

        g_signal_connect (G_OBJECT
                          (PRIVATE (a_this)->node_editor),
                          "element-changed",
                          G_CALLBACK (xml_node_changed_cb),
                          a_this);

        gtk_paned_pack2 (GTK_PANED
                         (PRIVATE (a_this)->main_paned),
                         GTK_WIDGET (PRIVATE (a_this)->node_editor),
                         FALSE, TRUE);

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_this)->node_editor));

        gtk_widget_show_all (GTK_WIDGET
                             (PRIVATE (a_this)->main_paned));
        
        PRIVATE (a_this)->app_context = a_app_context ;

        /* Build trees */
        nb_pages = gtk_notebook_get_n_pages (PRIVATE (a_this)->tree_editors);
        for (i = 0; i < nb_pages; i++) {
                tree_editor = MLVIEW_TREE_EDITOR
                        (gtk_notebook_get_nth_page (PRIVATE (a_this)->tree_editors, i));
                mlview_tree_editor_edit_xml_doc
                        (tree_editor, a_mlview_xml_doc, NULL);         
        }

        g_signal_connect (G_OBJECT (a_mlview_xml_doc),
                          "file-path-changed",
                          G_CALLBACK
                          (doc_path_changed_cb),
                          a_this) ;
        g_signal_connect (G_OBJECT (PRIVATE (a_this)->completion_widget),
                          "map",
                          G_CALLBACK (completion_widget_mapped_cb),
                          a_this) ;
        g_signal_connect (G_OBJECT 
                          (PRIVATE (a_this)->xml_doc),
                          "ext-subset-changed",
                          G_CALLBACK (document_ext_subset_changed_cb),
                          a_this);

        g_signal_connect (G_OBJECT (a_app_context),
                          "contextual-menu-requested",
                          G_CALLBACK (contextual_menu_requested_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (a_app_context),
                          "view-swapped",
                          G_CALLBACK (view_swapped_cb),
                          a_this) ;

        g_signal_connect (G_OBJECT (PRIVATE (a_this)->raw_tree_editor),
                          "ungrab-focus-requested",
                          G_CALLBACK (tree_editor_ungrab_focus_requested_cb),
                          a_this) ;
        g_signal_connect (G_OBJECT (PRIVATE (a_this)->icon_tree_editor),
                          "ungrab-focus-requested",
                          G_CALLBACK (tree_editor_ungrab_focus_requested_cb),
                          a_this) ;
        g_signal_connect (G_OBJECT (PRIVATE (a_this)->node_editor),
                          "ungrab-focus-requested",
                          G_CALLBACK (node_editor_ungrab_focus_requested_cb) ,
                          a_this) ;
	g_signal_connect (G_OBJECT (a_this),
			  "realize",
			  G_CALLBACK (set_tree_view_proportions_cb),
			  NULL) ;

        file_desc = mlview_xml_document_get_file_descriptor (a_mlview_xml_doc) ;
        if (file_desc) {
                path = g_strdup (a_name);
        }
        
        if (!path) {
                path = (gchar*)"untitled" ;
        }
        mlview_tree_view_set_xml_document_path (a_this, path) ;

}

/**
 *Creates an internal subset node 
 *(<DOCTYPE elname PUBLIC "blah" "blabla">)
 *@param a_this the current instance of #MlViewTreeView
 *@return MLVIEW_OK upon successful completion, 
 *an error code otherwise.
 */
enum MlViewStatus 
mlview_tree_view_create_internal_subset_node_interactive (MlViewTreeView *a_this)
{
        xmlDoc *native_doc = NULL ;
        gchar *name = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this)
                              && PRIVATE (a_this)->app_context,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (!PRIVATE (a_this)->xml_doc)
                return MLVIEW_OK;
        native_doc = mlview_xml_document_get_xml_document
                (PRIVATE (a_this)->xml_doc) ;
        g_return_val_if_fail (native_doc, MLVIEW_ERROR) ;
        if (native_doc->intSubset) {
                mlview_app_context_error 
                        (PRIVATE (a_this)->app_context, 
                         _("The document already has an internal subset defined !")) ;
                return MLVIEW_OK ;
        }
        mlview_app_context_ask_internal_subset_node_name (&name) ;
        if (name) {
                return mlview_xml_document_create_internal_subset 
                        (PRIVATE (a_this)->xml_doc, name, 
                         (xmlChar*)"default-public-id", 
                         (xmlChar*)"default-system-id",
                         TRUE) ;
        }
        return MLVIEW_OK ;
}

/**
 *Interactively add a child node to the currently selected xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_add_child_node_interactive (MlViewTreeView * a_this)
{
        MlViewTreeEditor *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_current_tree_editor
                (a_this);
        if (tree_editor)
                mlview_tree_editor_add_child_node_interactive
                        (tree_editor);
}

void
mlview_tree_view_add_child_text_node (MlViewTreeView *a_this,
                                      const gchar *a_text)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && a_text) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        
        mlview_tree_editor_add_child_text_node (tree_editor, a_text, TRUE) ;
}


void
mlview_tree_view_insert_prev_sibling_text_node (MlViewTreeView *a_this,
                                                const gchar *a_text)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && a_text) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        
        mlview_tree_editor_insert_prev_text_node (tree_editor, a_text, TRUE) ;
}

void
mlview_tree_view_insert_next_sibling_text_node (MlViewTreeView *a_this,
                                                const gchar *a_text)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && a_text) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        
        mlview_tree_editor_insert_next_text_node (tree_editor, a_text, TRUE) ;
}

void
mlview_tree_view_add_child_element_node (MlViewTreeView *a_this,
                                         const gchar *a_element_name)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          &&a_element_name) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        mlview_tree_editor_add_child_element_node (tree_editor,
                                                   a_element_name,
                                                   FALSE) ;
        
}

void
mlview_tree_view_insert_prev_sibling_element_node (MlViewTreeView *a_this,
                                                   const gchar *a_element_name)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && a_element_name) ;

        
        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        mlview_tree_editor_insert_prev_sibling_element_node (tree_editor, 
                                                             a_element_name,
                                                             FALSE) ;
}


void
mlview_tree_view_insert_next_sibling_element_node (MlViewTreeView *a_this,
                                                   const gchar *a_element_name)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && a_element_name) ;

        
        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        g_return_if_fail (tree_editor) ;
        mlview_tree_editor_insert_next_sibling_element_node (tree_editor, 
                                                             a_element_name,
                                                             FALSE) ;
}

/**
 *Interactively adds a previous sibling node to the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_insert_prev_sibling_node_interactive (MlViewTreeView * a_this)
{
        MlViewTreeEditor *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_current_tree_editor
                (a_this);

        if (tree_editor) {
                mlview_tree_editor_insert_prev_sibling_node_interactive
                        (tree_editor);
        }
}


/**
 *Interactively insert a next sibling node to the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_insert_next_sibling_node_interactive (MlViewTreeView * a_this) 
{
        MlViewTreeEditor *tree_editor = NULL;
        tree_editor =
                mlview_tree_view_get_current_tree_editor
                (a_this);

        if (tree_editor) {
                mlview_tree_editor_insert_next_sibling_node_interactive
                        (tree_editor);
        }
}


/**
 *Cuts the currently selected xml node.
 *That is, unlink it from the tree and put it in
 *the xml node clipboard.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_cut_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL;
        GtkTreeIter cur_sel_start = {0};
        enum MlViewStatus status = MLVIEW_OK ;

        tree_editor =
                mlview_tree_view_get_current_tree_editor
                (a_this);

        if (tree_editor == NULL)
                return;
        status = mlview_tree_editor_get_cur_sel_start_iter 
                (tree_editor, &cur_sel_start);
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor_cut_node (tree_editor,
                                      &cur_sel_start);
}

enum MlViewStatus 
mlview_tree_view_focus_on_node_editor (MlViewTreeView *a_this)
{
        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;
        g_return_val_if_fail (PRIVATE (a_this)->node_editor,
                              MLVIEW_BAD_PARAM_ERROR) ;

        if (GTK_WIDGET_HAS_GRAB (PRIVATE (a_this)->node_editor)) {
#ifdef MLVIEW_VERBOSE
                g_print ("In mlview_tree_view_focus_on_node_editor(): "
                         "Node editor is already in the grab stack\n") ;
#endif
                return MLVIEW_OK ;
        } else {
#ifdef MLVIEW_VERBOSE
                g_print ("In mlview_tree_view_focus_on_node_editor(): "
                         "making node editor grab the focus\n") ;
#endif          
                mlview_node_editor_grab_focus 
                        (PRIVATE (a_this)->node_editor) ;
        }
        return MLVIEW_OK ;
}


enum MlViewStatus 
mlview_tree_view_focus_on_tree_editor (MlViewTreeView *a_this)
{
        return MLVIEW_OK ;
}


void
mlview_tree_view_comment_current_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        if (!tree_editor) {
                mlview_utils_trace_info ("Could not retrieve tree editor") ;
                return ;
        }
        mlview_tree_editor_comment_current_node (tree_editor) ;
}


void
mlview_tree_view_uncomment_current_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        if (!tree_editor) {
                mlview_utils_trace_info ("Could not retrieve tree editor") ;
                return ;
        }
        mlview_tree_editor_uncomment_current_node (tree_editor) ;
}

void
mlview_tree_view_select_next_sibling_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        if (!tree_editor)
                return ;
        mlview_tree_editor_select_next_sibling_node (tree_editor) ;
}

void
mlview_tree_view_select_prev_sibling_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        if (!tree_editor)
                return ;
        mlview_tree_editor_select_prev_sibling_node (tree_editor) ;
}

void
mlview_tree_view_select_parent_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL ;

        g_return_if_fail (a_this && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)) ;

        tree_editor = mlview_tree_view_get_current_tree_editor (a_this) ;
        if (!tree_editor)
                return ;
        mlview_tree_editor_select_parent_node (tree_editor) ;
}

/**
 *Copy the currently selected xml node into the clipboard.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_copy_node (MlViewTreeView *a_this)
{
        MlViewTreeEditor *tree_editor = NULL;
        GtkTreeIter cur_sel_start = {0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this != NULL);

        tree_editor =
                mlview_tree_view_get_current_tree_editor
                (a_this);
        if (tree_editor == NULL)
                return;
        status = mlview_tree_editor_get_cur_sel_start_iter
                (tree_editor, &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor_copy_node (tree_editor,
                                       &cur_sel_start) ;
}


/**
 *Paste the last node found in the clipboard into the xml document
 *model.
 *The pasted node is pasted as a child node of the currently selected
 *xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_paste_node_as_child (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status = MLVIEW_OK;

        g_return_if_fail (a_this
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this)
                          && PRIVATE (a_this)->current_tree_editor);

        status = mlview_tree_editor_get_cur_sel_start_iter 
                (PRIVATE (a_this)->current_tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor_paste_node_as_child
                (PRIVATE (a_this)->current_tree_editor,
                 &cur_sel_start);
}

/**
 *Paste the last node found in the clipboard into the xml document
 *model.
 *The pasted node is pasted as a previous sibling node of the 
 *currently selected xml node.
 *@param a_this the current instance of #MlViewTreeView
 */
void
mlview_tree_view_paste_node_as_prev_sibling (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status=MLVIEW_OK ;

        g_return_if_fail (a_this != NULL
                          && MLVIEW_IS_TREE_VIEW (a_this)
                          && PRIVATE (a_this));

        status = mlview_tree_editor_get_cur_sel_start_iter
                (PRIVATE (a_this)->current_tree_editor, 
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        status = mlview_tree_editor_paste_node_as_sibling
                (PRIVATE (a_this)->current_tree_editor, 
                 &cur_sel_start, TRUE) ;
}


/**
 *Pastes the last node found in the clipboard at
 *the place of the the currently selected xml node.
 *@param a_this a pointer to the current instance
 *of MlViewTreeView.
 */
void 
mlview_tree_view_paste_node_as_next_sibling (MlViewTreeView * a_this) 
{
        GtkTreeIter cur_sel_start={0} ;
        enum MlViewStatus status = MLVIEW_OK ;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        status = mlview_tree_editor_get_cur_sel_start_iter
                (PRIVATE (a_this)->current_tree_editor,
                 &cur_sel_start) ;
        g_return_if_fail (status == MLVIEW_OK) ;
        mlview_tree_editor_paste_node_as_sibling
                (PRIVATE (a_this)->current_tree_editor,
                 &cur_sel_start, FALSE) ;
}


/**
 *Sets the proportions of the the upper paned of the tree view.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_percentage of width taken by the leftmost part of
 *the upper paned1
 */
void
mlview_tree_view_set_upper_paned1_proportions (MlViewTreeView * a_this,
                                               const guint a_percentage) 
{
        gint separator_position = 0;
        GtkWidget *top_level_widget = NULL;
        
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        top_level_widget =
                gtk_widget_get_toplevel (GTK_WIDGET
                                         (a_this));
        g_return_if_fail (top_level_widget != NULL);

        separator_position =
                top_level_widget->allocation.width *
                a_percentage / 100;

        gtk_paned_set_position (PRIVATE (a_this)->
                                upper_paned1,
                                separator_position);

        gtk_widget_show_all (GTK_WIDGET (a_this));
}


/**
 *Sets the GtkPaned top/bottom proportions of the main 
 *paned to a_percentage. This method gets the bounds of the top level widget
 *in order to determine the total size of the GtkVPaned 
 *(which actually has the same boundaries as the top level widget) and do
 *the calculation of the proportions of the top part vs bottom 
 *part of the GtkVPaned.
 *Therefore, this method must be called 
 *only *AFTER* the document has been opened and 
 *the top level of the document is shown. 
 *
 */
void
mlview_tree_view_set_main_paned_proportions (MlViewTreeView * a_this,
                                             const guint a_percentage) 
{
        guint separator_position = 0;
        GtkWidget *top_level_widget = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);
        top_level_widget =
                gtk_widget_get_toplevel (GTK_WIDGET
                                         (a_this));
        g_return_if_fail (top_level_widget != NULL);

        PRIVATE (a_this)->main_paned_percentage =
                a_percentage;

        separator_position =
                top_level_widget->allocation.height *
                a_percentage / 100;

        gtk_paned_set_position (PRIVATE (a_this)->
                                main_paned, separator_position);

        gtk_widget_show_all (GTK_WIDGET (a_this));
}


/**
 *Sets percentage taken by the upper part of the
 *view (tree editor and completion list) vs 
 *the lower part (node editor) , and, the percentage taken
 *by the left hand part (tree editor) vs right hand part 
 *(completion list)
 *@param a_this the current instance of #MlViewTreeView
 *@param a_top_down_percentage the percentage of height taken
 *by the upper part of the main paned
 *@param a_left_right_percentage the percentage of width taken
 *by the left part of the upper paned.
 */
void
mlview_tree_view_set_all_paned_proportions (MlViewTreeView * a_this,
                                            const guint a_top_down_percentage,
                                            const guint a_left_right_percentage) 
{
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW
                          (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);
        g_return_if_fail (PRIVATE (a_this)->node_editor !=
                          NULL);

        mlview_tree_view_set_upper_paned1_proportions
                (a_this, a_left_right_percentage);
        mlview_tree_view_set_main_paned_proportions
                (a_this, a_top_down_percentage);
}


/**
 *Sets the title of the xml DOM to @a_file_path.
 *Updates this information in the tree editor and in the
 *XML DOM. 
 *@param a_this the current instance of #MlViewTreeView.
 *@param a_file_path the new file path.
 */
void
mlview_tree_view_set_xml_document_path (MlViewTreeView * a_this, 
                                        gchar * a_file_path) 
{
        guchar *base_name = NULL;

        g_return_if_fail (a_file_path != NULL);
        g_return_if_fail (a_this != NULL);
        g_return_if_fail (PRIVATE (a_this) != NULL);

        if (PRIVATE (a_this)->tree_editors) {
                gint i =0 , nb_pages = 0 ;
                MlViewTreeEditor *tree_editor = NULL ;

                nb_pages = gtk_notebook_get_n_pages (PRIVATE (a_this)->tree_editors);
                for (i = 0; i < nb_pages; i++) {
                        tree_editor = MLVIEW_TREE_EDITOR 
                                (gtk_notebook_get_nth_page (PRIVATE (a_this)->tree_editors, i));
                        g_return_if_fail (tree_editor 
                                          && MLVIEW_IS_TREE_EDITOR (tree_editor));
                        mlview_tree_editor_set_xml_document_path
                                (tree_editor, a_file_path) ;
                }
        }
        base_name = (guchar *) g_basename (a_file_path);

        mlview_iview_set_name
                (MLVIEW_IVIEW (a_this), base_name);
}


/**
 *Opens a dialog box to ask the user the depth to which the tree
 *should be expanded and expands the tree to that depths. 
 *@param a_this the current instance of #MlViewTreeView
 */
void 
mlview_tree_view_expand_tree_to_depth_interactive (MlViewTreeView * a_this) 
{
        GtkDialog *dialog = NULL;
        GtkWidget *expand_to_leaves = NULL,
                *depth_entry = NULL;
        gint button = 0,
                depth = 0;
        gchar *depth_str = NULL;

        g_return_if_fail (a_this != NULL);
        g_return_if_fail (MLVIEW_IS_TREE_VIEW (a_this));
        g_return_if_fail (PRIVATE (a_this) != NULL);

        dialog = get_expand_tree_dialog (a_this);
        g_return_if_fail (dialog != NULL);

        button = gtk_dialog_run (dialog);

        expand_to_leaves =
                gtk_object_get_data (GTK_OBJECT (dialog),
                                     "expand-to-leaves");
        g_return_if_fail (expand_to_leaves != NULL);

        depth_entry = gtk_object_get_data (GTK_OBJECT (dialog),
                                           "depth-entry");
        g_return_if_fail (depth_entry != NULL);

        switch (button) {
        case GTK_RESPONSE_ACCEPT: /*user clicked OK */
                if (gtk_toggle_button_get_active
                    (GTK_TOGGLE_BUTTON (expand_to_leaves)) ==
                    TRUE) {
                        mlview_tree_editor_expand_tree_to_depth
                                (PRIVATE (a_this)->current_tree_editor,
                                 -1);
                } else {
                        depth_str = (guchar *)
                                gtk_entry_get_text (GTK_ENTRY
                                                    (depth_entry));

                        if (depth_str) {
                                depth = atoi (depth_str);
                                mlview_tree_editor_expand_tree_to_depth
                                        (PRIVATE
                                         (a_this)->current_tree_editor, 
                                         depth);
                        }
                }
                break;
        case GTK_RESPONSE_REJECT: /*user clicked CANCEL */
        case GTK_RESPONSE_CLOSE: /*closed the dialog box */
        default:               /*the sky falls on our heads */
                break;
        }

        gtk_widget_hide (GTK_WIDGET (dialog));
}

/**
 *Finds the graphical node that contains the given string.
 *@param a_this a pointer to the current instance of
 *#MlViewTreeView.
 *@return the graphical xml node that contains the given string.
 */
void
mlview_tree_view_find_xml_node_that_contains_str_interactive (MlViewTreeView * a_this) 
{
        MlViewTreeEditor *tree_editor = NULL;

        g_return_if_fail (a_this != NULL);

        tree_editor =
                mlview_tree_view_get_current_tree_editor (a_this);
        g_return_if_fail (tree_editor != NULL);
        mlview_tree_editor_search_interactive (tree_editor) ;
}

/**
 *Updates the contextual menu
 *@param a_this the current instance of #MlViewTreeView
 *@param a_menu_ptr out paramater the contextual menu.
 */
void
mlview_tree_view_update_contextual_menu (MlViewTreeView * a_this, 
                                         GtkMenu ** a_menu_ptr) 
{
        
}

/**
 *Gets the contextual menu exposed by the current view.
 *@param a_this the current instance of #MlViewTreeView
 *@param a_menu_ptr out parameter the menu.
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
mlview_tree_view_get_contextual_menu (MlViewTreeView *a_this,
                                      GtkWidget **a_menu_ptr) 
{
        GtkUIManager *ui_manager = NULL ;
        GtkWidget *tmp_widget = NULL, *menu = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_val_if_fail (ui_manager, MLVIEW_ERROR) ;

        tmp_widget = gtk_ui_manager_get_widget (ui_manager,
                                          "/TreeViewPopupEditMenu/CommentNodeMenuitem") ;
        if (!tmp_widget) {
                mlview_tree_view_build_contextual_menu2 (a_this) ;
               
        }
        menu = gtk_ui_manager_get_widget (ui_manager,
                                           "/TreeViewPopupEditMenu") ;
        g_return_val_if_fail (menu, MLVIEW_ERROR) ;

        gtk_widget_show_all (menu) ;

        activate_or_deactivate_proper_menu_items2 (a_this, "/TreeViewPopupEditMenu") ;

        *a_menu_ptr = menu ;

        if (!*a_menu_ptr)
                return MLVIEW_ERROR ;

        return MLVIEW_OK ;
}

enum MlViewStatus
mlview_tree_view_get_edit_menu_for_application (MlViewTreeView *a_this,
                                                GtkWidget **a_menu_ptr)
{
        GtkUIManager *ui_manager = NULL ;
        GtkWidget *tmp_widget = NULL, *menu = NULL ;

        g_return_val_if_fail (a_this
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this),
                              MLVIEW_BAD_PARAM_ERROR) ;

        ui_manager = mlview_tree_view_get_ui_manager (a_this) ;
        g_return_val_if_fail (ui_manager, MLVIEW_ERROR) ;

        tmp_widget = gtk_ui_manager_get_widget (ui_manager,
                                          "/MainMenubar/EditMenu/CommentNodeMenuitem") ;
        if (!tmp_widget) {
                mlview_tree_view_build_app_edit_menu (a_this) ;
               
        }
        menu = gtk_ui_manager_get_widget (ui_manager,
                                          "/MainMenubar/EditMenu") ;
        g_return_val_if_fail (menu, MLVIEW_ERROR) ;

        gtk_widget_show_all (menu) ;

        activate_or_deactivate_proper_menu_items2 (a_this, "/MainMenubar/EditMenu") ;

        *a_menu_ptr = menu ;

        if (!*a_menu_ptr)
                return MLVIEW_ERROR ;

        return MLVIEW_OK ;
}

/**
 *Gets the instance of GtkUIManager
 *associated to the current instance of
 *#MlViewTreeView. As of today, it's
 *the global one used across all the
 *application. It's has just been cached
 *here.
 *@param a_this the current instance of #MlViewTreeView.
 *@return the insntance of GtkUIManager associated to
 *the treeview, or NULL if something bad happened. 
 */
GtkUIManager *
mlview_tree_view_get_ui_manager (MlViewTreeView *a_this)
{
        MlViewAppContext *app_context = NULL ;
        gchar *file_path = NULL ;
        GtkActionGroup *action_group = NULL ;

        g_return_val_if_fail (a_this 
                              && MLVIEW_IS_TREE_VIEW (a_this)
                              && PRIVATE (a_this),
                              NULL) ;

        if (!PRIVATE (a_this)->ui_manager) {
                app_context = PRIVATE (a_this)->app_context ;
                g_return_val_if_fail (app_context, NULL) ;
                PRIVATE (a_this)->ui_manager = 
                        mlview_app_context_get_element (app_context,
                                                        "MlViewUIManager") ;                
                g_return_val_if_fail (PRIVATE (a_this)->ui_manager, NULL) ;

                action_group = gtk_action_group_new ("TreeViewEditMenuActions") ;
		gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE) ;
                gtk_action_group_add_actions (action_group, gv_edit_menu_actions,
                                              sizeof (gv_edit_menu_actions)/sizeof (GtkActionEntry),
                                              a_this) ;
                PRIVATE (a_this)->action_group = action_group ;
                gtk_ui_manager_insert_action_group (PRIVATE (a_this)->ui_manager,
                                                    action_group, 0) ;
                file_path = mlview_utils_locate_file ("tree-view-edit-menu.xml") ;
                g_return_val_if_fail (file_path, NULL) ;
                PRIVATE (a_this)->tree_view_popup_edit_menu_merge_id = 
                        gtk_ui_manager_add_ui_from_file (PRIVATE (a_this)->ui_manager,
                                                         file_path, 0) ;
                if (file_path) {
                        g_free (file_path)  ;
                        file_path = NULL ;
                }
                g_return_val_if_fail (PRIVATE (a_this)->tree_view_popup_edit_menu_merge_id,
                                      NULL) ;                
        }
        return PRIVATE (a_this)->ui_manager ;
}
