///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include <sstream>
#include <fstream>

#include <libxml++/libxml++.h>

#include "textframe.h"
#include "document.h"
#include "pptout/config.h" // Todo
#include "util/tempfile.h"
#include <util/warning.h>
#include <util/stringutil.h>
#include <util/filesys.h>
#include <ps/misc.h>

TextFrame::TextFrame(Group *parent, TextStream *stream,
		       float w, float h) :
  CachedFrame(parent, w, h, "Text " + (stream ? stream->get_name() : "")),
  parsed_file(Tempfile::find_new_name()), text_stream(stream),
  num_columns(1), gutter_width(12.0)
{
  resizable = true;
  white_is_transparent = true;
  ps_exists = false;
  parsed_file_lock = false;

  if(text_stream)
    text_stream->add_frame(this);
  signal_end_write_ps.connect
    (slot(*this,  &TextFrame::_end_write_ps));
}

TextFrame::TextFrame(Group *parent, const xmlpp::Element& node)
  : CachedFrame(parent, node),
    parsed_file(Tempfile::find_new_name()),
    text_stream(0), num_columns(1), gutter_width(12.0)
{
  resizable = true;
  white_is_transparent = true;
  ps_exists = false;
  parsed_file_lock = false;

  if(!node.get_attribute("type") 
     || node.get_attribute("type")->get_value() != "text")
    throw Error::FrameType("Bad text-frame type: " 
			    + node.get_attribute("type")->get_value());

  if(const xmlpp::Attribute* cols = node.get_attribute("num_columns"))
    num_columns = to<int>(cols->get_value());
  
  // Note: This is the gutter width as a plain numeric value, it /should/ be
  // possible to give it as a value with a unit, as "1em" or "5mm" ...
  if(const xmlpp::Attribute* gutter = node.get_attribute("gutter_width"))
    gutter_width = to<float>(gutter->get_value());
  
  if(const xmlpp::Attribute* stream = node.get_attribute("stream")) {
    set_stream(Document::containing(*this)
	       .get_text_stream(stream->get_value()));
    if(!text_stream)
      throw Error::Read("Failed to get text stream \"" + stream->get_value()
			+ "\" for text frame");
  } // If stream is not present text_stream is null. This is ok.
  signal_end_write_ps.connect
    (slot(*this,  &TextFrame::_end_write_ps));
}

TextFrame::~TextFrame() {
  if(text_stream)
    text_stream->remove_frame(this);
  try { unlink(parsed_file); }
  catch(std::exception& err) {
    warning << "Failed to delete parsed_file: " << err.what()
	    << std::endl;
  }
}

std::string TextFrame::getTypeName() const { return "text"; }

xmlpp::Element *TextFrame::save(xmlpp::Element& parent_node,
				const FileContext &context) const 
{
  xmlpp::Element *node = CachedFrame::save(parent_node, context);
  node->set_attribute("type", "text");

  node->set_attribute("num_columns", tostr(num_columns));
  node->set_attribute("gutter_width", tostr(gutter_width));
  if(text_stream)
    node->set_attribute("stream", text_stream->get_name());

  return node;
}

void TextFrame::print(std::ostream &out, bool grayscale) const {
  if(text_stream) {
    out << "% " << name << ", from " << text_stream->get_association() << '\n'
	<< "gsave\n"
	<< PS::Concat(get_matrix());
    text_stream->outputPageRaw(out, this);
    out << "grestore\n" << std::endl;

  } else {
    out << "% " << name << ", text stream without data\n";
  }
}

void TextFrame::set_stream(TextStream *new_stream, 
			   bool remove_from_old) 
{
  if(text_stream == new_stream)
    return;
  
  if(remove_from_old && text_stream)
    text_stream->remove_frame(this);
  
  text_stream = new_stream;
  ps_exists = false;
  if(text_stream)
    text_stream->add_frame(this);
  
  props_changed_signal(this);
  //ready_to_draw_signal(this);
}

void TextFrame::set_num_columns(unsigned int columns) { 
  if(num_columns != columns) {
    num_columns = columns;
    //content_is_broken = false;	// Assume it's ok until we try it
    props_changed_signal(this);
    if(text_stream)
      text_stream->generate_ps_request(this);
  }
}

void TextFrame::set_gutter(const float& gutter) { 
  if(gutter_width != gutter) {
    gutter_width = gutter; 
    props_changed_signal(this);
    if(text_stream)
      text_stream->generate_ps_request(this);
  }
}

void TextFrame::set_size(float w, float h) {
  // override is just so we can know when to run the typesetter again
  if(w != width || h != height) {
    ps_exists = false;
  }
  Pagent::set_size(w, h);
}

void TextFrame::begin_write_ps() {
  parsed_file_lock = true;
}

void TextFrame::end_write_ps(bool _ps_exists, bool _truncated) {
  parsed_file_lock = false;
  ps_exists = _ps_exists;
  truncated = _truncated;
  signal_end_write_ps();
}

void TextFrame::_end_write_ps() {
  object_changed_signal(this);
  ready_to_draw_signal(this);
}

void TextFrame::generate_picture(FileCallback callback) {
  if(!text_stream)
    throw GenPicError(ASSOCIATION, "No associated stream");
  
  if(!ps_exists) {
    text_stream->generate_ps_request(this);
    return;
  }
  
  { // create the PostScript file
    std::ofstream out(parsed_file.c_str());
    text_stream->outputPageEps(out, this);
  }

  // don't try to start gs if TextStream is writing the file
  if(parsed_file_lock) 
    return;
  
  callback(parsed_file);
}
