#region PDFsharp - A .NET library for processing PDF
//
// Authors:
//   Stefan Lange (mailto:Stefan.Lange@pdfsharp.com)
//
// Copyright (c) 2005-2008 empira Software GmbH, Cologne (Germany)
//
// http://www.pdfsharp.com
// http://sourceforge.net/projects/pdfsharp
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
// DEALINGS IN THE SOFTWARE.
#endregion

using System;
using System.Diagnostics;
using System.Collections;
using System.Text;
using System.IO;
using PdfSharp.Internal;
using PdfSharp.Pdf.IO;

namespace PdfSharp.Pdf
{
  /// <summary>
  /// Represents an indirect name value. This type is not used by PDFsharp. If it is imported from
  /// an external PDF file, the value is converted into a direct object. Acrobat sometime uses indirect
  /// names to save space, because an indirect reference to a name may be shorter than a long name.
  /// </summary>
  [DebuggerDisplay("({Value})")]
  public sealed class PdfNameObject : PdfObject
  {
    /// <summary>
    /// Initializes a new instance of the <see cref="PdfNameObject"/> class.
    /// </summary>
    public PdfNameObject()
    {
      this.value = "/";  // Empty name.
    }

    /// <summary>
    /// Initializes a new instance of the <see cref="PdfNameObject"/> class.
    /// </summary>
    /// <param name="document">The document.</param>
    /// <param name="value">The value.</param>
    public PdfNameObject(PdfDocument document, string value)
      : base(document)
    {
      if (value == null)
        throw new ArgumentNullException("value");
      if (value.Length == 0 || value[0] != '/')
        throw new ArgumentException(PSSR.NameMustStartWithSlash);

      this.value = value;
    }

    /// <summary>
    /// Determines whether the specified object is equal to the current object.
    /// </summary>
    public override bool Equals(object obj)
    {
      return this.value.Equals(obj);
    }

    /// <summary>
    /// Serves as a hash function for this type.
    /// </summary>
    public override int GetHashCode()
    {
      return this.value.GetHashCode();
    }

    /// <summary>
    /// Gets or sets the name value.
    /// </summary>
    public string Value
    {
      get { return this.value; }
      set { this.value = value; }
    }
    string value;

    /// <summary>
    /// Returns the name. The string always begins with a slash.
    /// </summary>
    public override string ToString()
    {
      return this.value;
    }

    /// <summary>
    /// Determines whether a name is equal to a string.
    /// </summary>
    public static bool operator ==(PdfNameObject name, string str)
    {
      return name.value == str;
    }

    /// <summary>
    /// Determines whether a name is not equal to a string.
    /// </summary>
    public static bool operator !=(PdfNameObject name, string str)
    {
      return name.value != str;
    }

#if leads_to_ambiguity
    public static bool operator ==(string str, PdfName name)
    {
      return str == name.value;
    }

    public static bool operator !=(string str, PdfName name)
    {
      return str == name.value;
    }

    public static bool operator ==(PdfName name1, PdfName name2)
    {
      return name1.value == name2.value;
    }

    public static bool operator !=(PdfName name1, PdfName name2)
    {
      return name1.value != name2.value;
    }
#endif

    /// <summary>
    /// Writes the name including the leading slash.
    /// </summary>
    internal override void WriteObject(PdfWriter writer)
    {
      writer.WriteBeginObject(this);
      writer.Write(new PdfName(this.value));
      writer.WriteEndObject();
    }
  }
}
