# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: pyb.py,v 1.8 1998/10/29 13:06:13 gobry Exp $

from string import *
import copy
import pybhelp
import re

# ----- Boolean test -----

		
class Bool:
	def __init__ (self):
		self.neg = 0
		
	def __or__ (self, other):
		return OrConnect (self, other)

	def __and__ (self, other):
		return AndConnect (self, other)

	def __neg__ (self):
		self.neg = not self.neg
		return self

class Tester (Bool):

	def __init__ (self, field, value):
		Bool.__init__ (self)
		self.__test = re.compile (value, re.IGNORECASE)
		self.field = lower (field)
		self.value = value

	def match (self, entry):
		if self.neg:
			if entry.has_key (self.field):
				return self.__test.search (entry [self.field]) == None
			else:
				return 1
		else:
			if entry.has_key (self.field):
				return self.__test.search (entry [self.field]) != None
			else:
				return 0
			
class Connecter (Bool):
	def __init__ (self, left, right):
		Bool.__init__ (self)
		self.left = left
		self.right = right


class OrConnect (Connecter):
	def match (self, entry):
		ret = (self.left.match (entry) or
		       self.right.match (entry)) 
		if self.neg:
			ret = not ret

		return ret

class AndConnect (Connecter):
	def match (self, entry):
		ret = (self.left.match (entry) and
			self.right.match (entry))  
		if self.neg:
			ret = not ret

		return ret


# ----- A class that holds references from several databases -----

pybhelp.register ('references', """

References are generic holders of subsets or complete databases. One
can apply several method on a reference :

 - ref.add (base, items) : adds `items' from a given `base' in the
   reference
 
 - ref.foreach (method, argument) : calls `method' on every entry
 - ref.where ( condition ) : returns the reference of entries matching
   the condition

 - ref.bases () : a list of all the bases that compose the reference
 - ref.refs (base) : entries from a given database
 - ref.base_foreach (base) : like `foreach' but only on a given database

In addition, references can be merged with `+'.
""")
class Reference:

	def __init__ (self):
		self.__refs = []

	def add (self, base, liste = None):
		"Ajouter des references liees a une base"

		for b in self.__refs:
			if b [0] == base:
				if liste <> None:
					for i in liste:
						b.append (i)
				return

		if liste <> None:
			local = copy.copy (liste)
			local.insert (0, base)
		else:
			local = [ base ]
			
		self.__refs.append (local)

	def __repr__ (self):
		""
		return `self.__refs`

	def bases (self):
		"Returns the databases available in the Reference object"
		bases = []
		for b in self.__refs:
			bases.append (b [0])
			
		return bases

	def refs (self, base):
		"Returns the references available in a given database"
		for b in self.__refs:
			if b [0] == base:
				if len (b) > 1:
					return b [1:]
				else:
					return None
		
		return []

	def base_foreach (self, base, function, arg):
		"Foreach entry in a base"

		liste = self.refs (base)

		if liste != None:
			for e in liste:
				function (base [e], arg)
		else:
			# On teste toute la base
			base.foreach (function, arg)

			
	def foreach (self, function, arg):
		"Foreach entry"

		for b in self.bases ():
			self.base_foreach (b, function, arg)

			
	def __add__ (self, other):

		r = Reference ()

		for b in self.bases ():
			r.add (b, self.refs (b))

		for b in other.bases ():
			r.add (b, other.refs (b))

		return r


	def where (self, test):
		r = Reference ()

		def __adder (entry, arg):
			test, res, base = arg

			if test.match (entry):
				res.add (base, entry.name ())

				
		for b in self.bases ():
			list = self.refs (b)

			# Si on doit tester toute une base, on
			# utilise la commande appropriee
			if list == None:
				r = r + b.where (test)
			else:
				# Sinon on test cas par cas
				self.base_foreach (b, __adder, (test, r, b))

		return r
				
KnownFields = {}
KnownTypes  = {}
FieldsOrder = []

def format_string (string, width, first, next):
	"Format a string on a given width"


	out = []
	current = first
	
	while len (string) > width - current:
		
		pos = width - current - 1
		
		while pos > 0 and string [pos] <> ' ':
			pos = pos - 1

		if pos == 0:
			pos = width - current
			taille = len (string)
			while pos < taille and string [pos] <> ' ':
				pos = pos + 1

		out.append (' ' * current + string [0:pos])
		string = string [pos+1:]
		current = next

	out.append (' ' * current + string)

	return strip (join (out, '\n'))

class Entry:

	def __init__ (self, name, type, dict):
		self.__name = name
		self.__type = type
		self.__dict = dict
		
	def keys (self):
		return self.__dict.keys ()

	def has_key (self, key):
		return self.__dict.has_key (key)

	def __getitem__ (self, key):
		return self.__dict [key]

	# Retourne une entree pour BibTeX
	def quoted (self, key):
		value = self.__dict [key]
		
		if find (value, '"') != -1:
			value = '{' + value + '}'
		else:
			value = '"' + value + '"'

		return value

	def __setitem__ (self, name, value):
		self.__dict [name] = value

	def __delitem__ (self, name):
		del self.__dict [name]

	def type (self):
		return self.__type

	def name (self):
		return self.__name

	def __repr__ (self):
		return self.__str__ ()
	
	def __str__ (self):
		"Print an entry as BiBTeX"

		type = self.type ()
		if KnownTypes.has_key (type):
			type = KnownTypes [type]
			
		text = '@%s{%s,\n\n' % (type, self.name ())
		dico = self.keys ()
		
		for entry in FieldsOrder:
			field = KnownFields [entry]
			if self.has_key (field):
				text = text + '  ' + format_string (
					'%-14s = %s' % (entry, self.quoted (field)), 75, 0, 19) + ',\n'
				dico.remove (field)

		for entry in dico:
			text = text + '  ' + format_string (
				'%-14s = %s' % (entry, self.quoted (entry)), 75, 0, 19) + ',\n'
			
		text = text + '}'
		
		return text


	def __add__ (self, other):
		"Merges two entries"

		ret = Entry (self.name (), self.type (), {})

		# Prendre ses propres entrees
		for f in self.keys ():
			ret [f] = self [f]

		# et ajouter celles qu'on n'a pas
		for f in other.keys ():
			if not self.has_key (f):
				ret [f] = other [f]

		return ret
	
# Une base de references bibliographiques, comme un dictionnaire avec des extensions...

class DBase:
	name = ''
	
	def __init__ (self, basename):
		self.name = basename

	def keys (self):
		""
		raise NameError, "not inherited"

	def has_key (self, field, key):
		""
		raise NameError, "not inherited"

	def __getitem__ (self, name):
		""
		raise NameError, "not inherited"

	def __setitem__ (self, name, value):
		""
		raise NameError, "not inherited"

	def __delitem__ (self, name):
		""
		raise NameError, "not inherited"

	def __len__ (self, name):
		""
		return len (self.keys ())

	def __repr__ (self):
		""
		return "generic bibliographic database (" + `len (self)` + " entries)"
		
	# Specifique a la base biblio
	def where (self, key):
		"Chercher une entree selon des criteres"
		raise NameError, "not inherited"

	def foreach (self, function, args):
		"Parcourt de toutes les entrees"
		raise NameError, "not inherited"

	def update (self):
		"Remise a jour des donnes"
		raise NameError, "not inherited"


