/*
 This file is part of pybliographer
 
 Copyright (C) 1998-1999 Frederic GOBRY
 Email : gobry@idiap.ch
 	   
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2 
 of the License, or (at your option) any later version.
   
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details. 
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 
 $Id: bibtex.h,v 1.21 1999/07/20 14:53:47 gobry Exp $
*/

#ifndef __bibtex_h__
#define __bibtex_h__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>

#include <glib.h>
#include "logging.h"
    
    /* 
       General structure for BibTeX content storing
    */
    typedef struct _BibtexStruct BibtexStruct;

    typedef enum {
	/* List holder */
	BIBTEX_STRUCT_LIST,
	/* Text body */
	BIBTEX_STRUCT_TEXT,
	/* Reference to alias */
	BIBTEX_STRUCT_REF,
	/* Sublevel, enclosed in braces or quotes */
	BIBTEX_STRUCT_SUB,
	/* \ like command */
	BIBTEX_STRUCT_COMMAND,
	/* space */
	BIBTEX_STRUCT_SPACE,
    }
    BibtexStructType;

    typedef enum {
	BIBTEX_ENCLOSER_BRACE,
	BIBTEX_ENCLOSER_QUOTE,
	BIBTEX_ENCLOSER_NONE
    }
    BibtexEncloserType;

    typedef struct {
	BibtexEncloserType encloser;
	BibtexStruct * content;
    }
    BibtexStructSub;

    struct _BibtexStruct {
	BibtexStructType type;

	union {
	    GList * list;
	    gchar * text;
	    gchar * ref;
	    gchar * com;
	    BibtexStructSub * sub;
	    gboolean unbreakable;
	} value;
    };

    /* 
       Available elementary field types 
    */

    typedef enum {
	BIBTEX_OTHER = 0,
	BIBTEX_AUTHOR,
	BIBTEX_TITLE,
	BIBTEX_DATE,
    } 
    BibtexFieldType;

    /* Single author */
    typedef struct {
	gchar * honorific;
	gchar * first;
	gchar * last;
	gchar * lineage;
    }
    BibtexAuthor;

    /* Group of authors */

    typedef GArray BibtexAuthorGroup;


    /* Date */
    typedef struct {
	gint16 year;
	gint16 month;
	gint16 day;
    }
    BibtexDateField;


    /* 
       General field declaration
    */

    typedef struct {
	BibtexFieldType    type;
	BibtexStruct *     structure;

	gboolean converted;
	gboolean loss;

	gchar * text;

	union {
	    BibtexAuthorGroup * author;
	    BibtexDateField     date;
	} field;
    }
    BibtexField;

    /*
      Full BibTeX entry
    */

    typedef struct {
	gchar * type;
	gchar * name;
	
	int length;
	int offset;

	BibtexStruct * preamble;

	GHashTable * table;
    } 
    BibtexEntry;

    /*
      Full BibTeX database
    */

    typedef struct {
	gchar * filename;
	FILE * file;

	int line;
	int offset;


	GHashTable * table;
	gpointer buffer;

	int debug;
	gboolean eof, error;
    }
    BibtexFile;

    /* -------------------------------------------------- 
       High level interface
       -------------------------------------------------- */

    BibtexFile * bibtex_open  (gchar * file);
    void         bibtex_close (BibtexFile *);

    BibtexEntry * bibtex_next_entry     (BibtexFile * file);
    void    	  bibtex_first_entry    (BibtexFile * file);
    void    	  bibtex_offset_entry   (BibtexFile * file, gint);
    gint    	  bibtex_current_offset (BibtexFile * file);

    BibtexEntry * bibtex_entry_new      (void);
    void          bibtex_entry_destroy  (BibtexEntry * entry, gboolean content);


    /* Fields manipulation */

    BibtexField * bibtex_field_new     (BibtexFieldType type);
    void          bibtex_field_destroy (BibtexField * field, gboolean content);
    BibtexField * bibtex_field_parse   (BibtexField * field, GHashTable * dico);


    /* Authors manipulation */

    BibtexAuthor * bibtex_author_new     (void);
    void           bibtex_author_destroy (BibtexAuthor * author);

    BibtexAuthorGroup * bibtex_author_group_new     (void);
    void                bibtex_author_group_destroy (BibtexAuthorGroup * authors);
    BibtexAuthorGroup * bibtex_author_parse         (BibtexStruct * authors, 
						     GHashTable * dico);


    /* Structure allocation / manipulation */

    BibtexStruct * bibtex_struct_new     (BibtexStructType type);
    void           bibtex_struct_destroy (BibtexStruct * structure, 
					  gboolean content);

    BibtexStruct * bibtex_struct_copy    (BibtexStruct * source);
    BibtexStruct * bibtex_struct_flatten (BibtexStruct * source);
    BibtexStruct * bibtex_struct_append  (BibtexStruct *, BibtexStruct *);


    /* Structure convertions */

    gchar *       bibtex_struct_as_string (BibtexStruct * s, 
					   BibtexFieldType type, 
					   GHashTable * dico,
					   gboolean * loss);

    gchar *       bibtex_struct_as_bibtex (BibtexStruct * s);

    BibtexField * bibtex_struct_as_field  (BibtexStruct * s, 
					   BibtexFieldType type);

    BibtexField * bibtex_string_as_field (gchar * string,
					  BibtexFieldType type);

    /* --------------------------------------------------
       Low level function
       -------------------------------------------------- */

    gchar * bibtex_accent_string (BibtexStruct * s, GList ** flow, gboolean * loss);
    void    bibtex_capitalize    (gchar * text, gboolean is_noun, gboolean at_start);

    /* Temporary strings */
    gchar * bibtex_tmp_string      (gchar *);
    void    bibtex_tmp_string_free (void);
    
    /* Parse next entry */
    BibtexEntry * parse_bibtex (BibtexFile * file);

    /*  analyser on a file */
    void bibtex_analyzer_initialize (BibtexFile * file);
    void bibtex_analyzer_finish     (BibtexFile * file);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#ifdef __DEBUGGING__
#include "debugging.h"
#endif /* __DEBUGGING__ */

#endif /* bibtex.h */
