/*	$NetBSD: pci_machdep.c,v 1.10 2003/11/07 17:12:56 augustss Exp $	*/

/*
 * Copyright (c) 1996 Christopher G. Demetriou.  All rights reserved.
 * Copyright (c) 1994 Charles M. Hannum.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Charles M. Hannum.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Machine-specific functions for PCI autoconfiguration.
 *
 * On PCs, there are two methods of generating PCI configuration cycles.
 * We try to detect the appropriate mechanism for this machine and set
 * up a few function pointers to access the correct method directly.
 *
 * The configuration method can be hard-coded in the config file by
 * using `options PCI_CONF_MODE=N', where `N' is the configuration mode
 * as defined section 3.6.4.1, `Generating Configuration Cycles'.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: pci_machdep.c,v 1.10 2003/11/07 17:12:56 augustss Exp $");

#include <sys/types.h>
#include <sys/param.h>
#include <sys/device.h>
#include <sys/errno.h>
#include <sys/extent.h>
#include <sys/malloc.h>
#include <sys/queue.h>
#include <sys/systm.h>
#include <sys/time.h>
#include <machine/pcb.h>

#include <uvm/uvm.h>

#define _POWERPC_BUS_DMA_PRIVATE
#include <machine/bus.h>
#include <machine/pio.h>
#include <machine/intr.h>

#include <dev/ic/cpc700reg.h>
#include <machine/pmppc.h>

#include <dev/pci/pcivar.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pciconf.h>

/*
 * Address conversion as seen from a PCI master.
 * XXX Shouldn't use 0x80000000, the actual value
 * should come from the BAR.
 */
#define PHYS_TO_PCI_MEM(x)	((x) + 0x80000000)
#define PCI_MEM_TO_PHYS(x)	((x) - 0x80000000)

static bus_addr_t phys_to_pci(bus_dma_tag_t, bus_addr_t);
static bus_addr_t pci_to_phys(bus_dma_tag_t, bus_addr_t);

struct powerpc_bus_dma_tag pci_bus_dma_tag = {
	0,			/* _bounce_thresh */
	_bus_dmamap_create,
	_bus_dmamap_destroy,
	_bus_dmamap_load,
	_bus_dmamap_load_mbuf,
	_bus_dmamap_load_uio,
	_bus_dmamap_load_raw,
	_bus_dmamap_unload,
	NULL,			/* _dmamap_sync */
	_bus_dmamem_alloc,
	_bus_dmamem_free,
	_bus_dmamem_map,
	_bus_dmamem_unmap,
	_bus_dmamem_mmap,
	phys_to_pci,
	pci_to_phys,
};

static bus_addr_t
phys_to_pci(bus_dma_tag_t t, bus_addr_t a)
{
	return PHYS_TO_PCI_MEM(a);
}

static bus_addr_t pci_to_phys(bus_dma_tag_t t, bus_addr_t a)
{
	return PCI_MEM_TO_PHYS(a);
}

void
pci_attach_hook(struct device *parent, struct device *self,
		struct pcibus_attach_args *pba)
{
}

int
pci_bus_maxdevs(pci_chipset_tag_t pc, int busno)
{

	/*
	 * Bus number is irrelevant.  Configuration Mechanism 1 is in
	 * use, can have devices 0-32 (i.e. the `normal' range).
	 */
	return (32);
}

pcitag_t
pci_make_tag(pci_chipset_tag_t pc, int bus, int device, int function)
{
	pcitag_t tag;

	if (bus >= 256 || device >= 32 || function >= 8)
		panic("pci_make_tag: bad request");

	tag = CPC_PCI_CONFIG_ENABLE |
		    (bus << 16) | (device << 11) | (function << 8);
	return (tag);
}

void
pci_decompose_tag(pci_chipset_tag_t pc, pcitag_t tag, int *bp, int *dp, int *fp)
{

	if (bp != NULL)
		*bp = (tag >> 16) & 0xff;
	if (dp != NULL)
		*dp = (tag >> 11) & 0x1f;
	if (fp != NULL)
		*fp = (tag >> 8) & 0x7;
}

#define SP_PCI(tag, reg) ((tag) | (reg))

pcireg_t
pci_conf_read(pci_chipset_tag_t pc, pcitag_t tag, int reg)
{
	pcireg_t data;
	struct faultbuf env, *oldfault;

	oldfault = curpcb->pcb_onfault;
	if (setfault(&env)) {
		curpcb->pcb_onfault = oldfault;
		return 0;
	}

/*printf("pci_conf_read %x %x\n", tag, reg);*/
	out32rb(CPC_PCICFGADR, SP_PCI(tag, reg));
	data = in32rb(CPC_PCICFGDATA);
	/*out32rb(CPC_PCICFGADR, 0);*/

	curpcb->pcb_onfault = oldfault;
	return data;
}

void
pci_conf_write(pci_chipset_tag_t pc, pcitag_t tag, int reg, pcireg_t data)
{
	struct faultbuf env, *oldfault;

	oldfault = curpcb->pcb_onfault;
	if (setfault(&env)) {
		curpcb->pcb_onfault = oldfault;
		return;
	}

/*printf("pci_conf_write %x %x %x\n", tag, reg, data);*/
	out32rb(CPC_PCICFGADR, SP_PCI(tag, reg));
	out32rb(CPC_PCICFGDATA, data);
	/*out32rb(CPC_PCICFGADR, 0);*/
	curpcb->pcb_onfault = oldfault;
}

int
pci_intr_map(struct pci_attach_args *pa, pci_intr_handle_t *ihp)
{
	int	pin = pa->pa_intrpin;
	int	line = pa->pa_intrline;

	if (pin == 0) {
		/* No IRQ used. */
		goto bad;
	}

	if (pin > 4) {
		printf("pci_intr_map: bad interrupt pin %d\n", pin);
		goto bad;
	}

	if (line == 255) {
		printf("pci_intr_map: no mapping for pin %c\n", '@' + pin);
		goto bad;
	}
	/*printf("pci_intr_map pin=%d line=%d\n", pin, line);*/

	switch (line & 3) {	/* XXX what should this be? */
	case 0: *ihp = PMPPC_I_BPMC_INTA; break;
	case 1: *ihp = PMPPC_I_BPMC_INTB; break;
	case 2: *ihp = PMPPC_I_BPMC_INTC; break;
	case 3: *ihp = PMPPC_I_BPMC_INTD; break;
	}
	return 0;

bad:
	*ihp = -1;
	return 1;
}

const char *
pci_intr_string(pci_chipset_tag_t pc, pci_intr_handle_t ih)
{
	static char irqstr[8];		/* 4 + 2 + NULL + sanity */

	if (ih < 0 || ih >= ICU_LEN)
		panic("pci_intr_string: bogus handle 0x%x", ih);

	sprintf(irqstr, "irq %d", ih);
	return (irqstr);
	
}

const struct evcnt *
pci_intr_evcnt(pci_chipset_tag_t pc, pci_intr_handle_t ih)
{

	/* XXX for now, no evcnt parent reported */
	return NULL;
}

void *
pci_intr_establish(pci_chipset_tag_t pc, pci_intr_handle_t ih, int level,
		   int (*func)(void *), void *arg)
{
	/*
	 * ih is the value assigned in pci_intr_map(), above.
	 */
	return intr_establish(ih, IST_LEVEL, level, func, arg);
}

void
pci_intr_disestablish(pci_chipset_tag_t pc, void *cookie)
{
	intr_disestablish(cookie);
}

void
pci_conf_interrupt(pci_chipset_tag_t pc, int bus, int dev, int pin, int swiz,
    int *iline)
{
	int line;

	line = (swiz + dev) & 3;
	/* XXX UGLY UGLY, figure out the real interrupt mapping */
	if (bus==3&&dev==2&&pin==1&&swiz==3) line=2;
/*
	printf("pci_conf_interrupt: bus=%d dev=%d pin=%d swiz=%d => line=%d\n",
		bus, dev, pin, swiz, line);
*/
	*iline = line;
}
