/*
 * Copyright 1999-2001 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details. 
 */

package com.vizdom.dbd.jdbc;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;


/**
 * This class uses Runtime.exec to start a process and passes
 * arguments to it in the same fashion as a web server passes 
 * arguments to a CGI process. The following environment variables
 * are set when the process starts: 
 * <ul>
 * <li>REQUEST_METHOD (always POST)
 * <li>CONTENT_TYPE
 * <li>CONTENT_LENGTH
 * </ul>
 * In addition, if a JDBC Connection object is provided, a 
 * Server object is instantiated using that Connection and set 
 * to listening for connections in a separate thread.
 * <p>
 * This class has been tested with Perl scripts using CGI.pm
 * to read the input.
 *
 * @author Gennis Emerson
 * @version $Revision: 1.5 $
 * @see Server
 */
public class CgiProcess extends java.lang.Process
{
    /** 
     * This key is used by users of this class to specify a port
     * number for the JDBC server, and by this class to send the
     * port number as part of the arguments to the subprocess
     * in {@link #CgiProcess(String, Hashtable, java.sql.Connection)}
     */
    public static final String PORT_KEY = "com.vizdom.dbd.jdbc.Server.port";
    /** The actual Process object returned by exec. */
    private Process mProcess;
    /** The thread in which the Server object is running, if there is one. */
    private Thread mServerThread;
    /** The port to which the Server is listening. */
    private Integer mPort;

    /**
     * Starts a process using the given command line and passes the values
     * in <code>aParameterList</code> to that process. Calls
     * <code>CgiProcess(aCommandLine, aParameterList, null)</code>.
     *
     * @param aCommandLine a complete command line to be passed to 
     *      <code>Runtime.exec</code>
     * @param aParameterList the name-value pairs to be sent to the
     *      script
     * @exception IOException if an error occurs starting the
     *      process
     * @exception Exception used to report other failures
     */
    public CgiProcess(String aCommandLine, Hashtable aParameterList) 
        throws IOException, Exception
    {
        this(aCommandLine, aParameterList, null);
    }
    
    /**
     * Starts a process using the given command line and passes the values
     * in <code>aParameterList</code> to that process. If 
     * <code>aJdbcConnection</code> is not null, a {@link Server}
     * object will be created and started in a new thread.
     * <p>
     * If a Server object is started, the port it listens to may be
     * specified by adding an Integer object to <code>aParameterList</code>
     * with a key of {@link #PORT_KEY}. If no port is specified,
     * the server will be started on a random port. The port used
     * will be added to the set of parameters and passed to the
     * started process in the query string.
     * <p>
     * The keys in <code>aParameterList</code> are assumed to be strings.
     * The Objects in <code>aParameterList</code> will be converted
     * to strings by calling their <code>toString</code> methods.
     * The data is passed to the script on <code>stdin</code> in the 
     * form of a query string. The following environment variables
     * are set: 
     * <ul>
     * <li>CONTENT_TYPE = application/x-www-form-urlencoded
     * <li>CONTENT_LENGTH is set to the number of bytes in the
     *      query string constructed from the elements in 
     *      <code>aParameterList</code> (using the platform default
     *      character encoding to convert characters to bytes)
     * </ul>
     *
     * @param aCommandLine a complete command line to be passed to 
     *      <code>Runtime.exec</code>
     * @param aParameterList the name-value pairs to be sent to the
     *      script
     * @param aJdbcConnection a JDBC Connection object; may be null
     * @exception IOException if an error occurs starting the
     *      process
     * @exception Exception used to report other failures
     */
    public CgiProcess(String aCommandLine, Hashtable aParameterList, 
        java.sql.Connection aJdbcConnection) throws java.io.IOException, 
        Exception
    {
        boolean addedKey = false;
        Server server = null;
        if (aJdbcConnection != null)
        {
            mPort = (Integer) aParameterList.get(PORT_KEY);
            if (mPort == null)
            {
                server = new Server(aJdbcConnection);
                mPort = new Integer(server.getPort());
                aParameterList.put(PORT_KEY, mPort);
                addedKey = true;
            }
            else
                server = new Server(aJdbcConnection, mPort.intValue());
        }
        
        String params = getCgiParamString(aParameterList);
        byte[] paramBytes = params.getBytes(); // Platform default encoding.
        if (addedKey)
            aParameterList.remove(PORT_KEY);

        String[] env = 
        { 
            "REQUEST_METHOD=POST",
            "CONTENT_TYPE=application/x-www-form-urlencoded",
            "CONTENT_LENGTH=" + paramBytes.length,
        };
                
        mProcess = Runtime.getRuntime().exec(aCommandLine, env);
        OutputStream anOut= mProcess.getOutputStream();
        anOut.write(paramBytes); 
        anOut.close();
        
        mServerThread = null;
        if (server != null)
        {
            mServerThread = new Thread(server);
            if (mServerThread == null)
                throw new Exception("Failed to create client thread.");
            mServerThread.start();
        }
    }
 
    
    // We probably also need versions of these with port arguments.
    /**
     * Starts a process using the given command line, with the
     * provided content type and content length in the process
     * environment. Calls {@link #CgiProcess(aCommandLine, aContentType,
     * aContentLength, null)}.
     * 
     * @param aCommandLine a complete command line to be passed to 
     *      <code>Runtime.exec</code>
     * @param aContentType the value to be used for the CONTENT_TYPE
     *      environment variable
     * @param aContentLength the value to be used for the CONTENT_LENGTH
     *      environment variable
     * @exception IOException if an error occurs starting the
     *      process
     * @exception Exception used to report other failures
     */
    public CgiProcess(String aCommandLine, String aContentType,
        int aContentLength) throws java.io.IOException, Exception
    {
        this(aCommandLine, aContentType, aContentLength, null);
    }
    
    /**
     * Starts a process using the given command line, with the
     * provided content type and content length in the process
     * environment. If <code>aJdbcConnection</code> is provided,
     * a Server object will be started using the provided Connection.
     * This Server will be started on a random port.
     * <p>
     * It is the user's responsibility to get the Process output stream
     * and write the number of bytes specified by 
     * <code>aContentLength</code>.
     * <p>
     * This method is intended to allow a client to use
     * a content-type other than application/x-www-form-urlencoded
     * and non-textual data.
     * 
     * @param aCommandLine a complete command line to be passed to 
     *      <code>Runtime.exec</code>
     * @param aContentType the value to be used for the CONTENT_TYPE
     *      environment variable
     * @param aContentLength the value to be used for the CONTENT_LENGTH
     *      environment variable
     * @param aJdbcConnection a JDBC Connection object; may be null
     * @exception IOException if an error occurs starting the
     *      process
     * @exception Exception used to report other failures
     */
    public CgiProcess(String aCommandLine, String aContentType,
        int aContentLength, java.sql.Connection aJdbcConnection) 
        throws java.io.IOException, Exception
    {
        Server server = null;
        if (aJdbcConnection != null)
        {
            server = new Server(aJdbcConnection);
            mPort = new Integer(server.getPort());
        }
        
        String[] env = 
        { 
            "REQUEST_METHOD=POST",
            "CONTENT_TYPE=" + aContentType,
            "CONTENT_LENGTH=" + aContentLength,
        };
        
        mProcess = Runtime.getRuntime().exec(aCommandLine, env);
        
        mServerThread = null;
        if (server != null)
        {
            mServerThread = new Thread(server);
            if (mServerThread == null)
                throw new Exception("Failed to create client thread.");
            mServerThread.start();
        }
    }
    
    

    /**
     * Builds a query string from the data in <code>aHashtable</code>
     * The Hashtable keys must be strings. The Hashtable 
     * values will have their <code>toString</code> methods called.
     * Note that the returned query string will have a trailing "&"
     * appended.
     *
     * @param aHashtable the data to be turned into a query string
     * @return a query string; empty if there are no values in 
     *      the hashtable
     */
    private String getCgiParamString(Hashtable aHashtable)
    {
        if (aHashtable == null)
            return "";
        StringBuffer params = new StringBuffer(1024); // size???
        Enumeration keys = aHashtable.keys();
        while (keys.hasMoreElements())
        {
            String key = (String) keys.nextElement();
            String value = aHashtable.get(key).toString();
            if (value == null)  // Somebody's toString could do this. ???
                value = ""; // URLEncoder doesn't like null arguments.
            
            params.append(java.net.URLEncoder.encode(key));
            params.append("=");
            params.append(java.net.URLEncoder.encode(value));
            params.append("&");
        }
        return params.toString();      
    }

    /**
     * Returns the port to which the Server object, if created, is 
     * listening.
     *
     * @return the Server port number
     * @see Server
     */
    public int getServerPort()
    {
        return mPort.intValue();
    }

    /**
     * Returns the thread in which the Server object is running, or null
     * if no Server object was created.
     * 
     * @return the thread in which the Server object is running, or null
     *  if no Server object was created
     * @see Server
     */
    public Thread getServerThread()
    {
        return mServerThread;
    }


    // The Process methods.

   /**
     * Gets the output stream of the subprocess.
     * This stream is usually buffered.
     *
     * @return  the output stream connected to the normal input of the
     *          subprocess.
     */
    public OutputStream getOutputStream()
    {
        return mProcess.getOutputStream();
    }

    /** 
     * Gets the input stream of the subprocess.
     * This stream is usually buffered.
     *
     * @return  the input stream connected to the normal output of the
     *          subprocess.
     */
    public InputStream getInputStream()
    {
        return mProcess.getInputStream();
    }

    /**
     * Gets the error stream of the subprocess.
     * This stream is usually unbuffered.
     *
     * @return  the input stream connected to the error stream of the
     *          subprocess.
     */
    public InputStream getErrorStream()
    {
        return mProcess.getErrorStream();
    }

    /**
     * Waits for the subprocess to complete. This method returns 
     * immediately if the subprocess has already terminated. If the
     * subprocess has not yet terminated, the calling thread will be
     * blocked until the subprocess exits.
     *
     * @return     the exit value of the process.
     * @exception  InterruptedException  if the <code>waitFor</code> was
     *               interrupted.
     */
    public int waitFor() throws InterruptedException
    {
        return mProcess.waitFor();
    }

    /**
     * Returns the exit value for the subprocess.
     *
     * @return  the exit value of the subprocess.
     * @exception  IllegalThreadStateException  if the subprocess has not yet
     *               terminated.
     */
    public int exitValue()
    {
        return mProcess.exitValue();
    }

    /**
     * Kills the subprocess. 
     */
    public void destroy()
    {
        mProcess.destroy();
    }
}

