#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;
use lib 'lib';

use Hypersonic::Future;
use Hypersonic::Future::Pool;

# Compile Future (includes Pool)
Hypersonic::Future->compile();

Hypersonic::Future::Pool->init;

# Test error handling - die
{
    my $f = Hypersonic::Future->new;

    Hypersonic::Future::Pool->submit($f, sub {
        die "Intentional error";
    }, []);

    select(undef, undef, undef, 0.1);
    Hypersonic::Future::Pool->process_ready;

    ok($f->is_failed, 'Future failed when code dies');
    my ($msg) = $f->failure;
    like($msg, qr/Intentional error/, 'Failure message contains error');
}

# Test error handling - croak
{
    my $f = Hypersonic::Future->new;

    Hypersonic::Future::Pool->submit($f, sub {
        require Carp;
        Carp::croak("Croaked error");
    }, []);

    select(undef, undef, undef, 0.1);
    Hypersonic::Future::Pool->process_ready;

    ok($f->is_failed, 'Future failed when code croaks');
    my ($msg) = $f->failure;
    like($msg, qr/Croaked error/, 'Failure message contains croak');
}

# Test error with catch recovery
{
    my $f = Hypersonic::Future->new;
    my $recovered;

    my $chain = $f->catch(sub {
        my ($err) = @_;
        $recovered = $err;
        return 'recovered';
    });

    Hypersonic::Future::Pool->submit($f, sub {
        die "Error to catch";
    }, []);

    select(undef, undef, undef, 0.1);
    Hypersonic::Future::Pool->process_ready;

    ok($f->is_failed, 'Original future failed');
    ok($chain->is_done, 'Chained future recovered');
    like($recovered, qr/Error to catch/, 'Catch received error');
    is($chain->result, 'recovered', 'Catch returned recovery value');
}

Hypersonic::Future::Pool->shutdown;

done_testing;
