
#############################################################################
## $Id: TemplateEngine.pm 8038 2006-11-09 21:12:39Z spadkins $
#############################################################################

package App::TemplateEngine;
$VERSION = (q$Revision: 8038 $ =~ /(\d[\d\.]*)/)[0];  # VERSION numbers generated by svn

use App;
use App::Service;
@ISA = ( "App::Service" );

use strict;

=head1 NAME

App::TemplateEngine - Interface for rendering HTML templates

=head1 SYNOPSIS

    use App;

    $context = App->context();
    $template_engine = $context->service("TemplateEngine");  # or ...
    $template_engine = $context->template_engine();

    $template = "index.html";
    $text = $template_engine->render($template);

=head1 DESCRIPTION

A TemplateEngine Service is a means by which a template (such as an
HTML template) may be rendered (with variables interpolated).

=cut

#############################################################################
# CLASS GROUP
#############################################################################

=head1 Class Group: TemplateEngine

The following classes might be a part of the TemplateEngine Class Group.

=over

=item * Class: App::TemplateEngine

=item * Class: App::TemplateEngine::TemplateToolkit

=item * Class: App::TemplateEngine::Embperl

=item * Class: App::TemplateEngine::Mason

=item * Class: App::TemplateEngine::AxKit

=item * Class: App::TemplateEngine::ASP

=item * Class: App::TemplateEngine::CGIFastTemplate

=item * Class: App::TemplateEngine::TextTemplate

=item * Class: App::TemplateEngine::HTMLTemplate

=back

=cut

#############################################################################
# CLASS
#############################################################################

=head1 Class: App::TemplateEngine

A TemplateEngine Service is a means by which a template (such as an
HTML template) may be rendered (with variables interpolated).

 * Throws: App::Exception::TemplateEngine
 * Since:  0.01

=head2 Class Design

...

=cut

#############################################################################
# CONSTRUCTOR METHODS
#############################################################################

=head1 Constructor Methods:

=cut

#############################################################################
# new()
#############################################################################

=head2 new()

The constructor is inherited from
L<C<App::Service>|App::Service/"new()">.

=cut

#############################################################################
# PUBLIC METHODS
#############################################################################

=head1 Public Methods:

=cut

#############################################################################
# render()
#############################################################################

=head2 render()

    * Signature: $text = $template_engine->render($template);
    * Param:     $template          string
    * Return:    $text              text
    * Throws:    App::Exception::TemplateEngine
    * Since:     0.01

    Sample Usage:

    $text = $template_engine->render($template);

=cut

sub render {
    &App::sub_entry if ($App::trace);
    my ($self, $template) = @_;

    my ($template_text, $values, $text);
    $template_text = $self->read_template($template);
    $values = $self->prepare_values();
    $text = $self->substitute($template_text, $values);
    &App::sub_exit($text) if ($App::trace);
    $text;
}

#############################################################################
# PROTECTED METHODS
#############################################################################

=head1 Protected Methods:

=cut

#############################################################################
# Method: service_type()
#############################################################################

=head2 service_type()

Returns 'TemplateEngine';

    * Signature: $service_type = App::TemplateEngine->service_type();
    * Param:     void
    * Return:    $service_type  string
    * Since:     0.01

    $service_type = $template_engine->service_type();

=cut

sub service_type () { 'TemplateEngine'; }

#############################################################################
# read_template()
#############################################################################

=head2 read_template()

    * Signature: $template_text = $template_engine->read_template($template);
    * Param:     $template          string
    * Return:    $template_text     text
    * Throws:    App::Exception::TemplateEngine
    * Since:     0.01

    Sample Usage:

    $template_text = $template_engine->read_template($template);

=cut

sub read_template {
    &App::sub_entry if ($App::trace);
    my ($self, $template) = @_;

    my $context = $self->{context};
    my $theme = $context->get_user_option("theme");
    my $options = $context->{options};

    my ($template_dir, $template_text);
    local(*App::FILE);
    $template_dir = $self->{template_dir} || $options->{template_dir} || "$options->{prefix}/templates";

    if ($theme && open(App::FILE,"< $template_dir/$theme/$template")) {
        local($/) = undef;
        $template_text = <App::FILE>;
        close(App::FILE);
    }
    elsif (open(App::FILE,"< $template_dir/$template")) {
        local($/) = undef;
        $template_text = <App::FILE>;
        close(App::FILE);
    }
    else {
        # maybe we should throw an exception here
        $template_text = "Template [$template_dir/$template] not found.";
    }

    &App::sub_exit($template_text) if ($App::trace);
    return($template_text);
}

#############################################################################
# prepare_values()
#############################################################################

=head2 prepare_values()

    * Signature: $values = $template_engine->prepare_values();
    * Param:     void
    * Return:    $values            {}
    * Throws:    App::Exception::TemplateEngine
    * Since:     0.01

    Sample Usage:

    $values = $template_engine->prepare_values();

=cut

sub prepare_values {
    &App::sub_entry if ($App::trace);
    my ($self) = @_;

    my $context = $self->{context};
    my $values = $context->session_object("default");

    #my (%values);
    #%values = %$values if ($values);
    #my $options = $context->{options};
    #foreach my $key (keys %$options) {
    #    $values{$key} = $options->{$key} if (!defined $values{$key});
    #}

    &App::sub_exit($values) if ($App::trace);
    return($values);
}

#############################################################################
# substitute()
#############################################################################

=head2 substitute()

    * Signature: $text = $template_engine->substitute($template_text, $values);
    * Param:     $template_text     string
    * Param:     $values            {}
    * Return:    $text              text
    * Throws:    App::Exception::TemplateEngine
    * Since:     0.01

    Sample Usage:

    $text = $template_engine->substitute($template_text, $values);

=cut

sub substitute {
    &App::sub_entry if ($App::trace);
    my ($self, $template_text, $values) = @_;

    my ($phrase, $var, $value, $context, $expand);
    $context = $self->{context};
    $values = {} if (! defined $values);
    my $options = $context->{options};

    # strict syntax usage is [%var%] rather than {var} which can get you in trouble
    if (! $self->{strict}) {
        while ( $template_text =~ /\{(\+?)([a-zA-Z][a-zA-Z0-9_.-]*)\}/ ) {  # vars of the form {var} or {+var}
            $expand = $1;
            $var    = $2;
            if ($expand) {
                eval {
                    $value = $context->widget($var)->html();
                };
                $value = "[$var: $@]" if ($@);
            }
            elsif (defined $values->{$var}) {
                $value = $values->{$var};
            }
            elsif (defined $options->{$var}) {
                $value = $options->{$var};
            }
            else {
                $value = "";
            }
            if ($expand) {
                $template_text =~ s/\{\+$var\}/$value/g;
            }
            else {
                $template_text =~ s/\{$var\}/$value/g;
            }
        }
    }

    while ( $template_text =~ /\[%(\+?)([^%]+)%\]/ ) {  # vars of the form [%var%] or [%+var%]
        $expand = $1;
        $var    = $2;
        if ($expand) {
            eval {
                $value = $context->widget($var)->html();
            };
            $value = "[$var: $@]" if ($@);
        }
        elsif (defined $values->{$var}) {
            $value = $values->{$var};
        }
        elsif (defined $options->{$var}) {
            $value = $options->{$var};
        }
        else {
            $value = "";
        }

        if ($expand) {
            $template_text =~ s/\[%\+$var%\]/$value/g;
        }
        else {
            $template_text =~ s/\[%$var%\]/$value/g;
        }
    }

    &App::sub_exit($template_text) if ($App::trace);
    $template_text;
}

=head1 ACKNOWLEDGEMENTS

 * Author:  Stephen Adkins <spadkins@gmail.com>
 * License: This is free software. It is licensed under the same terms as Perl itself.

=head1 SEE ALSO

L<C<App::Context>|App::Context>,
L<C<App::Service>|App::Service>

=cut

1;

