/*
 * lookup_yp.c
 *
 * Module for Linux automountd to access a YP (NIS) automount map
 */

#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include <unistd.h>
#include <dlfcn.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <rpc/rpc.h>
#include <rpc/xdr.h>
#include <rpcsvc/yp_prot.h>
#include <rpcsvc/ypclnt.h>

#define MODULE_LOOKUP
#include "automount.h"

#define MAPFMT_DEFAULT "sun"

#define MODPREFIX "lookup(yp): "

struct lookup_context {
  char *domainname;
  char *mapname;
  void *parse_dlhandle;
  parse_init_t parse_init;
  parse_mount_t parse_mount;
  parse_done_t parse_done;
  void *parse_context;
};

int lookup_version = AUTOFS_LOOKUP_VERSION; /* Required by protocol */

int lookup_init(char *mapfmt, int argc, char **argv, void **context)
{
  struct lookup_context *ctxt;
  int err;
  int *parse_version;
  char path[PATH_MAX];
  void *dlhandle;

  if ( !(*context = ctxt = malloc(sizeof(struct lookup_context))) ) {
    syslog(LOG_CRIT, MODPREFIX "%m");
    return 1;
  }
  
  if ( argc < 1 ) {
    syslog(LOG_CRIT, MODPREFIX "No map name");
    return 1;
  }
  ctxt->mapname = argv[0];

  err = yp_get_default_domain(&ctxt->domainname);
  if ( err ) {
    syslog(LOG_CRIT, MODPREFIX "map %s: %s\n", ctxt->mapname, yperr_string(err));
    return 1;
  }

  if ( !mapfmt )
    mapfmt = MAPFMT_DEFAULT;

  sprintf(path, "%s//parse_%s.so", AUTOFS_LIB_DIR, mapfmt);
  if ( !(dlhandle = dlopen(path, RTLD_NOW)) ) {
    syslog(LOG_CRIT, MODPREFIX "cannot open %s parse library", mapfmt);
    return 1;
  }

  if ( !(parse_version = (int *) dlsym(dlhandle, "parse_version")) ||
       *parse_version != AUTOFS_PARSE_VERSION ) {
    syslog(LOG_CRIT, MODPREFIX "%s parse library version mismatch", mapfmt);
    dlclose(dlhandle);
    return 1;
  }
  ctxt->parse_init = (parse_init_t) dlsym(dlhandle, "parse_init");
  ctxt->parse_mount = (parse_mount_t) dlsym(dlhandle, "parse_mount");
  ctxt->parse_done = (parse_done_t) dlsym(dlhandle, "parse_done");

  if ( !ctxt->parse_init || !ctxt->parse_mount || !ctxt->parse_done ) {
    syslog(LOG_CRIT, MODPREFIX "%s parse library corrupt", mapfmt);
    dlclose(dlhandle);
    return 1;
  }

  if ( ctxt->parse_init(argc-1, argv+1, &ctxt->parse_context) ) {
    dlclose(dlhandle);
    return 1;
  }

  ctxt->parse_dlhandle = dlhandle;
  return 0;
}

int lookup_mount(char *root, char *name, int name_len, void *context)
{
  struct lookup_context *ctxt = (struct lookup_context *) context;
  int err;
  char *mapent;
  int mapent_len;

  syslog(LOG_DEBUG, MODPREFIX "looking up %s", name);

  err = yp_match(ctxt->domainname,ctxt->mapname,name,name_len,&mapent,&mapent_len);
  if ( err == YPERR_KEY ) {
    /* Try to get the "*" entry if there is one - note that we *don't*
       modify "name" so & -> the name we used, not "*" */
    err = yp_match(ctxt->domainname,ctxt->mapname,"*",1,&mapent,&mapent_len);
  }
  if ( err ) {
    syslog(LOG_NOTICE, MODPREFIX "lookup for %s failed: %s", name, yperr_string(err));
    return 1;
  }

  mapent[mapent_len] = '\0';

  syslog(LOG_DEBUG, MODPREFIX "%s -> %s", name, mapent);

  return ctxt->parse_mount(root,name,name_len,mapent,ctxt->parse_context); 
}

int lookup_done(void *context)
{
  struct lookup_context *ctxt = (struct lookup_context *) context;

  ctxt->parse_done(ctxt->parse_context);
  dlclose(ctxt->parse_dlhandle);
  free(ctxt->domainname);
  free(ctxt);
  return 0;
}
