/* SPDX-License-Identifier: GPL-2.0 */
/* Copyright (C) 2021 Gerhard Engleder <gerhard@engleder-embedded.com> */

/* Hardware definition of TSNEP and EtherCAT MAC device */

#ifndef _TSNEP_HW_H
#define _TSNEP_HW_H

#include <linux/types.h>

/* type */
#define ECM_TYPE 0x0000
#define ECM_REVISION_MASK 0x000000FF
#define ECM_REVISION_SHIFT 0
#define ECM_VERSION_MASK 0x0000FF00
#define ECM_VERSION_SHIFT 8
#define ECM_QUEUE_COUNT_MASK 0x00070000
#define ECM_QUEUE_COUNT_SHIFT 16
#define ECM_GATE_CONTROL 0x02000000

/* system time */
#define ECM_SYSTEM_TIME_LOW 0x0008
#define ECM_SYSTEM_TIME_HIGH 0x000C

/* clock */
#define ECM_CLOCK_RATE 0x0010
#define ECM_CLOCK_RATE_OFFSET_MASK 0x7FFFFFFF
#define ECM_CLOCK_RATE_OFFSET_SIGN 0x80000000

/* interrupt */
#define ECM_INT_ENABLE 0x0018
#define ECM_INT_ACTIVE 0x001C
#define ECM_INT_ACKNOWLEDGE 0x001C
#define ECM_INT_LINK 0x00000020
#define ECM_INT_TX_0 0x00000100
#define ECM_INT_RX_0 0x00000200
#define ECM_INT_ALL 0x7FFFFFFF
#define ECM_INT_DISABLE 0x80000000

/* reset */
#define ECM_RESET 0x0020
#define ECM_RESET_COMMON 0x00000001
#define ECM_RESET_CHANNEL 0x00000100
#define ECM_RESET_TXRX 0x00010000

/* counter */
#define ECM_COUNTER_LOW 0x0028
#define ECM_COUNTER_HIGH 0x002C

/* control and status */
#define ECM_STATUS 0x0080
#define ECM_LINK_MODE_OFF 0x01000000
#define ECM_LINK_MODE_100 0x02000000
#define ECM_LINK_MODE_1000 0x04000000
#define ECM_NO_LINK 0x01000000
#define ECM_LINK_MODE_MASK 0x06000000

/* management data */
#define ECM_MD_CONTROL 0x0084
#define ECM_MD_STATUS 0x0084
#define ECM_MD_PREAMBLE 0x00000001
#define ECM_MD_READ 0x00000004
#define ECM_MD_WRITE 0x00000002
#define ECM_MD_ADDR_MASK 0x000000F8
#define ECM_MD_ADDR_SHIFT 3
#define ECM_MD_PHY_ADDR_MASK 0x00001F00
#define ECM_MD_PHY_ADDR_SHIFT 8
#define ECM_MD_BUSY 0x00000001
#define ECM_MD_DATA_MASK 0xFFFF0000
#define ECM_MD_DATA_SHIFT 16

/* statistic */
#define ECM_STAT 0x00B0
#define ECM_STAT_RX_ERR_MASK 0x000000FF
#define ECM_STAT_RX_ERR_SHIFT 0
#define ECM_STAT_INV_FRM_MASK 0x0000FF00
#define ECM_STAT_INV_FRM_SHIFT 8
#define ECM_STAT_FWD_RX_ERR_MASK 0x00FF0000
#define ECM_STAT_FWD_RX_ERR_SHIFT 16

/* tsnep */
#define TSNEP_MAC_SIZE 0x4000
#define TSNEP_QUEUE_SIZE 0x1000
#define TSNEP_QUEUE(n) ({ typeof(n) __n = (n); \
			  (__n) == 0 ? \
			  0 : \
			  TSNEP_MAC_SIZE + TSNEP_QUEUE_SIZE * ((__n) - 1); })
#define TSNEP_MAX_QUEUES 8
#define TSNEP_MAX_FRAME_SIZE (2 * 1024) /* hardware supports actually 16k */
#define TSNEP_DESC_SIZE 256
#define TSNEP_DESC_OFFSET 128

/* tsnep register */
#define TSNEP_INFO 0x0100
#define TSNEP_INFO_TX_TIME 0x00010000
#define TSNEP_CONTROL 0x0108
#define TSNEP_CONTROL_TX_RESET 0x00000001
#define TSNEP_CONTROL_TX_ENABLE 0x00000002
#define TSNEP_CONTROL_TX_DMA_ERROR 0x00000010
#define TSNEP_CONTROL_TX_DESC_ERROR 0x00000020
#define TSNEP_CONTROL_RX_RESET 0x00000100
#define TSNEP_CONTROL_RX_ENABLE 0x00000200
#define TSNEP_CONTROL_RX_DISABLE 0x00000400
#define TSNEP_CONTROL_RX_DMA_ERROR 0x00001000
#define TSNEP_CONTROL_RX_DESC_ERROR 0x00002000
#define TSNEP_TX_DESC_ADDR_LOW 0x0140
#define TSNEP_TX_DESC_ADDR_HIGH 0x0144
#define TSNEP_RX_DESC_ADDR_LOW 0x0180
#define TSNEP_RX_DESC_ADDR_HIGH 0x0184
#define TSNEP_RESET_OWNER_COUNTER 0x01
#define TSNEP_RX_STATISTIC 0x0190
#define TSNEP_RX_STATISTIC_NO_DESC_MASK 0x000000FF
#define TSNEP_RX_STATISTIC_NO_DESC_SHIFT 0
#define TSNEP_RX_STATISTIC_BUFFER_TOO_SMALL_MASK 0x0000FF00
#define TSNEP_RX_STATISTIC_BUFFER_TOO_SMALL_SHIFT 8
#define TSNEP_RX_STATISTIC_FIFO_OVERFLOW_MASK 0x00FF0000
#define TSNEP_RX_STATISTIC_FIFO_OVERFLOW_SHIFT 16
#define TSNEP_RX_STATISTIC_INVALID_FRAME_MASK 0xFF000000
#define TSNEP_RX_STATISTIC_INVALID_FRAME_SHIFT 24
#define TSNEP_RX_STATISTIC_NO_DESC 0x0190
#define TSNEP_RX_STATISTIC_BUFFER_TOO_SMALL 0x0191
#define TSNEP_RX_STATISTIC_FIFO_OVERFLOW 0x0192
#define TSNEP_RX_STATISTIC_INVALID_FRAME 0x0193
#define TSNEP_RX_ASSIGN 0x01A0
#define TSNEP_RX_ASSIGN_ETHER_TYPE_ACTIVE 0x00000001
#define TSNEP_RX_ASSIGN_ETHER_TYPE_MASK 0xFFFF0000
#define TSNEP_RX_ASSIGN_ETHER_TYPE_SHIFT 16
#define TSNEP_MAC_ADDRESS_LOW 0x0800
#define TSNEP_MAC_ADDRESS_HIGH 0x0804
#define TSNEP_RX_FILTER 0x0806
#define TSNEP_RX_FILTER_ACCEPT_ALL_MULTICASTS 0x0001
#define TSNEP_RX_FILTER_ACCEPT_ALL_UNICASTS 0x0002
#define TSNEP_GC 0x0808
#define TSNEP_GC_ENABLE_A 0x00000002
#define TSNEP_GC_ENABLE_B 0x00000004
#define TSNEP_GC_DISABLE 0x00000008
#define TSNEP_GC_ENABLE_TIMEOUT 0x00000010
#define TSNEP_GC_ACTIVE_A 0x00000002
#define TSNEP_GC_ACTIVE_B 0x00000004
#define TSNEP_GC_CHANGE_AB 0x00000008
#define TSNEP_GC_TIMEOUT_ACTIVE 0x00000010
#define TSNEP_GC_TIMEOUT_SIGNAL 0x00000020
#define TSNEP_GC_LIST_ERROR 0x00000080
#define TSNEP_GC_OPEN 0x00FF0000
#define TSNEP_GC_OPEN_SHIFT 16
#define TSNEP_GC_NEXT_OPEN 0xFF000000
#define TSNEP_GC_NEXT_OPEN_SHIFT 24
#define TSNEP_GC_TIMEOUT 131072
#define TSNEP_GC_TIME 0x080C
#define TSNEP_GC_CHANGE 0x0810
#define TSNEP_GCL_A 0x2000
#define TSNEP_GCL_B 0x2800
#define TSNEP_GCL_SIZE SZ_2K

/* tsnep gate control list operation */
struct tsnep_gcl_operation {
	u32 properties;
	u32 interval;
};

#define TSNEP_GCL_COUNT (TSNEP_GCL_SIZE / sizeof(struct tsnep_gcl_operation))
#define TSNEP_GCL_MASK 0x000000FF
#define TSNEP_GCL_INSERT 0x20000000
#define TSNEP_GCL_CHANGE 0x40000000
#define TSNEP_GCL_LAST 0x80000000
#define TSNEP_GCL_MIN_INTERVAL 32

/* tsnep TX/RX descriptor */
#define TSNEP_DESC_SIZE 256
#define TSNEP_DESC_SIZE_DATA_AFTER 2048
#define TSNEP_DESC_OFFSET 128
#define TSNEP_DESC_OWNER_COUNTER_MASK 0xC0000000
#define TSNEP_DESC_OWNER_COUNTER_SHIFT 30
#define TSNEP_DESC_LENGTH_MASK 0x00003FFF
#define TSNEP_DESC_INTERRUPT_FLAG 0x00040000
#define TSNEP_DESC_EXTENDED_WRITEBACK_FLAG 0x00080000
#define TSNEP_DESC_NO_LINK_FLAG 0x01000000

/* tsnep TX descriptor */
struct tsnep_tx_desc {
	__le32 properties;
	__le32 more_properties;
	__le32 reserved[2];
	__le64 next;
	__le64 tx;
};

#define TSNEP_TX_DESC_OWNER_MASK 0xE0000000
#define TSNEP_TX_DESC_OWNER_USER_FLAG 0x20000000
#define TSNEP_TX_DESC_LAST_FRAGMENT_FLAG 0x00010000
#define TSNEP_TX_DESC_DATA_AFTER_DESC_FLAG 0x00020000

/* tsnep TX descriptor writeback */
struct tsnep_tx_desc_wb {
	__le32 properties;
	__le32 reserved1;
	__le64 counter;
	__le64 timestamp;
	__le32 dma_delay;
	__le32 reserved2;
};

#define TSNEP_TX_DESC_UNDERRUN_ERROR_FLAG 0x00010000
#define TSNEP_TX_DESC_DMA_DELAY_FIRST_DATA_MASK 0x0000FFFC
#define TSNEP_TX_DESC_DMA_DELAY_FIRST_DATA_SHIFT 2
#define TSNEP_TX_DESC_DMA_DELAY_LAST_DATA_MASK 0xFFFC0000
#define TSNEP_TX_DESC_DMA_DELAY_LAST_DATA_SHIFT 18
#define TSNEP_TX_DESC_DMA_DELAY_NS 64

/* tsnep RX descriptor */
struct tsnep_rx_desc {
	__le32 properties;
	__le32 reserved[3];
	__le64 next;
	__le64 rx;
};

#define TSNEP_RX_DESC_BUFFER_SIZE_MASK 0x00003FFC

/* tsnep RX descriptor writeback */
struct tsnep_rx_desc_wb {
	__le32 properties;
	__le32 reserved[7];
};

/* tsnep RX inline meta */
struct tsnep_rx_inline {
	__le64 counter;
	__le64 timestamp;
};

#define TSNEP_RX_INLINE_METADATA_SIZE (sizeof(struct tsnep_rx_inline))

#endif /* _TSNEP_HW_H */
